#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>
#include <sys/param.h>
#include <dirent.h>
#define SYSCONFIG
#include "../include/sysconfig.h"
#undef SYSCONFIG

int testmode = 0;                                     /* Run in test mode? */
int verbose = 0;                                  /* Should we be verbose? */
char *progname;                       /* What is the name of this program? */


struct killpath
{
   char   pathspec[ MAXPATHLEN ];
   long   hours;
   char   action[ MAXPATHLEN ];
   struct killpath *next;
   struct killpath *prev;
};

struct killpath *first = NULL;
struct killpath *final = NULL;
 
               


/***************************************************************************
	TRIM
***************************************************************************/

char *
trim( char *str )
{
	char *ibuf;
	char *obuf;

   if ( str )
   {
   	for ( ibuf = obuf = str; *ibuf; )
      {
        	while ( *ibuf && ( isspace ( *ibuf ) ) )
         	ibuf++;
        	if ( *ibuf && ( obuf != str ) )
            *( obuf++ ) = ' ';
        	while ( *ibuf && ( !isspace ( *ibuf ) ) )
            *( obuf++ ) = *( ibuf++ );
      }
      *obuf = '\0';
   }
   return( str );
}
/*-- end of trim() -------------------------------------------------------*/



/****************************************************************************
    DELTREE                                                 07/24/96
****************************************************************************/

void
deltree( const char *dirname )
{
	DIR    *dirf;
	struct stat st;
	struct dirent *dr;
   char   curpath[ MAXPATHLEN ];

	dirf = opendir( dirname );
   while ( 1 )
   {
	  	dr = readdir( dirf );
		if ( !dr )
			break;
   
      if ( !strcmp( dr->d_name, "." ) || !strcmp( dr->d_name, ".." ) )
         continue;

      sprintf( curpath, "%s/%s", dirname, dr->d_name );

		if ( stat( curpath, &st ) < 0 )
         continue;                                 /* Could not stat? */

		if ( S_ISDIR( st.st_mode ) )
         deltree( curpath );
      else						 
			unlink( curpath );
   }
  	(void)closedir( dirf ); 
	
	rmdir( dirname );

   return;
}
/*--- end of deltree() ----------------------------------------------------*/



/****************************************************************************
	KILLPATH_STORE
	Adds a pathspec to the killpath structure.
****************************************************************************/													 
												  
void 
killpath_store( struct killpath *i, struct killpath **first, struct killpath **final )
{
	struct killpath *old = NULL;
	struct killpath *p;
						
	if ( *final == NULL )
	{
		/* 
		 * First element in list?
		 */
		i->next = NULL;
		i->prev = NULL;
		*final = i;
		*first = i;
		return;
	}
	
	/*
	 * Start at top of list
	 */
	p = *first;
	
	while ( p )
	{
		if ( strcmp( p->pathspec, i->pathspec ) < 0 )
		{
			old = p;
			p = p->next;
		}
		else
		{
			if ( p->prev )
			{
				p->prev->next = i;
				i->next = p;
				i->prev = p->prev;
				p->prev = i;
				return;
			}
			i->next = p;
			i->prev = NULL;
			p->prev = i;
			*first = i;
			return;
		}
	}
	
	old->next = i;
	i->next = NULL;
	i->prev = old;
	*final = i;
}
/*-- end of killpath_store() -----------------------------------------------*/	
                


/****************************************************************************
   FREE_KILLPATH_STRUCT
****************************************************************************/

void
free_killpath_struct( void )
{
 	struct killpath *ips;
							 
	ips = first;
	
	while ( ips )
	{          
		ips = ips->next;
      free( ips );
	}
   
   free( ips );
   
   first = final = NULL;
}
/*-- END of free_killpath_struct() ----------------------------------------*/

                               

/***************************************************************************
   LOAD_KILLPATHS                                
***************************************************************************/
                                                                            
void
load_killpaths( void )
{     
	char *comment;
   FILE *fp;
	char linebuf[ BUFSIZ ];
   char tempbuf[ 20 ];
   char tempname[ MAXPATHLEN+1 ];
   struct killpath *newpath;
	char 	 *tokptr;
   char linenum = 0;
                         
   /* Clear out the structure if one exists */
   free_killpath_struct();

   /* Open the file */
   sprintf( tempname, "%s/sitekill.data", cf.datapath );
   if ( ( fp = fopen( tempname, "r" ) ) == NULL )
   {
      fprintf( stderr, "%s: can't open '%s': %s\n", progname, tempname, strerror( errno ) );
      exit( EXIT_FAILURE );
   }

   /* Read /.ftp-data/sitekill.data */
   while ( 1 )
   {               
      linenum++;
      if ( fgets( linebuf, sizeof( linebuf ), fp ) == NULL )
      {
         fclose( fp );
         return;
      }

		/*
		 * Clip out comments
		 */
		if ( ( comment = strchr( linebuf, '#' ) ) != NULL )
			*comment = '\0';

		/*
		 * Trim the buffer 
		 */
		(void)trim( linebuf );

      if ( strlen( linebuf ) > 5 )
      {                   
         /* Tokenize string and move into killpath struct */
  	      newpath = (struct killpath *)malloc( sizeof( struct killpath ) );
         
      	if ( ( tokptr = strtok( linebuf, " " ) ) )
            strncpy( newpath->pathspec, tokptr, sizeof( newpath->pathspec ) );
         else
         {  
            fprintf( stderr, "%s: line %d: no pathspec\n", progname, linenum );
            exit( EXIT_FAILURE );
         }
               
         if ( ( tokptr = strtok( '\0', " " ) ) )
         {
            strncpy( tempbuf, tokptr, sizeof( tempbuf ) );
            newpath->hours = atol( tempbuf );
         }
         else
         {  
            fprintf( stderr, "%s: line %d: no time spec\n", progname, linenum );
            exit( EXIT_FAILURE );
         }
            
         newpath->action[ 0 ] = '\0';
         while ( ( tokptr = strtok( '\0', " " ) ) )
         {
            strcat( newpath->action, tokptr );
            strcat( newpath->action, " " );
         }
         
         if ( newpath->action[ 0 ] == '\0' ) 
         {  
            fprintf( stderr, "%s: line %d: no action stated\n", progname, linenum );
            exit( EXIT_FAILURE );
         }
         
         if ( verbose )
         {
            fprintf( stdout, "newpath->pathspec=[%s]\n", newpath->pathspec );
            fprintf( stdout, "newpath->hours=[%ld]\n", newpath->hours );
            fprintf( stdout, "newpath->action=[%s]\n", newpath->action );
         }
            
        	killpath_store( newpath, &first, &final );
      }
   }
}
/*-- end of load_killpaths() ----------------------------------------------*/


                                              
/***************************************************************************
	CLEAR_CONFIG_STRUCT
***************************************************************************/
												
void
clear_config_struct( void )
{						 
	/* Set default configuration values */
	strncpy( cf.datapath, "/.ftp-data", sizeof( cf.datapath ) );
	cf.max_users = 15;
	cf.tag_file[ 0 ] = '\0';
	cf.welcome_msg[ 0 ] = '\0';
	cf.show_diz = 1;
	cf.default_ratio = 6;
	cf.free_ratio_amount = (ulong)51200000L;
	cf.dupe_check_days = 7;
	strncpy( cf.default_restrict, "/tmp", sizeof( cf.default_restrict ) );
	strncpy( cf.default_tagline, "I love bleach-ftpd!", sizeof( cf.default_restrict ) );
	cf.default_num_logins = 2;
	cf.default_level = 1;
	cf.caps_first_letter = 0;
	strncpy( cf.email, "[UNKNOWN]", sizeof( cf.email ) );
	cf.banner[ 0 ] = '\0';
	strncpy( cf.sitename, "UNKNOWN", sizeof( cf.email ) );
	strncpy( cf.dividerline, "--=-------------------------------------------------------------------=--", sizeof( cf.dividerline ) );
   cf.show_newsfile = 1;
}
/*-- end of clear_cf_struct() ----------------------------------------*/


/***************************************************************************
	SYSCONFIG_PARSEVAL
	Parses values in "sysconfig" file.
***************************************************************************/

void 
sysconfig_parseval( char *lvalue, char *rvalue )
{
   /* Here we parse all values */
   if ( strcasecmp( lvalue, "datapath" ) == 0 )
   {
		strncpy( cf.datapath, rvalue, sizeof( cf.datapath ) );
      return;
   }
   
   if ( strcasecmp( lvalue, "max_users" ) == 0 )
   {												 
		cf.max_users = atoi( rvalue );
      return;
   }

   if ( strcasecmp( lvalue, "tag_file" ) == 0 )
   {
		strncpy( cf.tag_file, rvalue, sizeof( cf.tag_file ) );
      return;
   }

   if ( strcasecmp( lvalue, "welcome_msg" ) == 0 )
   {
		strncpy( cf.welcome_msg, rvalue, sizeof( cf.welcome_msg ) );
      return;
   }
	
   if ( strcasecmp( lvalue, "show_diz" ) == 0 )
   {											 
		cf.show_diz = atoi( rvalue );
		if ( cf.show_diz != 0 && cf.show_diz != 1 )
			cf.show_diz = 1;  			/* Default */
      return;
   }	  
	
   if ( strcasecmp( lvalue, "default_ratio" ) == 0 )
   {											 
		cf.default_ratio = atoi( rvalue );
      return;
   }										 
	
   if ( strcasecmp( lvalue, "free_ratio_amount" ) == 0 )
   {											 
      cf.free_ratio_amount = strtoul( rvalue, (char **)NULL, 0 );
      return;
   }										 		 
	
   if ( strcasecmp( lvalue, "dupe_check_days" ) == 0 )
   {											 
      cf.dupe_check_days = atoi( rvalue );
      return;
   }
	
   if ( strcasecmp( lvalue, "default_restrict" ) == 0 )
   {								  
		strncpy( cf.default_restrict, rvalue, sizeof( cf.default_restrict ) );
      return;
   }										 
	
   if ( strcasecmp( lvalue, "default_tagline" ) == 0 )
   {											 
		strncpy( cf.default_tagline, rvalue, sizeof( cf.default_tagline ) );
      return;
   }			 
	
   if ( strcasecmp( lvalue, "default_num_logins" ) == 0 )
   {											 
		cf.default_num_logins = atoi( rvalue );
      return;
   }									 
	
   if ( strcasecmp( lvalue, "default_level" ) == 0 )
   {											 
		cf.default_level = atoi( rvalue );
      return;
   }
	
   if ( strcasecmp( lvalue, "caps_first_letter" ) == 0 )
   {											   
		cf.caps_first_letter = atoi( rvalue );
      return;
   }

   if ( strcasecmp( lvalue, "email" ) == 0 )
   {											 
		strncpy( cf.email, rvalue, sizeof( cf.email ) );
      return;
   }

   if ( strcasecmp( lvalue, "banner" ) == 0 )
   {								  
		strncpy( cf.banner, rvalue, sizeof( cf.banner ) );
      return;
   }										
	
   if ( strcasecmp( lvalue, "sitename" ) == 0 )
   {								  
		strncpy( cf.sitename, rvalue, sizeof( cf.sitename ) );
      return;
   }
   
   if ( strcasecmp( lvalue, "dividerline" ) == 0 )
   {
		strncpy( cf.dividerline, rvalue, sizeof( cf.dividerline ) );
      return;
   }
   
   if ( strcasecmp( lvalue, "show_newsfile" ) == 0 )
   {											 
		cf.show_newsfile = atoi( rvalue );
      return;
   }										 
}
/*-- end of syscf_parseval() ------------------------------------------*/



/***************************************************************************
	LOAD_SYSCONFIG
	Loads data from system configuration file.
***************************************************************************/

void 
load_sysconfig( void )
{
   FILE *configfile;
   char lvalue[ 64 ];
   char rvalue[ MAXPATHLEN ];
   int  x, y;
	char work_buff[ MAXPATHLEN ];

   clear_config_struct();

   sprintf( work_buff, "/etc/phear.conf" );

   if ( ( configfile = fopen( work_buff, "r" ) ) == NULL )
	{
      fprintf( stderr, "%s: invalid or missing '/etc/pheard.conf' -- using defaults\n", progname );
      return;
	}

   while ( 1 )
   {
      if ( fgets( work_buff, sizeof( work_buff ), configfile ) == NULL )
      {
         fclose( configfile );
         return;
      }

      /* Clip out comments */
      for ( x = 0; x < strlen( work_buff ); x++ )
         if ( work_buff[ x ] == '#' )
            work_buff[ x ] = '\0';
				
		/* Trim */
		(void)trim( work_buff );
      
      /* Clear out old values */
      memset( lvalue, '\0', sizeof( lvalue ) );
      memset( rvalue, '\0', sizeof( rvalue ) );
													 
		/* The file allows spaces embedded, but clears leading & trailing 
			spaces */
		
      /* Parse lvalue */
      y = 0;
      for ( x = 0; x < strlen( work_buff ) && work_buff[ x ] != ' '; x++ )
         if ( isprint( work_buff[ x ] ) )
            lvalue[ y++ ] = work_buff[ x ];

      /* Parse rvalue */
      y = 0; x++;
      for ( ; x < strlen( work_buff ); x++ )
         if ( isprint( work_buff[ x ] ) )
            rvalue[ y++ ] = work_buff[ x ];

      sysconfig_parseval( lvalue, rvalue );
   }

   return;
}
/*-- end of load_sysconfig() ----------------------------------------------*/



/***************************************************************************
	SCAN_AND_KILL
***************************************************************************/

void
scan_and_kill( char *dirname, long killhours, char *action )
{
	DIR    *dirf;
	struct dirent *dn;
	struct stat st;
   char   fullpath[ MAXPATHLEN ];
   time_t timenow = time( NULL );
   time_t killtime = time( NULL );
   char   actionbuf[ MAXPATHLEN ];
   int    x, y;

   killtime -= (time_t)( (time_t)killhours * (time_t)3600 );

   if ( verbose || testmode )
   {                    
      fprintf( stdout, "\n--= Scanning directory '%s'.\n", dirname );
      fprintf( stdout, "--= You told me to kill if the dir is %ld hours old.\n", killhours );
      fprintf( stdout, "--= It is now %.24s.\n", ctime( &timenow ) );
      fprintf( stdout, "--= Killing < %.24s.\n", ctime( &killtime ) );
      fprintf( stdout, "--= The action line is \"%s\".\n\n", action );
   }

	dirf = opendir( dirname );
	if ( !dirf )
	{
		fprintf( stderr, "%s: error reading dir '%s': %s", progname, dirname, strerror( errno ) );
		return;
	}
	
	/*
	 * Get first filename containing ZIP
	 */

   while ( 1 )
   {
		dn = readdir( dirf );
		if ( !dn )
			break;
			      
      /* We ignore "." and ".." */         
      if ( !strcmp( dn->d_name, "." ) || !strcmp( dn->d_name, ".." ) )
         continue;
         
      if ( dirname[ strlen( dirname ) - 1 ] != '/' )
         sprintf( fullpath, "%s/%s", dirname, dn->d_name );
      else
         sprintf( fullpath, "%s%s", dirname, dn->d_name );
         
		if ( stat( fullpath, &st ) < 0 )
         fprintf( stderr, "%s: '%s': stat failed: %s\n", progname, dn->d_name,
            strerror( errno ) );
      
		if ( S_ISDIR( st.st_mode ) )
		{
         if ( st.st_mtime < killtime )
         {   
            memset( actionbuf, '\0', sizeof( actionbuf ) );                    
            for ( x = 0, y = 0; x < strlen( action ); x++ )
            {
               if ( action[ x ] == '@' )
               {
                  strcat( actionbuf, "'" );
                  strcat( actionbuf, fullpath );
                  strcat( actionbuf, "'" );
                  y += strlen( fullpath ) + 2;
               }
               else  
                  actionbuf[ y++ ] = action[ x ];
            }
            trim( actionbuf );
         
            if ( verbose || testmode )
               fprintf( stdout, "%8d hours old: \"%s\"\n", 
                  (int)( (time_t)(timenow - st.st_mtime) / (time_t)3600 ), actionbuf );

            /* Remove the directory via a system call */
            if ( !testmode )
            {
					if ( strncasecmp( actionbuf, "rm", 2 ) == 0 )
						deltree( fullpath );
					else
	               system( actionbuf );
               wait( 0 );
            }
         }
		}
   }
	
  	(void)closedir( dirf );
   
   return;
}
/*-- end of scan_and_kill() -----------------------------------------------*/

                 
/****************************************************************************
   USAGE
****************************************************************************/

void
usage( void )
{
   fprintf( stdout, "USAGE: %s [-t -v]\n", progname );
   fprintf( stdout, "\t-t\tTest mode (no directory kills).\n" );
   fprintf( stdout, "\t-v\tVerbose mode.\n" );
   exit( EXIT_SUCCESS );
}
/*-- end of usage() -------------------------------------------------------*/
        


/****************************************************************************
   MAIN
****************************************************************************/
                                   
int
main( int argc, char **argv )
{  
 	struct killpath *ips;
   int    x;

   if ( strrchr( argv[ 0 ], '/' ) != NULL )
      progname = strrchr( argv[ 0 ], '/' ) + 1;
   else
      progname = argv[ 0 ];
      
   for ( x = 1; x < argc; x++ )
   {
      if ( toupper( argv[ x ][ 1 ] ) == 'T' )
         testmode = 1;
      else if ( toupper( argv[ x ][ 1 ] ) == 'V' )
         verbose = 1;
      else
         usage();
   }

   load_sysconfig();
   load_killpaths();

	ips = first;
	while ( ips )
	{          
      scan_and_kill( ips->pathspec, ips->hours, ips->action );
		ips = ips->next;
	}
 
   return EXIT_SUCCESS;
}
/*-- end of main() --------------------------------------------------------*/

