#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <syslog.h>
#ifdef __linux__
#include <linux/unistd.h>
#endif
#include <sys/file.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <dirent.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>

char datapath[ 1024+1 ];
char searchgroup[20];
int  num_users;

struct UF
{
  char   name[ 24 ];         /* Username */
  int    ratio;              /* Ratio multiplier value */
  char   tagline[ 64 ];      /* The user's tagline */
  int    files_up;           /* Total # files upped */
  ulong  bytes_up;           /* Total # bytes upped */
  int    files_down;         /* Total # files downed */
  ulong  bytes_down;         /* Total # bytes downed */
  char   group_user[ 10 ][ 15 ];  /* Groups this user belongs to */
  char   group_priv[ 5 ][ 15 ];  /* Groups this user belongs to */
};

struct UF uf[ 400 ];

/***************************************************************************
	TRIM
***************************************************************************/

char *
trim( char *str )
{
	char *ibuf;
	char *obuf;
	
   if ( str )
   {
   	for ( ibuf = obuf = str; *ibuf; )
      {
        	while ( *ibuf && ( isspace ( *ibuf ) ) )
         	ibuf++;
        	if ( *ibuf && ( obuf != str ) )
            *( obuf++ ) = ' ';
        	while ( *ibuf && ( !isspace ( *ibuf ) ) )
            *( obuf++ ) = *( ibuf++ );
      }
      *obuf = '\0';
   }
   return( str );
}
/*-- end of trim() -------------------------------------------------------*/
		  


/***************************************************************************
	ISINGROUP
***************************************************************************/
										  
int
isingroup( int which, char *whichgrp )
{
	int ct;
	
	for ( ct = 0; ct < 10 && uf[which].group_user[ct][0] != '\0'; ct++ )
	{
		if ( !strcasecmp( whichgrp, uf[which].group_user[ct] ) )
			return 1;
	}
			
	for ( ct = 0; ct < 5 && uf[which].group_priv[ct][0] != '\0'; ct++ )
	{
		if ( !strcasecmp( whichgrp, uf[which].group_priv[ct] ) )
			return 1;
	}
			
	return 0;
}
/*-- end of isingroup() --------------------------------------------------*/


/***************************************************************************
	LOAD_SYSCONFIG
	Loads data from system configuration file.
***************************************************************************/

void 
load_sysconfig( void )
{
   FILE *configfile;
   char lvalue[ 64 ];
   char rvalue[ 1024 ];
   int  x, y;
	char work_buff[ 1024 ];
		
   sprintf( work_buff, "/etc/phear.conf" );

   if ( ( configfile = fopen( work_buff, "r" ) ) == NULL )
	{
		fprintf( stderr, "bad or missing '/etc/phear.conf', using defaults" );
      return;
	}

   while ( 1 )
   {
      if ( fgets( work_buff, sizeof( work_buff ), configfile ) == NULL )
      {
         fclose( configfile );
         return;
      }

      /* Clip out comments */
      for ( x = 0; x < strlen( work_buff ); x++ )
         if ( work_buff[ x ] == '#' )
            work_buff[ x ] = '\0';
				
		/* Trim */
		(void)trim( work_buff );
      
      /* Clear out old values */
      memset( lvalue, '\0', sizeof( lvalue ) );
      memset( rvalue, '\0', sizeof( rvalue ) );
													 
		/* The file allows spaces embedded, but clears leading & trailing 
			spaces */
		
      /* Parse lvalue */
      y = 0;
      for ( x = 0; x < strlen( work_buff ) && work_buff[ x ] != ' '; x++ )
         if ( isprint( work_buff[ x ] ) )
            lvalue[ y++ ] = work_buff[ x ];

      /* Parse rvalue */
      y = 0; x++;
      for ( ; x < strlen( work_buff ); x++ )
         if ( isprint( work_buff[ x ] ) )
            rvalue[ y++ ] = work_buff[ x ];
      
      if ( strcasecmp( lvalue, "datapath" ) == 0 )
	   	strncpy( datapath, rvalue, sizeof( datapath ) );
   }

   return;
}
/*-- end of load_sysconfig() ----------------------------------------------*/



/***************************************************************************
  CLEAR_USERS
***************************************************************************/

void 
clear_users( void )
{			
	int x;
  	int ct;
					
	for ( x = 0; x < 400; x++ )
	{							  
	  	bzero( uf[x].name, sizeof( uf[x].name ) );
	  	bzero( uf[x].tagline, sizeof( uf[x].tagline ) );
		uf[x].ratio = 0;
	   uf[x].files_up = 0;
   	uf[x].bytes_up = 0L;
   	uf[x].files_down = 0;
	   uf[x].bytes_down = 0L;
	
	   for ( ct = 0; ct < 10; ct++ )
   	   bzero( uf[x].group_user[ ct ], sizeof( uf[x].group_user[ct]) );
					  
	   for ( ct = 0; ct < 5; ct++ )
   	   bzero( uf[x].group_priv[ ct ], sizeof( uf[x].group_priv[ct]) );
	}				
	
	num_users = 0;
		
   return;
}
/*-- end of clear_users() ------------------------------------------------*/






/***************************************************************************
	QS_LIST_UP
***************************************************************************/

void 
qs_list_up( struct UF item[], int left, int right )
{
   register int i;
   register int j;
   size_t x;
   struct UF temp;
	
   i = left;
   j = right;
   x = item[ ( left + right ) / 2 ].bytes_up;

   do
   {
      while ( item[ i ].bytes_up > x && i < right )
         i++;
      while ( item[ j ].bytes_up < x && j > left )
         j--;
      if ( i <= j )
      {
         temp = item[ i ];
         item[ i ] = item[ j ];
         item[ j ] = temp;
         i++;
         j--;
      }
   } while ( i <= j );

   if ( left < j )
      qs_list_up( item, left, j );

   if ( i < right )
      qs_list_up( item, i, right );
}
/*-- end of qs_list_up() -------------------------------------------------*/



/***************************************************************************
	PARSEVAL
	Parses values in user datafile.
***************************************************************************/

void 
userfile_parseval( int x, char *lvalue, char *rvalue )
{
	int ct;									 

   if ( strcasecmp( lvalue, "ratio" ) == 0 )
   {
      uf[x].ratio = atoi( rvalue );
      return;
   }  

   if ( strcasecmp( lvalue, "tagline" ) == 0 )
   {
      strncpy( uf[x].tagline, rvalue, sizeof( uf[x].tagline ) );
      return;
   }

   if ( strcasecmp( lvalue, "files_up" ) == 0 )
   {
      uf[x].files_up = atoi( rvalue );
      return;
   }

   if ( strcasecmp( lvalue, "bytes_up" ) == 0 )
   {
      uf[x].bytes_up = strtoul( rvalue, (char **)NULL, 0 );
      return;
   }
	
   if ( strcasecmp( lvalue, "files_down" ) == 0 )
   {
      uf[x].files_down = atoi( rvalue );
      return;
   }

   if ( strcasecmp( lvalue, "bytes_down" ) == 0 )
   {
      uf[x].bytes_down = strtoul( rvalue, (char **)NULL, 0 );
      return;
   }

   if ( strcasecmp( lvalue, "group_user" ) == 0 )
	{
		for ( ct = 0; ct < 10; ct++ )
			if ( uf[x].group_user[ ct ][ 0 ] == '\0' )
			{
				strncpy( uf[x].group_user[ ct ], rvalue, sizeof( uf[x].group_user[ ct ] ) );
            trim( uf[x].group_user[ ct ] ); 
				return;
			}
	}
	 
   if ( strcasecmp( lvalue, "group_priv" ) == 0 )
	{
		for ( ct = 0; ct < 5; ct++ )
			if ( uf[x].group_priv[ ct ][ 0 ] == '\0' )
			{
				strncpy( uf[x].group_priv[ ct ], rvalue, sizeof( uf[x].group_priv[ ct ] ) );
            trim( uf[x].group_priv[ ct ] ); 
				return;
			}
	}
}  
/*-- end of userfile_parseval() -------------------------------------------*/



/***************************************************************************
	LOAD_USERFILE
	Loads user data for user "username".
***************************************************************************/

void 
load_userfile( int which, char *username )
{
	char *comment;
   FILE *userfile;
   char lvalue[ 64 ];
   char rvalue[ BUFSIZ ];
   int  x, y;
	char linebuf[ BUFSIZ ];
	char fname_buff[ 1024+1 ];
	
	/*
	 * Set the uf.name to the user's name
	 */
   strncpy( uf[which].name, username, sizeof( uf[which].name ) ) ;
									  
	/*
	 * Open the userfile
	 */		 
   sprintf( fname_buff, "%s/users/%s", datapath, username );
   if ( ( userfile = fopen( fname_buff, "r" ) ) == NULL )
	{
		/* We need code to read "default" user */
		strcpy( uf[which].name, "!!ERR!!" );
      return;
	}

	/*
	 * Read lines one at a time, parse lvalue & rvalue, and fill in struct
	 */
   while ( 1 )
   {
      if ( fgets( linebuf, sizeof( linebuf ), userfile ) == NULL )
      {				 
         fclose( userfile );
         return;
      }

		/*
		 * Clip out comments, they'll eventually be lost anyway :)
		 */
		if ( ( comment = strchr( linebuf, '#' ) ) != NULL )
			*comment = '\0';

		/*
		 * Trim the buffer 
		 */					
		(void)trim( linebuf ); 

		/* 
		 * Make sure our lvalue & rvalue are empty
		 */
      bzero( lvalue, sizeof( lvalue ) );
      bzero( rvalue, sizeof( rvalue ) );

		/*
		 * Parse lvalue 
		 */										 
      y = 0;
      for ( x = 0; x < strlen( linebuf ) && linebuf[ x ] != ' '; x++ )
         if ( isprint( linebuf[ x ] ) )
            lvalue[ y++ ] = linebuf[ x ];

		/*
		 * Parse rvalue
		 */				
      y = 0; x++;
      for ( ; x < strlen( linebuf ); x++ )
         if ( isprint( linebuf[ x ] ) )
            rvalue[ y++ ] = linebuf[ x ];

		/*
		 * Call 'parseval' to load structure based on lvalue & rvalue
		 */					
      userfile_parseval( which, lvalue, rvalue );
   }

   return;
}
/*-- end of load_userfile() ----------------------------------------------*/



/***************************************************************************
	BUILD_INFO
***************************************************************************/

int 
build_info( void )
{
	int ct,x;
	DIR    *dirf;
	struct dirent *dn;
	struct stat st;
	char   temppath[ 1024 + 1 ];
	
   sprintf( temppath, "%s/users", datapath );
	dirf = opendir( temppath );
	if ( !dirf )
		return 1;

   while ( 1 )
   {
		dn = readdir( dirf );
		if ( !dn )
			break;		  
			
		sprintf( temppath, "%s/users/%s", datapath, dn->d_name );
				
		stat( temppath, &st );
		
		if ( temppath[ strlen( temppath ) ] == '~' )
			continue;

	   for ( ct = 0; ct < 10; ct++ )
   	   bzero( uf[x].group_user[ ct ], sizeof( uf[x].group_user[ct]) );
	   for ( ct = 0; ct < 5; ct++ )
   	   bzero( uf[x].group_priv[ ct ], sizeof( uf[x].group_priv[ct]) );

      if ( S_ISREG (st.st_mode) && st.st_uid != 99 && st.st_gid != 99 )
		{
			load_userfile( num_users, dn->d_name );
			if ( isingroup( num_users, searchgroup ) )
			{
				num_users++;
			}
		}
   }

  	(void)closedir( dirf );	

   return 0;
}
/*-- end of build_info() -------------------------------------------------*/

			



/***************************************************************************
	OUTPUT_LIST
***************************************************************************/

void
output_list( void )
{							  
  	char temppath[ 1024 + 1 ];
  	DIR *dirf;
	int ct;
  	struct dirent *dn;
  	struct stat st;
  	char sendstr[80] = "\0";
  	uid_t oldid = geteuid();
	char whichgrp[ BUFSIZ ];
	char tempbuf[ BUFSIZ ];
  	char username[1024 + 1];
	char currentuser[ 40 ];
	int numusers = 0;
	int leeches = 0;
	char *dotptr;
	ulong total_bytes_dn = 0UL;
	ulong total_bytes_up = 0UL;
	int   total_files_up = 0;
	int   total_files_dn = 0;
	char  dnbuf[ 8 ];
	char  upbuf[ 8 ];
	char  numusersbuf[ 8 ];
	char  leechesbuf[ 8 ];
						 
	fprintf( stdout, "                                            ________\n" );
	fprintf( stdout, "_____________.______________________________\\     _/___________\n" );
	fprintf( stdout, "\\      _     |     _      /    _       /     \\     \\      _    \\\n" );
	fprintf( stdout, " \\     \\     |     /_____/     \\      /      |\\     \\     \\    /     XFER\n" );
	fprintf( stdout, "  \\______    |______|    \\___________/|______________\\    ____/   STATISTICS\n" );
	fprintf( stdout, ".---- /______| ------------------------------------ |_____| -----------------.\n" );
	fprintf( stdout, "| Username |   Up |   Megs |   Dn |   Megs | Ratio | Tagline                 |\n" );
	fprintf( stdout, "|----------+------+--------+------+--------+-------+-------------------------|\n" );
								
	for ( ct = 0; ct < num_users; ct++ )
   {
		if ( uf[ct].ratio )
	   	sprintf( tempbuf, " 1:%1d ", uf[ct].ratio );
		else
			strcpy( tempbuf, "leech" );
			
		numusers++;
		if ( !uf[ct].ratio )
			leeches++;

		fprintf( stdout, "| %-8.8s | %4d | %6.1f | %4d | %6.1f | %-5.5s | %-23.23s |\n",
					uf[ct].name, uf[ct].files_up, uf[ct].bytes_up / 1024.0,
					uf[ct].files_down, uf[ct].bytes_down / 1024.0, tempbuf, uf[ct].tagline );
					
		total_bytes_dn += uf[ct].bytes_down;
		total_bytes_up += uf[ct].bytes_up;
		total_files_dn += uf[ct].files_down;
		total_files_up += uf[ct].files_up;
	}
	
	sprintf( dnbuf, "%d", total_files_dn );
	sprintf( upbuf, "%d", total_files_up );
	sprintf( numusersbuf, "%d", numusers );
	sprintf( leechesbuf, "%d", leeches );
	
	fprintf( stdout, "|----------+------+--------+------+--------+-------+-------------------------|\n" );
	fprintf( stdout, "|  Total Megs Up: %9.2f   Total Files Up: %-6.6s     Users in group: %-3.3s |\n",
		total_bytes_up / 1024.0, upbuf, numusersbuf );
	fprintf( stdout, "|  Total Megs Dn: %9.2f   Total Files Dn: %-6.6s   Users with leech: %-3.3s |\n",
		total_bytes_dn / 1024.0, dnbuf, leechesbuf );
	fprintf( stdout, "`----------------------------------------------------------------------------'\n" );

	return;
}
/*-- end of output_list() ------------------------------------------------*/


/***************************************************************************
	MAIN
***************************************************************************/

int 
main( int argc, char **argv )
{     
   if ( argc < 2 )
   {				 
     fprintf( stdout, "USAGE: groupnfo [groupname]\n" );
     return( EXIT_SUCCESS );
   }
	
   strcpy( searchgroup, argv[1] );

   load_sysconfig();
   clear_users();
   build_info();
   qs_list_up( uf, 0, num_users - 1 );
   output_list();
	
   return 0;   
}
/*-- end of main() -------------------------------------------------------*/
