#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <unistd.h>
#include <syslog.h>
#ifdef __linux__
#include <linux/unistd.h>
#endif
#include <sys/file.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/param.h>
#include "../include/sysconfig.h"
#include "../include/userfile.h"

struct gpath
{				 
   char pathspec[ BUFSIZ ];
   struct gpath *next;
   struct gpath *prev;
};

static struct gpath *first = NULL;
static struct gpath *last = NULL;

extern uid_t userid; 
extern char *trim( char *str );
extern int isingroup( char *whichgrp );
   
 
/***************************************************************************
   CHECK_GPATH
   If "checkpath" is in the '<datapath>/gpath' file, and the user is not
	in the group specified by "gpath", the function returns 0, else the
	function returns 1.  i.e. 0=deny 1=allow
***************************************************************************/
                                  
int
check_gpath( char *checkpath )
{
  	int  x, y;
  	char lvalue[ BUFSIZ ];
  	char rvalue[ BUFSIZ ];
  	struct gpath *npath;

#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] show_gpath()" );
	debugfn();
#endif
  
  	npath = first;
   
  	while ( npath )
   {
      bzero( lvalue, sizeof( lvalue ) );
      bzero( rvalue, sizeof( rvalue ) );
      
      y = 0;
      for ( x = 0; x < strlen( npath->pathspec ) 
	      	&& npath->pathspec[ x ] != ' '; x++ )
			if ( isprint( npath->pathspec[ x ] ) )
	  			lvalue[ y++ ] = npath->pathspec[ x ];

      y = 0; x++;
      for ( ; x < strlen( npath->pathspec ); x++ )
			if ( isprint( npath->pathspec[ x ] ) )
	  			rvalue[ y++ ] = npath->pathspec[ x ];

      if ( !memcmp( checkpath, lvalue, strlen( lvalue ) ) )
		{										 
			if ( isingroup( rvalue ) )
				return 1;
			else
				return 0;
		}

      npath = npath->next;
   }
	
	return 1;
}
/*-- end of show_gpath() -----------------------------------------------*/
             



/****************************************************************************
	MSGPATH_STORE
	Adds a pathspec to the gpath structure.
****************************************************************************/

void 
gpath_store( struct gpath *i, struct gpath **start, struct gpath **finish )
{
  struct gpath *old = NULL;
  struct gpath *p;
  													
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] gpath_store()" );
	debugfn();
#endif
  
  if ( *finish == NULL )
    {
      /* 
       * First element in list?
       */
      i->next = NULL;
      i->prev = NULL;
      *finish = i;
      *start = i;
      return;
    }
  
  /*
   * Start at top of list
   */
  p = *start;
  
  while ( p )
    {
      if ( strcmp( p->pathspec, i->pathspec ) < 0 )
	{
	  old = p;
	  p = p->next;
	}
      else
	{
	  if ( p->prev )
	    {
	      p->prev->next = i;
	      i->next = p;
	      i->prev = p->prev;
	      p->prev = i;
	      return;
	    }
	  i->next = p;
	  i->prev = NULL;
	  p->prev = i;
	  *start = i;
	  return;
	}
    }
  
  old->next = i;
  i->next = NULL;
  i->prev = old;
  *finish = i;
}
/*-- end of gpath_store() -----------------------------------------------*/	
                


/****************************************************************************
   FREE_GPATH_STRUCT
****************************************************************************/

void
free_gpath_struct( void )
{
  struct gpath *ips;
  													 
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] free_gpath_struct()" );
	debugfn();
#endif
  
  ips = first;
  
  while ( ips )
    {          
      ips = ips->next;
      free( ips );
    }
  
  free( ips );
  
  first = last = NULL;
}
/*-- END of free_gpath_struct() ----------------------------------------*/

                               

/***************************************************************************
   LOAD_MSGPATHS
   Loads messagepaths from /.ftp-data/gpath
***************************************************************************/
                                                                            
void
load_gpaths( void )
{     
  char *comment;
  FILE *fp;
  char linebuf[ BUFSIZ ];
  char tempname[ MAXPATHLEN+1 ];
  struct gpath *newpath;
  uid_t oldid = geteuid();
  															 
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] load_gpaths()" );
	debugfn();
#endif
 
  /* Clear out the structure if one exists */
  free_gpath_struct();
  				  
  seteuid( 0 );
   /* Open the file */
   sprintf( tempname, "%s/gpath", cf.datapath );
   if ( ( fp = fopen( tempname, "r" ) ) == NULL )
	{
		seteuid( oldid );
      return;
	}

   /* Read /.ftp-data/gpath */
   while ( 1 )
   {
      if ( fgets( linebuf, sizeof( linebuf ), fp ) == NULL )
      {
         fclose( fp );
			seteuid( oldid );
         return;
      }

		/*
		 * Clip out comments
		 */
		if ( ( comment = strchr( linebuf, '#' ) ) != NULL )
			*comment = '\0';

		/*
		 * Trim the buffer 
		 */
		(void)trim( linebuf );

      if ( strlen( linebuf ) > 5 )
      {
  	      newpath = (struct gpath *)malloc( sizeof( struct gpath ) );
         if ( newpath == NULL )
         {
            syslog( LOG_ERR, "[pheard] Out of memory in load_gpaths()" );
         }
         else
         {
            strncpy( newpath->pathspec, linebuf, sizeof( newpath->pathspec ) );
           	gpath_store( newpath, &first, &last );
         }
      }
   }
	
	seteuid( oldid );
}
/*-- end of load_gpaths() ----------------------------------------------*/
 
