#define USERFILE 

#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <syslog.h>
#ifdef __linux__
#include <linux/unistd.h>
#endif
#include <sys/file.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <dirent.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>
#include "../include/userfile.h"
#include "../include/sysconfig.h"

#undef USERFILE

extern char	mystatus[ BUFSIZ ];	                   /* My current "status" */


char hideuser[25][25];					/* Users we don't list in listings */

int load_userfile( char *username );
void update_user( void );
char *trim( char *str );
time_t daystart( void );
time_t weekstart( void );
time_t monthstart( void );
int isuser( char *checkname );

extern void reply( int n, char *fmt, ... );
extern void lreply( int n, char *fmt, ... );
extern void send_msg( int msgcode, char *msgstr, int send_reply );
size_t commafmt( char *buf, int	bufsize, unsigned long N );
         
extern void sysoplog( char *fmt, ... );
								



/***************************************************************************
	LOAD_HIDEUSERS
	Loads hidden users from "<datapath>/hideusers"
***************************************************************************/

void 
load_hideusers( void )
{
	char *comment;
   FILE *hidefile;  
	char *tokptr;
   int  x;
	char linebuf[ BUFSIZ ];
	char fname_buff[ MAXPATHLEN+1 ];
  	uid_t oldid = geteuid();
	
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] load_hideusers()" );
	debugfn();
#endif
	
	/*
	 * Clear hideusers:
	 */				 
	for ( x = 0; x < 25; x++ )
		bzero( hideuser[ x ], sizeof( hideuser[ x ] ) ); 
				  
	/*
	 * Open the hideusers file
	 */		 
	seteuid( 0 );
   sprintf( fname_buff, "%s/hideusers", cf.datapath );
   if ( ( hidefile = fopen( fname_buff, "r" ) ) == NULL )
	{									
		syslog( LOG_ERR, "[B/B] error reading hidefile: %s: %m", fname_buff );
		seteuid( oldid );
      return;
	}
									
	/*
	 * Read lines one at a time and fill in struct
	 */
   while ( 1 )
   {
      if ( fgets( linebuf, sizeof( linebuf ), hidefile ) == NULL )
      {				 
         fclose( hidefile );
			seteuid( oldid );
         return;
      }
				 						  
		/*
		 * Clip out comments, they'll eventually be lost anyway :)
		 */
		if ( ( comment = strchr( linebuf, '#' ) ) != NULL )
			*comment = '\0';
										
		/*
		 * Trim the buffer 
		 */					
		(void)trim( linebuf ); 
											 
		if ( !isalpha( linebuf[ 0 ] ) )
			continue;
					
		for ( x = 0; x < 25 && hideuser[x][0] != '\0'; x++ )
			;
						
		strncpy( hideuser[x], linebuf, sizeof( hideuser[x] ) );
/*		syslog( LOG_INFO, "Hideuser [%d] is [%s]", x, linebuf ); */
   }
									
	seteuid( oldid );
   return;
}
/*-- end of load_hideusers() ---------------------------------------------*/



/***************************************************************************
   ISUSER
	Is this user a valid user? (0=no, 1=yes)
***************************************************************************/

int
isuser( char *checkname )
{
	char userfile[ 45 ];
	struct stat st;
   uid_t oldid = geteuid();

#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] isuser()" );
	debugfn();
#endif

	/*
	 * No username will be this long...
	 */
	if ( strlen( checkname ) > 23 )
		return 0;								  
		
	seteuid( 0 );
		
   sprintf( userfile, "%s/users/%s", cf.datapath, checkname );
							  
	/*
	 * Every user will have a "userfile"..
	 */
	if ( !stat( userfile, &st ) )
	{		 
			seteuid( oldid );
			return 1;
	}
	
	seteuid( oldid );
				  
	return 0;		
}
/*-- end of isuser() ------------------------------------------------------*/




/***************************************************************************
   ISDELUSER
	Is this user a deleted user? (0=no, 1=yes)
***************************************************************************/

int
isdeluser( char *checkname )
{
	char userfile[ 45 ];
	struct stat st;
   uid_t oldid = geteuid();

#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] isdeluser()" );
	debugfn();
#endif

	/*
	 * No username will be this long...
	 */
	if ( strlen( checkname ) > 23 )
		return 0;								  
		
	seteuid( 0 );
		
   sprintf( userfile, "%s/users/%s", cf.datapath, checkname );
							  
	/*
	 * Every user will have a "userfile"..
	 */
	if ( !stat( userfile, &st ) )
	{		 
		if ( st.st_uid == 99 || st.st_gid == 99 )
		{
			seteuid( oldid );
			return 1;
		}
	}
	
	seteuid( oldid );
				  
	return 0;		
}
/*-- end of isdeluser() ---------------------------------------------------*/




/***************************************************************************
	MONTHSTART
***************************************************************************/

time_t
monthstart( void )
{
        struct tm *timebuf;                     /* "struct tm" time buffer */
        time_t    curtime;                      /* Current time */
        time_t   midnight;                      /* Time at midnight */
        int       offset;                               /* # of days since sunday */
        char      daybuf[ 4 ];          /* Day of week expressed as a string */
        char      *days[7] = { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };

#ifdef DEBUG
        syslog( LOG_INFO, "[pheard] monthstart()" );
        debugfn();
#endif

        time( &curtime );                                                       /* Get current time */

        timebuf = localtime( &curtime );                /* Fill tm struct */

	timebuf->tm_mday = 1;
        timebuf->tm_hour = 0;                                   /* Set tm struct to midnight */
        timebuf->tm_min = 0;
        timebuf->tm_sec = 0;

        midnight = mktime( timebuf );                   /* Get time_t from tm struct */

        sprintf( daybuf, "%.3s",
                ctime( &midnight ) );                           /* Get day of week as string */

        for ( offset = 0;
                        strcmp( days[ offset ], daybuf ) && offset < 7;
                        offset++ );

        return ( midnight - ( offset * 86400 ) );
}
/*-- end of monthstart() --------------------------------------------------*/

/***************************************************************************
	WEEKSTART
	Returns the time_t value for Midnight of this previous Sunday.
***************************************************************************/

time_t 
weekstart( void )
{
	struct tm *timebuf;			/* "struct tm" time buffer */
	time_t    curtime;			/* Current time */
	time_t 	 midnight;			/* Time at midnight */
	int       offset;				/* # of days since sunday */
	char      daybuf[ 4 ];		/* Day of week expressed as a string */
	char      *days[7] = { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };
												
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] weekstart()" );
	debugfn();
#endif
	
	time( &curtime );							/* Get current time */
  
	timebuf = localtime( &curtime );		/* Fill tm struct */
	
	timebuf->tm_hour = 0;					/* Set tm struct to midnight */
	timebuf->tm_min = 0;
	timebuf->tm_sec = 0;
	
	midnight = mktime( timebuf );			/* Get time_t from tm struct */
	
	sprintf( daybuf, "%.3s", 
		ctime( &midnight ) ); 				/* Get day of week as string */
													/* Get days since Sunday */
	for ( offset = 0; 
			strcmp( days[ offset ], daybuf ) && offset < 7; 
			offset++ );

	return ( midnight - ( offset * 86400 ) );
}
/*-- end of weekstart() --------------------------------------------------*/

/***************************************************************************
	DAYSTART
***************************************************************************/

time_t
daystart( void )
{
        struct tm *timebuf;                     /* "struct tm" time buffer */
        time_t    curtime;                      /* Current time */
        time_t   midnight;                      /* Time at midnight */
        int       offset;                               /* # of days since sunday */
        char      daybuf[ 4 ];          /* Day of week expressed as a string */
        char      *days[7] = { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };

#ifdef DEBUG
        syslog( LOG_INFO, "[pheard] daystart()" );
        debugfn();
#endif

        time( &curtime );                                                       /* Get current time */

        timebuf = localtime( &curtime );                /* Fill tm struct */

        timebuf->tm_hour = 0;                                   /* Set tm struct to midnight */
        timebuf->tm_min = 0;
        timebuf->tm_sec = 0;

        midnight = mktime( timebuf );                   /* Get time_t from tm struct */

        return ( midnight );
}
/*-- end of daystart() --------------------------------------------------*/


/***************************************************************************
	COMMAFMT
	Puts commas in an unsigned long.
	Code stolen from SNIPPETS.
***************************************************************************/

size_t 
commafmt( char   		*buf,			/* Buffer for formatted string  */
          int     	bufsize, 	/* Size of buffer               */
          unsigned 	long N ) 	/* Number to convert            */
{
	int 	len = 1;
	int 	posn = 1;
	int 	sign = 1;
   char 	*ptr = buf + bufsize - 1;

	if ( 2 > bufsize )
   {
ABORT: 
		*buf = '\0';
		return 0;
   }

   *ptr-- = '\0';
   --bufsize;
	
   if ( 0L > N )
   {
   	sign = -1;
      N = -N;
   }

   for ( ; len <= bufsize; ++len, ++posn )
   {
   	*ptr-- = (char)( ( N % 10L ) + '0' );
   	if ( 0L == ( N /= 10L ) )
      	break;
   	if ( 0 == ( posn % 3 ) )
   	{
      	*ptr-- = ',';
         ++len;
   	}
   	if ( len >= bufsize )
         goto ABORT;
   }

   if ( 0 > sign )
   {
   	if ( len >= bufsize )
      	goto ABORT;
      *ptr-- = '-';
      ++len;
   }

   memmove( buf, ++ptr, len + 1 );
   return (size_t)len;
}
/*-- end of commafmt() ---------------------------------------------------*/

									 
/***************************************************************************
	TRIM
***************************************************************************/

char *
trim( char *str )
{
	char *ibuf;
	char *obuf;

#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] trim()" );
	if ( uf.name[ 0 ] == '\0' )
		syslog( LOG_INFO, "[pheard] uf.name was NULL in the preceding function." );
#endif
												 
   if ( str )
   {
   	for ( ibuf = obuf = str; *ibuf; )
      {
        	while ( *ibuf && ( isspace ( *ibuf ) ) )
         	ibuf++;
        	if ( *ibuf && ( obuf != str ) )
            *( obuf++ ) = ' ';
        	while ( *ibuf && ( !isspace ( *ibuf ) ) )
            *( obuf++ ) = *( ibuf++ );
      }
      *obuf = '\0';
   }
   return( str );
}
/*-- end of trim() -------------------------------------------------------*/


/***************************************************************************
	UPDATE_USER
	Writes a user's info file back out.
***************************************************************************/

void 
update_user(void)
{
	int fdes = -1;		/* File descriptor */
	int ct;		/* Generic counter */
	char work_buff[MAXPATHLEN + 1];		/* Work buffer */
	uid_t oldid = geteuid();

#ifdef DEBUG
	syslog(LOG_INFO, "[pheard] update_user()");
	debugfn();
#endif


	/*
	 * (Attempt to) open the file
	 */
	seteuid(0);
	sprintf(work_buff, "%s/users/%s", cf.datapath, uf.name);
	fdes = open(work_buff, O_CREAT | O_WRONLY | O_TRUNC, 0666);
	if (fdes < 0) {
		syslog(LOG_ERR, "[pheard] %s: opening '%s': %m", uf.name, work_buff);
		seteuid(oldid);
		return;
	}

	/*
	 * Lock the file, keep trying until we are successful
	 */
	while (flock(fdes, LOCK_EX)) {
		syslog(LOG_ERR, "[pheard] %s: flocking '%s': %m", uf.name, work_buff);
		sleep(1);
	}

	/*
	 * Return to beginning of file
	 */
	lseek(fdes, 0, L_SET);

	/*
	 * General information fields:
	 */
	sprintf(work_buff, "# General information:\n");
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "ratio %d\n", uf.ratio);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "level %d\n", uf.level);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "restrict %s\n", uf.restrict);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "tagline %s\n", uf.tagline);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "num_logins %d\n", uf.num_logins);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "exempt_from_limit %d\n", uf.exempt_from_limit);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "credits %lu\n", uf.credits);
	write(fdes, work_buff, strlen(work_buff));
	
	/*
	 * Xfer information fields:
	 */
	sprintf(work_buff, "# Xfer information:\n");
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "files_up %d\n", uf.files_up);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "bytes_up %lu\n", uf.bytes_up);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "seconds_up %ld\n", uf.seconds_up);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "files_down %d\n", uf.files_down);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "bytes_down %lu\n", uf.bytes_down);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "seconds_down %ld\n", uf.seconds_down);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "files_up_wk %d\n", uf.files_up_wk);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "bytes_up_wk %lu\n", uf.bytes_up_wk);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "seconds_up_wk %ld\n", uf.seconds_up_wk);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "files_down_wk %d\n", uf.files_down_wk);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "bytes_down_wk %lu\n", uf.bytes_down_wk);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "seconds_down_wk %ld\n", uf.seconds_down_wk);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "files_up_day %d\n", uf.files_up_day);
        write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "bytes_up_day %lu\n", uf.bytes_up_day);
        write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "seconds_up_day %ld\n", uf.seconds_up_day);
	write(fdes, work_buff, strlen(work_buff));

        sprintf(work_buff, "files_down_day %d\n", uf.files_down_day);
        write(fdes, work_buff, strlen(work_buff));

        sprintf(work_buff, "bytes_down_day %lu\n", uf.bytes_down_day);
        write(fdes, work_buff, strlen(work_buff));

        sprintf(work_buff, "seconds_down_day %ld\n", uf.seconds_down_day);
        write(fdes, work_buff, strlen(work_buff));

        sprintf(work_buff, "files_up_month %d\n", uf.files_up_month);
        write(fdes, work_buff, strlen(work_buff));

        sprintf(work_buff, "bytes_up_month %lu\n", uf.bytes_up_month);
        write(fdes, work_buff, strlen(work_buff));

        sprintf(work_buff, "seconds_up_month %ld\n", uf.seconds_up_month);
        write(fdes, work_buff, strlen(work_buff));

        sprintf(work_buff, "files_down_month %d\n", uf.files_down_month);
        write(fdes, work_buff, strlen(work_buff));

        sprintf(work_buff, "bytes_down_month %lu\n", uf.bytes_down_month);
        write(fdes, work_buff, strlen(work_buff));

        sprintf(work_buff, "seconds_down_month %ld\n\n", uf.seconds_down_month);
        write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "# Time-related information:\n");
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "login_times %d\n", uf.login_times);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "last_on %ld\n", uf.last_on);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "time_limit %d\n", uf.time_limit);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "time_on_today %d\n", uf.time_on_today);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "last_nuked %ld\n", uf.last_nuked);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "use_color %d\n", uf.use_color);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "glock %d\n", uf.glock);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "gadmin %s\n\n", uf.gadmin);
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "# Groups for %s:\n", uf.name);
	write(fdes, work_buff, strlen(work_buff));

	/*
	 * GROUPS:
	 */
	for (ct = 0; ct < 10; ct++)
		if (uf.group_user[ct][0] != '\0') {
			sprintf(work_buff, "group_user %s\n", uf.group_user[ct]);
			write(fdes, work_buff, strlen(work_buff));
		}

	for (ct = 0; ct < 10; ct++)
		if (uf.group_priv[ct][0] != '\0') {
			sprintf(work_buff, "group_priv %s\n", uf.group_priv[ct]);
			write(fdes, work_buff, strlen(work_buff));
		}

	sprintf(work_buff, "\n");
	write(fdes, work_buff, strlen(work_buff));

	sprintf(work_buff, "# Approved IP's for %s:\n", uf.name);
	write(fdes, work_buff, strlen(work_buff));

	/*
	 * User IPs:
	 */
	for (ct = 0; ct < 20; ct++)
		if (uf.ip[ct][0] != '\0') {
			sprintf(work_buff, "ip %s\n", uf.ip[ct]);
			write(fdes, work_buff, strlen(work_buff));
		}

	/*
	 * Unlock & close
	 */
	flock(fdes, LOCK_UN);
	close(fdes);

	seteuid(oldid);
}
/*-- end of update_user() ------------------------------------------------*/



/***************************************************************************
  CLEAR_UF_STRUCT
  Clears uf struct, setting default values where applicable.
***************************************************************************/

void 
clear_uf_struct( void )
{
  int ct;
												  
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] clear_uf_struct()" );
	debugfn();
#endif
  
  bzero( uf.name, sizeof( uf.name ) );

   uf.ratio = cf.default_ratio;
	uf.level = cf.default_level;
   strncpy( uf.restrict, cf.default_restrict, sizeof( uf.restrict ) );
	strncpy( uf.tagline, cf.default_tagline, sizeof( uf.tagline ) );
	uf.num_logins = cf.default_num_logins;
	uf.exempt_from_limit = 0;
	uf.credits = cf.free_ratio_amount;
   uf.files_up = 0;
   uf.bytes_up = 0L;
   uf.seconds_up = (time_t)0;
	
   uf.files_down = 0;
   uf.bytes_down = 0L;
   uf.seconds_down = (time_t)0;
	
   uf.files_up_wk = 0;
   uf.bytes_up_wk = 0L;
   uf.seconds_up_wk = (time_t)0;
	           
   uf.files_down_wk = 0;
   uf.bytes_down_wk = 0L;
   uf.seconds_down_wk = (time_t)0;

   uf.files_up_day = 0;
   uf.bytes_up_day = 0L;
   uf.seconds_up_day = (time_t)0;

   uf.files_down_day = 0;
   uf.bytes_down_day = 0L;
   uf.seconds_down_day = (time_t)0;

   uf.files_up_month = 0;
   uf.bytes_up_month = 0L;
   uf.seconds_up_month = (time_t)0;

   uf.files_down_month = 0;
   uf.bytes_down_month = 0L;
   uf.seconds_down_month = (time_t)0;

   uf.glock = 0;
   uf.login_times = 0;						  
   uf.last_on = (time_t)0;
   uf.time_limit = 0;
   uf.time_on_today = 0;
   uf.last_nuked = 0;
	uf.use_color = 0;
					 
/*   uf.gadmin = "none"; */
strncpy( uf.gadmin, "none", sizeof( uf.gadmin ) );
 for ( ct = 0; ct < 10; ct++ )
      bzero( uf.group_user[ ct ], sizeof( uf.group_user[ct]) );
					  
   for ( ct = 0; ct < 10; ct++ )
      bzero( uf.group_priv[ ct ], sizeof( uf.group_priv[ct]) );
		
   for ( ct = 0; ct < 20; ct++ )
      bzero( uf.ip[ ct ], sizeof( uf.ip[ct]) );
	
   return;
}
/*-- end of clear_uf_struct() --------------------------------------------*/
										  


/***************************************************************************
	RATIO_OKAY
	Returns 1 if the ratio is okay, 0 if it is not.
	04/09/96: Took out calculations, we now use "credit" system...
***************************************************************************/

int 
ratio_okay( ulong filesize )
{
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] ratio_okay()" );
	debugfn();
#endif


   if ( filesize < (size_t)51200 || uf.ratio == 0 )
      return 1;
						 
	if ( filesize > uf.credits * 1024U )
      return 0;

   return 1;
}
/*-- end of dupechk() ----------------------------------------------------*/
	 

/***************************************************************************
	PARSEVAL
	Parses values in user datafile.
***************************************************************************/

void 
userfile_parseval( char *lvalue, char *rvalue )
{
	int ct;									 
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] userfile_parseval()" );
	debugfn();
#endif
	
	
   if ( strcasecmp( lvalue, "ratio" ) == 0 )
   {
      uf.ratio = atoi( rvalue );
      return;
   }  
	
   if ( strcasecmp( lvalue, "level" ) == 0 )
   {
      uf.level = atoi( rvalue );
      return;
   }

   if ( strcasecmp( lvalue, "restrict" ) == 0 )
   {
      strncpy( uf.restrict, rvalue, sizeof( uf.restrict ) );
		/* Strip trailing '/' */
      if ( uf.restrict[ strlen( uf.restrict ) - 1 ] == '/' )
         uf.restrict[ strlen( uf.restrict ) ] = '\0';
      return;
   }									
	
   if ( strcasecmp( lvalue, "tagline" ) == 0 )
   {
      strncpy( uf.tagline, rvalue, sizeof( uf.tagline ) );
      return;
   }
	
   if ( strcasecmp( lvalue, "num_logins" ) == 0 )
   { 												
		uf.num_logins = atoi( rvalue );
      return;
   }											 
	
   if ( strcasecmp( lvalue, "exempt_from_limit" ) == 0 )
   { 												
		uf.exempt_from_limit = atoi( rvalue );
      return;
   }
	
   if ( strcasecmp( lvalue, "credits" ) == 0 )
   {
      uf.credits = strtoul( rvalue, (char **)NULL, 0 );
		if ( errno == ERANGE )
			syslog( LOG_INFO, "[pheard] *WARNING*: credits out of range for '%s'", uf.name );
      return;
   }
   if ( strcasecmp( lvalue, "files_up" ) == 0 )
   {
      uf.files_up = atoi( rvalue );
      return;
   }

   if ( strcasecmp( lvalue, "bytes_up" ) == 0 )
   {
      uf.bytes_up = strtoul( rvalue, (char **)NULL, 0 );
		if ( errno == ERANGE )
			syslog( LOG_INFO, "[pheard] *WARNING*: bytes_up out of range for '%s'",	uf.name );
      return;
   }
	
   if ( strcasecmp( lvalue, "seconds_up" ) == 0 )
   {
      uf.seconds_up = strtol( rvalue, (char **)NULL, 0 );
      return;
   }
	
   if ( strcasecmp( lvalue, "files_down" ) == 0 )
   {
      uf.files_down = atoi( rvalue );
      return;
   }

   if ( strcasecmp( lvalue, "bytes_down" ) == 0 )
   {
      uf.bytes_down = strtoul( rvalue, (char **)NULL, 0 );
		if ( errno == ERANGE )
			syslog( LOG_INFO, "[pheard] *WARNING*: bytes_down out of range for '%s'",	uf.name );
      return;
   }

   if ( strcasecmp( lvalue, "seconds_down" ) == 0 )
   { 
      uf.seconds_down = strtol( rvalue, (char **)NULL, 0 );
      return;
   }
	
   if ( strcasecmp( lvalue, "files_up_wk" ) == 0 )
   {
      uf.files_up_wk = atoi( rvalue );
      return;
   }

   if ( strcasecmp( lvalue, "bytes_up_wk" ) == 0 )
   {
      uf.bytes_up_wk = strtoul( rvalue, (char **)NULL, 0 );
		if ( errno == ERANGE )
			syslog( LOG_INFO, "[pheard] *WARNING*: bytes_up_wk out of range for '%s'",	uf.name );
      return;
   }
	
   if ( strcasecmp( lvalue, "seconds_up_wk" ) == 0 )
   {
      uf.seconds_up_wk = strtol( rvalue, (char **)NULL, 0 );
      return;
   }
                                     

   if ( strcasecmp( lvalue, "files_down_wk" ) == 0 )
   {
      uf.files_down_wk = atoi( rvalue );
      return;
   }

   if ( strcasecmp( lvalue, "bytes_down_wk" ) == 0 )
   {
      uf.bytes_down_wk = strtoul( rvalue, (char **)NULL, 0 );
		if ( errno == ERANGE )
			syslog( LOG_INFO, "[pheard] *WARNING*: bytes_down_wk out of range for '%s'",	uf.name );
      return;
   }

   if ( strcasecmp( lvalue, "seconds_down_wk" ) == 0 )
   { 
      uf.seconds_down_wk = strtol( rvalue, (char **)NULL, 0 );
      return;
   }
  
   if ( strcasecmp( lvalue, "files_up_day" ) == 0 )
   {
      uf.files_up_day = atoi( rvalue );
      return;
   }

   if ( strcasecmp( lvalue, "bytes_up_day" ) == 0 )
   {
      uf.bytes_up_day = strtoul( rvalue, (char **)NULL, 0 );
                if ( errno == ERANGE )
                        syslog( LOG_INFO, "[pheard] *WARNING*: bytes_up_day out of range for '%s'",    uf.name );
      return;
   }

   if ( strcasecmp( lvalue, "seconds_up_day" ) == 0 )
   {
      uf.seconds_up_day = strtol( rvalue, (char **)NULL, 0 );
      return;
   }
   if ( strcasecmp( lvalue, "files_down_day" ) == 0 )
   {
      uf.files_down_day = atoi( rvalue );
      return;
   }

   if ( strcasecmp( lvalue, "bytes_down_day" ) == 0 )
   {
      uf.bytes_down_day = strtoul( rvalue, (char **)NULL, 0 );
                if ( errno == ERANGE )
                        syslog( LOG_INFO, "[pheard] *WARNING*: bytes_down_day out of range for '%s'",    uf.name );
      return;
   }

   if ( strcasecmp( lvalue, "seconds_down_day" ) == 0 )
   {
      uf.seconds_down_day = strtol( rvalue, (char **)NULL, 0 );
      return;
   }		
   if ( strcasecmp( lvalue, "files_up_month" ) == 0 )
   {
      uf.files_up_month = atoi( rvalue );
      return;
   }

   if ( strcasecmp( lvalue, "bytes_up_month" ) == 0 )
   {
      uf.bytes_up_month = strtoul( rvalue, (char **)NULL, 0 );
                if ( errno == ERANGE )
                        syslog( LOG_INFO, "[pheard] *WARNING*: bytes_up_month out of range for '%s'",    uf.name );
      return;
   }

   if ( strcasecmp( lvalue, "seconds_up_month" ) == 0 )
   {
      uf.seconds_up_month = strtol( rvalue, (char **)NULL, 0 );
      return;
   }
   if ( strcasecmp( lvalue, "files_down_month" ) == 0 )
   {
      uf.files_down_month = atoi( rvalue );
      return;
   }

   if ( strcasecmp( lvalue, "bytes_down_month" ) == 0 )
   {
      uf.bytes_down_month = strtoul( rvalue, (char **)NULL, 0 );
                if ( errno == ERANGE )
                        syslog( LOG_INFO, "[pheard] *WARNING*: bytes_down_month out of range for 's'",    uf.name );
      return;
   }

   if ( strcasecmp( lvalue, "seconds_down_month" ) == 0 )
   {
      uf.seconds_down_month = strtol( rvalue, (char **)NULL, 0 );
      return;
   }       
   if ( strcasecmp( lvalue, "glock" ) == 0 )
   {
     uf.glock = atoi( rvalue );
     return;
   }
   if ( strcasecmp( lvalue, "gadmin" ) == 0 )
   {
     strncpy( uf.gadmin, rvalue, sizeof( uf.gadmin ) );
     return;
   }
   if ( strcasecmp( lvalue, "login_times" ) == 0 )
   {
      uf.login_times = atoi( rvalue );
      return;
   }
	
   if ( strcasecmp( lvalue, "last_on" ) == 0 )
   {
      uf.last_on = atol( rvalue );
      return;
   }

   if ( strcasecmp( lvalue, "time_limit" ) == 0 )
   {
      uf.time_limit = atoi( rvalue );
      return;
   }

   if ( strcasecmp( lvalue, "time_on_today" ) == 0 )
   {					 
		if ( uf.last_on )
		{
   		char datethen[30];
			char datenow[30];
			time_t timenow = time(NULL);
		
			sprintf( datethen, "%-24.24s", ctime(&uf.last_on) );
			sprintf( datenow, "%-24.24s", ctime(&timenow) );
			if ( !memcmp( datenow, datethen, 10 ) )
				uf.time_on_today = atoi( rvalue );
			else
				uf.time_on_today = 0;
		}
		else uf.time_on_today = 0;
	
      return;
   }                         
   
   if ( strcasecmp( lvalue, "last_nuked" ) == 0 )
   {
      uf.last_nuked = atol( rvalue );
      return;
   }									
	
   if ( strcasecmp( lvalue, "use_color" ) == 0 )
   {
      uf.use_color = atoi( rvalue );
      return;
   }
	
   								  
   if ( strcasecmp( lvalue, "group_user" ) == 0 )
	{
		for ( ct = 0; ct < 10; ct++ )
			if ( uf.group_user[ ct ][ 0 ] == '\0' )
			{
				strncpy( uf.group_user[ ct ], rvalue, sizeof( uf.group_user[ ct ] ) );
            trim( uf.group_user[ ct ] ); 
				return;
			}
	}
	 
   if ( strcasecmp( lvalue, "group_priv" ) == 0 )
	{
		for ( ct = 0; ct < 10; ct++ )
			if ( uf.group_priv[ ct ][ 0 ] == '\0' )
			{
				strncpy( uf.group_priv[ ct ], rvalue, sizeof( uf.group_priv[ ct ] ) );
            trim( uf.group_priv[ ct ] ); 
				return;
			}
	}
	
   if ( strcasecmp( lvalue, "ip" ) == 0 )
	{
		for ( ct = 0; ct < 20; ct++ )
			if ( uf.ip[ ct ][ 0 ] == '\0' )
			{
				strncpy( uf.ip[ ct ], rvalue, sizeof( uf.ip[ ct ] ) );
            trim( uf.ip[ ct ] ); 
				return;
			}
	}
}  
/*-- end of userfile_parseval() -------------------------------------------*/



/***************************************************************************
	LOAD_USERFILE
	Loads user data for user "username".
	returns 1 on error, 0 otherwise.
***************************************************************************/

int 
load_userfile( char *username )
{
	char *comment;
	FILE *userfile;
	char lvalue[64];
	char rvalue[BUFSIZ];
	int x, y;
	char linebuf[BUFSIZ];
	char fname_buff[MAXPATHLEN + 1];
	char uname2[14];
	uid_t oldid = geteuid();

#ifdef DEBUG
	syslog(LOG_INFO, "[pheard] load_userfile()");
	debugfn();
#endif

	strncpy(uname2, username, sizeof(uname2));

	clear_uf_struct();

	/*
	 * Set the uf.name to the user's name
	 */
	strncpy(uf.name, uname2, sizeof(uname2)); 


	seteuid(0);
	sprintf(fname_buff, "%s/users/%s", cf.datapath, username);
	if ((userfile = fopen(fname_buff, "r")) == NULL) {
		/* We need code to read "default" user */
		syslog(LOG_ERR, "[pheard] %s: (ERR) error reading userfile '%s': %m",
				username, fname_buff);
		seteuid(oldid);
		strcpy(uf.name, "LOCK.ERR");
		return 1;
	}

	/*
	 * Read lines one at a time, parse lvalue & rvalue, and fill in struct
	 */
	while (1) {
		if (fgets(linebuf, sizeof(linebuf), userfile) == NULL) {
			fclose(userfile);
			seteuid(oldid);
			return 0;
		}

		/*
		 * Clip out comments, they'll eventually be lost anyway :)
		 */
		if ((comment = strchr(linebuf, '#')) != NULL)
			*comment = '\0';

		/*
		 * Trim the buffer 
		 */
		(void) trim(linebuf);

		/* 
		 * Make sure our lvalue & rvalue are empty
		 */
		bzero(lvalue, sizeof(lvalue));
		bzero(rvalue, sizeof(rvalue));

		/*
		 * Parse lvalue 
		 */
		y = 0;
		for (x = 0; x < strlen(linebuf) && linebuf[x] != ' '; x++)
			if (isprint(linebuf[x]))
				lvalue[y++] = linebuf[x];

		/*
		 * Parse rvalue
		 */
		y = 0; x++;
		for (; x < strlen(linebuf); x++)
			if (isprint(linebuf[x]))
				rvalue[y++] = linebuf[x];

		/*
		 * Call 'parseval' to load structure based on lvalue & rvalue
		 */
		userfile_parseval(lvalue, rvalue);
	}

	seteuid(oldid);
	return 0;
}
/*-- end of load_userfile() ----------------------------------------------*/
