#include "../include/userfile.h"
#include "../include/sysconfig.h"
#include <pwd.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <unistd.h>
#include <syslog.h>
#ifdef __linux__
#include <linux/unistd.h>
#endif
#include <stdarg.h>
#include <sys/file.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>

extern void reply (int n, char *fmt,...);
extern void lreply (int n, char *fmt,...);
extern void send_msg (int msgcode, char *msgstr, int send_reply);
extern int isuser (char *checkname);
extern size_t commafmt (char *buf, int bufsize, unsigned long N);
extern int load_userfile (char *username);
extern void update_user (void);
extern char *trim (char *str);
extern time_t weekstart (void);
extern void sysoplog (char *fmt,...);
extern void add_oneliner( char *oneliner );
extern void show_text( int msgcode, const char *helpname );

extern uid_t userid; 
extern char hideuser[25][25];

/***************************************************************************
  STRREV
  Reverses a string in place
***************************************************************************/

char *
strrev (char *str)
{
  char *p1, *p2;

  if (!str || !*str)
    return str;

  for (p1 = str, p2 = str + strlen (str) - 1; p2 > p1; ++p1, --p2)
    {
      *p1 ^= *p2;
      *p2 ^= *p1;
      *p1 ^= *p2;
    }
  return str;
}
/*-- end of strrev() -----------------------------------------------------*/



/***************************************************************************
   SHOW_BYTES
   Displays number of files & bytes in a dir
***************************************************************************/

void
show_bytes (int msgcode)
{
  DIR *dirf;
  struct dirent *dn;
  struct stat st;
  char numbuf[64];
  int tot_files = 0;
  ulong tot_bytes = 0L;
  int numnames = 0;
  char names[10][36];
  ulong bytes[10];
  int files[10];
  struct passwd *p;
  int ct;

#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] show_bytes()" );
	debugfn();
#endif

  /*
   * Clear names!
   */
  for (ct = 0; ct < 10; ct++)
    names[ct][0] = '\0';

  dirf = opendir (".");
  if (!dirf)
    {
      syslog (LOG_ERR, "[pheard] %s: show_bytes, error opening dir: %m",
	      uf.name);
      return;
    }

  while (1)
    {
      dn = readdir (dirf);
      if (!dn)
	break;

      stat (dn->d_name, &st);

      if (S_ISREG (st.st_mode))
	{
	  /* Regular file, accumulate... */
	  tot_files++;
	  tot_bytes += (ulong) st.st_size;

	  if ((p = getpwuid (st.st_uid)) == NULL)
	    {
	      /* Could not find user ID? */
	      syslog (LOG_ERR, "[pheard] get_bytes, user unknown: %d",
		      (int) st.st_uid);
	    }
	  else
	    {
	      for (ct = 0; ct < 10; ct++)
		{
		  if (strcmp (names[ct], p->pw_name) == 0)
		    {
		      bytes[ct] += (ulong) st.st_size;
		      files[ct]++;
		      ct = 10;
		    }
		  else if (names[ct][0] == '\0')
		    {
		      bytes[ct] = (ulong) st.st_size;
		      strcpy (names[ct], p->pw_name);
		      files[ct] = 1;
		      numnames++;
		      ct = 10;
		    }
		}
	    }
	}
    }
  (void) closedir (dirf);

  if (numnames != 0)
    lreply (msgcode, "!B!5Dir Stats                                           !0");
    lreply (msgcode, "!G!5----------------------------------------------------!0");

  for (ct = 0; ct < 10; ct++)
    {
      if (names[ct][0] != '\0')
	{
	  commafmt (numbuf, 64, bytes[ct]);
	  (void) strrev (numbuf);
	  while (strlen (numbuf) < 12)
	    strcat (numbuf, " ");
	  (void) strrev (numbuf);
	  lreply (msgcode, "!E!5 <!D!5%-8.8s!E!5>!H!5 %02d files / %18s bytes     !0",
		  names[ct], files[ct], numbuf);
	}
    }

  commafmt (numbuf, 64, tot_bytes);
  (void) strrev (numbuf);
  while (strlen (numbuf) < 12)
    strcat (numbuf, " ");
  (void) strrev (numbuf);
  lreply (msgcode, "!E!5 <!D!5Total   !E!5> !H!5%02d files / %18s bytes     !0", tot_files, numbuf);
  lreply (msgcode, "!G!5----------------------------------------------------!0");

  return;
}
/*-- end of show_bytes() --------------------------------------------------*/



/***************************************************************************
   SITEDUPE
   Dupe checker facility, requires dupebd
   USAGE: SITE DUPE <searchstring>
***************************************************************************/

long
sitedupe (int msgcode, char *search_str)
{
  int ct;
  char *lfptr;
  char searchstr[BUFSIZ];
  char linebuf[BUFSIZ];
  char tempname[MAXPATHLEN + 1];
  long numhits = 0L;
  FILE *dbf;
  uid_t oldid = geteuid();
												 
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] sitedupe()" );
#endif
  
  strncpy (searchstr, search_str, sizeof (searchstr));
  for (ct = 0; ct < strlen (searchstr); ct++)
    searchstr[ct] = toupper (searchstr[ct]);
				 
	seteuid( 0 );	 
	 
  sprintf (tempname, "%s/dupe.database", cf.datapath);
  dbf = fopen (tempname, "r");

  if (dbf)
    {
      while (fgets (linebuf, 255, dbf) != NULL)
	{
	  if ((lfptr = strchr (linebuf, '\r')) != NULL)
	    *lfptr = '\0';
	  if ((lfptr = strchr (linebuf, '\n')) != NULL)
	    *lfptr = '\0';
	  for (ct = 0; ct < strlen (linebuf); ct++)
	    linebuf[ct] = toupper (linebuf[ct]);

	  if (strstr (linebuf, searchstr) != NULL)
	    {
	      lreply (msgcode, linebuf);
	      numhits++;
	    }
	}
      fclose (dbf);
    }
					 
	seteuid( oldid );	 
	
	lreply( msgcode, "!C%ld !Hmatches found.!0", numhits );
	 
  return numhits;
}
/*-- end of sitedupe() ---------------------------------------------------*/





/***************************************************************************
   LIST_DELETED_USERS
	List users on site which may be added via SITE READD.
   Returns number of users listed.
***************************************************************************/

int
list_deleted_users (int msgcode)
{
  	char *dotptr;
  	DIR *dirf;
  	struct dirent *dn;
  	struct stat st;
  	char temppath[MAXPATHLEN + 1];
  	char username[MAXPATHLEN + 1];
  	char sendstr[80] = "!H!5 \0";
  	char tempbuf[BUFSIZ];
  	char on_item = 0;
  	int numusers = 0;
  	uid_t oldid = geteuid();
											 
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] list_deleted users()" );
	debugfn();
#endif
  				 
	seteuid( 0 );

  	sprintf (temppath, "%s/users", cf.datapath);
  	dirf = opendir (temppath);
  	if (!dirf)
   {
   	syslog (LOG_ERR, "[pheard] %s: list_deleted_users, error opening user dir: %m",
	      uf.name);
		seteuid( oldid );
      return 0;
   }

  	while (1)
   {
      dn = readdir (dirf);
      if (!dn)
			break;

      sprintf (temppath, "%s/users/%s", cf.datapath, dn->d_name);

      strncpy (username, dn->d_name, sizeof (username));

      stat (temppath, &st);

      if ( S_ISREG (st.st_mode) && ( st.st_uid == 99 || st.st_gid == 99 ) )
		{
		  	numusers++;
		  	if ((dotptr = strchr (username, '.')) != NULL)
	   		*dotptr = '\0';
								  
		   sprintf (tempbuf, " %-8.8s ", username);

		  	if (on_item < 7)
	   	{
	      	strcat (sendstr, tempbuf);
	      	on_item++;
		   }
		  	else
	   	{	
				strcat( sendstr, " " );
	      	lreply (msgcode, sendstr);
	      	sendstr[0] = '!';
	      	sendstr[1] = 'B';
	      	sendstr[2] = '!';
	      	sendstr[3] = '5';
	      	sendstr[4] = ' ';
	      	sendstr[5] = '\0';
		   	on_item = 0;
		   }
		}
	}

  	if (sendstr[6] != '\0')
	{
		while ( strlen( sendstr ) < 76 )
			strcat( sendstr, " " );
		strcat( sendstr, " " );
    	lreply (msgcode, sendstr);	
	}
		
  	(void) closedir (dirf);
  
  	seteuid( oldid );

  	return (numusers);
}
/*-- end of list_deleted_users() -----------------------------------------*/




/***************************************************************************
   USERLIST
	List users on site.
   Returns number of users listed.
***************************************************************************/

int
userlist (int msgcode)
{		
	int hidden;
  	char *dotptr;
  	DIR *dirf;
  	struct dirent *dn;
  	struct stat st;
  	char temppath[MAXPATHLEN + 1];
  	char username[MAXPATHLEN + 1];
  	char sendstr[80] = "!H!5 \0";
  	char tempbuf[BUFSIZ];
  	char on_item = 0;
  	int numusers = 0;
  	uid_t oldid = geteuid();
	int ct;
											 
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] userlist()" );
	debugfn();
#endif
  				 
	seteuid( 0 );

  	sprintf (temppath, "%s/users", cf.datapath);
  	dirf = opendir (temppath);
  	if (!dirf)
   {
   	syslog (LOG_ERR, "[pheard] %s: userlist, error opening user dir: %m",
	      uf.name);
		seteuid( oldid );
      return 0;
   }
	show_text( msgcode, "user.head");
  	while (1)
   {
      dn = readdir (dirf);
      if (!dn)
			break;

      sprintf (temppath, "%s/users/%s", cf.datapath, dn->d_name);

      strncpy (username, dn->d_name, sizeof (username));
      stat (temppath, &st);
		hidden=0;			
		for ( ct = 0; ct < 25 && *hideuser[ct]; ct++ )
		{
			if ( !strcmp( username, hideuser[ct] ) )
			{
				hidden = 1;
			}
		}
		
      if ( S_ISREG (st.st_mode) && st.st_uid != 99 && st.st_gid != 99 && 
			 ( !hidden || (hidden && uf.level >= 10)) )
		{
		  	numusers++;
		  	if ((dotptr = strchr (username, '.')) != NULL)
	   		*dotptr = '\0';
								  
		   sprintf (tempbuf, " %-8.8s ", username);

		  	if (on_item < 7)
	   	{
	      	strcat (sendstr, tempbuf);
	      	on_item++;
		   }
		  	else
	   	{	
				strcat( sendstr, " " );
	      	lreply (msgcode, sendstr);
	      	sendstr[0] = '!';
	      	sendstr[1] = 'H';
	      	sendstr[2] = '!';
	      	sendstr[3] = '5';
	      	sendstr[4] = ' ';
	      	sendstr[5] = '\0';
		   	on_item = 0;
		   }
		}
	}

  	if (sendstr[6] != '\0')
	{
		while ( strlen( sendstr ) < 76 )
			strcat( sendstr, " " );
    	lreply (msgcode, sendstr);	
	}
		
	if ( uf.level >= 10 )
	{
		show_text( msgcode, "site.user.middle" );
	   if ( !list_deleted_users( msgcode ) )
			lreply( msgcode, "<None>" );
	}
		
	show_text( msgcode, "user.foot");
  	(void) closedir (dirf);
  
  	seteuid( oldid );
	
  	return (numusers);
}
/*-- end of userlist() ---------------------------------------------------*/

/***************************************************************************
   SHOWUSER 
	Show detailed info about a user.
***************************************************************************/

void
showuser (int msgcode, char *args)
{
  char currentuser[24];
  char queryuser[24];
  char *tokptr;
  char numbuff[30];    
  int userlevel = uf.level;
  char workbuff[80];
  int isme = 0;
  int ct;
  char workbuff2[80];											  
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] showuser()" );
	debugfn();
#endif
  
  tokptr = strtok (args, " ");
  sprintf (queryuser, "%.23s", tokptr);

  if (!isuser (queryuser))
    {
      reply (msgcode, "!H'!C%s!H': !BUnknown user!H.!0", queryuser);
      return;
    }
	 
	if ( !strcmp( queryuser, uf.name ) )
		isme = 1;

  	strncpy (currentuser, uf.name, sizeof (currentuser));
  	if ( load_userfile (queryuser) )
		syslog( LOG_ERR, "[pheard] Calling function was showuser(1)" );

  	lreply (msgcode, "!G!5-------------------------------------------------------------------------!0");
  	lreply (msgcode, "!D!5User!B!5: !H!5%-67.67s!0", queryuser);
  	lreply (msgcode, "!G!5-------------------------------------------------------------------------!0");
  	lreply (msgcode, "!D!5Tagline!B!5: !H!5%-64.64s!0", uf.tagline);
  	if ( uf.group_user[0][0] != '\0' )
		lreply (msgcode, "!D!5Group(s)!B!5: !H!5%-63.63s!0", uf.group_user[0] );
  	if ( uf.level >= 10 || isme )
  	{
   	for ( ct = 0; ct < 10 && uf.group_priv[ct][0] != '\0'; ct++ )
	   	lreply (msgcode, "!D!5Private Group(s)!B!5: !H!5%-56.56s!0", uf.group_priv[ct] );
  	}
  	lreply (msgcode, "!G!5-------------------------------------------------------------------------!0");
  	commafmt (numbuff, 30, uf.bytes_up * 1024U);
  	if (uf.seconds_up != 0)
    	sprintf (workbuff, "%.2f",
	            (float) ((uf.bytes_up / uf.seconds_up)));
  	else
    	strcpy (workbuff, "---");
    	lreply (msgcode, "!D!5Uploads!B!5: !H!5%6dMb - %4d !D!5Files.  Avg Speed!B!5: !H!5%6s k/sec                 !0", (uf.bytes_up / 1024), uf.files_up, workbuff); 
  	commafmt (numbuff, 30, uf.bytes_down * 1024U);
  	if (uf.seconds_down != 0)
    	 sprintf (workbuff, "%.2f",
	             (float) ((uf.bytes_down / uf.seconds_down)));
  	else
    	strcpy (workbuff, "---");
    	lreply (msgcode, "!D!5Downloads!B!5: !H!5%4dMb - %4d !D!5Files.  Avg Speed!B!5: !H!5%6s k/sec                 !0", (uf.bytes_down / 1024), uf.files_down, workbuff); 
  	lreply (msgcode, "!G!5-------------------------------------------------------------------------!0");
  	/* Information will be displayed to ppl with NUKE access */
  	if (userlevel >= 5 || isme ) {
      	if (uf.ratio == 0)
			lreply (msgcode, "!D!5Ratio!B!5: !H!5Unlimited                                                         !0");
      	else   
			lreply(msgcode, "!D!5Ratio!B!5: !H!5%d                                                                 !0", uf.ratio);
	lreply ( msgcode, "!D!5Level!B!5: !H!5%-3d                                                               !0", uf.level);
	}
  	lreply (msgcode, "!G!5-------------------------------------------------------------------------!0");
	if (userlevel >= 10 || isme)
 	{
      		for (ct = 0; ct < 10; ct++)
			if (uf.ip[ct][0] != '\0')
				lreply (msgcode, "!D!5IP %2d!B!5: !H!5%-15s                                                   ", ct, uf.ip[ct]);
   }
  	lreply(msgcode, "!G!5-------------------------------------------------------------------------!0");
	sprintf(workbuff2, "%-24.24s", ctime (&uf.last_on));
  	lreply(msgcode, "!D!5Seen!B!5: !H!5%s                                           !0", workbuff2);
  	lreply(msgcode, "!G!5-------------------------------------------------------------------------!0");
  	reply(msgcode, "!HCommand successful.!0");

  	if ( load_userfile (currentuser) )
		syslog( LOG_ERR, "[pheard] Calling function was showuser(2)" );

  	return;
}
/*-- end of showuser() ---------------------------------------------------*/



/***************************************************************************
   MSG_WAITING
	Does this user have any msgs waiting? (0=no, 1=yes)
***************************************************************************/

int
msg_waiting (void)
{
  char msgfile[45];
  char userbuf[24];
  struct stat st;
  uid_t oldid = geteuid();
												
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] msg_waiting()" );
	debugfn();
#endif		 

	seteuid( 0 );
  
  sprintf (msgfile, "%s/msgs/%s", cf.datapath, uf.name);

  /* 
   * If "msgfile" exists, the user has messages...
   */
  if (!stat (msgfile, &st))
    {
      /*
       * We reload the userfile in case the message is about some credits
       * given, so that if the user is online it the credit give will take
       * effect immediately...
       */
      strcpy (userbuf, uf.name);
      if ( load_userfile (userbuf) )
			syslog( LOG_ERR, "[pheard] Calling function was msg_waiting()" );
		seteuid( oldid );
      return 1;
    }
	 
	seteuid( oldid );

  return 0;
}
/*-- end of msg_waiting() ------------------------------------------------*/


/***************************************************************************
   GIVE_CREDITS
	Allows one user to give their credits to another...
	USAGE: SITE GIVE <megabytes> <username> <message>
***************************************************************************/

void
give_credits (int msgcode, char *args)
{
  char *tokptr;
  ulong givecreds;
  char numcredits[10];
  char touser[24];
  char currentuser[24];
  char message[BUFSIZ] = "\0";
  int ct;
												  
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] give_credits()" );
	debugfn();
#endif
  
  /*
   * Users without ratios may NOT transfer credits..
   */
  if (uf.ratio == 0 && uf.level < 10)
    {
      reply (200, "!BNonratioed users may not transfer credits.!0");
      return;
    }
	 
		 
  /*
   *    Parse 'touser'...
   */
  tokptr = strtok (args, " ");
  if (!tokptr)
    {
      reply (200, "!BYou must specify a username to give credits!!!0");
      return;
    }
  if (!isuser (tokptr))
    {
      reply (200, "!C%s!h: !Bno such user!h.!0", touser);
      return;
    }
  strncpy (touser, tokptr, sizeof (touser));
	 
  /*
   * Get # of credits to xfer
   */
  tokptr = strtok ('\0', " ");
  strncpy (numcredits, tokptr, sizeof (numcredits));
  for (ct = 0; ct < strlen (numcredits); ct++)
    if (!isdigit (numcredits[ct]))
      {
	reply (200, "!H'!C%s!H' !Hmegabytes?  !B?SYNTAX ERROR!0", numcredits);
	return;
      }
		
  givecreds = strtol (numcredits, (char **) NULL, 0);
  if (uf.credits < givecreds && uf.level < 10)
    {
      reply (200, "!BYou don't have enough credits to do that!!!0");
      return;
    }

  /*
   * Perform the GIVE!
   */
  strncpy (currentuser, uf.name, sizeof (currentuser));
  if ( load_userfile (touser) )
		syslog( LOG_ERR, "[pheard] Calling function was give_credits(1)" );
  uf.credits += givecreds;
  update_user ();
  if ( load_userfile (currentuser) )
		syslog( LOG_ERR, "[pheard] Calling function was give_credits(2)" );
  if (uf.level < 10)
    {
      uf.credits -= givecreds;
      update_user ();
    }

  syslog (LOG_INFO, "[pheard] %s transferred %sk to %s",
	  uf.name, numcredits, touser);

  sysoplog ("'%s' %s transferred %sk to %s",
	    uf.name, (uf.level >= 10) ? "(sysop)" : "(non-sysop)",
	    numcredits, touser);

  /*
   * Get message
   */
  do
    {
      tokptr = strtok ('\0', " ");
      if (tokptr)
	{
	  strcat (message, tokptr);
	  strcat (message, " ");
	}
    }
  while (tokptr);
  
  if (message[0] != '\0')
    send_msg (200, message, 0);

  sprintf (message, "%s (%s gave you %sk credit!)",
	   touser, uf.name, numcredits );
  send_msg (200, message, 2);

  reply (200, "!HSuccessfully transferred !C%s!Hk to !C%s!H.!0",
	 numcredits, touser);
}
/*-- end of give_credits() -----------------------------------------------*/




/***************************************************************************
	SEND_MSG
	Sends a message to another user.
***************************************************************************/

void
send_msg (int msgcode, char *msgstr, int send_reply)
{
  int empty_file = 0;
  static char lastname[24] = "\0";
  int append = 0;
  int fdes = -1;		/* File descriptor */
  register int ct;		/* Generic counter */
  char work_buff[MAXPATHLEN + 1];	/* Work buffer */
  char send_to[1024];
  char msg_text[1024];
  mode_t oldmode;
  time_t curtime = time (NULL);
  struct stat st;
  uid_t oldid = geteuid();
													
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] send_msg()" );
	debugfn();
#endif
  
  /*
   * Parse out name, message
   */
  strncpy (send_to, msgstr, 1024);
  for (ct = 0; ct < 1024; ct++)
    if (send_to[ct] == ' ')
      send_to[ct] = '\0';

  if (strchr (msgstr, ' ') == NULL)
    {
      reply (200, "!BEmpty message text, nothing sent.!0");
      return;
    }

  strncpy (msg_text, strchr (msgstr, ' ') + 1, 1024);
  for (ct = 0; ct < 1024; ct++)
    if (msg_text[ct] == '\n')
      msg_text[ct] = '\0';
  trim (send_to);

  if (!isuser (send_to))
    {
      if (send_reply == 1)
	reply (msgcode, "!H'!C%s!H' is not a valid user on this site!!!0", send_to);
      return;
    }

  /*
   * (Attempt to) open the file
   */			 
	seteuid( 0 );
  oldmode = umask (0000);
  sprintf (work_buff, "%s/msgs/%s", cf.datapath, send_to);

  if (stat (work_buff, &st))
    empty_file = 1;

  fdes = open (work_buff, O_CREAT | O_WRONLY | O_APPEND, 0666);
  if (fdes < 0)
    {									 
	 	syslog( LOG_ERR, "[pheard] %s: Could not open msg file '%s': %m",
			uf.name, work_buff );
      (void) umask (oldmode);
      if (send_reply == 1)
	reply (msgcode, "Couldn't open msg file: %s", strerror (errno));
	seteuid( oldid );
      return;
    }

  /*
   * Lock the file
   */
  while (flock (fdes, LOCK_EX))
    {
      syslog (LOG_ERR, "[pheard] %s: flocking '%s': %s",
	      uf.name, work_buff, strerror (errno));
      sleep (1);
    }

  /*
   * Write out message...
   */

  if (lastname[0] == '\0' || !empty_file)
    write (fdes, "\n\n", 2);

  if (empty_file || lastname[0] == '\0')
    {
      strcpy (lastname, send_to);
    }
  else
    {
      if (strcmp (lastname, send_to) != 0)
	{
	  bzero (lastname, sizeof(lastname));
	  write (fdes, "\n\n", 2);
	}
      else
	append = 1;
    }

  if (!append)
    {
      sprintf (work_buff, "!HFrom: !C%s !H(!C%.24s!H)!0\n",
	     (send_reply == 2) ? "PhearD" : uf.name, ctime (&curtime));
      write (fdes, work_buff, strlen (work_buff));

      bzero (work_buff, MAXPATHLEN + 1);
      memset (work_buff, '-', sizeof( work_buff ));
      work_buff[74] = '\n';
		work_buff[75] = '\0';
      write (fdes, work_buff, strlen (work_buff));
    }

  sprintf (work_buff, "!H%s!0\n", msg_text);
  write (fdes, work_buff, strlen (work_buff));

  /*
   * Unlock & close
   */
  flock (fdes, LOCK_UN);
  close (fdes);

  if (send_reply == 1)
    {
      if (empty_file && append)
	reply (msgcode, "!HMessage sent to !C%s!H. (Not appended, user had purged inbox)!0", send_to);
      else if (append)
	reply (msgcode, "!HMessage to !C%s !Happended to last.!0", send_to);
      else
	reply (msgcode, "!HMessage sent successfully to !C%s!H.!0", send_to);
    }						
	 seteuid( oldid );
  (void) umask (oldmode);
}
/*-- end of send_msg() ---------------------------------------------------*/




/***************************************************************************
	READ_MSGS
	Read message(s) from other user(s).
***************************************************************************/

void
read_msgs (int msgcode)
{
  char showname[MAXPATHLEN + 1];
  FILE *infile;
  char *crptr;
  char linebuf[1024];
  uid_t oldid = geteuid();
											  
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] read_msgs()" );
	debugfn();
#endif
  
  if (!msg_waiting ())
    {
      reply (msgcode, "!HYou do not have any messages waiting to be read.!0");
    }
  else
    {
      lreply (msgcode, "");
		
		seteuid( 0 );

      sprintf (showname, "%s/msgs/%s", cf.datapath, uf.name);
      infile = fopen (showname, "r");
      if (infile)
	{
	  while (fgets (linebuf, 255, infile) != NULL)
	    {
	      /*
	       * Strip trailing newline, if any
	       */
	      if ((crptr = strchr (linebuf, '\n')) != NULL)
		*crptr = '\0';

	      lreply (msgcode, "%s", linebuf);
	      if (strstr (linebuf, "credit") != NULL)
		{
		  strcpy (linebuf, uf.name);
		  if ( load_userfile (linebuf) )
			syslog( LOG_ERR, "[pheard] Calling function was read_msgs()" );
		}
	    }
	  fclose (infile);
	}
					  
	seteuid( 0 );
      if (unlink (showname) != 0)
	syslog (LOG_ERR, "[pheard] %s: could not purge inbox: %m", uf.name);
						 
		seteuid( oldid );
	
      reply (msgcode, "!HMessages purged from inbox.!0");
    }
}
/*-- end of read_msgs() --------------------------------------------------*/
/**************************************************************************
	SEEN
**************************************************************************/
void
seen_user( int msgcode, char *user )
{
	  char currentuser[24];
	    char queryuser[24];
	int isme = 0;
	strncpy(queryuser, user, sizeof (queryuser));
	if (!isuser (queryuser))
	{
		reply (msgcode, "!H'!C%s!H': !BUnknown user!H.!0", queryuser);
		return;
	}
	if ( !strcmp( user, uf.name ) )
		isme = 1;
	strncpy (currentuser, uf.name, sizeof (currentuser));
	if ( load_userfile (queryuser) )
		syslog( LOG_ERR, "[pheard] Calling function was showuser(1)" );
	lreply(msgcode, "I last saw %s on %s", queryuser, ctime (&uf.last_on));
	if ( load_userfile (currentuser) )
		syslog( LOG_ERR, "[pheard] Calling function was showuser(1)" );
	return;
}