/*
 * pids.c
 */

#include <dirent.h>							 
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include "../include/userfile.h"	  
#include "../include/sysconfig.h"

extern char mystatus[];       /* Current STAT */
extern char remotehost[];     /* Remote hostname */
extern void sysoplog( char *fmt, ... );
extern void reply( int n, char *fmt, ... );
extern void lreply( int n, char *fmt, ... );
extern int load_userfile( char *username );
extern char *trim( char *str );
extern void show_text( int msgcode, const char *helpname );


/* Structure which holds information regarding users currently online. */
/* This is the same structure written to/read from pid files. */
struct ONLINE
{
  char   username[35];
  char   status[41];
  char   host[256];
  char   currentdir[ MAXPATHLEN ];
  time_t login_time;
  time_t last_update;
  pid_t  procid;
  struct ONLINE *next;
  struct ONLINE *prev;
};

static struct ONLINE *first = NULL;
static struct ONLINE *last = NULL;

extern uid_t userid; 
extern time_t logintime;
extern char authuser[100];

char hmsbuf[ 20 ];

extern char hideuser[25][25];

/***************************************************************************
  UPDATE_PID
  Updates a user's PID file.
***************************************************************************/

void
update_pid (void)
{
  FILE *fp;
  char fname[256];
  struct ONLINE online;
  uid_t oldid = geteuid();
  char currentwd[ MAXPATHLEN ];
  char hostbuf[ 1024 ];

#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] update_pid()" );
	debugfn();
#endif
  
  	if (uf.name[0] == '\0')
  	{
  		return;	  			 
  	}
  
   /*
    * Clear structure
    */
  	bzero( online.username, sizeof( online.username ) );
  	bzero( online.status, sizeof( online.status ) );
  	bzero( online.host, sizeof( online.host ) );
  	online.login_time = (time_t)0;
  	online.last_update = (time_t)0;
  	online.procid = 0;
  	online.next = NULL;
  	online.prev = NULL;
  
   /*
    * Fill structure
    */		 
  	strncpy( online.username, uf.name, sizeof( online.username ) );
  	strncpy( online.status, mystatus, sizeof( online.status ) );
	sprintf( hostbuf, "%s@%s", authuser, remotehost );
  	strncpy( online.host, hostbuf, sizeof( online.host ) );
  	online.login_time = logintime;
  	online.last_update = time(NULL);
  	online.procid = getpid();
  
  	getcwd( currentwd, MAXPATHLEN );
  	if ( strrchr( currentwd, '/' ) == NULL )
     	strncpy( online.currentdir, currentwd, sizeof( online.currentdir ) );
  	else
     	strncpy( online.currentdir, strrchr( currentwd, '/' ) + 1, sizeof( online.currentdir ) );
  
  
  	sprintf (fname, "%s/pids/%d", cf.datapath, online.procid);
  				 
  	seteuid( 0 );
  
  	fp = fopen (fname, "w");
  	if (fp == NULL)
   {
   	syslog (LOG_ERR, "[pheard] %s: Error opening (\"w\") pid file '%s': %m", 
	      cf.datapath, uf.name);
		seteuid( oldid );
      return;
   }

  	fwrite( &online, sizeof( struct ONLINE ), 1, fp );
  
  	fclose (fp);
  
  	seteuid( oldid );
}
/*-- end of update_pid() -------------------------------------------------*/


/***************************************************************************
  DOKILL
  Kill a PID!
***************************************************************************/

void
dokill (int msgcode, char *args)
{
  int pidtokill;
  struct ONLINE *ips;
  char filebuf[ BUFSIZ ];
  uid_t oldid = geteuid();
  		
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] dokill()" );
	debugfn();
#endif
  
  pidtokill = atoi (args);
  
  ips = first;
  
  while (ips)
    {
      if (ips->procid == pidtokill)
	{
	  seteuid (0);
	  kill (pidtokill, SIGKILL);
	  sysoplog ("'%s' killed PID %d (%s).", uf.name, pidtokill,
		    ips->username);
	  reply (msgcode, "!HPID !C%d !H(!C%s!H) killed!0", pidtokill, ips->username);
	  
	  sprintf( filebuf, "%s/pids/%d", cf.datapath, pidtokill );
	  if ( unlink( filebuf ) < 0 )
	  {
	   	syslog( LOG_ERR, "[pheard] error unlinking '%s': %m", filebuf );
	  }
	  
	  seteuid (oldid);
	  return;
	}
      
      ips = ips->next;
    }
  
  reply (msgcode, "!C%d!h: !BInvalid PID.!0", pidtokill);
}
/*-- end of dokill() -----------------------------------------------------*/


/****************************************************************************
  USERSTORE
  Adds a user to the list (in sorted order)
****************************************************************************/

void
userstore (struct ONLINE *i, struct ONLINE **start, struct ONLINE **finish)
{
  struct ONLINE *old = NULL;
  struct ONLINE *p;

#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] userstore()" );
	debugfn();
#endif

  if (*finish == NULL)
    {
      /* 
       * First element in list?
       */
      i->next = NULL;
      i->prev = NULL;
      *finish = i;
      *start = i;
      return;
    }

  /*
   * Start at top of list
   */
  p = *start;

  while (p)
    {
      if (strcmp (p->username, i->username) < 0)
	{
	  old = p;
	  p = p->next;
	}
      else
	{
	  if (p->prev)
	    {
	      p->prev->next = i;
	      i->next = p;
	      i->prev = p->prev;
	      p->prev = i;
	      return;
	    }
	  i->next = p;
	  i->prev = NULL;
	  p->prev = i;
	  *start = i;
	  return;
	}
    }
  
  old->next = i;
  i->next = NULL;
  i->prev = old;
  *finish = i;
}
/*-- end of userstore() ---------------------------------------------------*/



/****************************************************************************
   FREE_ONLINE_STRUCT
****************************************************************************/

void
free_online_struct(void)
{
	struct ONLINE *online, *ptr;

#ifdef DEBUG
	syslog(LOG_INFO, "[pheard] free_online_struct()");
	debugfn();
#endif

   if (first == NULL)
		return;

	ptr = online = first;

	while (ptr != NULL) {
		if (online->next != NULL)
			online = online->next;
		else
			online = NULL;

		free(ptr);
		ptr = online;
	}

	first = last = NULL;
}
/*-- END of free_online_struct() -----------------------------------------*/



/***************************************************************************
  BUILD_ONLINE_INFO
  Returns # of users currently online.
***************************************************************************/

int
build_online_info(void)
{
	DIR *dirf;
	struct dirent *dn;
	struct stat st;
	char temppath[256];
	FILE *fp;
	struct ONLINE *newuser = NULL;
	int found = 0;
	uid_t oldid = geteuid();

#ifdef DEBUG
	syslog(LOG_INFO, "[pheard] build_online_info()");
	debugfn();
#endif

	/*
	 * Clear out current structure if need be, freeing memory
	 */
/*
	if (first != NULL || last != NULL) {
		tmpptr = newuser = first;
		newuser = newuser->next;

		while (tmpptr != NULL) {
			free(tmpptr);
			if (newuser != NULL)
				tmpptr = newuser;
			else
				tmpptr = NULL;
			if (newuser->next != NULL)
				newuser = newuser->next;
			else
				newuser = NULL;
		}

		free(newuser);
		first = last = NULL;
	}
*/
	first = last = NULL;

	/*
	 * Open directory
	 */
	seteuid(0);
	sprintf(temppath, "%s/pids", cf.datapath);
	dirf = opendir(temppath);
	if (!dirf) {
		syslog(LOG_ERR, "[pheard]: Error opening dir '%s': %m", temppath);
		seteuid(oldid);
		return 0;
	}

	/*
	 * Scroll thru /.ftp-data/pids looking for files (processes)
	 */
	while (dn = readdir(dirf)) {
		if (!strcmp(dn->d_name, ".") || !strcmp(dn->d_name, ".."))
			continue;

		sprintf(temppath, "%s/pids/%s", cf.datapath, dn->d_name);
		/* should check if PID is stale or not before doing this */
		stat(temppath, &st);		/* Stat file */

		if (S_ISREG(st.st_mode)) {		/* Found one? */
			/* open the pidfile */
			if ((fp = fopen(temppath, "r")) == NULL) {
				if (errno != ENOENT) {
					syslog(LOG_ERR, "[pheard]: Error opening file '%s': %m", temppath);
				}
				continue;
			}

			if ((newuser = (struct ONLINE *) malloc(sizeof (struct ONLINE)))
					== NULL) {
				syslog(LOG_ERR, "[pheard] Out of memory in build_online_info()");
				continue;
			}
			newuser->next = NULL;

			fread(newuser, sizeof(struct ONLINE), 1, fp);
			fclose(fp);

			userstore(newuser, &first, &last);
			found++;
		}
	}

	(void) closedir(dirf);
	seteuid(oldid);
	return (found);
}
/*-- end of build_online_info() ------------------------------------------*/



/***************************************************************************
  HOWMANYTIMES
  Returns number of times a user is logged in...
***************************************************************************/

int
howmanytimes (char *which)
{
  struct ONLINE *online;
  int numtimes = 0;
															
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] howmanytimes()" );
	debugfn();
#endif
  
  online = first;
  
  while (online)
    {
      if (!strcmp (online->username, which))
	numtimes++;
      online = online->next;
    }

  return numtimes;
}
/*-- end of howmanytimes() -----------------------------------------------*/


																									  


/***************************************************************************
   PID_HMS_FORMAT
***************************************************************************/
                              
void
pid_hms_format( time_t testtime )
{
   time_t timenow = time( NULL );
   time_t difftime;
   int    days = 0;
   int    hours = 0;
   int    minutes = 0;
	int    seconds = 0;

#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] hms_format()" );
	debugfn();
#endif
   
   difftime = timenow - testtime;
   
   while ( difftime >= (time_t)3600 )
   {
      difftime -= (time_t)3600;
      hours++;
   }          
   
   while ( difftime >= (time_t)60 )
   {
      difftime -= (time_t)60;
      minutes++;
   }
   
   if ( hours != 0 )
      sprintf( hmsbuf, "%02d:%02d:%02d", hours, minutes, difftime );
   else
      sprintf( hmsbuf, "   %02d:%02d", minutes, difftime );
}
/*-- end of pid_hms_format() ---------------------------------------------*/



/***************************************************************************
  WHO
  Lists the users currently online (wow!)
***************************************************************************/

void
who(int msgcode)
{
	int which = 1;
	int me = 0;
	int nu, ct, hidden, mylvl, fh;
	struct ONLINE *online, *tmponline, *myname;
	char username[35];
	char currentname[40];
	char action[32];
	char actionbuf[35];
	char bc = '|';

	mylvl = uf.level;
	fh = 0;
        myname = online;
#ifdef DEBUG
	syslog(LOG_INFO, "[pheard] who()");
	debugfn();
#endif

	strncpy(currentname, uf.name, sizeof(currentname));
	nu = build_online_info();
	online = first;

	while (online) {
		if (which > nu)
			break;

		hidden = 0;
		me = 0;

		if (getpid() == online->procid)
			me = 1;

		/* 
		 * Get "username" or blank if logging in 
		 */
		strncpy(username, online->username, sizeof(username));
		for ( ct = 0; ct < 25 && *hideuser[ct]; ct++ )
		{
			if ( !strcmp( username, hideuser[ct] ) )
			{
				hidden = 1;
			}
		}
			if ( load_userfile(username) )
				syslog( LOG_ERR, "[pheard] Calling function was who(1)" );

		snprintf(actionbuf, 25, "\%.31s", online->status);
		trim(actionbuf);
		sprintf(action, "\"%.*s\"", sizeof(action),  actionbuf);
		pid_hms_format(online->last_update);

		/* Uploading */
		if ( !hidden || ( hidden && mylvl >= 10 ) )
		{
		
			if (memcmp(online->status, "STOR", 4) == 0) {
 				lreply(msgcode, "!5!D%c!H%2d!D%c !E!5<!D!5%-9.9s!E!5 -!H!5 %-22.22s!E!5> !E!5!5!D%-28.28s !5!e%c!0", (me) ? '>' : ' ', which, (me) ? '<' : ' ', username, uf.tagline, online->currentdir);
			}

			/* Downloading */
			else if (memcmp(online->status, "RETR", 4) == 0) {
				lreply(msgcode, "!D!5%c!H%2d!D%c !E!5<!D!5%-9.9s!E!5 -!H!5 %-22.22s!E!5> !E!5!5!D%-28.28s !5!e%c!0", (me) ? '>' : ' ', which, (me) ? '<' : ' ', username, uf.tagline, online->currentdir);
			}

			/* Logging in */
			else if ((*username == ' ')
					|| (memcmp(online->status, "PASS", 4) == 0)) {
				lreply(msgcode, "!D!5%c!H%2d!D%c !E!5<!D!5%-9.9s!E!5 -                      > !E!5Logging on...               !5!e%c!0", (me) ? '>' : ' ', which, (me) ? '<' : ' ', username);
			}

			/* Idling */
			else if (memcmp(online->status, "IDLE", 4) == 0) {
				lreply(msgcode, "!5!D%c!5!H%2d!D%c !E!5<!D!5%-9.9s !E!5- !H!5%-22.22s!E!5> !5!EIdle for !5!D%-21.2s !5!e%c!0", (me) ? '>' : ' ', which, (me) ? '<' : ' ', username, uf.tagline, hmsbuf+3);
			}

			/* Doing something else... */
			else {
				lreply(msgcode, "!5!D%c!H%2d!D%c !E!5<!D!5%-9.9s!E!5 -!H!5 %-22.22s!E!5> !5!EDoing !5!D%-24.24s !5!e%c!0", (me) ? '>' : ' ', which, (me) ? '<' : ' ', username, uf.tagline, action);		
			}
		}
		else 
			fh++;

		which++;
		tmponline = online->next;
		free(online);
		online = tmponline;
	}
  
	first = last = NULL;		/* we're now freeing the structure in the while 
										loop, this is here so that free_online_struct() 
										doesn't break. */
	if ( load_userfile(currentname) )
		syslog( LOG_ERR, "[pheard] Calling function was who(2)" );
	show_text(200, "who.foot");
	lreply(msgcode, " ");
	reply(msgcode, "!HCommand successful. (!C%d!H user%s online)!0", which-1-fh, (which-1-fh == 1) ? "" : "s");
	return;
}
/*-- end of who() ---------------------------------------------------------*/

						  

/***************************************************************************
  IWHO
  Sysop-only version of the WHO command, shows idle time, etc.
***************************************************************************/

void
swho (int msgcode)
{
  	int which = 1;
  	struct ONLINE *ips;
  	char curruser[40];
  	int me = 0;
	char libuf[ 10 ];
	char idlebuf[ 10 ];
	char actionbuf[40];
										
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] iwho()" );
	debugfn();
#endif
  
  	strncpy (curruser, uf.name, sizeof (curruser));

  	lreply (msgcode, "!H  ## !e| !HPID    !e| !HUsername !e| !HOnline   !e| !HIdle     !e| !HAction!0" );
  	lreply (msgcode, "!e-----+--------+----------+----------+----------+-------------------------!0" );
	
  	ips = first;

  	while (ips)
	{
  		me = 0;

      /*
       * Is this me?
       */
      if ( getpid() == ips->procid )
	  		me = 1;

		pid_hms_format( ips->login_time );
		strcpy( libuf, hmsbuf );
		pid_hms_format( ips->last_update );
		strcpy( idlebuf, hmsbuf );

      sprintf (actionbuf, "\%.31s", ips->status);
      trim (actionbuf);

      lreply (msgcode, "!D%c !H%2d !e| !H%6d !e| !H%-8.8s !e| !H%-8.8s !e| !H%-8.8s !e| !H%-24.24s!0",
	      	  (me) ? '*' : ' ', which, ips->procid,
	           ips->username, libuf, idlebuf, actionbuf );

		which++;

      ips = ips->next;
	}

  	lreply (msgcode, "");
  	reply (msgcode, "!D%s!h: !C%d !Huser(s) online.!0", cf.sitename, which - 1);

  	return;
}
/*-- end of swho() --------------------------------------------------------*/


/***************************************************************************
  IWHO
  Sysop-only version of the WHO command, shows IP's
***************************************************************************/

void
iwho (int msgcode)
{
  	int which = 1;
  	struct ONLINE *ips;
  	char curruser[40];
  	int me = 0;
  	char libuf[ 9 ];
										
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] swho()" );
	debugfn();
#endif
  
  	strncpy (curruser, uf.name, sizeof (curruser));

  	lreply (msgcode, "!H  ## !e|!H PID    !e| !HUsername !e| !HConnected from!0" );
  	lreply (msgcode, "!e-----+--------+----------+-----------------------------------------------!0" );

  	ips = first;

  	while (ips)
	{
  		me = 0;

      /*
       * Is this me?
       */
      if ( getpid() == ips->procid )
	  		me = 1;
								 
      lreply (msgcode, "!D%c !H%2d !e| !H%6d !e| !H%-8.8s !e| !H%-45.45s!0",
	      	  (me) ? '*' : ' ', which, ips->procid,
	           ips->username, ips->host );

	   which++;

      ips = ips->next;
	}

  	lreply (msgcode, "");
  	reply (msgcode, "!D%s!h: !C%d!H user(s) online.!0", cf.sitename, which - 1);

  	return;
}
/*-- end of iwho() --------------------------------------------------------*/
