
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <unistd.h>
#include <syslog.h>
#ifdef __linux__
#include <linux/unistd.h>
#endif
#include <sys/file.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/param.h>
#include "../include/sysconfig.h"
#include "../include/userfile.h"

struct fivepath
{
   char   fpdirname[ MAXPATHLEN ];              /* The name of the directory */
   int    fpfiles;                                 /* Number of files in dir */
   ulong  fpbytes;                                 /* Total size of the ware */
   time_t fpuptime;                           /* What time_t it was upped at */
   struct fivepath *nxt;
   struct fivepath *prv;
};

struct fivepath *top_list = NULL;
struct fivepath *bottom_list = NULL;
 				 
extern uid_t userid;
extern char *trim( char *str );
extern void debugfn( void );
void free_fivepath_struct( void );
void load_fivepaths( char *reqdir );

extern void lreply( int n, char *fmt, ... );
                             
static char dhmbuf[ 20 ];

/***************************************************************************
   HMS_FORMAT
***************************************************************************/
                              
void
hms_format( time_t dirtime )
{
   time_t timenow = time( NULL );
   time_t difftime;
   int    days = 0;
   int    hours = 0;
   int    minutes = 0;

#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] hms_format()" );
	debugfn();
#endif
   
   difftime = timenow - dirtime;
   
   while ( difftime >= (time_t)86400 )
   {
      difftime -= (time_t)86400;
      days++;
   }
   
   while ( difftime >= (time_t)3600 )
   {
      difftime -= (time_t)3600;
      hours++;
   }          
   
   while ( difftime >= (time_t)60 )
   {
      difftime -= (time_t)60;
      minutes++;
   }
   
   if ( days != 0 )
      sprintf( dhmbuf, "%2dd %2dh %2dm", days, hours, minutes );
   else if ( hours != 0 )
      sprintf( dhmbuf, "    %2dh %2dm", hours, minutes );
   else
      sprintf( dhmbuf, "        %2dm", minutes );
}
/*-- end of hms_format() -------------------------------------------------*/

   

/***************************************************************************
   SHOW_NEWX
***************************************************************************/
                                  
void
show_newx( int msgcode, char *reqdir, char *numtimes )
{
   struct fivepath *npath;
   int    shown = 0;
   char   tempbuf[ 20 ];
   char   tempbuf2[ BUFSIZ ];
	int    howmany;
	
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] show_newfive()" );
	debugfn();
	#endif

	howmany = atoi( numtimes );
	
	if ( howmany < 1 )
	{
	 	lreply( msgcode, "!BInvalid number of files specified.!0" );
		return;
	}
	
   load_fivepaths( reqdir );

	npath = bottom_list;

   while ( npath && shown < howmany )
	{                          /* 11d 11m 11s */
      shown++;
      sprintf( tempbuf, "%3.0f", (double)( npath->fpbytes / 1024000L ) );
      trim( tempbuf );
      while ( strlen( tempbuf ) < 3 )
         strcat( tempbuf, " " );

      hms_format( npath->fpuptime );

#ifdef VDR
	sprintf( tempbuf2, "!h!5 %2d  %s  %-3d %s %-46.46s !0",
					shown, dhmbuf, npath->fpfiles, tempbuf, npath->fpdirname );
#else
	sprintf( tempbuf2, "!H!5  %2d   %s   %-3d %s   %-40.40s  !0",
					shown, dhmbuf, npath->fpfiles, tempbuf, npath->fpdirname );
#endif
      lreply( msgcode, "%s", tempbuf2 );

		npath = npath->prv;
   }
   free_fivepath_struct();
}
/*-- bottom_list of show_newx() -------------------------------------------*/


/****************************************************************************
	FIVEPATH_STORE
	Adds a fpdirname to the fivepath structure.
****************************************************************************/													 
												  
void 
fivepath_store( struct fivepath *i, struct fivepath **top_list, struct fivepath **bottom_list )
{
	struct fivepath *old = NULL;
	struct fivepath *p;
													
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] fivepath_store()" );
	debugfn();
#endif
	
	if ( *bottom_list == NULL )
	{
		/* 
		 * First element in list?
		 */
		i->nxt = NULL;
		i->prv = NULL;
		*bottom_list = i;
		*top_list = i;
		return;
	}
	
	/*
	 * Start at top of list
	 */
	p = *top_list;
	
	while ( p )
	{
		if ( p->fpuptime < i->fpuptime )
		{
			old = p;
			p = p->nxt;
		}
		else
		{
			if ( p->prv )
			{
				p->prv->nxt = i;
				i->nxt = p;
				i->prv = p->prv;
				p->prv = i;
				return;
			}
			i->nxt = p;
			i->prv = NULL;
			p->prv = i;
			*top_list = i;
			return;
		}
	}
	
	old->nxt = i;
	i->nxt = NULL;
	i->prv = old;
	*bottom_list = i;
}
/*-- bottom_list of fivepath_store() -----------------------------------------------*/	
                


/****************************************************************************
   FREE_FIVEPATH_STRUCT
****************************************************************************/

void
free_fivepath_struct( void )
{
 	struct fivepath *wdm;
													  
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] free_fivepath_struct()" );
	debugfn();
#endif
	
	wdm = top_list;

	while ( wdm )
	{
	   wdm = wdm->prv;
	   free( wdm );
	}

   free( wdm );

   top_list = bottom_list = NULL;
}
/*-- END of free_fivepath_struct() ----------------------------------------*/

                               

/***************************************************************************
   LOAD_FIVEPATHS
   Loads up the fivepaths in cwd
***************************************************************************/
                                            
void
load_fivepaths( char *whichdir )
{
   struct fivepath *newpath;
	DIR    *dirf;
	DIR    *dirfr;
	struct dirent *dn;
	struct dirent *dr;
	struct stat st;
	struct stat stb;
   int    tot_files = 0;
   ulong  tot_bytes = 0L; 
   char   curpath[ MAXPATHLEN ];
   char   namebuf[ MAXPATHLEN];
  	uid_t oldid = geteuid();
   														  
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] load_fivepaths()" );
	debugfn();
#endif
	
   free_fivepath_struct();
   			 
	seteuid( 0 );
	dirfr = opendir( whichdir );
   while ( 1 )
   {
		dr = readdir( dirfr );
		if ( !dr )
			break;
                                       
      /* We ignore "." and ".." */         
      if ( !strcmp( dr->d_name, "." ) || !strcmp( dr->d_name, ".." ) || strstr( dr->d_name, "NUKED" ) )
         continue;

      sprintf( curpath, "%s/%s", whichdir, dr->d_name );

		if ( stat( curpath, &stb ) < 0 )
	 		syslog( LOG_ERR, "[pheard] %s: stat failed on '%s': %m", dr->d_name,
	 			uf.name );

		if ( S_ISDIR( stb.st_mode ) )
		{     
         /* Get number of files/bytes in dir */
	  	 	dirf = opendir( curpath );
	  	 	if ( !dirf )
	  	 	{
	  	 		syslog( LOG_ERR, "[pheard] %s: newfive, error opening dir: %m",
	  	 			uf.name );
				seteuid( oldid );
	  	 		return;
	  	 	}
		   
		
         tot_files = 0;
         tot_bytes = 0L;
         while ( 1 )
         {
	  	 		dn = readdir( dirf );
	  	 		if ( !dn )
	  	 			break;		  
				sprintf( namebuf, "%s/%s", curpath, dn->d_name );
	    
	  	 		stat( namebuf, &st );

	  	 		if ( S_ISREG( st.st_mode ) )
	  	 		{           
               if ( strcmp( dn->d_name, "FILE_ID.DIZ" ) != 0 )
               {
                  /* Regular file, accumulate... */
                  tot_files++;
                  tot_bytes += (ulong)st.st_size;
               }
            }
         }
  	    	(void)closedir( dirf );

  	    	newpath = (struct fivepath *)malloc( sizeof( struct fivepath ) );
         if ( newpath == NULL )
         {
            syslog( LOG_ERR, "[pheard] Out of memory in load_fivepaths()" );
         }
         else
         {
            strncpy( newpath->fpdirname, dr->d_name, sizeof( newpath->fpdirname ) );
            newpath->fpfiles = tot_files;
            newpath->fpbytes = tot_bytes;
            newpath->fpuptime = stb.st_mtime;
            fivepath_store( newpath, &top_list, &bottom_list );
         }
		}
   }
  	(void)closedir( dirfr );
	
	seteuid( oldid );

   return;
}
/*-- bottom_list of load_fivepaths() ----------------------------------------------*/

