#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <syslog.h>
#ifdef __linux__
#include <linux/unistd.h>
#endif
#include <sys/file.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <dirent.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>
#include "../include/userfile.h"
#include "../include/sysconfig.h"
						  
char usergroup[ 40 ][ 15 ];
char usergroup_desc[ 40 ][ 40 ];
char privgroup[ 20 ][ 15 ];
char privgroup_desc[ 20 ][ 40 ];

extern int isuser( char *checkname );
extern char *trim( char *str );
extern int load_userfile( char *username );
extern void update_user( void );
extern void reply( int n, char *fmt, ... );
extern void lreply( int n, char *fmt, ... );

void list_user_groups( int msgcode, int sysop );

/***************************************************************************
	LOAD_USERGROUPS
	Loads group data from "<datapath>/groups.user"
***************************************************************************/

void 
load_usergroups( void )
{
	char *comment;
  	char *tokptr;
   FILE *groupfile;
   int  x;
	char linebuf[ BUFSIZ ];
	char fname_buff[ MAXPATHLEN+1 ];
  	uid_t oldid = geteuid();
	
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] load_usergroups()" );
	debugfn();
#endif
	
	/*
	 * Clear groups:
	 */				 
	for ( x = 0; x < 40; x++ )
		bzero( usergroup[ x ], sizeof( usergroup[ x ] ) ); 
				  
	/*
	 * Open the groupfile
	 */		 
	seteuid( 0 );
   sprintf( fname_buff, "%s/groups.user", cf.datapath );
   if ( ( groupfile = fopen( fname_buff, "r" ) ) == NULL )
	{									
		syslog( LOG_INFO, "[pheard] error reading groupfile: %m", fname_buff );
		seteuid( oldid );
      return;
	}
									
	/*
	 * Read lines one at a time and fill in struct
	 */
   while ( 1 )
   {
      if ( fgets( linebuf, sizeof( linebuf ), groupfile ) == NULL )
      {				 
         fclose( groupfile );
			seteuid( oldid );
         return;
      }
				 						  
		/*
		 * Clip out comments, they'll eventually be lost anyway :)
		 */
		if ( ( comment = strchr( linebuf, '#' ) ) != NULL )
			*comment = '\0';

		/*
		 * Trim the buffer 
		 */					
		(void)trim( linebuf ); 

		if ( !isalpha( linebuf[ 0 ] ) )
			continue;

		for ( x = 0; x < 40 && usergroup[x][0] != '\0'; x++ )
			;

  		tokptr = strtok (linebuf, " ");
		if ( tokptr )
			strncpy( usergroup[x], tokptr, sizeof( usergroup[x] ) );
		else
			continue;
				 
		strcpy( usergroup_desc[x], "\0" );
	  	tokptr = strtok ('\0', " ");
		while ( tokptr )
		{
			if ( usergroup_desc[0] != '\0' )
				strcat( usergroup_desc[x], " " );
			strcat( usergroup_desc[x], tokptr );
	  		tokptr = strtok ('\0', " ");
		}
   }

	seteuid( oldid );
   return;
}
/*-- end of load_usergroups() --------------------------------------------*/
						


/***************************************************************************
	LOAD_PRIVGROUPS
	Loads group data from "<datapath>/groups.priv"
***************************************************************************/

void 
load_privgroups( void )
{
	char *comment;
   FILE *groupfile;  
	char *tokptr;
   int  x;
	char linebuf[ BUFSIZ ];
	char fname_buff[ MAXPATHLEN+1 ];
  	uid_t oldid = geteuid();
	
#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] load_usergroups()" );
	debugfn();
#endif
	
	/*
	 * Clear groups:
	 */				 
	for ( x = 0; x < 20; x++ )
		bzero( privgroup[ x ], sizeof( privgroup[ x ] ) ); 
				  
	/*
	 * Open the groupfile
	 */		 
	seteuid( 0 );
   sprintf( fname_buff, "%s/groups.priv", cf.datapath );
   if ( ( groupfile = fopen( fname_buff, "r" ) ) == NULL )
	{									
		syslog( LOG_INFO, "[pheard] error reading groupfile: %m", fname_buff );
		seteuid( oldid );
      return;
	}
									
	/*
	 * Read lines one at a time and fill in struct
	 */
   while ( 1 )
   {
      if ( fgets( linebuf, sizeof( linebuf ), groupfile ) == NULL )
      {				 
         fclose( groupfile );
			seteuid( oldid );
         return;
      }
				 						  
		/*
		 * Clip out comments, they'll eventually be lost anyway :)
		 */
		if ( ( comment = strchr( linebuf, '#' ) ) != NULL )
			*comment = '\0';
										
		/*
		 * Trim the buffer 
		 */					
		(void)trim( linebuf ); 
											 
		if ( !isalpha( linebuf[ 0 ] ) )
			continue;
					
		for ( x = 0; x < 20 && privgroup[x][0] != '\0'; x++ )
			;
						
  		tokptr = strtok (linebuf, " ");
		if ( tokptr )
			strncpy( privgroup[x], tokptr, sizeof( privgroup[x] ) );
		else
			continue;
			
		strcpy( privgroup_desc[x], "\0" );
	  	tokptr = strtok ('\0', " ");
		while ( tokptr )
		{
			if ( privgroup_desc[0] != '\0' )
				strcat( privgroup_desc[x], " " );
			strcat( privgroup_desc[x], tokptr );
	  		tokptr = strtok ('\0', " ");
		}
   }
									
	seteuid( oldid );
   return;
}
/*-- end of load_privgroups() --------------------------------------------*/



/***************************************************************************
	LOAD_GROUPS
	Loads group data from <datapath>/groups.user and <datapath>/groups.priv
***************************************************************************/

void
load_groups( void )
{
	load_privgroups();
	load_usergroups();

	return;
}
/*-- end of load_groups() ------------------------------------------------*/


/***************************************************************************
	SETGROUP
	Sets the user's group(s).
***************************************************************************/

void
setgroup( int msgcode, char *args )
{
	int ct;
	int ok = 0;
	int member = 0;
	int added = 0;

	trim( args );
	
	/* Is the requested group a legitimate group? */

	for ( ct = 0; ct < 40 && usergroup[ct][0] != '\0'; ct++ )
		if ( !strcasecmp( args, usergroup[ct] ) )
			ok = 1;

	if ( !ok )
	{
	 	reply( msgcode, "'%s' is not a valid group.  SITE GROUP for a list.", args );
		return;
	}
	 
	/* Is the user currently a member of this group? */
	for ( ct = 0; ct < 10 && uf.group_user[ct][0] != '\0'; ct++ )
		if ( !strcasecmp( args, uf.group_user[ct] ) )
			member = 1;
	
	/* Are they already in a group? */
	for ( ct = 0; ct < 10 && uf.group_user[ct][0] != '\0'; ct++ )
		;
	if ( ct && !member )
	{
		reply( msgcode, "You must leave \"%s\" before you may join \"%s\".",
				 uf.group_user[0], args );
		return;
	}				 
			
	/* Toggle group membership & update user */
	if ( !member )
	{
		/* Add new group */
		for ( ct = 0; ct < 10; ct++ )
		{
			if ( uf.group_user[ct][0] == '\0' )
			{
				strncpy( uf.group_user[ct], args, sizeof( uf.group_user[ct] ) );
				reply( msgcode, "You have been successfully added to '%s'", uf.group_user[ct] );
				added = 1;
				break;
			}
		}
		
		if ( !added )
		{
			reply( msgcode, "You may only be a member in 10 groups." );
			return;
		}
	}
	else
	{
		/* Remove from group */
		for ( ct = 0; ct < 10; ct++ )
		{
			if ( !strcasecmp( uf.group_user[ct], args ) )
			{
				bzero( uf.group_user[ct], sizeof( uf.group_user[ct] ) );
				reply( msgcode, "You have been successfully removed from '%s'", args );
				break;
			}
		}
	}
	
   update_user();                            /* Update user */
}
/*-- end of setgroup() ---------------------------------------------------*/



/***************************************************************************
	SGROUP
	Sysop version of setgroup.
***************************************************************************/

void
sgroup( int msgcode, char *argus )
{
	int  ct;
	int  ok = 0;
	int  member = 0;
	int  added = 0;
	char *tokptr; 
   char user_name[ BUFSIZ ];
	char args[ BUFSIZ ];			/* Really the group */
   char currentuser[ 40 ];

#ifdef DEBUG
	syslog( LOG_INFO, "[pheard] sgroup()" );
	debugfn();
#endif
	
	trim( argus );
	tokptr = strtok( argus, " " );	 
	if ( tokptr == NULL )
	{
		reply( msgcode, "You must provide a username." );
		return;
	}
	strncpy( user_name, tokptr, sizeof( user_name ) );
	
	if ( !isuser( user_name ) )
	{
		reply( msgcode, "'%s': user not found.", user_name );
		return;
	}
	
   tokptr = strtok( '\0', " " );
   if ( tokptr == NULL )
   {         	 		 
		strcpy( currentuser, uf.name );
		if ( load_userfile( user_name ) )
			syslog( LOG_ERR, "[pheard] Calling function was sgroup(1)" );
		list_user_groups( msgcode, 1 );
		if ( load_userfile( currentuser ) )
			syslog( LOG_ERR, "[pheard] Calling function was sgroup(2)" );
		reply( msgcode, "Use SITE SGRP <username> <group> to toggle groups." );
      return;
   }
	strncpy( args, tokptr, sizeof( args ) );

	/* Is the requested group a legitimate group? */

	for ( ct = 0; ct < 40 && usergroup[ct][0] != '\0'; ct++ )
		if ( !strcasecmp( args, usergroup[ct] ) )
			ok = 1;		/* Add to user groups */
	for ( ct = 0; ct < 20 && privgroup[ct][0] != '\0'; ct++ )
		if ( !strcasecmp( args, privgroup[ct] ) )
			ok = 2;		/* Add to priv groups */
			
	if ( !ok )
	{
	 	reply( msgcode, "'%s' is not a valid group.  SITE GROUP for a list.", args );
		return;
	}
										
	strcpy( currentuser, uf.name );
	if ( load_userfile( user_name ) )
	   syslog( LOG_ERR, "[pheard] Calling function was sgroup(3)" );
	
	
	/* Is the user currently a member of this group? */
	if ( ok == 1 )
	{
		for ( ct = 0; ct < 10 && uf.group_user[ct][0] != '\0'; ct++ )
			if ( !strcasecmp( args, uf.group_user[ct] ) )
				member = 1;
	}
	else
	{
		for ( ct = 0; ct < 10 && uf.group_priv[ct][0] != '\0'; ct++ )
			if ( !strcasecmp( args, uf.group_priv[ct] ) )
				member = 1;
	}

	/* Toggle group membership & update user */
	if ( ok == 1 )
	{
		if ( !member )
		{
			/* Add new group */
			for ( ct = 0; ct < 10; ct++ )
			{		  
				if ( uf.group_user[ct][0] == '\0' )
				{
					strncpy( uf.group_user[ct], args, sizeof( uf.group_user[ct] ) );
					reply( msgcode, "'%s' has been successfully added to '%s'", 
							 user_name, uf.group_user[ct] );
					added = 1;
					break;
				}
			}
		
			if ( !added )
			{
				reply( msgcode, "A user may only be a member in 10 groups." );
				if ( load_userfile( currentuser ) )
					syslog( LOG_ERR, "[pheard] Calling function was sgroup(4)" );
				return;
			}
		}
		else
		{
			/* Remove from group */
			for ( ct = 0; ct < 10; ct++ )
			{
				if ( !strcasecmp( uf.group_user[ct], args ) )
				{
					bzero( uf.group_user[ct], sizeof( uf.group_user[ct] ) );
					reply( msgcode, "'%s' has been successfully removed from '%s'", 
							 user_name, args );
					break;
				}
			}
		}
	}
	else
	{
		if ( !member )
		{
			/* Add new group */
			for ( ct = 0; ct < 10; ct++ )
			{		  
				if ( uf.group_priv[ct][0] == '\0' )
				{
					strncpy( uf.group_priv[ct], args, sizeof( uf.group_priv[ct] ) );
					reply( msgcode, "'%s' has been successfully added to '%s'", 
							 user_name, uf.group_priv[ct] );
					added = 1;
					break;
				}
			}
		
			if ( !added )
			{
				reply( msgcode, "A user may only be a member in 5 priv groups." );
				if ( load_userfile( currentuser ) )
					syslog( LOG_ERR, "[pheard] Calling function was sgroup(5)" );
				return;
			}
		}
		else
		{
			/* Remove from group */
			for ( ct = 0; ct < 10; ct++ )
			{
				if ( !strcasecmp( uf.group_priv[ct], args ) )
				{
					bzero( uf.group_priv[ct], sizeof( uf.group_priv[ct] ) );
					reply( msgcode, "'%s' has been successfully removed from '%s'", 
							 user_name, args );
					break;
				}
			}
		}
	}
	 
   update_user();                            /* Update user */
	if ( load_userfile( currentuser ) )
		syslog( LOG_ERR, "[pheard] Calling function was sgroup(6)" );
}
/*-- end of sgroup() -----------------------------------------------------*/


/***************************************************************************
	LIST_AVAILABLE_GROUPS
	Lists all available groups.
***************************************************************************/

void
list_available_groups( int msgcode )
{
	int ct;
	int user_total = 0;
	int priv_total = 0;
	char outbuf[ 100 ];

	lreply( 200, "The following groups are available:" );
	lreply( 200, cf.dividerline );

	bzero( outbuf, 100 );

	for ( ct = 0; ct < 40 && usergroup[ct][0] != '\0'; ct++ )
	{
		user_total++;
		lreply( msgcode, " %-15.15s  %s", usergroup[ct], usergroup_desc[ct] );
	}

	lreply( msgcode, "" );

	if ( uf.level >= 10 )
	{
		for ( ct = 0; ct < 20 && privgroup[ct][0] != '\0'; ct++ )
		{
			priv_total++;
			lreply( msgcode, " *%-15.15s %s", privgroup[ct], privgroup_desc[ct] );
		}

		lreply( msgcode, "" );
		lreply( msgcode, "A '*' denotes a priveliged group.  Only sysop-level users may add" );
		lreply( msgcode, "or remove users from these groups.  Type SITE SGRP to see how." );
		lreply( msgcode, "" );
		lreply( msgcode, "%d user groups, %d privileged groups.", user_total, priv_total );
	}
	else
		lreply( msgcode, "%d groups listed.", user_total );
}
/*-- end of list_available_groups() --------------------------------------*/


/***************************************************************************
	LIST_USER_GROUPS
	Lists groups the user is a member of.
***************************************************************************/

void
list_user_groups( int msgcode, int sysop )
{
	int shown;
	int ct;
	int total = 0;
	char outbuf[ 100 ];
	char namebuf[ 20 ];
													 
	if ( sysop )
		lreply( 200, "%s is a member of the following groups:", uf.name );
	else
		lreply( 200, "You are a member of the following groups:" );
	lreply( 200, cf.dividerline );

	shown = 0;
	bzero( outbuf, 100 );

	for ( ct = 0; ct < 10 && uf.group_user[ct][0] != '\0'; ct++ )
	{
		shown++;
		total++;

		if ( shown > 4 )
		{
			lreply( msgcode, outbuf );
			bzero( outbuf, 100 );
			shown = 0;
		}

		sprintf( namebuf, " %-14.14s ", uf.group_user[ct] );
		strcat( outbuf, namebuf );
	}
	if ( outbuf[ 0 ] != '\0' )
		lreply( msgcode, outbuf );

	lreply( msgcode, "" );
	
	if ( sysop )
	{
		lreply( 200, "%s is a member of the following priv groups:", uf.name );
		lreply( 200, cf.dividerline );
	
		shown = 0;
		bzero( outbuf, 100 );

		for ( ct = 0; ct < 10 && uf.group_priv[ct][0] != '\0'; ct++ )
		{
			shown++;
			total++;

			if ( shown > 4 )
			{
				lreply( msgcode, outbuf );
				bzero( outbuf, 100 );
				shown = 0;
			}

			sprintf( namebuf, " %-14.14s ", uf.group_priv[ct] );
			strcat( outbuf, namebuf );
		}
		if ( outbuf[ 0 ] != '\0' )
			lreply( msgcode, outbuf );

		lreply( msgcode, "" );
	}
}
/*-- end of list_user_groups() -------------------------------------------*/


/***************************************************************************
	LIST_GROUPS
	Lists groups.
***************************************************************************/

void
list_groups( int msgcode )
{
	list_user_groups( msgcode, 0 );
	list_available_groups( msgcode );

	return;
}
/*-- end of list_groups() ------------------------------------------------*/



/***************************************************************************
	ISINGROUP
	Returns 1 if the current uf. person is in the group "whichgrp", else 0.
***************************************************************************/
										  
int
isingroup( char *whichgrp )
{
	int ct;
	
	for ( ct = 0; ct < 10 && uf.group_user[ct][0] != '\0'; ct++ )
		if ( !strcasecmp( whichgrp, uf.group_user[ct] ) )
			return 1;
			
	for ( ct = 0; ct < 10 && uf.group_priv[ct][0] != '\0'; ct++ )
		if ( !strcasecmp( whichgrp, uf.group_priv[ct] ) )
			return 1;
			
	return 0;
}
/*-- end of isingroup() --------------------------------------------------*/


/***************************************************************************
	ISGROUP
	Returns 1 if the group is a valid one, 0 otherwise.
***************************************************************************/
										  
int
isgroup( char *whichgrp )
{
	int ct;
					
	for ( ct = 0; ct < 40 && usergroup[ct][0] != '\0'; ct++ )
		if ( !strcasecmp( whichgrp, usergroup[ct] ) )
			return 1;
			
	if ( uf.level < 10 )
		return 0;
		
	for ( ct = 0; ct < 20 && privgroup[ct][0] != '\0'; ct++ )
		if ( !strcasecmp( whichgrp, privgroup[ct] ) )
			return 1;
	
	return 0;
}
/*-- end of isgroup() ----------------------------------------------------*/


/***************************************************************************
	LISTMEMBERS
	List members in a specific group.  Usage: SITE LGRP <groupname>
	if <groupname> is "none", lists users NOT any any *group_priv* groups
***************************************************************************/

void
listmembers( int msgcode, char *args )
{							  
  	char temppath[ MAXPATHLEN + 1 ];
  	DIR *dirf;
  	struct dirent *dn;
  	struct stat st;
  	char on_item = 0;
  	char sendstr[80] = "\0";
  	uid_t oldid = geteuid();
	char whichgrp[ BUFSIZ ];
	char tempbuf[ BUFSIZ ];
  	char username[MAXPATHLEN + 1];
	char currentuser[ 40 ];
	int numusers = 0;
	int leeches = 0;
	char *dotptr;
	int  none = 0;
				
	strcpy( currentuser, uf.name );
	
	strncpy( whichgrp, args, sizeof( whichgrp ) );
	trim( whichgrp );			 
	
	if ( !strcasecmp( whichgrp, "none" ) )
		none = 1;
	
	if ( uf.level < 10 )
		none = 0;
			
	if ( !isgroup( whichgrp ) && !none )
	{
	 	/* Not a valid group */
	 	reply( msgcode, "'%s' is not a valid group.  SITE GROUP for a list.", whichgrp );
		return;
	}
												 
	if ( none )
		lreply( msgcode, "The following users are in not in any priv groups:" );
	else
		lreply( msgcode, "The following users are in group '%s':", whichgrp );
	lreply( msgcode, cf.dividerline );

	seteuid( 0 );

  	sprintf (temppath, "%s/users", cf.datapath);
  	dirf = opendir (temppath);
  	if (!dirf)
   {
   	syslog (LOG_ERR, "[pheard] %s: userlist, error opening user dir: %m",
	      uf.name);
		seteuid( oldid );
	   reply( msgcode, "Error reading userfiles.  Aborted." );
      return;
   }
	
  	while (1)
   {
      dn = readdir (dirf);
      if (!dn)
			break;

      sprintf (temppath, "%s/users/%s", cf.datapath, dn->d_name);

      strncpy (username, dn->d_name, sizeof (username));

      stat (temppath, &st);

      if ( S_ISREG (st.st_mode) && st.st_uid != 99 && st.st_gid != 99 )
		{
		  	if ((dotptr = strchr (username, '.')) != NULL)
	   		*dotptr = '\0';

   		sprintf (tempbuf, " %-8.8s ", username);
			
			if ( load_userfile( username ) )
				syslog( LOG_ERR, "[pheard] Calling function was listmembers(1)" );
											  
			if ( isingroup(whichgrp) || (uf.group_priv[0][0] == '\0' && none))
			{
				if ( uf.ratio == 0 )
				   leeches++;
			  	numusers++;
			  	if (on_item < 7)
		   	{
	   	   	strcat (sendstr, tempbuf);
	      		on_item++;
			   }
			  	else
	   		{	
	      		lreply (msgcode, sendstr);
	      		sendstr[0] = '\0';
			   	on_item = 0;
			   }
			}
		}
	}

	if ( load_userfile( currentuser ) )
		syslog( LOG_ERR, "[pheard] Calling function was listmembers(2)" );

  	if (sendstr[0] != '\0')
    	lreply (msgcode, sendstr);				
		
	if ( numusers == 0 )
		lreply( msgcode, "<none>" );

  	(void) closedir (dirf);

  	seteuid( oldid );

	lreply( msgcode, cf.dividerline );
	lreply( msgcode, "" );
	
	if ( none )
	{
		reply( msgcode, "%d users (%d leeches) are not in a group.", numusers, leeches );
	}
	else
	{
		if ( uf.level >= 10 )
			reply( msgcode, "%d users (%d leeches) in '%s'.", numusers, leeches, whichgrp );
		else
			reply( msgcode, "%d users in '%s'.", numusers, whichgrp );
	}

	return;
}
/*-- end of listmembers() -------------------------------------------------*/






/***************************************************************************
	GROUPINFO
	List members in a specific group with detailed usage stats. 
   Usage: SITE GNFO <groupname>
***************************************************************************/

void
groupinfo( int msgcode, char *args )
{							  
  	char temppath[ MAXPATHLEN + 1 ];
  	DIR *dirf;
  	struct dirent *dn;
  	struct stat st;
  	char sendstr[80] = "\0";
  	uid_t oldid = geteuid();
	char whichgrp[ BUFSIZ ];
	char tempbuf[ BUFSIZ ];
  	char username[MAXPATHLEN + 1];
	char currentuser[ 40 ];
	int numusers = 0;
	int leeches = 0;
	char *dotptr;
	ulong total_bytes_dn = 0UL;
	ulong total_bytes_up = 0UL;
	int   total_files_up = 0;
	int   total_files_dn = 0;
	char  dnbuf[ 8 ];
	char  upbuf[ 8 ];
	char  numusersbuf[ 8 ];
	char  leechesbuf[ 8 ];
				
	strcpy( currentuser, uf.name );
	
	strncpy( whichgrp, args, sizeof( whichgrp ) );
	trim( whichgrp );
	
	if ( !isgroup( whichgrp ) )
	{
	 	/* Not a valid group */
	 	reply( msgcode, "'%s' is not a valid group.  SITE GROUP for a list.", whichgrp );
		return;
	}

	lreply( msgcode, "                                            ________" );
	lreply( msgcode, "_____________.______________________________\\     _/___________" );
	lreply( msgcode, "\\      _     |     _      /    _       /     \\     \\      _    \\" );
	lreply( msgcode, " \\     \\     |     /_____/     \\      /      |\\     \\     \\    /XFER" );
	lreply( msgcode, "  \\______    |______|    \\___________/|______________\\    ____/ STATS" );
	lreply( msgcode, ".---- /______| ------------------------------------ |_____| ------------." );
	lreply( msgcode, "| Username |   Up |   Megs |   Dn |   Megs | Ratio | Tagline            |" );
	lreply( msgcode, "|----------+------+--------+------+--------+-------+--------------------|" );

	seteuid( 0 );																			

  	sprintf (temppath, "%s/users", cf.datapath);
  	dirf = opendir (temppath);
  	if (!dirf)
   {
   	syslog (LOG_ERR, "[pheard] %s: userlist, error opening user dir: %m",
	      uf.name);
		seteuid( oldid );
	   reply( msgcode, "Error reading userfiles.  Aborted." );
      return;
   }
	
  	while (1)
   {
      dn = readdir (dirf);
      if (!dn)
			break;

      sprintf (temppath, "%s/users/%s", cf.datapath, dn->d_name);

      strncpy (username, dn->d_name, sizeof (username));

      stat (temppath, &st);

      if ( S_ISREG (st.st_mode) && st.st_uid != 99 && st.st_gid != 99 )
		{
		  	if ((dotptr = strchr (username, '.')) != NULL)
	   		*dotptr = '\0';
									 
			if ( load_userfile( username ) )
				syslog( LOG_ERR, "[pheard] Calling function was groupinfo(1)" );
				  
			if ( uf.ratio )
	   		sprintf( tempbuf, " 1:%1d ", uf.ratio );
			else
				strcpy( tempbuf, "leech" );

			if ( isingroup( whichgrp ) )
			{
				if ( uf.ratio == 0 )
				   leeches++;
			  	numusers++;		  
				
				lreply( msgcode, "| %-8.8s | %4d | %6.1f | %4d | %6.1f | %-5.5s | %-18.18s |",
					uf.name, uf.files_up, uf.bytes_up / 1024.0,
					uf.files_down, uf.bytes_down / 1024.0, tempbuf, uf.tagline );
					
				total_bytes_dn += uf.bytes_down;
				total_bytes_up += uf.bytes_up;
				total_files_dn += uf.files_down;
				total_files_up += uf.files_up;
			}
		}
	}

	if ( load_userfile( currentuser ) )
		syslog( LOG_ERR, "[pheard] Calling function was groupinfo(2)" );

	sprintf( dnbuf, "%d", total_files_dn );
	sprintf( upbuf, "%d", total_files_up );
	sprintf( numusersbuf, "%d", numusers );
	sprintf( leechesbuf, "%d", leeches );

	lreply( msgcode, "|----------+------+--------+------+--------+-------+--------------------|" );
	lreply( msgcode, "|   Tot Megs Up: %9.2f   Tot Files Up: %-6.6s   Users in group: %-3.3s |",
		total_bytes_up / 1024.0, upbuf, numusersbuf );
	lreply( msgcode, "|   Tot Megs Dn: %9.2f   Tot Files Dn: %-6.6s       With leech: %-3.3s |",
		total_bytes_dn / 1024.0, dnbuf, leechesbuf );
	lreply( msgcode, "`-----------------------------------------------------------------------'" );
											  
	
  	(void) closedir (dirf);

  	seteuid( oldid );

	lreply( msgcode, "" );
	reply( msgcode, "Command successful." );

	return;
}
/*-- end of groupinfo() ---------------------------------------------------*/

