#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <syslog.h>
#ifdef __linux__
#include <linux/unistd.h>
#endif
#include <sys/file.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <dirent.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>
#include "../include/userfile.h"
#include "../include/sysconfig.h"

extern void reply( int n, char *fmt, ... );
extern void lreply( int n, char *fmt, ... );
		


/***************************************************************************
	ADD_DUPE
	Writes a "dupe" entry to the appropriate file.
***************************************************************************/

void 
add_dupe( char *filename_to_add, int is_a_dir )
{
   int    fdes = -1;
   int    ct = 0;  
   char   fname_buff[ 64 ];
	struct tm *ptr;
	time_t timenow = time( NULL );
	char   work_buff[ MAXPATHLEN+1 ]; 
	char   tu_buff[ MAXPATHLEN+1 ]; 
	mode_t oldmask;
	char   last_three[ 4 ];
	uid_t  oldid;
	char   addfilename[ MAXPATHLEN+1 ];
	
#ifdef DEBUG
	syslog( LOG_INFO, "-Z- add_dupe()" );
	debugfn();
#endif
	
	strcpy( addfilename, filename_to_add );
	for ( ct = 0; ct < strlen( addfilename ); ct++ )
		addfilename[ ct ] = tolower( filename_to_add[ ct ] );		 
	
	/*
	 * Extract extension of file
	 */	
	last_three[ 0 ] = tolower( addfilename[ strlen( addfilename ) - 3 ] );
	last_three[ 1 ] = tolower( addfilename[ strlen( addfilename ) - 2 ] );
	last_three[ 2 ] = tolower( addfilename[ strlen( addfilename ) - 1 ] );
	last_three[ 3 ] = '\0';
	 
	/*
	 * Add only archive files, unless it's a directory
	 */
	if ( is_a_dir == 0 )
	{
   	if ( memcmp( last_three, "zip", 3 ) &&
        	  memcmp( last_three, "arj", 3 ) && 
           memcmp( last_three, "exe", 3 ) &&
           memcmp( last_three, "lha", 3 ) &&
           memcmp( last_three, "lzh", 3 ) )
        return;
	}
	
   /* Build filename to use */
	ptr = localtime( &timenow );
	strftime( fname_buff, 64, "%m-%d-%Y.dupes", ptr );

   /* Attempt to open the file */
	oldid = geteuid();
	seteuid( 0 );
   sprintf( work_buff, "%s/dupes/%s", cf.datapath, fname_buff );
  	oldmask = umask(0000);
   fdes = open( work_buff, O_CREAT | O_WRONLY | O_APPEND, 0666 );
  	umask( oldmask );
   if ( fdes < 0 )
   {
		seteuid( oldid );
      return;
   }

   /* Attempt to lock the file */
	ct = 0;
   while ( flock( fdes, LOCK_EX ) && ct < 20 ) 
   {
      ct++;
      syslog( LOG_ERR, "%s: flocking '%s': %m", uf.name, work_buff );
      sleep( 1 );
   }

   /* 20 seconds without lock, give up! */
   if ( ct == 10 )
   {
      syslog( LOG_ERR, "%s: flock failed: '%s'", uf.name, work_buff );
		seteuid( oldid );
      return;
   }

   if ( strrchr( addfilename, '/' ) != (char *)NULL )
      sprintf( work_buff, "%s", strrchr( addfilename, '/' ) + 1 );
   else
      sprintf( work_buff, "%s", addfilename );
		
	if ( is_a_dir )
		strcat( work_buff, "/" );		
					  					  
	/*
	 * Also save the username & time uploaded so we can rag on dupers
	 */		
	if ( !is_a_dir )
	{
		sprintf( tu_buff, "|%s|%ld\n", uf.name, time( NULL ) );
		strcat( work_buff, tu_buff );
	}
	else
	{
		strcat( work_buff, "\n" );
	}
		
   write( fdes, work_buff, strlen( work_buff ) );

   flock( fdes, LOCK_UN );
   close( fdes );
	
	seteuid( oldid );
	
   return;
}
/*-- end of add_dupe() ---------------------------------------------------*/

				
/***************************************************************************
	DUPECHK		
	Returns 0 on no dupe, 1 on a dupe.
	If an error is encountered, we return 0, giving the user the benefit
	of the doubt.
***************************************************************************/
int
dupechk(char *dupename, time_t datenfo)
{
	FILE *fp;
	char dupebuff[MAXPATHLEN + 1];
	char dizcheckbuff[MAXPATHLEN + 1];
	char datename[MAXPATHLEN + 1];
	char work_buff[MAXPATHLEN + 1];
	char dupefile[MAXPATHLEN + 1];
	char userbuff[30];		/* User who upped */
	char timebuff[40];		/* Time file was upped */
	time_t upped;		/* Actual numeric time uploaded */
	int cnt;
	struct tm *ptr;
	uid_t oldid;

#ifdef DEBUG
	syslog(LOG_INFO, "dupechk()");
	debugfn();
#endif

	/*
	 * Build filename to use
	 */
	ptr = localtime(&datenfo);
	strftime(datename, MAXPATHLEN, "%m-%d-%Y.dupes", ptr);
	sprintf(dupefile, "%s/dupes/%s", cf.datapath, datename);

	oldid = geteuid();
	seteuid(0);

	/*
	 * Open file for reading
	 */
	if ((fp = fopen(dupefile, "r")) == NULL) {
		if (errno != ENOENT)
		seteuid(oldid);
		return 0;
	}

	while (!feof(fp)) {
		if (fgets(dupebuff, 127, fp) == NULL && !feof(fp)) {
			syslog(LOG_ERR, "%s: reading '%s': %m", uf.name, dupefile);
			fclose(fp);
			seteuid(oldid);
			return 0;
		}

		if (feof(fp)) {
			fclose(fp);
			seteuid(oldid);
			return 0;
		}

		/*
		 * Strip newline
		 */
		dupebuff[strlen(dupebuff) - 1] = '\0';
		strcpy(work_buff, dupebuff);

		for (cnt = 0; cnt < MAXPATHLEN; cnt++)
			if (dupebuff[cnt] == '|') {
				dupebuff[cnt] = '\0';
				cnt = MAXPATHLEN;
			}

		/*
		 * Get name of user who upped this 
		 */
		if (strchr(work_buff, '|') != (char *) NULL)
			sprintf(userbuff, "%s", strchr(work_buff, '|') + 1);
		else
			strcpy(userbuff, "[UNKNOWN]");

		for (cnt = 0; cnt < 30; cnt++)
			if (userbuff[cnt] == '|')
				userbuff[cnt] = '\0';

		/*
		 * Get time uploaded
		 */		
		if (strrchr(work_buff, '|') != (char *) NULL)
			sprintf(timebuff, "%s", strrchr(work_buff, '|') + 1);
		else
			strcpy(timebuff, "0");

		upped = strtol(timebuff, (char **) NULL, 0);

		if (memcmp(userbuff, "[UNKNOWN]", 9) == 0)
			strcpy(timebuff, ".");
		else
			sprintf(timebuff, " !h(!E%ld!h minutes ago).", 
					(time_t) ((time(NULL) - upped)) / (time_t) 60);

		/*
		 * .message files shouldn't be checked for dupes.
		 */
		if (strlen(dupebuff) >= 8)
			if (strncmp(&dupebuff[strlen(dupebuff) - 8], ".message", 8) == 0) {
				seteuid(oldid);
				return 0;
			}

		/*
		 * file_id.diz files shouldn't be checked for dupes.
		 */
		if (strlen(dupebuff) >= 11) {
			/* create working copy */
			strcpy(dizcheckbuff, dupebuff);

			/* convert copy to all lower case */
			for (cnt = 0; cnt < sizeof(dizcheckbuff); cnt++)
				dizcheckbuff[cnt] = tolower(dizcheckbuff[cnt]);

			/* match filename */
			if (strncmp(&dizcheckbuff[strlen(dizcheckbuff) - 11], "file_id.diz",
					11) == 0) {
				seteuid(oldid);
				return 0;
			}
		}

		/*
		 * *.nfo files should not be checked for dupes.
		 */
		if (strlen(dupebuff) >= 4) {
			/* create working copy */
			strcpy(dizcheckbuff, dupebuff);

			/* convert last 4 chars to lower case */
			for (cnt = strlen(dizcheckbuff) - 4; cnt < strlen(dizcheckbuff); cnt++)
				dizcheckbuff[cnt] = tolower(dizcheckbuff[cnt]);

			if (strncmp(&dizcheckbuff[strlen(dizcheckbuff) - 4], ".nfo", 4) == 0) {
				seteuid(oldid);
				return 0;
			}
		}

		/*
		 * Is this a dupe?  If so, return 1.
		 */
		if (strcasecmp(dupebuff, dupename) == 0) {
/*			syslog(LOG_INFO, "%s: duped: %s", uf.name, dupename ); */
			lreply(553, "!H%s!h: !BThis file looks like a dupe!!!0", dupename);
			if (memcmp(userbuff, "[UNKNOWN]", 9) != 0)
				lreply(553, "!hIt was uploaded by %s%s!0", userbuff, timebuff);
			reply(553, " !FClean up empty directories!!!0");

			seteuid(oldid);
			return 1;
		}
	}

	fclose(fp);
	seteuid(oldid);
	return 0;
}
/*-- end of dupechk() ----------------------------------------------------*/


/***************************************************************************
	CHECK_DUPE
	This is the main dupe checking function.  It is passed the filename,
	and returns 1 if the file listed is a dupe, otherwise it returns 0.
	Note that the filename will be stripped from the pathname if a pathname
	is sent.
***************************************************************************/

int 
check_dupe( char *file_to_check )
{
   time_t curtime;							/* Current time */
   char   dnbuff[ MAXPATHLEN+1 ];  		/* File_to_check stripped of dir */
   int    cnt;									/* Counter */
	char   work_buff[ MAXPATHLEN+1 ];		/* Work buffer */
											 
#ifdef DEBUG
	syslog( LOG_INFO, "check_dupe()" );
	debugfn();
#endif
	
	/*
	 * Dupe checker can be DISABLED by setting dupe_check_days to 0
	 */
	if ( cf.dupe_check_days == 0 )
		return 0;
	
	/*		
    * Make "file_to_check" all-caps.
	 */
   strncpy( work_buff, file_to_check, sizeof( work_buff ) );
   for ( cnt = 0; cnt < strlen( work_buff ); cnt++ )
      work_buff[ cnt ] = tolower( work_buff[ cnt ] );
									  
	/* 
	 * We only check archive-type files, to avoid dupe busts on stuff like
	 * FILE_ID.DIZ...
	 */
   if ( strstr( work_buff, ".zip" ) == (char *)NULL &&
        strstr( work_buff, ".arj" ) == (char *)NULL && 
        strstr( work_buff, ".exe" ) == (char *)NULL &&
        strstr( work_buff, ".lha" ) == (char *)NULL &&
        strstr( work_buff, ".lzh" ) == (char *)NULL )
        return 0;
		  
	/*
	 * Strip any dir info, we just want the bare filename:
	 */		  
   if ( strrchr( file_to_check, '/' ) != (char *)NULL )
      sprintf( dnbuff, "%s", strrchr( file_to_check, '/' ) + 1 );
   else
      sprintf( dnbuff, "%s", file_to_check );
		
	/*
	 * Check the number of days required by system config
	 */		
	for ( cnt = 0; cnt < cf.dupe_check_days; cnt++ )
	{
	   time( &curtime );
   	curtime -= ( 86400 * cnt );
   	if ( dupechk( dnbuff, curtime ) )
      	return 1;      								/* A dupe!  */
	}

   return 0;												/* No dupe  */
}
/*-- end of check_dupe() -------------------------------------------------*/

							


/***************************************************************************
	DUPEDEL
	Returns 0 on no dupe, 1 on a dupe.
***************************************************************************/

int 
dupedel(char *dupename, time_t datenfo)
{
	FILE *fp;
	long position;
	char datename[MAXPATHLEN + 1];
	char dupefile[MAXPATHLEN + 1];
	char dupebuff[MAXPATHLEN + 1];
	struct tm *ptr;
	register int ct;
	uid_t oldid;

#ifdef DEBUG
	syslog(LOG_INFO, "dupedel()");
	debugfn();
#endif

	/*
	 * Build filename to use
	 */
	ptr = localtime(&datenfo);
	strftime(datename, MAXPATHLEN, "%m-%d-%Y.dupes", ptr);
	sprintf(dupefile, "%s/dupes/%s", cf.datapath, datename);

	/*
	 * Open file
	 */
	oldid = geteuid();
	seteuid(0);
	if ((fp = fopen(dupefile, "r+")) == NULL) {
		seteuid(oldid);
		return 0;
	}

	while (!feof(fp)) {
		position = ftell(fp);
		bzero(dupebuff, MAXPATHLEN);
		(void) fgets(dupebuff, MAXPATHLEN, fp);
		for (ct = 0; ct < strlen(dupebuff); ct++)
			dupebuff[ct] = tolower(dupebuff[ct]);

		if (strstr(dupebuff, dupename) != NULL) {		/* Found it!!! */
			fseek(fp, position, SEEK_SET);
			for (ct = 0; ct < strlen(dupename); ct++)
				dupename[ct] = 'X';
			fwrite(dupename, strlen(dupename), 1, fp);
			fclose(fp);
			seteuid(oldid);
			return 1;
		}
	}

	seteuid(oldid);
	return 0;
}
/*-- end of dupedel() ----------------------------------------------------*/



/***************************************************************************
	DELETE_DUPE
	This deletes a dupe from the database, if it can find it...
***************************************************************************/
	 
void
delete_dupe(char *file_to_delete)
{
	time_t curtime;		/* Current time */
	char dnbuff[MAXPATHLEN + 1];		/* File_to_check stripped of dir */
	int cnt;		/* Counter */
	char work_buff[MAXPATHLEN + 1];		/* Work buffer */

#ifdef DEBUG
	syslog(LOG_INFO, "delete_dupe()");
	debugfn();
#endif

	/*
	 * Dupe checker can be DISABLED by setting dupe_check_days to 0
	 */
	if (cf.dupe_check_days == 0)
		return;

	/*
	 * Make "file_to_delete" all-lower.
	 */
	strncpy(work_buff, file_to_delete, sizeof(work_buff));
	for (cnt = 0; cnt < strlen(work_buff); cnt++)
		work_buff[cnt] = tolower(work_buff[cnt]);

	/* 
	 * We only check archive-type files, to avoid dupe busts on stuff like
	 * FILE_ID.DIZ...
	 */
	/* should not matter what we're removing from dupe database--
	if (strstr(work_buff, ".zip") == (char *) NULL &&
			strstr(work_buff, ".arj") == (char *) NULL &&
			strstr(work_buff, ".exe") == (char *) NULL &&
			strstr(work_buff, ".lha") == (char *) NULL &&
			strstr(work_buff, ".lzh") == (char *) NULL)
		return;
	*/

	/*
	 * Strip any dir info, we just want the basename
	 */
	if (strrchr(work_buff, '/') != (char *) NULL)
		sprintf(dnbuff, "%s", strrchr(work_buff, '/') + 1);
	else
		sprintf(dnbuff, "%s", work_buff);

	/*
	 * Check the number of days required by system config
	 */
	for (cnt = 0; cnt < cf.dupe_check_days; cnt++) {
		time(&curtime);
		curtime -= (86400 * cnt);
		if (dupedel(dnbuff, curtime))
			break;		/* found match, we're done with for loop */
	}

	return;		/* No dupe */
}
/*-- end of delete_dupe() -------------------------------------------------*/

