{*******************************************************}
{                                                       }
{   Copyright (c) 1996 Classic Software                 }
{   All rights reserved                                 }
{                                                       }
{*******************************************************}

unit CSRankLB;

{$B-,P+,W-,X+}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls;

type
  TcsModifierKey = (mkShift, mkNone);

  { TcsRankListBox

    By Definition:
      Sorted = False

    Currently requires (these are ensured through read-only properties):
      Columns = 0
      DragMode = dmManual
      ExtendedSelect = False
      MultiSelect = False
  }
  TcsRankListBox = class(TListBox)
  private
    FModifierKey: TcsModifierKey; { only applies to dragging with mouse }
    FMoveOnDrag: Boolean;
    FMoving: Boolean;
    FOldIndex: Integer;
    FOldCursor: HCursor;
    FColumns: Integer; { used instead of inherited Columns }
    FDragCursor: TCursor; { used instead of inherited DragCursor }
    FDragMode: TDragMode; { used instead of inherited DragMode }
    FExtendedSelect: Boolean; { used instead of inherited ExtendedSelect }
    FMultiSelect: Boolean;  { used instead of inherited MultiSelect }
    FSorted: Boolean; { used instead of inherited Sorted }
    procedure SetMoveOnDrag(Value: Boolean);
  protected
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    { Define read-only properties to hide inherited properties }
    property Columns: Integer read FColumns default 0;  { Must be 0 }
    property DragCursor: TCursor read FDragCursor default crDrag;
    property DragMode: TDragMode read FDragMode default dmManual;  { Must be dmManual }
    property ExtendedSelect: Boolean read FExtendedSelect default False;  { Must be False }
    property MultiSelect: Boolean read FMultiSelect default False;  { Must be False }
    property Sorted: Boolean read FSorted default False; { Must be False }
    { New properties }
    property ModifierKey: TcsModifierKey read FModifierKey write FModifierKey default mkShift;
    property MoveOnDrag: Boolean read FMoveOnDrag write SetMoveOnDrag default True;
  end;

implementation

{$IFDEF EVALUATION} uses CSEval; {$ENDIF}

{ TcsRankListBox }

constructor TcsRankListBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FColumns := 0;
  FDragCursor := crDrag;
  FDragMode := dmManual;
  inherited ExtendedSelect := False;  { normally True for TListBox }
  FExtendedSelect := False;
  FMultiSelect := False;
  FSorted := False;
  FModifierKey := mkShift;
  FMoveOnDrag := True;
  FOldIndex := -1;
  FMoving := False;
end;

procedure TcsRankListBox.SetMoveOnDrag(Value: Boolean);
begin
  if FMoving then
    raise Exception.Create('MoveOnDrag property can''t be changed ' +
                           'while moving an item')
  else
    if (FMoveOnDrag <> Value) then
      FMoveOnDrag := Value;
end;

procedure TcsRankListBox.MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer);
begin
  inherited MouseDown(Button, Shift, X, Y);
  if (Button = mbLeft) and ((FModifierKey = mkNone) or ((FModifierKey = mkShift) and (ssShift in Shift))) then
  begin
    FOldIndex := ItemAtPos(Point(X, Y), True);
    if FOldIndex <> -1 then
    begin
      FOldCursor := WinProcs.GetCursor;
      WinProcs.SetCursor(Screen.Cursors[DragCursor]);
      FMoving := True;
    end;
  end;
end;

procedure TcsRankListBox.MouseMove(Shift: TShiftState; X, Y: Integer);
var NewIndex: Integer;
    ScreenPos: TPoint;
    Box: TRect;
    HitTop: Boolean;
begin
  inherited MouseMove(Shift, X, Y);
  if FMoving then
  begin
    HitTop := False;
    Box := ClientRect;

    if not PtInRect(Box, Point(X, Y)) then
    begin
      { Don't allow drag outside box; adjust position if outside box.
        Different adjustments necessary for right and bottom edges due
        to a point on either edge not being considered _inside_ the box
        when ItemAtPos() used.
      }
      if X < Box.Left then X := Box.Left
      else if X >= Box.Right then X := Box.Right - 1;
      if Y < Box.Top then
      begin
        Y := Box.Top;
        HitTop := True;
      end
      else if Y >= Box.Bottom then Y := Box.Bottom - 1;
      ScreenPos := ClientToScreen(Point(X, Y));
      SetCursorPos(ScreenPos.X, ScreenPos.Y);
    end;

    if FMoveOnDrag then
    begin
      NewIndex := ItemAtPos(Point(X, Y), True);
      if (NewIndex = -1) then
        { dragged above first item or below last item }
        if HitTop then NewIndex := 0
        else NewIndex := Items.Count - 1;

      if (NewIndex <> FOldIndex) then
      begin
        Items.Move(FOldIndex, NewIndex);
        ItemIndex := NewIndex;
        FOldIndex := ItemIndex;
      end;
    end;
  end;
end;

procedure TcsRankListBox.MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer);
var NewIndex: Integer;
begin
  inherited MouseUp(Button, Shift, X, Y);
  if FMoving then
  begin
    if not FMoveOnDrag then
    begin
      NewIndex := ItemAtPos(Point(X, Y), True);
      if (NewIndex = -1) then
        { dragged above first item or below last item }
        if Y < ClientRect.Top then NewIndex := 0
        else
         { IntegralHeight = False and dragged below last item into space }
          NewIndex := Items.Count - 1;

      if (NewIndex <> FOldIndex) then
      begin
        Items.Move(FOldIndex, NewIndex);
        ItemIndex := NewIndex;
      end;
    end;

    WinProcs.SetCursor(FOldCursor);
    FMoving := False;
  end;
end;

procedure TcsRankListBox.KeyDown(var Key: Word; Shift: TShiftState);
var NewIndex: Integer;
begin
  if (ssShift in Shift) and not FMoving then
  begin
    FOldIndex := ItemIndex;
    case Key of
      VK_DOWN:
      begin
        if ItemIndex < Items.Count - 1 then
        begin
          NewIndex := ItemIndex + 1;
          Items.Move(FOldIndex, NewIndex);
          ItemIndex := NewIndex;
          Key := 0;
        end;
      end;
      VK_UP:
      begin
        if ItemIndex > 0 then
        begin
          NewIndex := ItemIndex - 1;
          Items.Move(FOldIndex, NewIndex);
          ItemIndex := NewIndex;
          Key := 0;
        end;
      end;
      VK_HOME:
      begin
        if ItemIndex > 0 then
        begin
          NewIndex := 0;
          Items.Move(FOldIndex, NewIndex);
          ItemIndex := NewIndex;
          Key := 0;
        end;
      end;
      VK_END:
      begin
        if ItemIndex < Items.Count - 1 then
        begin
          NewIndex := Items.Count - 1;
          Items.Move(FOldIndex, NewIndex);
          ItemIndex := NewIndex;
          Key := 0;
        end;
      end;
    end;
  end;
  inherited KeyDown(Key, Shift);
end;

end.
