{***************************************************************}
{                                                               }
{   Copyright (c) 1996-1997 Classic Software                    }
{   All rights reserved                                         }
{                                                               }
{***************************************************************}

unit CSHRTime;

{$B-,P+,W-,X+}

interface

uses Classes, Messages, WinTypes, WinProcs, Forms, Controls, SysUtils,
      Consts, CSHRType;

type
  { The TcsCustomTimer component is the base class for timer components
    based on the multimedia timer services.
  }
  TcsCustomTimer = class(TComponent)
  private
    FEnabled: Boolean;
    FOneShot: Boolean;
    FInterval: UINT;
    FPeriod: UINT;
    FResolution: UINT;
    FTimerID: UINT;
    FOnTimer: TNotifyEvent;
    function GetDLLLoaded: Boolean;
    procedure SetEnabled(Value: Boolean);
    procedure SetInterval(Value: UINT);
    procedure SetOneShot(Value: Boolean);
    procedure SetOnTimer(Value: TNotifyEvent);
    procedure SetResolution(Value: UINT);
    procedure UpdateTimer;
  protected
    procedure Loaded; override;
    procedure Shutdown; virtual;
    function StartTimer(EventType: UINT): UINT; virtual; abstract;
    procedure Timer; dynamic;
    property DLLLoaded: Boolean read GetDLLLoaded;
    property Enabled: Boolean read FEnabled write SetEnabled default True;
    property Interval: UINT read FInterval write SetInterval default 100;
    property OneShot: Boolean read FOneShot write SetOneShot default False;
    property Resolution: UINT read FResolution write SetResolution default 100;
    property OnTimer: TNotifyEvent read FOnTimer write SetOnTimer;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  end;

  { The TcsHiResTimer component is similar to the standard TTimer component but
    can have a higher resolution and will not 'lose' timer event messages.
    It still however relies on messages going through the message queue so the
    precision of the timer events will vary depending on the current CPU load.
  }
  TcsHiResTimer = class(TcsCustomTimer)
  private
    FWindowHandle: HWND;
    procedure WndProc(var Msg: TMessage);
  protected
    procedure Shutdown; override;
    function StartTimer(EventType: UINT): UINT; override;
  public
    constructor Create(AOwner: TComponent); override;
    property DLLLoaded;
  published
    property Enabled;
    property Interval;
    property OneShot;
    property Resolution;
    property OnTimer;
  end;

{$IFDEF WIN32}
  { The TcsInterrupt component is probably about as close as you can get to
    a timer interrupt under Windows without using an actual device driver.
    Unlike the TcsHiResTimer component, TcsInterrupt doesn't rely on the
    message queue to send the timer events and thus will have even better
    precision than the TcsHiResTimer component but the precision will still
    depend on the current CPU load.
    Because the message queue is not used (which would normally stack up
    pending messages if a task is busy for a little while, e.g. while
    processing keyboard/mouse events) it is very important that the
    processing you do in the OnTimer handler takes less than the specified
    interval, otherwise you may cause the CPU load to reach 100% and you won't
    be able to do anything (perhaps even being unable to end the runaway task!).
  }
  TcsInterrupt = class(TcsCustomTimer)
  protected
    function StartTimer(EventType: UINT): UINT; override;
  public
    property DLLLoaded;
  published
    property Enabled;
    property Interval;
    property OneShot;
    property Resolution;
    property OnTimer;
  end;
{$ENDIF}

implementation

uses MMSystem {$IFDEF EVALUATION}, CSEval {$ENDIF};

{ Internal global variables, functions etc. }

{$IFDEF WIN32}
var TimerCallBackProc: TFNTimeCallBack;

procedure TimerCallBack(uTimerID, uMessage: UINT;
                        dwUser, dw1, dw2: DWORD);
                        stdcall;
begin
  PostMessage(HWND(dwUser), WM_HiResTIMER, uTimerID, 0);
end;

var InterruptCallBackProc: TFNTimeCallBack;

procedure InterruptCallBack(uTimerID, uMessage: UINT;
                            dwUser, dw1, dw2: DWORD);
                            stdcall;
begin
  { dwUser will contain a reference to an TcsInterrupt object (instance) }
  TcsInterrupt(dwUser).Timer;
end;

{$ELSE}
var
  RefCount: Integer;
  DLLHandle: THandle;
  TimerCallBackProc: TTimeCallBack;

procedure InitDLL;
var OldErrorMode: UINT;
begin
  if DLLHandle = 0 then
  begin
    OldErrorMode := SetErrorMode(SEM_NOOPENFILEERRORBOX);
    DLLHandle := LoadLibrary(CSTIME_DLL_NAME);
    SetErrorMode(OldErrorMode);
    if DLLHandle >= HINSTANCE_ERROR then
    begin
      @TimerCallBackProc := GetProcAddress(DLLHandle, 'CSHiResTimerCallBackProc');
      Inc(RefCount);
    end
    else
      DLLHandle := 0; { 0 indicates error }
  end
  else
    Inc(RefCount);
end;

procedure DoneDLL;
begin
  if RefCount > 0 then
    Dec(RefCount);
  if (RefCount = 0) and (DLLHandle <> 0) then
  begin
    FreeLibrary(DLLHandle);
    DLLHandle := 0;
  end;
end;
{$ENDIF}

function GetMinTimerPeriod: UINT;
var TimeCaps: TTimeCaps;
begin
  if timeGetDevCaps(@TimeCaps, SizeOf(TimeCaps)) = 0 then
    Result := TimeCaps.wPeriodMin
  else
    Result := 0;
end;

function GetMaxTimerPeriod: UINT;
var TimeCaps: TTimeCaps;
begin
  if timeGetDevCaps(@TimeCaps, SizeOf(TimeCaps)) = 0 then
    Result := TimeCaps.wPeriodMax
  else
    Result := 0;
end;

{ TcsCustomTimer }

constructor TcsCustomTimer.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FEnabled := True;
  FInterval := 100;
  FResolution := 100;
  FOneShot := False;
  FPeriod := 0;
  FTimerID := 0;
end;

destructor TcsCustomTimer.Destroy;
begin
  Shutdown;
  inherited Destroy;
end;

procedure TcsCustomTimer.Loaded;
begin
  inherited Loaded;
{$IFNDEF WIN32}
  InitDLL;
{$ENDIF}
  UpdateTimer;
end;

procedure TcsCustomTimer.Shutdown;
begin
  SetEnabled(False);
{$IFNDEF WIN32}
  DoneDLL;
{$ENDIF}
end;

function TcsCustomTimer.GetDLLLoaded: Boolean;
begin
{$IFDEF WIN32}
  Result := True;
{$ELSE}
  Result := (DLLHandle <> 0);
{$ENDIF}
end;

procedure TcsCustomTimer.UpdateTimer;
var ErrorCode, EventType: UINT;
begin
  if (FTimerID <> 0) then
  begin
    if (FOneShot and not FEnabled) or
      (timeKillEvent(FTimerID) = TIMERR_NOERROR) then
      FTimerID := 0;
  end;
  if (FPeriod <> 0) then
  begin
    if (timeEndPeriod(FPeriod) = TIMERR_NOERROR) then
      FPeriod := 0;
  end;
  if (FInterval <> 0) and FEnabled and Assigned(FOnTimer) then
  begin
    ErrorCode := timeBeginPeriod(FResolution);
    if ErrorCode = 0 then FPeriod := FResolution
    else FPeriod := 0;
    if FOneShot then EventType := TIME_ONESHOT
    else EventType := TIME_PERIODIC;
    if GetDLLLoaded then
    begin
      FTimerID := StartTimer(EventType);
      if FTimerID = 0 then
        raise EOutOfResources.Create(LoadStr(SNoTimers));
    end;
  end;
end;

procedure TcsCustomTimer.SetEnabled(Value: Boolean);
begin
  if Value <> FEnabled then
  begin
    FEnabled := Value;
    UpdateTimer;
  end;
end;

procedure TcsCustomTimer.SetOneShot(Value: Boolean);
begin
  if Value <> FOneShot then
  begin
    FOneShot := Value;
    UpdateTimer;
  end;
end;

procedure TcsCustomTimer.SetInterval(Value: UINT);
begin
  if Value <> FInterval then
  begin
    FInterval := Value;
    UpdateTimer;
  end;
end;

procedure TcsCustomTimer.SetOnTimer(Value: TNotifyEvent);
begin
  FOnTimer := Value;
  UpdateTimer;
end;

procedure TcsCustomTimer.SetResolution(Value: UINT);
var MinPeriod, MaxPeriod: UINT;
begin
  if Value <> FResolution then
  begin
    if FInterval < Value then FResolution := FInterval
    else FResolution := Value;
    MinPeriod := GetMinTimerPeriod;
    MaxPeriod := GetMaxTimerPeriod;
    if MinPeriod > FResolution then FResolution := MinPeriod
    else if MaxPeriod < FResolution then FResolution := MaxPeriod;
    UpdateTimer;
  end;
end;

procedure TcsCustomTimer.Timer;
begin
  if Assigned(FOnTimer) then FOnTimer(Self);
end;

{ TcsHiResTimer }

constructor TcsHiResTimer.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FWindowHandle := AllocateHWnd(WndProc);
end;

procedure TcsHiResTimer.Shutdown;
begin
  inherited Shutdown;
  DeallocateHWnd(FWindowHandle);
end;

function TcsHiResTimer.StartTimer(EventType: UINT): UINT;
begin
  Result := timeSetEvent(Interval, Resolution, TimerCallBackProc,
    FWindowHandle, EventType);
end;

procedure TcsHiResTimer.WndProc(var Msg: TMessage);
begin
  with Msg do
    if (Msg = WM_HiResTIMER) then
      try
        Timer;
        if FOneShot then SetEnabled(False);
      except
        Application.HandleException(Self);
      end
    else
      Result := DefWindowProc(FWindowHandle, Msg, wParam, lParam);
end;

{$IFDEF WIN32}
{ TcsInterrupt }

function TcsInterrupt.StartTimer(EventType: UINT): UINT;
begin
  { A reference to self is passed for later use by the callback procedure. }
  Result := timeSetEvent(Interval, Resolution, InterruptCallBackProc,
    Integer(Self), EventType);
end;
{$ENDIF}

begin
{$IFDEF WIN32}
  TimerCallBackProc := @TimerCallBack;
  InterruptCallBackProc := @InterruptCallBack;
{$ELSE}
  { Initialise internal global varrables;
    necessary because Delphi 1 doesn't allow
    initial values in global variable
    declarations.
  }
  RefCount := 0;
  DLLHandle := 0;
  TimerCallBackProc := nil;
{$ENDIF}
end.
