{*******************************************************}
{                                                       }
{   Copyright (c) 1996 Classic Software                 }
{   All rights reserved                                 }
{                                                       }
{*******************************************************}

unit CSGrid;

{$B-,P+,W-,X+}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Grids, CSStrTbl;

type
  TcsGrid = class(TDrawGrid)
  private
    FData: TcsStringTable;
    FUpdating: Boolean;
    FNeedsUpdating: Boolean;
    FEditUpdate: Integer;
    FDefaultDrawing: Boolean;
    FExtendedSelect: Boolean;
    FMultiSelect: Boolean;
    FExtendedStart: Integer;
    FExtendedFinish: Integer;
    FInDataCell: Boolean;
    procedure DisableEditUpdate;
    procedure EnableEditUpdate;
    function GetCell(ACol, ARow: Integer): String;
    function GetDataCell(ACol, ARow: Integer): String;
    function GetDataObject(ACol, ARow: Integer): TObject;
    function GetObject(ACol, ARow: Integer): TObject;
    procedure SetCell(ACol, ARow: Integer; Value: String);
    procedure SetDataCell(ACol, ARow: Integer; Value: String);
    procedure SetDataObject(ACol, ARow: Integer; Value: TObject);
    procedure SetExtendedSelect(Value: Boolean);
    procedure SetMultiSelect(Value: Boolean);
    procedure SetObject(ACol, ARow: Integer; Value: TObject);
    procedure SetRangeStatus(const RangeStart, RangeFinish: Integer; const AStatus: Boolean);
    procedure SetUpdateState(Updating: Boolean);
    { Update renamed to UpdateCell to prevent compiler warning:
      "Method 'Update' hides virtual method of base type 'TWinControl'"
      under Delphi 2
    }
    procedure UpdateCell(ACol, ARow: Integer);
  protected
    procedure ColumnMoved(FromIndex, ToIndex: Longint); override;
    procedure DrawCell(ACol, ARow: LongInt; ARect: TRect;
      AState: TGridDrawState); override;
    function GetEditText(ACol, ARow: Longint): string; override;
    function GetSelected(ARow: Integer): Boolean; virtual;
    procedure IncColCount; virtual;
    procedure IncRowCount; virtual;
    procedure InvalidateGrid; virtual;
    procedure InvalidateWholeRow(ARow: Longint); virtual;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure RowMoved(FromIndex, ToIndex: Longint); override;
    function SelectCell(ACol, ARow: Longint): Boolean; override;
    procedure SetEditText(ACol, ARow: Longint; const Value: string); override;
    procedure SetSelected(ARow: Integer; Value: Boolean); virtual;
    procedure SizeChanged(OldColCount, OldRowCount: Longint); override;
    procedure TopLeftChanged; override;
    procedure UpdateRowIndicator(ARow: LongInt); virtual;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure AddColumn; virtual;
    procedure AddRow; virtual;
    procedure ClearDataCells; virtual;
    procedure ClearSelected; virtual;
    procedure DeleteColumn(ACol: Integer); virtual;
    procedure DeleteRow(ARow: Integer); virtual;
    procedure InsertColumn(ACol: Integer); virtual;
    procedure InsertRow(ARow: Integer); virtual;
    procedure InvalidateRow(ARow: Integer); virtual;{ can't use override (not virtual in ancestor) }
    property Cells[ACol, ARow: Integer]: String read GetCell write SetCell;
    property Data: TcsStringTable read FData;
    property DataCells[ACol, ARow: Integer]: String read GetDataCell write SetDataCell;
    property DataObjects[ACol, ARow: Integer]: TObject read GetDataObject write SetDataObject;
    property Objects[ACol, ARow: Integer]: TObject read GetObject write SetObject;
    property Selected[ARow: Integer]: Boolean read GetSelected write SetSelected;
  published
    property DefaultDrawing: Boolean read FDefaultDrawing write FDefaultDrawing default True;
    property ExtendedSelect: Boolean read FExtendedSelect write SetExtendedSelect default False;
    property MultiSelect: Boolean read FMultiSelect write SetMultiSelect default False;
  end;

implementation

{$IFDEF EVALUATION} uses CSEval; {$ENDIF}

{ TcsGrid }

constructor TcsGrid.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FEditUpdate := 0;
  FData := TcsStringTable.Create(ColCount, RowCount);
  { default drawing for ancestor is handled by TcsGrid }
  inherited DefaultDrawing := False;
  FDefaultDrawing := True;
  FExtendedSelect := False;
  FMultiSelect := False;
  FExtendedStart := -1;
  FExtendedFinish := -1;
  FInDataCell := False;
end;

destructor TcsGrid.Destroy;
begin
  FData.Free;
  inherited Destroy;
end;

function TcsGrid.GetCell(ACol, ARow: Integer): String;
begin
  Result := FData.Cells[ACol, ARow];
end;

procedure TcsGrid.SetCell(ACol, ARow: Integer; Value: String);
begin
  FData.Cells[ACol, ARow] := Value;
  UpdateCell(ACol, ARow);
end;

{ DataCells[] applies to cells outside the fixed rows/cols and
  re-maps 0,0 to the origin of the data cells,
  i.e. DataCells[0,0] is equivalent to Cells[FixedCols, FixedRows].
}
function TcsGrid.GetDataCell(ACol, ARow: Integer): String;
begin
  Result := FData.Cells[ACol + FixedCols, ARow + FixedRows];
end;

procedure TcsGrid.SetDataCell(ACol, ARow: Integer; Value: String);
begin
  FData.Cells[ACol + FixedCols, ARow + FixedRows] := Value;
end;

function TcsGrid.GetObject(ACol, ARow: Integer): TObject;
begin
  Result := FData.Objects[ACol, ARow];
end;

procedure TcsGrid.SetObject(ACol, ARow: Integer; Value: TObject);
begin
  FData.Objects[ACol, ARow] := Value;
  UpdateCell(ACol, ARow);
end;

{ DataObjects[] applies to cells outside the fixed rows/cols and
  re-maps 0,0 to the origin of the data cells' objects,
  i.e. DataObjects[0,0] is equivalent to Objects[FixedCols, FixedRows].
}
function TcsGrid.GetDataObject(ACol, ARow: Integer): TObject;
begin
  Result := FData.Objects[ACol + FixedCols, ARow + FixedRows];
end;

procedure TcsGrid.SetDataObject(ACol, ARow: Integer; Value: TObject);
begin
  FData.Objects[ACol + FixedCols, ARow + FixedRows] := Value;
end;

procedure TcsGrid.InvalidateWholeRow(ARow: Longint);
var InvalidRect: TRect;
begin
  if not HandleAllocated then Exit;
  InvalidRect := BoxRect(0, ARow, ColCount - 1, ARow);
  WinProcs.InvalidateRect(Handle, @InvalidRect, False);
end;

procedure TcsGrid.InvalidateRow(ARow: Integer);
begin
  { The inherited InvalidateRow isn't used because it doesn't
    include partially visible columns in the area to be
    invalidated.
  }
  InvalidateWholeRow(ARow);
end;

procedure TcsGrid.ColumnMoved(FromIndex, ToIndex: LongInt);
begin
  FData.MoveColumn(FromIndex, ToIndex);
  Invalidate;
  inherited ColumnMoved(FromIndex, ToIndex);
end;

procedure TcsGrid.RowMoved(FromIndex, ToIndex: LongInt);
begin
  FData.MoveRow(FromIndex, ToIndex);
  Invalidate;
  inherited RowMoved(FromIndex, ToIndex);
end;

function TcsGrid.GetEditText(ACol, ARow: LongInt): string;
var Handler: TGetEditEvent;
begin
  Result := Cells[ACol, ARow];
  Handler := OnGetEditText;
  if Assigned(Handler) then Handler(Self, ACol, ARow, Result);
end;

procedure TcsGrid.SetEditText(ACol, ARow: LongInt; const Value: string);
begin
  DisableEditUpdate;
  try
    if Value <> Cells[ACol, ARow] then Cells[ACol, ARow] := Value;
  finally
    EnableEditUpdate;
  end;
  inherited SetEditText(ACol, ARow, Value);
end;


procedure TcsGrid.DisableEditUpdate;
begin
  Inc(FEditUpdate);
end;

procedure TcsGrid.EnableEditUpdate;
begin
  Dec(FEditUpdate);
end;

procedure TcsGrid.SetUpdateState(Updating: Boolean);
begin
  FUpdating := Updating;
  if not Updating and FNeedsUpdating then
  begin
    InvalidateGrid;
    FNeedsUpdating := False;
  end;
end;

procedure TcsGrid.UpdateCell(ACol, ARow: Integer);
begin
  if not FUpdating then InvalidateCell(ACol, ARow)
  else FNeedsUpdating := True;
  if (ACol = Col) and (ARow = Row) and (FEditUpdate = 0) then InvalidateEditor;
end;

procedure TcsGrid.InvalidateGrid;
begin
  Invalidate;
end;

procedure TcsGrid.SizeChanged(OldColCount, OldRowCount: Longint);
begin
  FData.ChangeSize(ColCount, RowCount); { new sizes }
  inherited SizeChanged(OldColCount, OldRowCount);
end;

procedure TcsGrid.SetSelected(ARow: Integer; Value: Boolean);
begin
  FData.Selected[ARow] := Value;
  InvalidateWholeRow(ARow);
end;

function TcsGrid.GetSelected(ARow: Integer): Boolean;
begin
  Result := FData.Selected[ARow];
end;

procedure TcsGrid.DrawCell(ACol, ARow: LongInt; ARect: TRect;
  AState: TGridDrawState);

  procedure DrawCellText;
  var
    Text: array[0..255] of Char;
  begin
    StrPCopy(Text, Cells[ACol, ARow]);
    ExtTextOut(Canvas.Handle, ARect.Left + 2, ARect.Top + 2, ETO_CLIPPED or
      ETO_OPAQUE, @ARect, Text, StrLen(Text), nil);
  end;

  procedure DrawCurrentRowIndicator;
  var OldBrushColor: TColor;
      OldPenColor: TColor;
      ArrowLeft, ArrowTop: Integer;
  begin
    OldPenColor := Canvas.Pen.Color;
    OldBrushColor := Canvas.Brush.Color;
    Canvas.Pen.Color := clBtnText;  { used for arrow outline }
    Canvas.Brush.Color := clBtnText; { used for inside of arrow }
    ArrowLeft := ARect.Right - 8; { arrow width is 6 (leave 2 for margin) }
    ArrowTop := ARect.Top + ((ARect.Bottom - ARect.Top - 11) div 2);  { arrow height is 11 }
    Canvas.Polygon([Point(ArrowLeft, ArrowTop),
                    Point(ArrowLeft + 5, ArrowTop + 5),
                    Point(ArrowLeft, ArrowTop + 10)]);
    Canvas.Pen.Color := OldPenColor;
    Canvas.Brush.Color := OldBrushColor;
  end;

begin
  if DefaultDrawing or (csDesigning in ComponentState) then
    with Canvas do
    begin
      Font := Self.Font;
      if ((not FMultiSelect) and
        (gdSelected in AState) and
        (not (gdFocused in AState) or
        ([goDrawFocusSelected, goRowSelect] * Options <> []))) or
        (FMultiSelect and (goRowSelect in Options) and
        (ACol >= FixedCols) and (ARow >= FixedRows) and
        Selected[ARow]) then
      begin
        Brush.Color := clHighlight;
        Font.Color := clHighlightText;
      end
      else
        if (gdFixed in AState) then Brush.Color := FixedColor
        else Brush.Color := Color;
      FillRect(ARect);
    end;

  if DefaultDrawing then DrawCellText;
  inherited DrawCell(ACol, ARow, ARect, AState);

  { now draw 3D effect for fixed cells and focus rect for edit cell }
  if DefaultDrawing and (gdFixed in AState) and
    Ctl3D then
    with Canvas do
    begin
      Pen.Color := clBtnHighlight;
      Polyline([Point(ARect.Left, ARect.Bottom - 1), ARect.TopLeft,
      Point(ARect.Right, ARect.Top)]);
    end;
  if DefaultDrawing and not (csDesigning in ComponentState) then
    if (gdFocused in AState) and
      ([goEditing, goAlwaysShowEditor] * Options <>
      [goEditing, goAlwaysShowEditor])
      and not (goRowSelect in Options) then
      DrawFocusRect(Canvas.Handle, ARect)
    else if FMultiSelect and (goRowSelect in Options) and
      (FixedCols > 0) and (ACol = FixedCols - 1) and
      (ARow = Row) then
      DrawCurrentRowIndicator;
end;

{ Determine if cell can be selected. }
function TcsGrid.SelectCell(ACol, ARow: Longint): Boolean;
begin
  Result := inherited SelectCell(ACol, ARow);
  if Result then
    UpdateRowIndicator(ARow);
end;

procedure TcsGrid.UpdateRowIndicator(ARow: LongInt);
const PrevRow: Integer = -1;
begin
  if FMultiSelect and (goRowSelect in Options) and
     (ARow <> PrevRow) and (FixedCols > 0) then
  begin { move current row indicator }
    InvalidateCell(FixedCols - 1, PrevRow);
    InvalidateCell(FixedCols - 1, ARow);
  end;
  PrevRow := ARow;
end;


procedure TcsGrid.SetRangeStatus(const RangeStart, RangeFinish: Integer; const AStatus: Boolean);
var I, RowMin, RowMax: Integer;
begin
  if (RangeFinish >= RangeStart) then
  begin
    RowMin := RangeStart;
    RowMax := RangeFinish;
  end
  else
  begin
    RowMin := RangeFinish;
    RowMax := RangeStart;
  end;
  for I := RowMin to RowMax do
    Selected[I] := AStatus;
end; { SetRangeStatus }

procedure TcsGrid.MouseDown(Button: TMouseButton; Shift: TShiftState;
     X, Y: Integer);
var MouseCol, MouseRow: LongInt;
    AnchorStatus: Boolean;
begin
  FInDataCell := False;
  MouseToCell(X, Y, MouseCol, MouseRow);
  if (MouseCol >= FixedCols) and (MouseRow >= FixedRows) and
    FMultiSelect and (goRowSelect in Options) then
  begin
      FInDataCell := True;
      if FExtendedSelect then
      begin
        if (FExtendedStart = -1) then { no selections made yet }
        begin
          if not ((ssCtrl in Shift) and (ssShift in Shift)) then
            Selected[MouseRow] := True;
          FExtendedStart := MouseRow;
          FExtendedFinish := MouseRow;
        end
        else if not ((ssCtrl in Shift) or (ssShift in Shift)) then
        begin
          ClearSelected;
          Selected[MouseRow] := True;
          FExtendedStart := MouseRow;
          FExtendedFinish := MouseRow;
        end
        else
        begin
          if (ssShift in Shift) and not (ssCtrl in Shift) then
          begin
            ClearSelected;
            FExtendedFinish := MouseRow;
            SetRangeStatus(FExtendedStart, FExtendedFinish, True)
          end
          else if (ssCtrl in Shift) and not (ssShift in Shift) then
          begin
            Selected[MouseRow] := not Selected[MouseRow];
            FExtendedStart := MouseRow;
            FExtendedFinish := MouseRow;
          end
          else { (ssCtrl in Shift) and (ssShift in Shift) }
          begin
            AnchorStatus := Selected[FExtendedStart];
            SetRangeStatus(FExtendedStart, FExtendedFinish, False);
            FExtendedFinish := MouseRow;
            SetRangeStatus(FExtendedStart, FExtendedFinish, AnchorStatus);
          end;
        end;
      end
      else { MultiSelect (and not ExtendedSelect) }
      begin
        Selected[MouseRow] := not Selected[MouseRow];
        FExtendedStart := MouseRow;
        FExtendedFinish := MouseRow;
      end;
  end;
  inherited MouseDown(Button, Shift, X, Y);
end;

procedure TcsGrid.MouseMove(Shift: TShiftState; X, Y: Integer);
var MouseCol, MouseRow: LongInt;
    AnchorStatus: Boolean;
begin
  if FInDataCell then
  begin
    MouseToCell(X, Y, MouseCol, MouseRow);
    if (ssLeft in Shift) and
      (MouseCol >= FixedCols) and
      FMultiSelect and (goRowSelect in Options) and FExtendedSelect and
      (FExtendedStart >= 0) then
    begin
      if (MouseRow < FixedRows) then
        MouseRow := FixedRows
      else if (MouseRow >= RowCount) then
        MouseRow := RowCount - 1;
      if (MouseRow <> FExtendedFinish) then
      begin
        AnchorStatus := Selected[FExtendedStart];
        SetRangeStatus(FExtendedStart, FExtendedFinish, False);
        FExtendedFinish := MouseRow;
        SetRangeStatus(FExtendedStart, FExtendedFinish, AnchorStatus);
      end;
    end;
  end;
  inherited MouseMove(Shift, X, Y);
end;

procedure TcsGrid.KeyDown(var Key: Word; Shift: TShiftState);
const PrevRow: Integer = -1;
var AnchorStatus: Boolean;
begin
  inherited KeyDown(Key, Shift);
  if FMultiSelect and (goRowSelect in Options) then
    if (Key = VK_SPACE) then
    begin
      if FExtendedSelect then
      begin { clear current selections and select current item only }
        ClearSelected;
        Selected[Row] := True;
      end
      else
        Selected[Row] := not Selected[Row]; { toggle }
    end
    else if FExtendedSelect then
      case Key of
        VK_UP, VK_DOWN, VK_PRIOR, VK_NEXT, VK_LEFT, VK_RIGHT, VK_HOME, VK_END:
        begin
          if (ssShift in Shift) then
          begin
            if (FExtendedStart >= 0) then
            begin
              AnchorStatus := Selected[FExtendedStart];
              SetRangeStatus(FExtendedStart, FExtendedFinish, False);
              FExtendedFinish := Row;
              SetRangeStatus(FExtendedStart, FExtendedFinish, AnchorStatus);
            end;
          end
          else if (Row <> PrevRow) or (FExtendedFinish <> FExtendedStart) then
          begin
            ClearSelected;
            Selected[Row] := True;
            FExtendedStart := Row;
            FExtendedFinish := Row;
          end;
        end;
      end;
  PrevRow := Row;
end;

procedure TcsGrid.ClearSelected;
var I: Integer;
begin
  if FData.SelectedRows.Count > 0 then
  begin
    for I := 0 to FData.SelectedRows.Count - 1 do
      InvalidateRow(StrToInt(FData.SelectedRows[I]));
    FData.ClearSelected;
  end;
end;

procedure TcsGrid.SetMultiSelect(Value: Boolean);
begin
  if (Value <> FMultiSelect) then
  begin
    FMultiSelect := Value;
    { simulate standard ListBox behaviour }
    ClearSelected;
    FExtendedStart := -1;
    FExtendedFinish := -1;
    if not FMultiSelect then
      Selected[Row] := True;
  end;
end;

procedure TcsGrid.SetExtendedSelect(Value: Boolean);
begin
  if (Value <> FExtendedSelect) then
  begin
    FExtendedSelect := Value;
    { simulate standard ListBox behaviour }
    ClearSelected;
    FExtendedStart := -1;
    FExtendedFinish := -1;
  end;
end;

procedure TcsGrid.TopLeftChanged;
begin
  inherited TopLeftChanged;
  InvalidateRow(Selection.Top);
end;

procedure TcsGrid.AddColumn;
begin
  IncColCount;
end;

procedure TcsGrid.IncColCount;
var OldWidth: Integer;
begin
  { the next line of code is only included to prevent the compiler
    warning: "'OldWidth' might not have been initialized" in Delphi 2
  }
  OldWidth := 0;
  if (goColSizing in Options) and (ColCount > 0) then
    OldWidth := ColWidths[ColCount - 1];
  ColCount := ColCount + 1;  { ChangeSize will adjust table cells }
  if (goColSizing in Options) and (ColCount > 1) then
    ColWidths[ColCount - 2] := OldWidth;
end;

procedure TcsGrid.AddRow;
begin
  IncRowCount;
end;

procedure TcsGrid.IncRowCount;
var OldHeight: Integer;
begin
  { the next line of code is only included to prevent the compiler
    warning: "'OldHeight' might not have been initialized" in Delphi 2
  }
  OldHeight := 0;
  if (goRowSizing in Options) and (RowCount > 0) then
    OldHeight := RowHeights[RowCount - 1];
  RowCount := RowCount + 1;  { ChangeSize will adjust table cells }
  if (goRowSizing in Options) and (RowCount > 1) then
    RowHeights[RowCount - 2] := OldHeight;
end;

procedure TcsGrid.DeleteColumn(ACol: Integer);
begin
  FData.DeleteColumn(ACol);
  ColCount := ColCount - 1;
end;

procedure TcsGrid.DeleteRow(ARow: Integer);
begin
  FData.DeleteRow(ARow);
  RowCount := RowCount - 1;
  Invalidate;
end;

procedure TcsGrid.InsertColumn(ACol: Integer);
begin
  FData.InsertColumn(ACol);
  IncColCount;
end;

procedure TcsGrid.InsertRow(ARow: Integer);
begin
  FData.InsertRow(ARow);
  IncRowCount;
end;

procedure TcsGrid.ClearDataCells;
var I, J: Integer;
begin
  ClearSelected;
  { blank all data cells }
  for I := FixedCols to ColCount - 1 do
    for J := FixedRows to RowCount - 1 do
      SetCell(I, J, '');
  Invalidate;
end;

end.
