{*******************************************************}
{                                                       }
{   Copyright (c) 1996-1997 Classic Software            }
{   All Rights Reserved                                 }
{                                                       }
{*******************************************************}

unit CSDBDtEd;

{$B-,P+,W-,X+}
{$IFDEF WIN32}
{$H+}
{$ENDIF}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, DBCtrls;

type
  TcsDBDateEdit = class(TDBEdit)
  private
    FUseEpoch: Boolean;
    procedure CMExit(var Message: TCMExit); message CM_EXIT;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property UseEpoch: Boolean read FUseEpoch write FUseEpoch default True;
  end;

{ AddEpochCentury has been made a separate public function so
  it is able to be used outside this unit.
}
function AddEpochCentury(const DateString: String): String;

{ The Epoch setting specifies the starting year for a 100-year
  period in which all dates containing only two year digits are
  assumed to fall when using AddEpochCentury.
  The default value is the century of the current date.
  If the year digits in a date string are greater than or equal to
  the year of the Epoch value, the date is assumed to fall within
  the same century as the century of the Epoch value.
  Otherwise, the date is assumed to fall in the century after that
  of the Epoch setting.

  For example:  Epoch  Year  Full Year
                -----  ----  ---------
                1900    00     1900
                1900    05     1905
                1900    96     1996
                1950    00     2000
                1950    05     2005
                1950    96     1996
}
var Epoch: Integer;

implementation

uses DB, DBTables {$IFDEF EVALUATION}, CSEval {$ENDIF};

{ This CurrentYear function is Delphi version independent;
  the one defined in the implementation section of SysUtils
  is different for each Delphi version.
}
function CurrentYear: Word;
var Y, M, D: Word;
begin
  DecodeDate(Date, Y, M, D);
  Result := Y;
end;

{ The date conversion routines (Scan... and GetDateOrder) come from the
  implementation section of SysUtils.
}
type
  TDateOrder = (doMDY, doDMY, doYMD);

procedure ScanBlanks(const S: string; var Pos: Integer);
var
  I: Integer;
begin
  I := Pos;
  while (I <= Length(S)) and (S[I] = ' ') do Inc(I);
  Pos := I;
end;

function ScanNumber(const S: string; var Pos: Integer;
  var Number: Word): Boolean;
var
  I: Integer;
  N: Word;
begin
  Result := False;
  ScanBlanks(S, Pos);
  I := Pos;
  N := 0;
  while (I <= Length(S)) and (S[I] in ['0'..'9']) and (N < 1000) do
  begin
    N := N * 10 + (Ord(S[I]) - Ord('0'));
    Inc(I);
  end;
  if I > Pos then
  begin
    Pos := I;
    Number := N;
    Result := True;
  end;
end;

function ScanChar(const S: string; var Pos: Integer; Ch: Char): Boolean;
begin
  Result := False;
  ScanBlanks(S, Pos);
  if (Pos <= Length(S)) and (S[Pos] = Ch) then
  begin
    Inc(Pos);
    Result := True;
  end;
end;

function GetDateOrder(const DateFormat: string): TDateOrder;
var
  I: Integer;
begin
  Result := doMDY;
  I := 1;
  while I <= Length(DateFormat) do
  begin
    case Chr(Ord(DateFormat[I]) and $DF) of
      'Y': Result := doYMD;
      'M': Result := doMDY;
      'D': Result := doDMY;
    else
      Inc(I);
      Continue;
    end;
    Break;
  end;
end;

{ This function has been changed from the original in SysUtils so
  that it uses the current Epoch value to determine the century of
  the date.
}
function ScanDate(const S: string; var Pos: Integer;
  var ADate: TDateTime): Boolean;
var
  DateOrder: TDateOrder;
  N1, N2, N3, Y, M, D: Word;
begin
  Result := False;
  { the next three assignments aren't actually necessary; they are
    only included to prevent the (Delphi 2) compiler from generating
    warnings that they might not have been initialized
  }
  Y := 0;
  M := 0;
  D := 0;
  DateOrder := GetDateOrder(ShortDateFormat);
  if not (ScanNumber(S, Pos, N1) and ScanChar(S, Pos, DateSeparator) and
    ScanNumber(S, Pos, N2)) then Exit;
  if ScanChar(S, Pos, DateSeparator) then
  begin
    if not ScanNumber(S, Pos, N3) then Exit;
    case DateOrder of
      doMDY: begin Y := N3; M := N1; D := N2; end;
      doDMY: begin Y := N3; M := N2; D := N1; end;
      doYMD: begin Y := N1; M := N2; D := N3; end;
    end;
    if Y <= 99 then
      if Y < (Epoch mod 100) then Inc(Y, ((CurrentYear div 100) + 1) * 100)
      else Inc(Y, (CurrentYear div 100) * 100);
  end else
  begin
    Y := CurrentYear;
    if DateOrder = doDMY then
    begin
      D := N1; M := N2;
    end else
    begin
      M := N1; D := N2;
    end;
  end;
  ScanChar(S, Pos, DateSeparator);
  ScanBlanks(S, Pos);
  Result := True;
  try
    ADate := EncodeDate(Y, M, D);
  except
    Result := False;
  end;
end;

{ Add the century digits for the current epoch to the date string }
function AddEpochCentury(const DateString: String): String;
var Pos: Integer;
    NewDate: TDateTime;
    DateFormat: String;
begin
  Pos := 1;
  if ScanDate(DateString, Pos, NewDate) and (Pos > Length(DateString)) then
  begin
    { Can't use pre-defined constants for DateFormat because the
      DateSeparator can be changed at run-time.
    }
    case GetDateOrder(ShortDateFormat) of
      doDMY: DateFormat := 'd' + DateSeparator + 'm' + DateSeparator + 'yyyy';
      doMDY: DateFormat := 'm' + DateSeparator + 'd' + DateSeparator + 'yyyy';
      doYMD: DateFormat := 'yyyy' + DateSeparator + 'm' + DateSeparator + 'd';
    end;
    Result := FormatDateTime(DateFormat, NewDate);
  end
  else Result := DateString;
end;

{ TcsDBDateEdit }

constructor TcsDBDateEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FUseEpoch := True;
end;

procedure TcsDBDateEdit.CMExit(var Message: TCMExit);
begin
  if ((Field is TDateField) and FUseEpoch and (DataSource.State in [dsEdit, dsInsert])) then
    Text := AddEpochCentury(Text);
  inherited;
end;

initialization
  { default epoch is the current century }
  Epoch := (CurrentYear div 100) * 100;

end.
