{*******************************************************}
{                                                       }
{   Copyright (c) 1996 Classic Software                 }
{   All rights reserved                                 }
{                                                       }
{*******************************************************}

unit CSADEdit;

{$B-,P+,W-,X+}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Buttons, Grids, ExtCtrls, CSADMain;

type
  TcsAutoDefaultsEditDlg = class(TForm)
    IgnoreBtn: TButton;
    HelpBtn: TBitBtn;
    CloseBtn: TBitBtn;
    ComponentPanel: TPanel;
    ColumnHeadings: THeader;
    CompGrid: TDrawGrid;
    AddBtn: TButton;
    SetBtn: TButton;
    ClassPanel: TPanel;
    Label1: TLabel;
    ClassLB: TListBox;
    DeleteBtn: TButton;
    DeleteAllBtn: TButton;
    Label2: TLabel;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure CompGridDrawCell(Sender: TObject; Col, Row: Longint;
      Rect: TRect; State: TGridDrawState);
    procedure ColumnHeadingsSized(Sender: TObject; ASection,
      AWidth: Integer);
    procedure AddBtnClick(Sender: TObject);
    procedure CompGridDblClick(Sender: TObject);
    procedure SetBtnClick(Sender: TObject);
    procedure IgnoreBtnClick(Sender: TObject);
{$IFDEF NEVER}
    procedure CompGridKeyPress(Sender: TObject; var Key: Char);
{$ENDIF}
    procedure DeleteBtnClick(Sender: TObject);
    procedure DeleteAllBtnClick(Sender: TObject);
  private
    { Private declarations }
    FOldHelpFile: String;
    FDesignForm: TForm;
    FDefaultsComponent: TcsAutoDefaults;
    FCompList: TStringList;
    FModified: Boolean;
    procedure SetDefaultsComponent(Value: TcsAutoDefaults);
    procedure SetDesignForm(Value: TForm);
    procedure FillClassLB;
  public
    { Public declarations }
    property Modified: Boolean read FModified write FModified;
    property DefaultsComponent: TcsAutoDefaults read FDefaultsComponent write SetDefaultsComponent;
    property DesignForm: TForm read FDesignForm write SetDesignForm;
  end;

var
  csAutoDefaultsEditDlg: TcsAutoDefaultsEditDlg;

implementation

{$R *.DFM}

uses StrEdit;

const
  hcEditAutoDefaultsDlgBox = 4;

procedure TcsAutoDefaultsEditDlg.FormCreate(Sender: TObject);
begin
  FModified := False;
  FOldHelpFile := Application.HelpFile;  { must restore when done }
  Application.HelpFile := 'CSCCS.HLP';
  HelpContext := hcEditAutoDefaultsDlgBox;
  with CompGrid do
  begin
    Col := 0;
    Row := 0;
    RowCount := 0;
    DefaultRowHeight := Canvas.TextHeight('W') {$IFDEF WIN32} + 4 {$ENDIF};
    DefaultColWidth := ClientWidth;
  end;
  FDesignForm := nil;
  FCompList := TStringList.Create;
  FCompList.Sorted := True;
end;

procedure TcsAutoDefaultsEditDlg.FormDestroy(Sender: TObject);
begin
  Application.HelpFile := FOldHelpFile;
  FDesignForm := nil;
  { Components in FCompList.Objects[] are owned by the form; don't free }
  FCompList.Free;
end;

procedure TcsAutoDefaultsEditDlg.SetDefaultsComponent(Value: TcsAutoDefaults);
begin
  FDefaultsComponent := Value;
  FillClassLB;
  ShowHint := FDefaultsComponent.ShowHint;
end;

procedure TcsAutoDefaultsEditDlg.CompGridDrawCell(Sender: TObject; Col,
  Row: Longint; Rect: TRect; State: TGridDrawState);
var DC: THandle;
    PText: array[0..255] of Char;
begin
  { draw cell for this row/col }
  if (Row < FCompList.Count) then
  begin
    DC := CompGrid.Canvas.Handle;
    { Clip text to specified region }
    Rect.Right := Rect.Left + ColumnHeadings.SectionWidth[0];

    { Component Name }
    ExtTextOut(DC, Rect.Left + 2, Rect.Top + 2, ETO_CLIPPED,
      @Rect, StrPCopy(PText, FCompList[Row]), Length(FCompList[Row]), nil);

    { Component Class }
    Rect.Left := Rect.Right + 1;
    Rect.Right := Rect.Left + ColumnHeadings.SectionWidth[1];
    ExtTextOut(DC, Rect.Left + 2, Rect.Top + 2, ETO_CLIPPED,
      @Rect, StrPCopy(PText, FCompList.Objects[Row].ClassName),
      Length(FCompList.Objects[Row].ClassName), nil);
  end;
end;

procedure TcsAutoDefaultsEditDlg.SetDesignForm(Value: TForm);
var I: Integer;
begin
  FDesignForm := Value;
  FCompList.Clear;
  if FDesignForm <> nil then
  begin
    { build sorted list of all components on the form }
    for I := 0 to FDesignForm.ComponentCount - 1 do
      FCompList.AddObject(FDesignForm.Components[I].Name, FDesignForm.Components[I]);
    CompGrid.RowCount := FDesignForm.ComponentCount;
    CompGrid.Repaint;
  end;
end;

procedure TcsAutoDefaultsEditDlg.ColumnHeadingsSized(Sender: TObject; ASection,
  AWidth: Integer);
begin
  CompGrid.Repaint;
end;

{ Save properties for component currently highlighted in grid }
procedure TcsAutoDefaultsEditDlg.AddBtnClick(Sender: TObject);
var Component: TComponent;
begin
  Component := TComponent(FCompList.Objects[CompGrid.Row]);
  DefaultsComponent.SaveAsDefault(Component);
  CompGrid.Repaint;
  FillClassLB;
end;


procedure TcsAutoDefaultsEditDlg.CompGridDblClick(Sender: TObject);
begin
  AddBtnClick(Self);
end;

procedure TcsAutoDefaultsEditDlg.SetBtnClick(Sender: TObject);
var Component: TComponent;
begin
  Component := TComponent(FCompList.Objects[CompGrid.Row]);
  DefaultsComponent.ApplyDefaults(Component);
end;

procedure TcsAutoDefaultsEditDlg.IgnoreBtnClick(Sender: TObject);
begin
  with TStrEditDlg.Create(Application) do
  try
    Caption := 'Properties to ignore (in <class>.<property> format)';
    DefaultsComponent.GetIgnoreOptions(Memo.Lines);
    UpdateStatus(nil);
    ActiveControl := Memo;
    if ShowModal = idOk then
      DefaultsComponent.SetIgnoreOptions(Memo.Lines);
    { changing Ignore Options doesn't change anything on form,
      i.e. Modified is unchanged
    }
  finally
    Free;
  end;
  CompGrid.SetFocus;
end;

{$IFDEF NEVER}
{ This will only work as expected when the key being pressed doesn't
  correspond to an accelerator key for one of the buttons!
}
procedure TcsAutoDefaultsEditDlg.CompGridKeyPress(Sender: TObject;
  var Key: Char);
var FindFirst: Boolean;
    Pos: Integer;
begin
  FindFirst := False;
  Pos := CompGrid.Row;
  if (AnsiCompareText(FCompList[Pos][1], Key) = 0) then
  begin
    { move to next item and see if it starts with same character }
    if (Pos < CompGrid.RowCount) and
       (AnsiCompareText(FCompList[Pos + 1][1], Key) = 0) then
      Inc(Pos)
    else
      FindFirst := True;
  end
  else
    FindFirst := True;

  if FindFirst then
    Pos := FCompList.IndexOf(Key);
    if Pos < 0 then
      { Not found; check using alternate (lower/upper) case }
      if Key = LowerCase(Key) then
        { Key is lower-case character; check for match using upper-case }
        Pos := FCompList.IndexOf(UpperCase(Key))
      else
        { Key is upper-case character; check for match using lower-case }
        Pos := FCompList.IndexOf(LowerCase(Key));

  if Pos >= 0 then
    CompGrid.Row := Pos;
end;
{$ENDIF}


{ Allows for multiple/extended selection of classes whose defaults are
  to be deleted.
}
procedure TcsAutoDefaultsEditDlg.DeleteBtnClick(Sender: TObject);
{var Component: TComponent; }
var I: Integer;
begin
  if (ClassLB.Items.Count > 0) and (ClassLB.SelCount > 0) then
  begin
    if (ClassLB.SelCount = ClassLB.Items.Count) and (ClassLB.SelCount > 1) then
      DeleteAllBtnClick(Sender)
    else
      for I := 0 to ClassLB.Items.Count - 1 do
        if ClassLB.Selected[I] then
          DefaultsComponent.ClearDefaults(FindClass(ClassLB.Items[I]));
    FillClassLB;
  end;
end;

procedure TcsAutoDefaultsEditDlg.DeleteAllBtnClick(Sender: TObject);
begin
  if (ClassLB.Items.Count > 0) then
  begin
    if (Application.MessageBox('Are you sure you want to delete all defaults?',
                               'TcsAutoDefaults',
                               MB_ICONQUESTION + MB_YESNOCANCEL) = idYes) then
    begin
      DefaultsComponent.ClearDefaults(nil);
      FillClassLB;
    end;
  end;
end;

procedure TcsAutoDefaultsEditDlg.FillClassLB;
var List: TStringList;
begin
  List := TStringList.Create; { this will be freed by ClassLB }
  DefaultsComponent.GetClassesWithDefaults(List);
  ClassLB.Items := List;
  DeleteBtn.Enabled := (ClassLB.Items.Count > 0);
  DeleteAllBtn.Enabled := (ClassLB.Items.Count > 0);
end;

end.
