/*
 * mental ray shader interface header file, version 1.9.
 * (C) Copyright 1995 by mental images GmbH & Co. KG
 */

#ifndef _SHADER_H_
#define _SHADER_H_

/* data types */

#ifndef PI
#define PI 3.14159265358979323846
#endif

#ifdef _WIN32
#define DLLEXPORT __declspec(dllexport)	/* symbols to be exported to ray */
#ifndef _MENTAL_RAY_
#define DLLIMPORT 			/* symbols to be imported from ray */
#else
#define DLLIMPORT __declspec(dllexport) /* seen from ray, they get exported */
#endif
#else /* !_WIN32 */
#define DLLEXPORT
#define DLLIMPORT
#endif

#ifdef __STDC__
#include <stdlib.h>	/* for drand48 */
#define Pr(args) args
#else
#define Pr(args) ()
#endif
#ifdef IRIX
#define MI_SQRT(A) sqrtf(A)
#else
#define MI_SQRT(A) ((miScalar) sqrt(A))
#endif

typedef enum {miFALSE=0, miTRUE=1}	miBoolean;
typedef unsigned char			miCBoolean;	/* if space is tight */
typedef int				miInteger;	/* for shader params */
typedef char				miSint1;
typedef unsigned char			miUint1;
typedef short				miSint2;
typedef unsigned short			miUint2;
typedef int				miSint4;
typedef unsigned int			miUint4;
typedef unsigned int			miUint;
typedef unsigned char			miUchar;
typedef unsigned short			miUshort;
typedef unsigned int			miUlong;	/* avoid 64-bit longs*/
typedef char				miSchar;
typedef float				miScalar;
typedef struct {miScalar u, v;}		miVector2d;
#ifdef _MENTAL_RAY_
typedef Vector				miVector;
#else
typedef struct {miScalar x, y, z;}	miVector;
#endif
typedef struct {double   x, y, z;}	miDVector;
typedef miScalar			miMatrix[16];
#ifdef _MENTAL_RAY_
typedef Color				miColor;
#else
typedef struct {float r, g, b, a;}	miColor;
#endif
typedef struct {miScalar min, max;}	miRange;
typedef unsigned short			miIndex;
typedef miBoolean		      (*miFunction_ptr)();

typedef int				miModule;
typedef char			       *miLock;
typedef void			       *miTag;

typedef enum {
	miIMG_TYPE_RGBA		= 0,	/* 4*8  bits, color with alpha */
	miIMG_TYPE_RGBA_16	= 1,	/* 4*16 bits, color with alpha */
	miIMG_TYPE_RGB		= 2,	/* 4*8  bits, color */
	miIMG_TYPE_RGB_16	= 3,	/* 4*16 bits, color */
	miIMG_TYPE_A		= 4,	/* 1*8  bits, alpha */
	miIMG_TYPE_A_16		= 5,	/* 1*16 bits, alpha */
	miIMG_TYPE_S		= 6,	/* 1*8  bits, intensity */
	miIMG_TYPE_S_16		= 7,	/* 1*16 bits, intensity */
	miIMG_TYPE_VTA		= 8,	/* 2*16 bits, bumpmap from alpha */
	miIMG_TYPE_VTS		= 9,	/* 2*16 bits, bumpmap from intensity */
	miIMG_TYPE_Z		= 10,	/* 1*32 bits, Z coordinates (float) */
	miIMG_TYPE_N		= 11,	/* 3*32 bits, normal vectors (float) */
	miIMG_TYPE_TAG		= 12,	/* 1*32 bits, material tags (u long) */
	miIMG_TYPE_RGBA_FP	= 13,	/* 4*32 bits, color with alpha, float*/
	miIMG_NTYPES		= 14,
	miIMG_TYPE_ANY		= 99	/* img_open: don't convert */
} miImg_type;


typedef struct miCamera {
    miBoolean	orthographic;		/* orthographic rendering? */
    float	focal;			/* focal length */
    float	aperture;		/* aperture */
    float	aspect;			/* aspect ratio of the image */
    miRange	clip;			/* min/max clipping */
    int		x_resolution;		/* x resolution */
    int		y_resolution;		/* y resolution */
    struct {
        int    	xl;			/* window: lower left x */
        int    	yl;			/* window: lower left y */
        int    	xh;			/* window: size x */
        int    	yh;			/* window: size y */
    }     	window;
    miTag	volume;			/* opt. volume shader */
    miTag	environment;		/* opt. environment shader */
    miTag	lens;			/* opt. lens shader */
    int		frame;			/* current frame number */
    float	frame_time;		/* frame number as time in seconds */
} miCamera;


typedef struct {
	void		*internal1;	/* internal to ray */
	void		*internal2;	/* internal to ray */
	void		*internal3;	/* internal to ray */
	char		*parameters;	/* shader parameters */
	miLock		lock;		/* function lock */
	void		*internal4;	/* internal to ray */
} miFunction;

#define miRC_MAX_IMAGES		16		/* max. number of images */

/*
 * Options for RC.
 */

enum miRc_face {				/* facing of primitives */
	miRC_FACE_FRONT = 1,			/* only front */
	miRC_FACE_BACK,				/* only back */
	miRC_FACE_BOTH				/* both */
};

enum miRc_field {				/* field rendering */
	miRC_FIELD_OFF = 0,			/* no field rendering */
	miRC_FIELD_EVEN,			/* even fields */
	miRC_FIELD_ODD				/* odd fields */
};

enum miRc_sampling {				/* image sampling */
	miRC_SAMPLING_RECURSIVE,		/* recursive */
	miRC_SAMPLING_CONSTANT,			/* constant */
	miRC_SAMPLING_ADAPTIVE			/* two level adaptive */
};

enum miRc_filter {				/* sample filter function */
	miRC_FILTER_BOX,			/* box filter */
	miRC_FILTER_TRIANGLE,			/* triangle (tent) filter */
	miRC_FILTER_GAUSS			/* gauss curve filter */
};

enum miRc_acceleration {			/* acceleration technique */
	miRC_ACCEL_RAY_CLASSIFICATION,		/* ray classification */
	miRC_ACCEL_SPACE_SUBDIVISION		/* binary space partition */
};

enum miRc_paper_size {				/* contour paper size */
	miRC_PAPER_SIZE_11X17,
	miRC_PAPER_SIZE_B,
	miRC_PAPER_SIZE_LETTER,
	miRC_PAPER_SIZE_A,
	miRC_PAPER_SIZE_A3,
	miRC_PAPER_SIZE_A4,
	miRC_PAPER_SIZE_A5,
	miRC_PAPER_SIZE_A6,
	miRC_PAPER_SIZE_B4,
	miRC_PAPER_SIZE_B5,
	miRC_PAPER_SIZE_B6,
	miRC_PAPER_SIZE_EXECUTIVE,
	miRC_PAPER_SIZE_LEGAL
};

typedef struct miRc_options {			/* options to RC */
						/* the comments state
						 * the default values */
	miBoolean		shadow;		/* shadow casting? : miTRUE */
	miBoolean		trace;		/* 2nd generation ray trace? */
						/* miTRUE */
	miBoolean		scanline;	/* 1st generation scanline? */
						/* miTRUE */
	miBoolean		shadow_sort;	/* sorted shadow calls? FALSE */
	miBoolean		contour;	/* contour rendering? */
						/* miFALSE */
	miBoolean		motion;		/* motion blur? : miFALSE */
	enum miRc_sampling	sampling;	/* image sampling */
						/* miRC_SAMPLING_RECURSIVE */
	enum miRc_filter	filter;		/* sample filter type : box */
	enum miRc_acceleration	acceleration;	/* acceleration */
					/* miRC_ACCEL_RAY_CLASSIFICATION */
	enum miRc_face		face;		/* primitive facing */
						/* miRC_FACE_BOTH */
	enum miRc_field		field;		/* field rendering? */
						/* miRC_FIELD_OFF */
	int			reflection_depth; /* refl. trace depth : 1*/
	int			refraction_depth; /* refr. trace depth : 1 */
	int			trace_depth;	/* refl. + refr. depth : 1 */
	int			min_samples;	/* min. sampling level : -2 */
	int			max_samples;	/* max. sampling level : 0 */
	miColor			contrast;	/* sampling contrast */
						/* 0.1 0.1 0.1 0.1 */
	miColor			time_contrast;	/* temporal sampling contrast */
						/* 0.1 0.1 0.1 0.1 */
	float			filter_size_x;	/* filter size in x : 1.0 */
	float			filter_size_y;	/* filter size in y : 1.0 */
	float			jitter;		/* sample jittering : 0.0 */
	float			shutter;	/* shutter speed for motion */
						/* 0 */
	int			subdivision;	/* ray class. adjustment : 0 */
	int			subdivision_2d;	/* eye + shadow adjustment:0 */
	int			subdivision_memory; /* ray class. memory (mb)*/
						/* 6 */
	int			space_max_size;	/* space subdiv. leaf size: 4*/
	int			space_max_depth;/* space subdiv. depth : 25*/
	miImg_type		image_types[miRC_MAX_IMAGES]; /* image types */
						/* miIMG_TYPE_RGBA, ...ANY...*/
	miBoolean		write_image[miRC_MAX_IMAGES]; /* write image?*/
						/* miTRUE ... */
	miBoolean		interp_image[miRC_MAX_IMAGES]; /* int. image?*/
						/* miTRUE, miFALSE ... */
	int			no_images;	/* # images : 1 */
						/* contour options: */
	enum miRc_paper_size	paper_size;	/* miRC_PAPER_SIZE_A4 */
	float			paper_scale;	/* 1.0 */
	float			contour_lnwidth_min;	/* 0.5 */
	float			contour_lnwidth_max;	/* -1.0 */
	float			contour_lnwidth_zmid;	/* -1.0 */
	float			contour_discontinuity;	/* 60.0 */
	float			contour_zdelta;	/* 0.0 */
	float			paper_transform_b;	/* 0.0 */
	float			paper_transform_d;	/* 1.0 */

	miBoolean		preview_mode;		/* miTRUE if shaderball */
	miBoolean		shadow_segments;/* shadow segments? FALSE */
} miRc_options;



typedef enum miRay_type {
	miRAY_EYE,		/* eye ray */
	miRAY_TRANSPARENT,	/* transparency ray */
	miRAY_REFLECT,		/* reflection ray */
	miRAY_REFRACT,		/* refraction ray */
	miRAY_LIGHT,		/* light ray */
	miRAY_SHADOW,		/* shadow ray */
	miRAY_ENVIRONMENT,	/* ray only into environment and volume */
	miRAY_NONE,		/* other ray */
	miRAY_NO_TYPES
} miRay_type;

/*
 * Determine if a ray given by miRay_type is an eye or first generation
 * transparency ray (primary), or a reflection or refraction (secondary) ray.
 */

#define miRAY_PRIMARY(r)	((r) <= miRAY_TRANSPARENT)
#define miRAY_SECONDARY(r)	((r) >  miRAY_TRANSPARENT)


/*
 * Types of shaders
 */

typedef enum {
	miSHADER_LENS,		/* lens shader */
	miSHADER_MATERIAL,	/* material shader */
	miSHADER_LIGHT,		/* light shader */
	miSHADER_SHADOW,	/* shadow shader */
	miSHADER_ENVIRONMENT,	/* environment shader */
	miSHADER_VOLUME,	/* volume shader */
	miSHADER_TEXTURE,	/* texture shader */
	miSHADER_OTHER,		/* shader not known by RC */
	miSHADER_NO_TYPES
} miShader_type;


/*
 * state structure passed to all shaders except output shaders
 */

typedef struct miState {
					    /* global */
	int		version;		/* version of state struct,
						 * and shader interface
						 * currently 2 */
	miCamera	*camera;		/* camera */
	miRc_options	*options;		/* options */
	miScalar	*world_transform;	/* world transformation */
	miScalar	*inv_world_transform;	/* inverse world transformation
						 */
	miScalar	*screen_transform;	/* screen transformation */
	miScalar	*inv_screen_transform;	/* inverse screen transformation
						 */
	miScalar	*raster_transform;	/* raster space transformation
						 */
	miScalar	*inv_raster_transform;	/* inverse raster space
						 * transformation */
	miLock		global_lock;		/* global lock for shaders */
	void		*accel;			/* acceleration data */
					    /* sample */
	float		raster_x;		/* raster x coordinate */
	float		raster_y;		/* raster y coordinate */
					    /* shader */
	miFunction	*shader;		/* current shader parameters */
					    /* ray */
	struct miState	*parent;		/* state of parent ray
						 * first eye ray/lens shader:
						 *   NULL
						 * subsequent eye rays/lens
						 * shaders: previous one
						 * reflection, refraction,
						 * dissolve rays: parent ray
						 * light rays: parent ray
						 * shadow rays: light ray
						 */
	miRay_type	type;			/* type of the ray */
	miCBoolean	contour;		/* contour rendering?
						 * allows to return
						 * transparency and skip
						 * illumination */
	miCBoolean	scanline;		/* intersect ray by scanline? */
	void		*cache;			/* cache holding the beam
						 * of the ray
						 * for light and shadow rays
						 * it's a shadow cache */
	void		*active;		/* active pri for scanline */
	int		reflection_level;	/* reflection depth of ray */
	int		refraction_level;	/* refraction depth of ray */
	miVector	org;			/* ray origin
						 * for light,shadow ray it's the
						 * light position */
	miVector	dir;			/* ray direction
						 * for shadow ray it points
						 * towards the intersection */
	double		dist;			/* length of the ray */
	float		time;			/* time of the ray */
	miTag		volume;			/* volume shader to be applied
						 */
	miTag		environment;		/* environment shader to be
						 * applied if no hit */
					    /* intersection point */
	miTag		refraction_volume;	/* volume shader to apply
						 * to refraction ray */
	unsigned int	label;			/* tag of hit object */
	miTag		instance;		/* instance */
	miScalar	*object_transform;	/* transformation of object
						 */
	miScalar	*inv_object_transform;	/* inverse transformation of
						 * object
						 */
	void		*pri;			/* hit Primitive * */
	double		bary[4];		/* barycentric coordinates
						 * of hit primitive
						 * triangles: 0 - 2,
						 * imp. patches: 0 - 3
						 */
	void		*material;		/* material of hit pri */
	miVector	point;			/* point of intersection */
	miVector	normal;			/* interpolated normal
						 * pointing to the side of the
						 * ray */
	miVector	normal_geom;		/* geometric normal of pri.
						 * pointing to the side of the
						 * ray */
	miScalar	dot_nd;			/* dot of normal, dir
						 * for light rays,
						 * it's the dot_nd of the
						 * parent (intersection) normal
						 * and the light ray dir */
	double		shadow_tol;		/* the minimum distance to a
						 * plane perpendicular to the
						 * shading normal for which the
						 * triangle is completely on
						 * the negative side. */
	miVector	*tex_list;		/* list of texture coordinates
						 */
	miVector	*bump_x_list;		/* list of bump map basis x
						 * (perturbation) vectors */
	miVector	*bump_y_list;		/* list of bump map basis y
						 * (perturbation) vectors */
	miVector	motion;			/* motion vector or 0 */
	miVector	u_deriv;		/* U surface derivative or 0 */
	miVector	v_deriv;		/* V surface derivative or 0 */
					    /* texture shader */
	miVector	tex;			/* texture coordinates for
						 * lookup */
					    /* light shader */
						/* instance points to
						 * light instance */
					    /*  shadow shader */

					    /* user */
	void		*user;			/* user data */
	int		user_size;		/* size of user data */
					    /* late additions */
	miScalar	ior;			/* ior for outgoing medium */
	miScalar	ior_in;			/* ior for incoming medium */
	miBoolean	inv_normal;		/* normals are inverted because
						 * ray hit the back side */
} miState;


/*
 * state structure passed to output shaders. The state argument must be
 * declared as "miOutstate *state;".
 */

typedef union {					/* Component order is */
	unsigned char	*c[4];			/* R-G-B-A, or NX-NY-NZ. */
	unsigned short	*s[4];
	unsigned int	*l[4];
	float		*f[4];
} *miImg_image;

typedef struct {			    /* state for output shaders */
	int		xres, yres;		/* image resolution */
	miImg_image	*frame_rgba;		/* RGBA color frame buffer */
	miImg_image	*frame_z;		/* depth channel frame buffer*/
	miImg_image	*frame_n;		/* normal vector frame buffer*/
	miImg_image	*frame_label;		/* label channel frame buffer */
	struct miCamera	*camera;		/* camera */
	struct miRc_options *options;		/* options */
	miMatrix	camera_to_world;	/* world transformation */
	miMatrix	world_to_camera;	/* inverse world transform */
} miOutstate;



/*
 * vector operations
 */

#define mi_vector_neg(r)	((r)->x = -(r)->x,\
				 (r)->y = -(r)->y,\
				 (r)->z = -(r)->z)

#define mi_vector_add(r,a,b)	((r)->x = (a)->x + (b)->x,\
				 (r)->y = (a)->y + (b)->y,\
				 (r)->z = (a)->z + (b)->z)

#define mi_vector_sub(r,a,b)	((r)->x = (a)->x - (b)->x,\
				 (r)->y = (a)->y - (b)->y,\
				 (r)->z = (a)->z - (b)->z)

#define mi_vector_mul(r,f)	((r)->x *= (f),\
				 (r)->y *= (f),\
				 (r)->z *= (f))

#ifdef __STDC__
#define mi_vector_div(r,_f)	do { if ((_f) != 0.0f) {\
					register miScalar _i = 1.0f/(_f);\
						   (r)->x *= _i;\
						   (r)->y *= _i;\
						   (r)->z *= _i; } } while (0)
#else
#define mi_vector_div(r,_f)	do { if ((_f) != 0.0) {\
					register miScalar _i = 1.0/(_f);\
						   (r)->x *= _i;\
						   (r)->y *= _i;\
						   (r)->z *= _i; } } while (0)
#endif

#define mi_vector_div_d(r,_f)	do { if ((_f) != 0.0) {\
					register double _i = 1.0/(_f);\
						   (r)->x *= _i;\
						   (r)->y *= _i;\
						   (r)->z *= _i; } } while (0)

#define mi_vector_prod(r,a,b)	((r)->x = (a)->y * (b)->z - (a)->z * (b)->y,\
				 (r)->y = (a)->z * (b)->x - (a)->x * (b)->z,\
				 (r)->z = (a)->x * (b)->y - (a)->y * (b)->x)

#define mi_vector_dot(a,b)	((a)->x * (b)->x +\
				 (a)->y * (b)->y +\
				 (a)->z * (b)->z)

#define mi_vector_dot_d(a,b)	((double)(a)->x * (double)(b)->x +\
				 (double)(a)->y * (double)(b)->y +\
				 (double)(a)->z * (double)(b)->z)

				/* sqrtf doesn't work if K&R-compiled */
#define mi_vector_norm(r)	MI_SQRT((r)->x * (r)->x +\
					(r)->y * (r)->y +\
					(r)->z * (r)->z)

#define mi_vector_norm_d(r)	sqrt((double)(r)->x * (double)(r)->x +\
				     (double)(r)->y * (double)(r)->y +\
				     (double)(r)->z * (double)(r)->z)

#define mi_vector_normalize(r)	do { register miScalar _n = mi_vector_norm(r);\
				     mi_vector_div((r), _n); } while (0)

#define mi_vector_normalize_d(r) do { register double _n= mi_vector_norm_d(r);\
				     mi_vector_div_d((r), _n); } while (0)

#define mi_vector_min(r,a,b)	((a)->x < (b)->x ? ((r)->x = (a)->x) \
						 : ((r)->x = (b)->x),\
				 (a)->y < (b)->y ? ((r)->y = (a)->y) \
						 : ((r)->y = (b)->y),\
				 (a)->z < (b)->z ? ((r)->z = (a)->z) \
						 : ((r)->z = (b)->z))

#define mi_vector_max(r,a,b)	((a)->x > (b)->x ? ((r)->x = (a)->x) \
						 : ((r)->x = (b)->x),\
				 (a)->y > (b)->y ? ((r)->y = (a)->y) \
						 : ((r)->y = (b)->y),\
				 (a)->z > (b)->z ? ((r)->z = (a)->z) \
						 : ((r)->z = (b)->z))

#define mi_vector_det(a,b,c)	((a)->x * ((b)->y * (c)->z - (c)->y * (b)->z)+\
				 (b)->x * ((c)->y * (a)->z - (a)->y * (c)->z)+\
				 (c)->x * ((a)->y * (b)->z - (b)->y * (a)->z))

				/* sqrtf doesn't work if K&R-compiled */
#define mi_vector_dist(a,b)	MI_SQRT(((a)->x - (b)->x) * ((a)->x - (b)->x)+\
					((a)->y - (b)->y) * ((a)->y - (b)->y)+\
					((a)->z - (b)->z) * ((a)->z - (b)->z))
#ifndef _MENTAL_RAY_
#endif /* !_MENTAL_RAY_ */


/*
 * matrix operations
 */

DLLIMPORT void	  mi_matrix_ident	Pr((miMatrix));
DLLIMPORT void	  mi_matrix_prod	Pr((miMatrix, miMatrix, miMatrix));
DLLIMPORT miBoolean mi_matrix_invert	Pr((miMatrix, miMatrix));
DLLIMPORT void	  mi_point_transform	Pr((miVector *const, miVector *const,\
					    miMatrix));
DLLIMPORT void	  mi_vector_transform	Pr((miVector *const, miVector *const,\
					    miMatrix));
DLLIMPORT void	  mi_matrix_rotate	Pr((miMatrix, const double,\
					    const double, const double));
DLLIMPORT void	 mi_point_to_world	Pr((miState * const, miVector * const,\
					    miVector * const));
DLLIMPORT void	 mi_point_to_camera	Pr((miState * const, miVector * const,\
					    miVector * const));
DLLIMPORT void	 mi_point_to_object	Pr((miState * const, miVector * const,\
					    miVector * const));
DLLIMPORT void	 mi_point_from_world	Pr((miState * const, miVector * const,\
					    miVector * const));
DLLIMPORT void	 mi_point_from_camera	Pr((miState * const, miVector * const,\
					    miVector * const));
DLLIMPORT void	 mi_point_from_object	Pr((miState * const, miVector * const,\
					    miVector * const));
DLLIMPORT void	 mi_vector_to_world	Pr((miState * const, miVector * const,\
					    miVector * const));
DLLIMPORT void	 mi_vector_to_camera	Pr((miState * const, miVector * const,\
					    miVector * const));
DLLIMPORT void	 mi_vector_to_object	Pr((miState * const, miVector * const,\
					    miVector * const));
DLLIMPORT void	 mi_vector_from_world	Pr((miState * const, miVector * const,\
					    miVector * const));
DLLIMPORT void	 mi_vector_from_camera	Pr((miState * const, miVector * const,\
					    miVector * const));
DLLIMPORT void	 mi_vector_from_object	Pr((miState * const, miVector * const,\
					    miVector * const));


/*
 * init/exit, spline, random-number, and 3D noise functions
 */

#define   mi_random()		(miScalar)drand48()

#ifdef __hpux
double drand48();	/* workaround for HP/UX compiler weirdness */
#endif
#ifdef _WIN32
DLLIMPORT double drand48();
#endif

DLLIMPORT double    mi_spline		Pr((double, const int,\
					    miScalar * const));
DLLIMPORT double    mi_noise_1d		Pr((const double));
DLLIMPORT double    mi_unoise_1d	Pr((const double));
DLLIMPORT double    mi_noise_2d		Pr((const double, const double));
DLLIMPORT double    mi_unoise_2d	Pr((const double, const double));
DLLIMPORT double    mi_noise_3d		Pr((miVector * const));
DLLIMPORT double    mi_unoise_3d	Pr((miVector * const));
DLLIMPORT double    mi_noise_1d_grad	Pr((const double, miScalar * const));
DLLIMPORT double    mi_unoise_1d_grad	Pr((const double, miScalar * const));
DLLIMPORT double    mi_noise_2d_grad	Pr((const double, const double,\
					    miScalar *const, miScalar *const));
DLLIMPORT double    mi_unoise_2d_grad	Pr((const double, const double,\
					    miScalar *const, miScalar *const));
DLLIMPORT double    mi_noise_3d_grad	Pr((miVector *const, miVector *const));
DLLIMPORT double    mi_unoise_3d_grad	Pr((miVector *const, miVector *const));



/*
 * memory allocation and database access
 */

DLLIMPORT void	*mi_mem_allocate	Pr((const int));
DLLIMPORT void	*mi_mem_reallocate	Pr((void * const, const int));
DLLIMPORT void	 mi_mem_release		Pr((void * const));
DLLIMPORT void	 mi_mem_check		Pr((void));
DLLIMPORT void	 mi_mem_dump		Pr((const miModule));

DLLIMPORT int	 mi_db_type		Pr((miTag tag));
DLLIMPORT void	*mi_db_access		Pr((miTag tag));
DLLIMPORT void	 mi_db_unpin		Pr((miTag tag));
DLLIMPORT void	 mi_db_flush		Pr((miTag tag));


/*
 * locking and threads
 */

DLLIMPORT void atomic_init		Pr((miLock * const));
DLLIMPORT void atomic_lock		Pr((const miLock));
DLLIMPORT void atomic_clear		Pr((const miLock));
DLLIMPORT int  get_no_procs		Pr((void));
DLLIMPORT int  get_no_threads		Pr((void));

#define  mi_init_lock			atomic_init
#define  mi_delete_lock(l)
#define  mi_lock			atomic_lock
#define  mi_unlock			atomic_clear
#define  mi_par_nthreads		get_no_threads
#define  mi_par_localvpu()		*(int *)state->accel
#define  miTHREAD(v)			((v) & 0xffff)


/*
 * image access. Do not use functions that don't begin with mi_ directly!
 */

DLLIMPORT void	 mi_img_put_color     Pr((miImg_image*, miColor  *, int, int));
DLLIMPORT void	 mi_img_get_color     Pr((miImg_image*, miColor  *, int, int));
DLLIMPORT void	 mi_img_put_depth     Pr((miImg_image*, float     , int, int));
DLLIMPORT void	 mi_img_get_depth     Pr((miImg_image*, float    *, int, int));
DLLIMPORT void	 mi_img_put_normal    Pr((miImg_image*, miVector *, int, int));
DLLIMPORT void	 mi_img_get_normal    Pr((miImg_image*, miVector *, int, int));
DLLIMPORT void	 mi_img_put_label     Pr((miImg_image*, miUint    , int, int));
DLLIMPORT void	 mi_img_get_label     Pr((miImg_image*, miUint   *, int, int));
DLLIMPORT void	 mi_img_get_scalar    Pr((miImg_image*, float	 *, int, int));
DLLIMPORT void	 mi_img_put_scalar    Pr((miImg_image*, float     , int, int));
DLLIMPORT void	 mi_img_get_vector    Pr((miImg_image*, miVector *, int, int));
DLLIMPORT void	 mi_img_put_vector    Pr((miImg_image*, miVector *, int, int));


/*
 * rendering functions
 */

DLLIMPORT miBoolean mi_trace_eye	Pr((miColor *, miState*, miVector *,\
					    miVector*));
DLLIMPORT miBoolean mi_trace_reflection	Pr((miColor *, miState *, miVector *));
DLLIMPORT miBoolean mi_trace_refraction	Pr((miColor *, miState *, miVector *));
DLLIMPORT miBoolean mi_trace_transparent Pr((miColor *, miState *));
DLLIMPORT miBoolean mi_trace_environment Pr((miColor *, miState *,miVector *));
DLLIMPORT miBoolean mi_trace_shadow	Pr((miColor *, miState *));
DLLIMPORT miBoolean mi_trace_shadow_seg Pr((miColor *, miState *));
DLLIMPORT miBoolean mi_trace_light	Pr((miColor *, miVector*, miScalar *,\
					    miState *, miTag));
DLLIMPORT miBoolean mi_sample_light	Pr((miColor *, miVector *, miScalar *,\
					    miState *, miTag, int *));

DLLIMPORT double  mi_fresnel		Pr((double, double, double, double));
DLLIMPORT double  mi_fresnel_reflection Pr((miState *, double, double));
DLLIMPORT double  mi_phong_specular	Pr((double, miState *, miVector *));
DLLIMPORT double  mi_blinn_specular	Pr((double, miState *, miVector *));
DLLIMPORT void	  mi_fresnel_specular	Pr((miScalar *, miScalar *, double,\
					    miState*, miVector *, double,\
					    double));
DLLIMPORT void	  mi_reflection_dir	Pr((miVector *, miState *));
DLLIMPORT int	  mi_refraction_dir	Pr((miVector *, miState *, double,\
					    double));

DLLIMPORT miBoolean mi_call_shader	Pr((miColor *, miShader_type,miState*,\
					    miTag));

DLLIMPORT void	  mi_light_info		Pr((miTag, miVector *, miVector *,\
					    void **));
DLLIMPORT void	  mi_texture_info	Pr((miTag, int *, int *, void **));
DLLIMPORT miBoolean mi_lookup_color_texture
				Pr((miColor  *, miState *, miTag, miVector*));
DLLIMPORT miBoolean mi_lookup_scalar_texture
				Pr((miScalar *, miState *, miTag, miVector*));
DLLIMPORT miBoolean mi_lookup_vector_texture
				Pr((miVector *, miState *, miTag, miVector*));

DLLIMPORT miBoolean mi_tri_vectors	Pr((miState*, int, int,\
					    miVector**,miVector**,miVector**));

/*
 * error functions (only mi_fatal doesn't return)
 */

#ifndef _MENTAL_RAY_
#define os_error mi_fatal	/* for mi_init_lock only, DO NOT USE! */
#endif

#if defined(IRIX6) && !defined(__STDC__)  /* for IRIX6 K&R vararg functions */
#include <varargs.h>
DLLIMPORT extern void mi_fatal		(va_list, ...);
DLLIMPORT extern void mi_error		(va_list, ...);
DLLIMPORT extern void mi_warning	(va_list, ...);
DLLIMPORT extern void mi_info		(va_list, ...);
DLLIMPORT extern void mi_progress	(va_list, ...);
DLLIMPORT extern void mi_debug		(va_list, ...);
DLLIMPORT extern void mi_vdebug		(va_list, ...);
#else
/* #ifndef _MENTAL_RAY_ */
DLLIMPORT void	 mi_fatal		Pr((const char * const, ...));
DLLIMPORT void	 mi_error		Pr((const char * const, ...));
DLLIMPORT void	 mi_warning		Pr((const char * const, ...));
DLLIMPORT void	 mi_info		Pr((const char * const, ...));
DLLIMPORT void	 mi_progress		Pr((const char * const, ...));
DLLIMPORT void	 mi_debug		Pr((const char * const, ...));
DLLIMPORT void	 mi_vdebug		Pr((const char * const, ...));
/* #endif */
#endif

#endif /* !_SHADER_H_ */
