//	Zinc Application Framework - Z_TEXT.HPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#if !defined(Z_TEXT_HPP)
#define Z_TEXT_HPP
#include <z_str1.hpp>

// --------------------------------------------------------------------------
// ----- ZafText ------------------------------------------------------------
// --------------------------------------------------------------------------

class ZAF_EXPORT ZafText : public ZafWindow
{
public:
	// --- General members ---
	ZafText(int left, int top, int width, int height,
		ZafStringData *stringData = ZAF_NULLP(ZafStringData));
	ZafText(int left, int top, int width, int height,
		const ZafIChar *text, int maxLength);
	ZafText(const ZafText &copy);
	virtual ~ZafText(void);

	// --- Persistent members ---
	ZafText(const ZafIChar *name, ZafObjectPersistence &persist);
	static ZafElement *Read(const ZafIChar *name, ZafObjectPersistence &persist);
	virtual void Write(ZafObjectPersistence &persist);

	// --- Attributes & data access ---
	// get
	bool AutoClear(void) const;
	int CursorOffset(void) const;
	ZafPositionStruct CursorPosition(void) const;
	ZafHzJustify HzJustify(void) const;
	bool Invalid(void) const;
	ZafStringData *StringData(void) const;
	bool Unanswered(void) const;
	bool ViewOnly(void) const;
	bool WordWrap(void) const;
	// set
	virtual bool SetAutoClear(bool autoClear);
	virtual ZafError SetCursorOffset(int position);
	virtual ZafHzJustify SetHzJustify(ZafHzJustify hzJustify);
	virtual bool SetInvalid(bool invalid);
	virtual ZafError SetStringData(ZafStringData *string);
	virtual ZafError SetCursorPosition(ZafPositionStruct position);
	virtual bool SetUnanswered(bool unanswered);
	virtual bool SetViewOnly(bool viewOnly);
	virtual bool SetWordWrap(bool wrappedData);

	// --- Window members ---
	virtual bool SetDestroyable(bool destroyable);
	virtual bool SetLocked(bool locked);
	virtual bool SetMaximized(bool maximized);
	virtual bool SetMinimized(bool minimized);
	virtual bool SetMoveable(bool moveable);
	virtual ZafSelectionType SetSelectionType(ZafSelectionType selectionType);
	virtual bool SetSizeable(bool sizeable);
	virtual bool SetTemporary(bool temporary);

	// --- Window object members ---
	virtual ZafEventType Event(const ZafEventStruct &event);
	ZafLogicalEvent LogicalEvent(const ZafEventStruct &event);
	virtual bool SetCopyDraggable(bool copyDraggable);
	virtual bool SetLinkDraggable(bool linkDraggable);
	virtual bool SetMoveDraggable(bool moveDraggable);
	virtual bool SetParentDrawBorder(bool parentDrawBorder);
	virtual ZafError SetText(const ZafIChar *text);
	virtual bool SetVisible(bool visible);
	virtual const ZafIChar *Text(void);

	// --- List members ---
	virtual ZafWindowObject *Add(ZafWindowObject *object, ZafWindowObject *position = ZAF_NULLP(ZafWindowObject));
	virtual ZafWindowObject *Subtract(ZafWindowObject *object);

	ZafText &operator+(ZafWindowObject *object) { Add(object); return(*this); }
	ZafText &operator-(ZafWindowObject *object) { Subtract(object); return(*this); }

	// --- Class identification ---
	static ZafClassID classID;
	static ZafClassNameChar ZAF_FARDATA className[];
	virtual ZafClassID ClassID(void) const { return (classID); }
	virtual ZafClassName ClassName(void) const { return (className); }
	virtual bool IsA(ZafClassID compareID) const { return ((compareID == classID) ? true : ZafWindow::IsA(compareID)); }
	virtual bool IsA(ZafClassName compareName) const { return ((compareName == className) ? true : ZafWindow::IsA(compareName)); }

	// --- Special OS members ---
	virtual OSWindowID OSScreenID(ZafScreenIDType type = ZAF_SCREENID) const;

protected:
	friend class ZAF_EXPORT OSText;

	// --- General members ---
	bool autoClear;
	bool invalid;
	bool unanswered;
	bool viewOnly;
	bool wordWrap;
	ZafHzJustify hzJustify;
	ZafStringData *stringData;

	virtual ZafWindowObject *Duplicate(void);
	virtual ZafEventType DragDropEvent(const ZafEventStruct &event);
	virtual ZafEventType Draw(const ZafEventStruct &event, ZafEventType ccode);
	virtual const ZafPaletteStruct *MapClassPalette(ZafPaletteType type, ZafPaletteState state);

	// --- Data access ---
	static ZafError Update(ZafText *text, ZafUpdateType type)
		{ if (type == ZAF_UPDATE_OBJECT) return text->OSSetText();
		  else if (type == ZAF_UPDATE_DATA) return text->OSGetText();
		  else return (ZAF_ERROR_INVALID); }

	// --- Special OS members ---
	virtual ZafEventType OSEvent(const ZafEventStruct &event);
	ZafError OSGetText(void);
	virtual void OSMapPalette(ZafPaletteStruct &palette, ZafPaletteType type, ZafPaletteState state);
	virtual void OSRegisterObject(void);
	virtual ZafError OSSetText(void);
	virtual void OSSize(void);

	// --- Persistent members ---
	void ReadData(ZafObjectPersistence &persist);
	virtual ZafObjectConstructor ReadFunction(void) { return (ZafText::Read); }
	void WriteData(ZafObjectPersistence &persist);

private:
	static ZafPaletteMap ZAF_FARDATA defaultPaletteMap[];
	static ZafEventMap ZAF_FARDATA defaultEventMap[];

#	if defined(ZAF_MSWINDOWS)
#		if !defined(ZAF_WIN32)
	HINSTANCE hEditDS;
#		endif
#	elif defined(ZAF_MACINTOSH)
	long selectPosition;
#	elif defined(ZAF_MSDOS) || defined(ZAF_CURSES)
	// Edit buffer data.
	ZafIChar *editBuffer;
	int editSize;
	int offsetCursor;
	int offsetMarkAnchor;
	int offsetMarkFloat;
	int offsetSave;
	int offsetSplit;
	int offsetTopLine;
	int splitSize;
	int textLength;

	// Cursor Data.
	ZafPositionStruct cursorPos;
	int lastCursorColumn;

	// Other Data.
	ZafRegionStruct scroll;
	ZafScrollStruct vScroll, hScroll;
	int lineHeight;
	int lines;
	int marginSize;
	int width;

	int GetTextWidth(ZafIChar *string);
	int CharWidth(int offset);
	int CursorOffset(int textOffset = -1);
	ZafIChar Delete(int *offset);
	void Draw(int startOffset, int endOffset = 0, bool erase = false,
		bool markOnly = false, const ZafRegionStruct *clipRegion = ZAF_NULLP(ZafRegionStruct));
	void Insert(int *offset, ZafIChar value, ZafIChar code = 0);
	int EditOffset(int textOffset);
	void FormatText(void);
	void GetPosition(int offset, ZafPositionStruct *position);
	int LineWidth(int offset, bool wrap = false);
	int Move(int *offset, int moveCode) const;
	void SetCursor(int offset);
	void SetCursor(const ZafPositionStruct &position);
	void SetMark(int offset, bool repaint = true);
	void SetSplit(int *offset);
	int TextOffset(int editOffset);
	int UnformatText(ZafIChar *string, int startOffset = 0, int endOffset = -1);
	void UpdateCursor(void);
	void UpdateScroll(bool redisplay = false, bool scrollCursor = true);
	void Wrap(int offset, bool testPrevLine = false, bool draw = true, bool drawToEnd = false);
#	endif
};

// --- inline functions ---
inline ZafStringData *ZafText::StringData(void) const { return stringData; }

inline bool ZafText::AutoClear(void) const { return autoClear; }
inline ZafHzJustify ZafText::HzJustify(void) const { return hzJustify; }
inline bool ZafText::Invalid(void) const { return invalid; }
inline bool ZafText::Unanswered(void) const { return unanswered; }
inline bool ZafText::ViewOnly(void) const { return viewOnly; }
inline bool ZafText::WordWrap(void) const { return wordWrap; }

#endif // Z_TEXT_HPP

