//	Zinc Application Framework - Z_STORE.HPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#if !defined(Z_STORE_HPP)
#define Z_STORE_HPP
#include <z_dskfil.hpp>

// --------------------------------------------------------------------------
// ----- ZafStorage ---------------------------------------------------------
// --------------------------------------------------------------------------

typedef ZafInt16 ZafVersion;

class ZAF_EXPORT ZafStorageFile;
class ZAF_EXPORT ZafDirectoryElement;
class ZAF_EXPORT ZafStorage : public ZafFileSystem
{
public:
	ZafStorage(void);
	ZafStorage(const ZafIChar *name, ZafFileMode mode = ZAF_FILE_READWRITE);
	virtual ~ZafStorage(void);

	// --- General Members ---
	int Save(int backups = 0);
	int SaveAs(const ZafIChar *newName, int backups = 0);
	ZafVersion SetVersion(ZafVersion version);

	// --- Modes ---
	bool Create(void) const;
	bool OpenCreate(void) const;
	bool ReadOnly(void) const;
	bool ReadWrite(void) const;

	// --- Directory Members ---
	virtual int ChDir(const ZafIChar *newPath,
		ZafStringID stringID = ZAF_NULLP(ZafIChar), ZafNumberID numberID = 0);
	virtual int FindFirst(ZafFileInfoStruct &fileInfo, const ZafIChar *searchName,
		ZafStringID stringID = ZAF_NULLP(ZafIChar), ZafNumberID numberID = 0);
	virtual int FindNext(ZafFileInfoStruct &fileInfo, const ZafIChar *searchName,
		ZafStringID stringID = ZAF_NULLP(ZafIChar), ZafNumberID numberID = 0);
	virtual int FindClose(ZafFileInfoStruct &fileInfo);
	virtual int GetCWD(ZafIChar *pathName, int pathLength);
	virtual int MkDir(const ZafIChar *pathName,
		ZafStringID stringID = ZAF_NULLP(ZafIChar), ZafNumberID numberID = 0);
	virtual int RmDir(const ZafIChar *pathName, bool deleteContents = false);

	// --- File Members ---
	virtual void Close(ZafFile *file);
	virtual ZafFile *Open(const ZafIChar *fileName, const ZafFileMode mode,
		ZafStringID stringID = ZAF_NULLP(ZafIChar), ZafNumberID numberID = 0);
	virtual int Remove(const ZafIChar *fileName);
	virtual int Rename(const ZafIChar *oldName, const ZafIChar *newName,
		ZafStringID stringID = ZAF_NULLP(ZafIChar), ZafNumberID numberID = 0);

protected:
	// --- Mode and Version info ---
	ZafFileMode mode;
	ZafVersion version;

	// --- File Access ---
//??? Only readFile is needed for readOnly mode.
//??? Should ZafStorageReadOnly be split out?
	ZafFile *readFile;
	ZafFile *tempFile;
	ZafFile *writeFile;

	// --- Disk Access ---
//??? Not needed for readOnly mode.
//??? Should ZafStorageReadOnly be split out?
	ZafDiskFileSystem diskFileSystem;
	ZafIChar *fileName;
	ZafIChar *tempFileName;

	// --- Directory Access ---
	ZafDirectoryElement *currentDirectory;
	ZafDirectoryElement *rootDirectory;

	// --- Directory Maintenance ---
	ZafDirectoryElement *WalkPath(const ZafIChar *path);
	ZafDirectoryElement *WalkPath(const ZafIChar *path, const ZafIChar **fileName);
};

// --- inline functions ---
inline bool ZafStorage::Create(void) const { return ((mode & ZAF_FILE_CREATE) ? true : false); }
inline bool ZafStorage::OpenCreate(void) const { return ((mode & ZAF_FILE_OPENCREATE) ? true : false); }
inline bool ZafStorage::ReadOnly(void) const { return ((mode & ZAF_FILE_READ) ? true : false); }
inline bool ZafStorage::ReadWrite(void) const { return ((mode & ZAF_FILE_READWRITE) ? true : false); }
inline ZafVersion ZafStorage::SetVersion(ZafVersion setVersion) { return (version = setVersion); }

// --------------------------------------------------------------------------
// ----- ZafStorageFile -----------------------------------------------------
// --------------------------------------------------------------------------

class ZAF_EXPORT ZafStorageFile : public ZafFile
{
	friend class ZAF_EXPORT ZafStorage;
public:
	// --- General Members ---
	bool Changed(void);
	const char *Data(void);

	// --- Search functions ---
	virtual ZafOffset Length(void) const;
	virtual int Seek(ZafOffset offset, ZafSeek location);
	virtual ZafOffset Tell(void) const;

	// --- I/O functions ---
	virtual int ReadData(void *buffer, int size);
	virtual int WriteData(const void *buffer, int size);

protected:
	// --- Data Management ---
	bool changed;
	char *dataBuffer;
	ZafOffset bufferSize;
	ZafOffset length;
	ZafOffset position;

	// --- Directory Access ---
	ZafDirectoryElement *directoryElement;

	// Created and destroyed only by ZafStorage Open() and Close().
	ZafStorageFile(ZafFileMode mode, char *initData = ZAF_NULLP(char), int initDataSize = 0);
	virtual ~ZafStorageFile(void);
};

// --- inline functions ---
inline bool ZafStorageFile::Changed(void) { return (changed); }
inline const char *ZafStorageFile::Data(void) { return (dataBuffer); }

#endif // Z_STORE_HPP
