//	Zinc Application Framework - Z_REGION.HPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#if !defined(Z_REGION_HPP)
#define Z_REGION_HPP
#include <z_pos.hpp>

// --------------------------------------------------------------------------
// ----- ZafRegionStruct ----------------------------------------------------
// --------------------------------------------------------------------------

struct ZAF_EXPORT ZafRegionStruct : public ZafCoordinateStruct
{
	int left, top, right, bottom;

	void Assign(int left, int top, int width, int height);
	bool Encompassed(const ZafRegionStruct &region) const;
	int Height(void) const;
	void HzShift(int shift);
	bool Overlap(const ZafRegionStruct &region) const;
	bool Overlap(const ZafPositionStruct &position) const;
	bool Overlap(const ZafPositionStruct &position, int columnShift, int lineShift) const;
	bool Overlap(const ZafRegionStruct &region, ZafRegionStruct &result) const;
	bool Touching(const ZafPositionStruct &position) const;
	void VtShift(int shift);
	int Width(void) const;

	// --- Conversion ---
	void ConvertCoordinates(ZafCoordinateType newType);
	ZafRegionStruct Region(ZafCoordinateType type);

	// --- Operators ---
	bool operator==(const ZafRegionStruct &region) const;
	bool operator!=(const ZafRegionStruct &region) const;
	ZafRegionStruct operator++(void);
	ZafRegionStruct operator++(int);
	ZafRegionStruct operator--(void);
	ZafRegionStruct operator--(int);
	ZafRegionStruct &operator+=(int offset);
	ZafRegionStruct &operator-=(int offset);

	// --- OS specific functions ---
#	if defined(ZAF_MSWINDOWS)
	void ExportPixel(OSRegion &rect) const { rect.left = left, rect.top = top, rect.right = right, rect.bottom = bottom; }
	void ExportPoint(OSRegion &rect) const { rect.left = left, rect.top = top, rect.right = right + 1, rect.bottom = bottom + 1; }
	void ImportPixel(const OSRegion &rect) { left = rect.left, top = rect.top, right = rect.right, bottom = rect.bottom; }
	void ImportPoint(const OSRegion &rect) { left = rect.left, top = rect.top, right = rect.right - 1, bottom = rect.bottom - 1; }
#	elif defined(ZAF_MACINTOSH)
	// Add or subtract one on the bottom and right to convert between pixels and points.
	// Zinc regions are always in pixel coordinates.
	void ExportPixel(OSRegion &rect) const { rect.left = left, rect.top = top, rect.right = right, rect.bottom = bottom; }
	void ExportPoint(OSRegion &rect) const { rect.left = left, rect.top = top, rect.right = right + 1, rect.bottom = bottom + 1; }
	void ImportPixel(const OSRegion &rect) { left = rect.left, top = rect.top, right = rect.right, bottom = rect.bottom; }
	void ImportPoint(const OSRegion &rect) { left = rect.left, top = rect.top, right = rect.right - 1, bottom = rect.bottom - 1; }
#	endif
};

// --- inline functions ---
inline void ZafRegionStruct::Assign(int tLeft, int tTop, int tWidth, int tHeight)
	{ left = tLeft, right = tLeft + tWidth - 1, top = tTop, bottom = tTop + tHeight - 1; }
inline bool ZafRegionStruct::Encompassed(const ZafRegionStruct &region) const
	{ return (left >= region.left && top >= region.top && right <= region.right && bottom <= region.bottom); }
inline int ZafRegionStruct::Height(void) const { return (bottom - top + 1); }
inline void ZafRegionStruct::HzShift(int shift)
	{ right += shift, left += shift; }
inline bool ZafRegionStruct::Overlap(const ZafRegionStruct &region) const
	{ return (((region.left > left) ? region.left : left) <= ((region.right < right) ? region.right : right) &&
	  ((region.top > top) ? region.top : top) <= ((region.bottom < bottom) ? region.bottom : bottom)); }
inline bool ZafRegionStruct::Overlap(const ZafPositionStruct &position) const
	{ return (position.column >= left && position.column <= right &&
	  position.line >= top && position.line <= bottom); }
inline bool ZafRegionStruct::Overlap(const ZafPositionStruct &position, int columnShift, int lineShift) const
	{ return (position.column + columnShift >= left && position.column + columnShift <= right &&
	  position.line + lineShift >= top && position.line + lineShift <= bottom); }
inline bool ZafRegionStruct::Overlap(const ZafRegionStruct &region, ZafRegionStruct &result) const
	{ result.left = (left > region.left) ? left : region.left;
	  result.top = (top > region.top) ? top : region.top;
	  result.right = (right < region.right) ? right : region.right;
	  result.bottom = (bottom < region.bottom) ? bottom : region.bottom;
	  return (result.left <= result.right && result.top <= result.bottom);
	}
inline bool ZafRegionStruct::Touching(const ZafPositionStruct &position) const
	{ return (((position.column == left || position.column == right) &&
			    position.line >= top && position.line <= bottom) ||
			  ((position.line == top || position.line == bottom) &&
			    position.column >= left && position.column <= right)); }
inline void ZafRegionStruct::VtShift(int shift)
	{ bottom += shift, top += shift; }
inline int ZafRegionStruct::Width(void) const { return (right - left + 1); }

inline bool ZafRegionStruct::operator==(const ZafRegionStruct &region) const
	{ return (region.left == left && region.top == top && region.right == right && region.bottom == bottom); }
inline bool ZafRegionStruct::operator!=(const ZafRegionStruct &region) const
	{ return (region.left != left || region.top != top || region.right != right || region.bottom != bottom); }
inline ZafRegionStruct &ZafRegionStruct::operator+=(int offset)
	{ left -= offset, top -= offset, right += offset, bottom += offset; return (*this); }
inline ZafRegionStruct &ZafRegionStruct::operator-=(int offset)
	{ left += offset, top += offset, right -= offset, bottom -= offset; return (*this); }

#endif // Z_REGION_HPP

