//	Zinc Application Framework - Z_DSP.HPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#if !defined(Z_DSP_HPP)
#define Z_DSP_HPP
#include <z_region.hpp>
#include <z_pal.hpp>
#include <z_list.hpp>

// --------------------------------------------------------------------------
// ----- ZafImageStruct -----------------------------------------------------
// --------------------------------------------------------------------------

struct ZAF_EXPORT ZafImageStruct
{
	int width, height;
	ZafLogicalColor *array;
	bool staticArray;

	// --- Attributes ---
	// get
	bool StaticArray(void) const;
	// set
	bool SetStaticArray(bool staticArray);
};

// --- inline functions ---
inline bool ZafImageStruct::StaticArray(void) const { return staticArray; }
inline bool ZafImageStruct::SetStaticArray(bool tStaticArray) { staticArray = tStaticArray; return (staticArray); }

// ----- ZafBitmapStruct ----------------------------------------------------

struct ZAF_EXPORT ZafBitmapStruct : public ZafImageStruct
{
#	if defined(ZAF_MSWINDOWS)
	HBITMAP handle, mask;
#	elif defined(ZAF_MOTIF)
	Pixmap handle, mask;
#	elif defined(ZAF_MACINTOSH)
	PixMapHandle colorBitmap;
	BitMapHandle monoBitmap, mask;
#	elif defined(ZAF_MSDOS)
	grafPort *handle, *mask;
#	endif
	bool staticHandle;

	ZafBitmapStruct(void);
	ZafBitmapStruct(const ZafImageStruct &data);
	ZafBitmapStruct(int width, int height, ZafLogicalColor *array,
		bool staticArray);

	// --- Attributes ---
	// get
	bool StaticHandle(void) const;
	// set
	bool SetStaticHandle(bool staticHandle);
};

// --- inline functions ---
inline bool ZafBitmapStruct::StaticHandle(void) const { return staticHandle; }
inline bool ZafBitmapStruct::SetStaticHandle(bool tStaticHandle) { staticHandle = tStaticHandle; return (staticHandle); }

// ----- ZafIconStruct ------------------------------------------------------

struct ZAF_EXPORT ZafIconStruct : public ZafImageStruct
{
#	if defined(ZAF_MSWINDOWS)
	HICON handle;
#	elif defined(ZAF_MOTIF)
	Pixmap handle, mask;
#	elif defined(ZAF_MACINTOSH)
	PixMapHandle colorBitmap;
	BitMapHandle monoBitmap, mask;
	Handle iconHandle;
#	elif defined(ZAF_MSDOS)
	grafPort *handle, *mask;
#	endif
	bool staticHandle;

	ZafIconStruct(void);
	ZafIconStruct(const ZafImageStruct &data);
	ZafIconStruct(int width, int height, ZafLogicalColor *array,
		bool staticArray);

	// --- Attributes ---
	// get
	bool StaticHandle(void) const;
	// set
	bool SetStaticHandle(bool staticHandle);
};

// --- inline functions ---
inline bool ZafIconStruct::StaticHandle(void) const { return staticHandle; }
inline bool ZafIconStruct::SetStaticHandle(bool tStaticHandle) { staticHandle = tStaticHandle; return (staticHandle); }

// ----- ZafMouseStruct -----------------------------------------------------

struct ZAF_EXPORT ZafMouseStruct : public ZafImageStruct
{
	// --- General members ---
	int hotSpotX, hotSpotY;

#	if defined(ZAF_MSWINDOWS)
	HCURSOR cursor;
#	elif defined(ZAF_MOTIF)
	Cursor cursor;
#	elif defined(ZAF_MACINTOSH)
	CursHandle cursorHandle;
#	elif defined(ZAF_MSDOS)
	grafPort *handle, *mask;
#	endif
	bool staticHandle;

	ZafMouseStruct(void);
	ZafMouseStruct(const ZafImageStruct &data);
	ZafMouseStruct(int width, int height, ZafLogicalColor *array,
		int hotSpotX, int hotSpotY, bool staticArray);

	// --- Attributes ---
	// get
	bool StaticHandle(void) const;
	// set
	bool SetStaticHandle(bool staticHandle);
};

// --- inline functions ---
inline bool ZafMouseStruct::StaticHandle(void) const { return staticHandle; }
inline bool ZafMouseStruct::SetStaticHandle(bool tStaticHandle) { staticHandle = tStaticHandle; return (staticHandle); }

// --------------------------------------------------------------------------
// ----- ZafDisplay ---------------------------------------------------------
// --------------------------------------------------------------------------

#if defined(ZAF_MSWINDOWS) 
#	if defined(ZAF_MSWINDOWS_3D)
const int ZAF_BORDER_WIDTH					= 2;
#	else
const int ZAF_BORDER_WIDTH					= 1;
#	endif
#elif defined(ZAF_MACINTOSH)
const int ZAF_BORDER_WIDTH					= 1;
#else
const int ZAF_BORDER_WIDTH					= 2;
#endif
const int ZAF_MARGIN_WIDTH					= 5;
const int ZAF_SHADOW_WIDTH					= 2;

const ZafLogicalColor		ZAF_MAXCOLORS	= 127;
const ZafLogicalFont		ZAF_MAXFONTS	= 10;
const int					ZAF_MAXIMAGES	= 2;
const ZafLogicalLineStyle	ZAF_MAXLINES	= 10;
const ZafLogicalMode		ZAF_MAXMODES	= 10;
const int					ZAF_MAXPALETTES	= 127;
const ZafLogicalFillPattern	ZAF_MAXPATTERNS	= 15;

#if defined(ZAF_MOTIF)
const ZafLogicalColor		ZAF_CLR_MDIINACTIVE	= ZAF_MAXCOLORS - 2;
const ZafLogicalColor		ZAF_CLR_MDIACTIVE	= ZAF_MAXCOLORS - 1;
#endif

enum ZafHzJustify
{
	ZAF_HZ_LEFT		= 0,
	ZAF_HZ_RIGHT	= 1,
	ZAF_HZ_CENTER	= 2
};

enum ZafVtJustify
{
	ZAF_VT_TOP		= 0,
	ZAF_VT_BOTTOM	= 1,
	ZAF_VT_CENTER	= 2
};

#if defined(ZAF_MSDOS) || defined(ZAF_CURSES)
enum ZafDeviceImageType
{
	ZAF_IMT_MOUSE_IMAGE,
	ZAF_IMT_CURSOR_IMAGE
};

#   if defined(ZAF_CURSES)
#	   	if defined(SCO_UNIX)
#	    	define attrib(foreground, background) (COLOR_PAIR(foreground << 3 | background))
#	   	else
#	    	define attrib(foreground, background) (0)
#	   	endif
#   elif defined(ZAF_MSDOS)
#	   	define attrib(foreground, background) (((background) << 4) + (foreground))
#   endif
#endif

// ----- Private class definition. Do not use! ------------------------------
class ZAF_EXPORT ZafRegionElement : public ZafElement
{
public:
	// --- General members ---
	int useCount;
	OSWindowID screenID;
	ZafRegionStruct region;

	ZafRegionElement(OSWindowID screenID, const ZafRegionStruct &_region);
	ZafRegionElement(OSWindowID screenID, int left, int top, int right,
		int bottom, ZafCoordinateType type);
	~ZafRegionElement(void);

	// --- Element members ---
	ZafRegionElement *Next(void) const { return((ZafRegionElement *)next); }
	ZafRegionElement *Previous(void) const { return((ZafRegionElement *)previous); }
};

// ----- Private class definition. Do not use! ------------------------------
class ZAF_EXPORT ZafRegionList : public ZafList
{
public:
	// --- General members ---
	ZafRegionList(void);
	~ZafRegionList(void);
	void Split(OSWindowID screenID, const ZafRegionStruct &region, int allocateBelow = false);

	// --- List members ---
	ZafRegionElement *Current(void) const { return((ZafRegionElement *)current); }
	ZafRegionElement *First(void) const { return((ZafRegionElement *)first); }
	ZafRegionElement *Last(void) const { return((ZafRegionElement *)last); }
};

// ----- ZafDisplay ---------------------------------------------------------

enum ZafFontWeight
{
	ZAF_FNT_WEIGHT_NORMAL,
	ZAF_FNT_WEIGHT_BOLD
};

enum ZafFontSlant
{
	ZAF_FNT_SLANT_NORMAL,
	ZAF_FNT_SLANT_ITALIC
};

enum ZafDisplayMode
{
	ZAF_DISPLAY_MONO,
	ZAF_DISPLAY_COLOR
};

class ZAF_EXPORT ZafDisplay : public ZafCoordinateStruct
{
public:
	int columns, lines;
	int preSpace, postSpace;

	static OSLineStyle ZAF_FARDATA lineTable[ZAF_MAXLINES];
	static OSFillPattern ZAF_FARDATA patternTable[ZAF_MAXPATTERNS];
	static OSColor ZAF_FARDATA colorTable[ZAF_MAXCOLORS];
	static OSMono ZAF_FARDATA monoTable[ZAF_MAXCOLORS];
	static OSFont ZAF_FARDATA fontTable[ZAF_MAXFONTS];
	static OSMode ZAF_FARDATA modeTable[ZAF_MAXMODES];

	ZafDisplay(int argc, char **argv);
	virtual ~ZafDisplay(void);

	// --- Drawing members ---
	// bitmap
	virtual ZafError Bitmap(int column, int line, ZafBitmapStruct &bitmap);
	virtual ZafError ConvertToOSBitmap(ZafBitmapStruct &bitmap);
	virtual ZafError ConvertToZafBitmap(ZafBitmapStruct &bitmap);
	virtual ZafError DestroyOSBitmap(ZafBitmapStruct &bitmap);
	virtual ZafError DestroyZafBitmap(ZafBitmapStruct &bitmap);
	static ZafError InitializeOSBitmap(ZafBitmapStruct &bitmap);
	static ZafError ResetOSBitmap(ZafBitmapStruct &bitmap, const ZafBitmapStruct &copy);
	// icon
	virtual ZafError Icon(int column, int line, ZafIconStruct &icon);
	virtual ZafError ConvertToOSIcon(ZafIconStruct &icon);
	virtual ZafError ConvertToZafIcon(ZafIconStruct &icon);
	virtual ZafError DestroyOSIcon(ZafIconStruct &icon);
	virtual ZafError DestroyZafIcon(ZafIconStruct &icon);
	static ZafError InitializeOSIcon(ZafIconStruct &icon);
	static ZafError ResetOSIcon(ZafIconStruct &icon, const ZafIconStruct &copy);
	// mouse
	virtual ZafError Mouse(int column, int line, ZafMouseStruct &mouse);
	virtual ZafError ConvertToOSMouse(ZafMouseStruct &mouse);
	virtual ZafError ConvertToZafMouse(ZafMouseStruct &mouse);
	virtual ZafError DestroyOSMouse(ZafMouseStruct &mouse);
	virtual ZafError DestroyZafMouse(ZafMouseStruct &mouse);
	static ZafError InitializeOSMouse(ZafMouseStruct &mouse);
	static ZafError ResetOSMouse(ZafMouseStruct &mouse, const ZafMouseStruct &copy);
	// text
	ZafError Text(int left, int top, const ZafIChar *text, int length = -1,
		int hotKeyIndex = -1, bool fill = false);
	ZafError Text(const ZafRegionStruct &region,
		const ZafIChar *text, int length = -1,
		ZafHzJustify hzJustify = ZAF_HZ_LEFT,
		ZafVtJustify vtJustify = ZAF_VT_CENTER,
		int hotKeyIndex = -1, bool fill = false)
		{ return (Text(region.left, region.top, region.right, region.bottom,
		  text, length, hzJustify, vtJustify, hotKeyIndex, fill)); }
	virtual ZafError Text(int left, int top, int right, int bottom,
		const ZafIChar *text, int length = -1,
		ZafHzJustify hzJustify = ZAF_HZ_LEFT,
		ZafVtJustify vtJustify = ZAF_VT_CENTER,
		int hotKeyIndex = -1, bool fill = false);
	virtual ZafRegionStruct TextSize(const ZafIChar *text, int length = -1);
	// graphic primitives
	virtual ZafError Ellipse(int left, int top, int right, int bottom,
		int startAngle, int endAngle, int width = 1, bool fill = false);
	virtual ZafError Line(int column1, int line1, int column2, int line2,
		int width = 1);
	virtual ZafError Pixel(int column, int line,
		ZafLogicalColor color = ZAF_CLR_DEFAULT, ZafLogicalColor mono = ZAF_MONO_DEFAULT);
	virtual ZafError Polygon(int numPoints, const int *polygonPoints,
		int width = 1, bool fill = false, bool close = false);
	ZafError Rectangle(const ZafRegionStruct &region, int width = 1,
		bool fill = false)
		{ return (Rectangle(region.left, region.top, region.right,
		  region.bottom, width, fill)); }
	virtual ZafError Rectangle(int left, int top, int right, int bottom,
		int width = 1, bool fill = false);
	virtual ZafError RegionCopy(const ZafRegionStruct &oldRegion,
		int newColumn, int newLine);
	virtual ZafError RectangleXORDiff(const ZafRegionStruct *oldRegion,
		const ZafRegionStruct *newRegion);

	// --- Drawing support ---
	ZafDisplayMode DisplayMode(void) const;
	// get
	ZafLogicalColor Background(void) const;
	ZafRegionStruct ClipRegion(void) const;
	ZafLogicalFont Font(void) const;
	ZafLogicalColor Foreground(void) const;
	ZafLogicalMode Mode(void) const;
	ZafLogicalColor MonoBackground(void) const;
	ZafLogicalColor MonoForeground(void) const;
	ZafPaletteStruct Palette(void) const;
	ZafLogicalFillPattern FillPattern(void) const;
	ZafLogicalLineStyle LineStyle(void) const;
	// set
	ZafLogicalColor SetBackground(ZafLogicalColor color);
	ZafRegionStruct SetClipRegion(const ZafRegionStruct &region);
	ZafLogicalFont SetFont(ZafLogicalFont font);
	ZafLogicalColor SetForeground(ZafLogicalColor color);
	ZafLogicalLineStyle SetLineStyle(ZafLogicalLineStyle line);
	ZafLogicalMode SetMode(ZafLogicalMode mode);
	ZafLogicalColor SetMonoBackground(ZafLogicalColor color);
	ZafLogicalColor SetMonoForeground(ZafLogicalColor color);
	ZafPaletteStruct SetPalette(ZafPaletteStruct palette);
	ZafLogicalFillPattern SetFillPattern(ZafLogicalFillPattern pattern);

	// --- Table support ---
	ZafLogicalColor AddColor(ZafLogicalColor index, ZafUInt8 red, ZafUInt8 green, ZafUInt8 blue);
	ZafLogicalFont AddFont(ZafLogicalFont index, char *fontFamily, int pointSize,
		ZafFontWeight weight = ZAF_FNT_WEIGHT_NORMAL,
		ZafFontSlant slant = ZAF_FNT_SLANT_NORMAL);
	ZafError DestroyColor(ZafLogicalColor color);
	ZafError DestroyFont(ZafLogicalFont font);

	// --- ScreenOS specific support members ---
	ZafError BeginDraw(OSDisplayContext displayContext,
		OSDrawContext drawContext, const ZafRegionStruct &draw,
		const ZafRegionStruct &clip);
	OSDisplayContext DisplayContext(void) const;
	OSDrawContext DrawContext(void) const;
	ZafError EndDraw(void);
	OSDisplayContext RestoreDisplayContext(void);
	OSDrawContext RestoreDrawContext(void);
	OSDisplayContext SetDisplayContext(OSDisplayContext context);
	OSDrawContext SetDrawContext(OSDrawContext context);

protected:
	// --- General members ---
	ZafRegionStruct clipRegion;
	ZafPaletteStruct palette;
	ZafLogicalMode mode;
	ZafDisplayMode displayMode;

	// --- OS specific support members ---
#	if defined(ZAF_MSWINDOWS)
public:
	bool paletteDevice;
	HPALETTE logicalPalette;

	HBRUSH GetGdiCacheBrush(ZafPaletteStruct palette);
	HPEN GetGdiCachePen(ZafPaletteStruct palette, int width);
	int ReleaseGdiCacheObject(HGDIOBJ hGdiObject);

private:
	HBRUSH hBrush;
	OSDisplayContext hDC;
	OSDrawContext hWnd;
	HPEN hPen;
	int virtualCount;

	void CreateLogicalPalette(void);
#	elif defined(ZAF_MOTIF)
public:
	XtAppContext xAppContext;
	int xScreenNumber;

	Widget topLevelShell;
	Display *xDisplay;
	GC xGC;
	Drawable xDrawable;

	XIC ic;

	bool ZafArrayToOSPixmap(int width, int height, ZafLogicalColor *array, Pixmap *handle, Pixmap *mask);
	bool OSPixmapToZafArray(int *width, int *height, ZafLogicalColor **array, Pixmap handle, Pixmap mask);
	XFontSet MtfGetFontSet(XmFontList fontList) const;
	XFontStruct *MtfGetFont(XmFontList fontList) const;
	ZafLogicalColor MtfGetColor(char *color);
#	elif defined(ZAF_MSDOS)
public:
#		if defined(ZAF_GRAPHICS_MODE)

	struct ZAF_EXPORT ZafDisplayImage
	{
		ZafRegionStruct region, clip;
		ZafLogicalColor *image;
		grafPort *screen;
		grafPort *backup;
	};
	ZafDisplayImage displayImage[ZAF_MAXIMAGES];
	grafPort *screenPort;

#		endif

	bool installed;
	int maxColors;
	signed char virtualCount;
	ZafRegionStruct virtualRegion;
	char stopDevice;
	int isMono;
	static ZafPaletteStruct *backgroundPalette;
	static ZafPaletteStruct *xorPalette;
	static ZafPaletteStruct *colorMap;
	static ZafPaletteStruct *markPalette;
	ZafRegionList *screenList;
	OSWindowID screenID;

	void RegionDefine(OSWindowID screenID, const ZafRegionStruct &region)
		{ RegionDefine(screenID, region.left, region.top, region.right, region.bottom); }
	virtual void RegionDefine(OSWindowID screenID, int left, int top, int right, int bottom);
	virtual ZafError UnvirtualizeRegion(ZafRegionStruct &region);
	virtual ZafError VirtualizeRegion(ZafRegionStruct &region);

	// ADVANCED functions for mouse and cursor --- DO NOT USE! ---
	virtual int DeviceMove(ZafDeviceImageType imageType, int newColumn, int newLine, ZafRegionStruct &clip);
	virtual int DeviceSet(ZafDeviceImageType imageType, int column, int line, int width, int height, ZafLogicalColor *image, ZafRegionStruct &clip);

	int RegionInitialize(ZafRegionStruct &region, const ZafRegionStruct *clipRegion,
		int left, int top, int right, int bottom);

private:
	ZafRegionStruct masterClip;
#	endif
};

inline ZafDisplayMode ZafDisplay::DisplayMode(void) const { return (displayMode); }

extern ZAF_EXPORT ZafDisplay *zafDisplay;

#endif // Z_DSP_HPP

