//	Zinc Application Framework - Z_UTIME1.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_utime1.hpp>
#include <z_string.hpp>
#include <z_error.hpp>
#include <z_app.hpp>
#define ZAF_UTIME_INFO
#include "lang_def.cpp"
#include "gbl_def.cpp"

static ZafIChar utimeSeparator[3] = { '.','.',0 };
static ZafIChar rangeSeparator[3] = { ',',' ',0 };
static ZafStringID _utimeDataName = ZAF_ITEXT("utimeData");

// ----- ZafUTime -----------------------------------------------------------

ZafUTime::ZafUTime(int left, int top, int width, ZafUTimeData *zUTimeData) :
	ZafString(left, top, width, new ZafStringData),
	utimeData(ZAF_NULLP(ZafUTimeData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Initialize the utime data.
	SetUTimeData(zUTimeData ? zUTimeData : new ZafUTimeData);
}

ZafUTime::ZafUTime(int left, int top, int width, int year, int month,
	int day, int hour, int minute, int second, int milliSecond) :
	ZafString(left, top, width, new ZafStringData),
	utimeData(ZAF_NULLP(ZafUTimeData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Initialize the utime data.
	SetUTimeData(new ZafUTimeData(year, month, day, hour, minute, second, milliSecond));
}

ZafUTime::ZafUTime(const ZafUTime &copy) :
	ZafString(copy), utimeData(ZAF_NULLP(ZafUTimeData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Copy the time data.
	if (copy.UTimeData()->Destroyable())
		SetUTimeData(new ZafUTimeData(*copy.UTimeData()));
	else
		SetUTimeData(copy.UTimeData());
}

ZafUTime::~ZafUTime(void)
{
	// Remove the data notification.
	utimeData->SubtractNotification(this, (ZafUpdateFunction)Update);

	// Check the language and local information.
	LanguageFree();

	// Restore the time information.
	if (utimeData->Destroyable() && utimeData->NotifyCount() == 0)
		delete utimeData;
}

ZafError ZafUTime::Decrement(ZafData *data)
{
	// Check for a valid object.
	ZafUTimeData *utime = DynamicPtrCast(data, ZafUTimeData);
	if (!utime)
		return (ZAF_ERROR_INVALID_CLASS);

	// Decrement the value.
	ZafUTimeData _utime(*utimeData);
	OSGetUTime();
	*utimeData -= *utime; // update is automatic.

	// Return success.
	return (ZAF_ERROR_NONE);
}

ZafWindowObject *ZafUTime::Duplicate(void)
{
	return (new ZafUTime(*this));
}

ZafEventType ZafUTime::Event(const ZafEventStruct &event)
{
	// Process OS specific messages.
	ZafEventType ccode = event.type;
	if (event.InputType() == E_KEY)
		ccode = LogicalEvent(event);

	// Check for zinc events.
	switch (ccode)
	{
	// ----- Create messages -----
	case S_INITIALIZE:
		// Check for an unanswered field.
		if (!Unanswered())
			OSSetUTime();
		ccode = ZafString::Event(event);
		break;

	// ----- Action messages -----
	case L_SELECT:
	case N_NON_CURRENT:
		{
		// Keep a temporary time in case of error.
		ZafUTimeData oldValue(*utimeData);
		OSGetUTime();

		// Call the associated user function.
		ccode = ZafString::Event(event);

		// Check for errors.
		if (ccode != 0 && ccode != ZAF_ERROR_LEAVE_INVALID)
			utimeData->SetUTime(oldValue);

		// Set the formatted bignum text into the field.
		OSSetUTime();
		}
		break;

	case S_COPY_DATA:
		{
		ZafWindowObject *object = event.windowObject;
		ZafUTime *utime = DynamicPtrCast(object, ZafUTime);
		if (utime)
			utimeData->SetUTime(*utime->UTimeData());
		}
		break;

	case S_SET_DATA:
		if (event.windowObject)
		{
			ZafWindowObject *object = event.windowObject;
			ZafUTime *utime = DynamicPtrCast(object, ZafUTime);
			if (utime)
				SetUTimeData(utime->UTimeData());
		}
		else
			SetUTimeData(new ZafUTimeData(*utimeData));
		break;

	// ----- Default or unknown messages -----
	default:
		ccode = ZafString::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

ZafError ZafUTime::Increment(ZafData *data)
{
	// Check for a valid object.
	ZafUTimeData *utime = DynamicPtrCast(data, ZafUTimeData);
	if (!utime)
		return (ZAF_ERROR_INVALID_CLASS);

	// Increment the value.
	ZafUTimeData _utime(*utimeData);
	OSGetUTime();
	*utimeData += *utime; // update is automatic.

	// Return success.
	return (ZAF_ERROR_NONE);
}

void ZafUTime::LanguageAllocate(void)
{
	// Check for initialization.
	if (errorStrings)
		return;
	ZafApplication::AddStaticModule(LanguageFree);

	// Try the zafDataManager.
	if (!errorStrings && zafDataManager)
		errorStrings = DynamicPtrCast(zafDataManager->AllocateData(className, ZafLanguageData::className, ZafLanguageData::classID), ZafLanguageData);

	// Default to code initialization.
	if (!errorStrings)
		errorStrings = new ZafLanguageData(defaultErrorStrings);
}

void ZafUTime::LanguageFree(bool globalRequest)
{
	// Destroy the static language.
	if (globalRequest && errorStrings && errorStrings->Destroyable() &&
		errorStrings->NotifyCount() == 0)
	{
		delete errorStrings;
		errorStrings = ZAF_NULLP(ZafLanguageData);
	}
}

ZafError ZafUTime::OSGetUTime(void)
{
	// Disable notification.
	ZafUpdateType update = utimeData->Update(this);
	utimeData->SetUpdate(this, ZAF_UPDATE_NONE);

	// Set the utime based on the string data.
	OSGetText();
	error = utimeData->SetUTime(stringData->Text(), InputFormatText());

	// Restore notification.
	utimeData->SetUpdate(this, update);
	return (error);
}

ZafError ZafUTime::OSSetUTime(void)
{
	// Set the string based on the utime data.
	ZafIChar text[ZAF_STRING_DATA_LENGTH];
	utimeData->FormattedText(text, ZAF_STRING_DATA_LENGTH, OutputFormatText());
	stringData->SetText(text);

	// Return the current status.
	return (ZAF_ERROR_NONE);
}

ZafError ZafUTime::SetUTimeData(ZafUTimeData *tUTimeData)
{
	// Remove the data notification.
	if (utimeData)
		utimeData->SubtractNotification(this, (ZafUpdateFunction)Update);

	// Reset the utime data.
	if (utimeData && utimeData != tUTimeData &&
		utimeData->Destroyable() && utimeData->NotifyCount() == 0)
		delete utimeData;
	utimeData = tUTimeData ? tUTimeData : new ZafUTimeData;
	if (!utimeData->StringID())
		utimeData->SetStringID(_utimeDataName);

	// Add the data notification.
	utimeData->AddNotification(this, (ZafUpdateFunction)Update);
	return (OSSetUTime());
}

bool ZafUTime::SetLowerCase(bool )
{
	// lowerCase is set to false by this class.
	lowerCase = false;
	return (lowerCase);
}

bool ZafUTime::SetPassword(bool )
{
	// password is set to false by this class.
	password = false;
	return (password);
}

ZafError ZafUTime::SetStringData(ZafStringData *)
{
	// String data cannot be set for this class.
	return (ZAF_ERROR_INVALID);
}

bool ZafUTime::SetUpperCase(bool )
{
	// upperCase is set to false by this class.
	upperCase = false;
	return (upperCase);
}

ZafError ZafUTime::Validate(bool processError)
{
	// Check for an absolute time error. Don't set the error member to ZAF_ERROR_NONE.
	ZafUTimeData currentTime;
	ZafIChar *stringTime = stringData->DynamicText();
	ZafError tError = currentTime.SetUTime(stringTime, InputFormatText());

	// Check for a range error.
	if (RangeText() && tError == ZAF_ERROR_NONE)
		tError = ZAF_ERROR_OUT_OF_RANGE;
	int numRanges = 0;
	ZafUTimeData low, high;
	ZafIChar minTime[ZAF_STRING_DATA_LENGTH];
	ZafIChar maxTime[ZAF_STRING_DATA_LENGTH];
	ZafIChar rBuffer[1024]; // Localized range string for error messages.
	rBuffer[0] = '\0';
	for (const ZafIChar *tRange = RangeText(); tRange && tError == ZAF_ERROR_OUT_OF_RANGE; numRanges++)
	{
		tRange = ParseRange(tRange, minTime, maxTime);
		low.SetUTime(minTime, OutputFormatText());
		high.SetUTime(maxTime, OutputFormatText());
		if ((!minTime[0] || currentTime >= low) && (!maxTime[0] || currentTime <= high))
			tError = ZAF_ERROR_NONE;
		else if (processError)
		{
			if (rBuffer[0])
				strcat(rBuffer, rangeSeparator);
			if (minTime[0])
				low.FormattedText(&rBuffer[strlen(rBuffer)], ZAF_STRING_DATA_LENGTH, OutputFormatText());
			strcat(rBuffer, utimeSeparator);
			if (maxTime[0])
				high.FormattedText(&rBuffer[strlen(rBuffer)], ZAF_STRING_DATA_LENGTH, OutputFormatText());
		}
	}

	// Process the error code.
	SetInvalid(false);
	if (tError == ZAF_ERROR_NONE)
	{
		// Set up the new time.
		currentTime.FormattedText(stringTime, ZAF_STRING_DATA_LENGTH, OutputFormatText());
		SetText(stringTime);
		delete []stringTime;
		return (ZAF_ERROR_NONE);
	}
	else
	{
		// Keep the error code.
		error = tError;
		if (!zafErrorSystem)
		{
			// Restore the original time.
			utimeData->FormattedText(stringTime, ZAF_STRING_DATA_LENGTH, OutputFormatText());
			SetText(stringTime);
			delete []stringTime;
			return (error);
		}
		else if (!processError)
		{
			delete []stringTime;
			return (error);
		}
	}

	// Check for open-ended range errors.
	if (error == ZAF_ERROR_OUT_OF_RANGE && numRanges == 1)
	{
		if (minTime[0] && !maxTime[0])
		{
			error = ZAF_ERROR_LESS_THAN_RANGE;
			low.FormattedText(rBuffer, ZAF_STRING_DATA_LENGTH, OutputFormatText());
		}
		else if (!minTime[0] && maxTime[0])
		{
			error = ZAF_ERROR_GREATER_THAN_RANGE;
			high.FormattedText(rBuffer, ZAF_STRING_DATA_LENGTH, OutputFormatText());
		}
	}

	// Generate the error message and wait for a response.
	ZafIChar *title = ZafLanguageData::blankString;
	ZafIChar *errorString = errorStrings->GetMessage((ZafNumberID)error, true);
	if (errorString &&
		zafErrorSystem->ReportError(windowManager, title, ZAF_DIALOG_OK | ZAF_DIALOG_CANCEL, errorString, stringTime, rBuffer) == S_DLG_OK)
		error = ZAF_ERROR_LEAVE_INVALID; // Keep the new value.
	delete []stringTime;
	return (error);
}

bool ZafUTime::SetVariableName(bool )
{
	// variableName is false for this class.
	variableName = false;
	return (variableName);
}

// ----- Persistent functions -----------------------------------------------

ZafUTime::ZafUTime(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafString(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	utimeData(ZAF_NULLP(ZafUTimeData))
{
	// Check the language and local information.
	LanguageAllocate();

	if (persist.Error() == ZAF_ERROR_NONE)
		ZafUTime::ReadData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

ZafElement *ZafUTime::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafUTime(name, persist));
}

void ZafUTime::ReadData(ZafObjectPersistence &persist)
{
	// Read the data.
	ZafIChar utimeName[ZAF_MAXNAMELEN];
	ZafFile *file = persist.CurrentFile();
	*file >> utimeName;
	SetUTimeData(utimeName[0] ? new ZafUTimeData(utimeName, persist) : new ZafUTimeData);
}

void ZafUTime::Write(ZafObjectPersistence &persist)
{
	// Write the object.
	ZafString::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafUTime::WriteData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafUTime::WriteData(ZafObjectPersistence &persist)
{
	// Write the data.
	const ZafIChar *utimeName = utimeData ? utimeData->StringID() : ZAF_NULLP(ZafIChar);
	ZafFile *file = persist.CurrentFile();
	*file << utimeName;
	if (utimeData)
		utimeData->Write(persist);
}

