//	Zinc Application Framework - Z_TABLE2.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_table.hpp>
#include <z_str1.hpp>
#include <z_ctype.hpp>

#if defined(ZAF_MSWINDOWS)
#include "w_app.hpp"
#endif

// ----- ZafTableRecord -----------------------------------------------------

ZafTableRecord::ZafTableRecord(int width, int height) :
	ZafWindow(0, 0, width, height),
	offset(-1)
{
	SetSelectionType(ZAF_MULTIPLE_SELECTION);
}

ZafTableRecord::ZafTableRecord(const ZafTableRecord &copy) :
	ZafWindow(copy),
	offset(-1)
{
	SetSelectionType(ZAF_MULTIPLE_SELECTION);
}

ZafTableRecord::~ZafTableRecord(void)
{
}

ZafWindowObject *ZafTableRecord::Add(ZafWindowObject *object, ZafWindowObject *position)
{
	// Check for a support or invalid object.
	if (object->parent)
		return (object);
	else if (object->SupportObject())
		return (ZafWindow::Add(object, position));

	// Add the object to the table record.
	object->SetSystemObject(false);
	object->SetParentDrawFocus(true);
	return (ZafWindow::Add(object, position));
}

void ZafTableRecord::ConvertRegion(ZafRegionStruct &region, ZafCoordinateType newType)
{
	// Go directly to region convert so preSpace and postSpace are not used.
	region.ConvertCoordinates(newType);
}

ZafEventType ZafTableRecord::Draw(const ZafEventStruct &event, ZafEventType ccode)
{
	// Compute the actual draw region.
	ZafRegionStruct drawRegion = BeginDraw();
	ZafRegionStruct clip = drawRegion;
	if (ccode == S_REDISPLAY_REGION)
		clip = event.region;

#	if defined(ZAF_MOTIF)
	if (ccode == E_MOTIF && event.data)
	{
		XRectangle xclip;
		XClipBox((_XRegion *)event.data, &xclip);
		clip.Assign(xclip.x, xclip.y, xclip.width, xclip.height);
		clip.Overlap(zafRegion, clip);
	}
#	endif

	clip.HzShift(-zafRegion.left);
	clip.VtShift(-zafRegion.top);

#	if defined(ZAF_MSWINDOWS)
	// Draw the table fields.
	for (ZafWindowObject *object = First(); object; object = object->Next())
		if (object->zafRegion.Overlap(clip) && !object->SystemObject())
		{
			ZafPaletteState state = PaletteState();
			if (object->Focus())
			{
				state &= !ZAF_PM_NOT_SELECTED;
				state |= ZAF_PM_SELECTED;
			}
			ZafPaletteStruct textPalette = LogicalPalette(ZAF_PM_TEXT, state);
			ZafPaletteStruct backgroundPalette = LogicalPalette(ZAF_PM_BACKGROUND, state);

			if (object->Focus())
			{
				if (textPalette.colorForeground == ZAF_CLR_DEFAULT)
				{
					display->colorTable[ZAF_MAXCOLORS - 1] = textPalette.osPalette.colorForeground;
					textPalette.colorForeground = ZAF_MAXCOLORS - 1;
				}
				if (textPalette.colorBackground == ZAF_CLR_DEFAULT)
				{
					display->colorTable[ZAF_MAXCOLORS - 2] = textPalette.osPalette.colorBackground;
					textPalette.colorBackground = ZAF_MAXCOLORS - 2;
				}
				if (backgroundPalette.colorForeground == ZAF_CLR_DEFAULT)
				{
					display->colorTable[ZAF_MAXCOLORS - 3] = backgroundPalette.osPalette.colorForeground;
					backgroundPalette.colorForeground = ZAF_MAXCOLORS - 3;
				}
				if (backgroundPalette.colorBackground == ZAF_CLR_DEFAULT)
				{
					display->colorTable[ZAF_MAXCOLORS - 4] = backgroundPalette.osPalette.colorBackground;
					backgroundPalette.colorBackground = ZAF_MAXCOLORS - 4;
				}
				ZafPaletteMap traverseModePaletteMap[3];
				traverseModePaletteMap[0].type = ZAF_PM_TEXT;
				traverseModePaletteMap[0].state = ZAF_PM_ANY_STATE;
				traverseModePaletteMap[0].palette = textPalette; 
				traverseModePaletteMap[1].type = ZAF_PM_BACKGROUND;
				traverseModePaletteMap[1].state = ZAF_PM_ANY_STATE;
				traverseModePaletteMap[1].palette = backgroundPalette; 
				traverseModePaletteMap[2].type = ZAF_PM_NONE; 
				traverseModePaletteMap[2].state = 0;
				ZafPaletteData traverseModePaletteData(traverseModePaletteMap, true);

				ZafPaletteData *saveUserPaletteData = object->userPaletteData;
				object->userPaletteData = &traverseModePaletteData;
				object->Draw(event, ccode);
				object->userPaletteData = saveUserPaletteData;
			}
			else
				object->Draw(event, ccode);
		}
#	elif defined(ZAF_MOTIF)
	// Draw the table fields.
	for (ZafWindowObject *object = First(); object; object = object->Next())
		if (object->zafRegion.Overlap(clip) && !object->SystemObject())
			object->Draw(event, ccode);
#	else
	// Draw the table fields.
	for (ZafWindowObject *object = First(); object; object = object->Next())
	{
		ZafRegionStruct objectRegion = ConvertToOSRegion(object);
		if (objectRegion.Overlap(clip) && !object->SystemObject())
			object->Draw(event, ccode);
	}
#	endif

	// Return the control code.
	EndDraw();
	return (ccode);
}

ZafEventType ZafTableRecord::DrawFocus(ZafRegionStruct &, ZafEventType ccode)
{
	// Table record does not draw its own focus.
	return (ccode);
}

ZafWindowObject *ZafTableRecord::Duplicate(void)
{
	return (new ZafTableRecord(*this));
}

ZafEventType ZafTableRecord::Event(const ZafEventStruct &event)
{
	// Check for zinc events.
	ZafEventType ccode = LogicalEvent(event);
	ZafWindowObject *object = Current();
	switch (ccode)
	{
	case L_BEGIN_SELECT:
		// Try to find an overlapping object.
		for (object = First(); object; object = object->Next())
			if (object->zafRegion.Overlap(event.position))
			{
				object->SetSystemObject(true);

#				if defined(ZAF_MSWINDOWS)
				POINT mousePoint;
				mousePoint.x = LOWORD(event.osEvent.lParam);
				mousePoint.y = HIWORD(event.osEvent.lParam);
				MapWindowPoints(event.osEvent.hwnd, object->screenID, &mousePoint,1);
				SendMessage(object->screenID, WM_LBUTTONDOWN, event.osEvent.wParam,
					MAKELONG(mousePoint.x, mousePoint.y));
#				elif defined(ZAF_MOTIF)
				XEvent xEvent = event.osEvent;
				xEvent.xbutton.display = XtDisplay(object->screenID);
				xEvent.xbutton.subwindow = 0;
				xEvent.xbutton.window = XtWindow(object->screenID);
				xEvent.xbutton.x = event.position.column - object->zafRegion.left;
				xEvent.xbutton.y = event.position.line - object->zafRegion.top;
				XtDispatchEvent(&xEvent);
#				endif

				break;
			}

		// Make sure the object has the focus.
		if (!object && !Focus())
			SetFocus(true);
		break;

	case N_SIZE:
		// Pre-size the children.
		for (object = First(); object; object = object->Next())
		{
			object->zafRegion.top = 0;
			object->zafRegion.bottom = zafRegion.bottom - zafRegion.top;
			if (!object->Previous() && object->Next())
				object->zafRegion.left = 0;
			if (object != Last())
				object->zafRegion.right = object->Next()->zafRegion.left - 2;
			else if (Table()->virtualRecord->Last())
			{
				object->zafRegion.right = Table()->virtualRecord->Last()->zafRegion.right;
				if (object->zafRegion.right < zafRegion.right - zafRegion.left)
					object->zafRegion.right = zafRegion.right - zafRegion.left;
			}
			else if (object->zafRegion.left < zafRegion.right - zafRegion.left)
				object->zafRegion.right = zafRegion.right - zafRegion.left;
			object->SetRegion(object->zafRegion);
		}

		// Now, defer to the base class.
		ccode = ZafWindow::Event(event);
		break;

	case S_REDISPLAY:
	case S_REDISPLAY_DATA:
	case S_REDISPLAY_REGION:
		ccode = ZafWindowObject::Event(event);
		break;

	default:
		// Check for key-activation of the current field.
		if (event.InputType() == E_KEY && (event.modifiers & S_KEYDOWN) &&
			IsPrint(event.key.value) && !object->SystemObject())
		{
			object->SetSystemObject(true);
			object->SetFocus(true);
#if defined(ZAF_MSWINDOWS)
			SendMessage(object->screenID, WM_KEYDOWN, event.osEvent.wParam, event.osEvent.lParam);
#endif
		}

		// Defer to the window.
		ccode = ZafWindow::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

ZafEventType ZafTableRecord::MoveEvent(const ZafEventStruct &event)
{
	// Determine the new selection.
	ZafEventType ccode = event.type;
	ZafWindowObject *object = Current();
	switch (ccode)
	{
	case L_BOL:
		if (object && object->SystemObject())
			ccode = S_CONTINUE;
		else
			object = First();
		break;

	case L_EOL:
		if (object && object->SystemObject())
			ccode = S_CONTINUE;
		else
			object = Last();
		break;

	case L_LEFT:
	case L_PREVIOUS:
		if (!object)
			object = Last();
		else if (ccode == L_LEFT && object->SystemObject())
			ccode = S_CONTINUE;
		else if (object->Previous())
			object = object->Previous();
		else
			object = Last();
		break;

	case L_RIGHT:
	case L_NEXT:
	case L_SELECT:
		if (!object)
			object = First();
		else if (ccode == L_RIGHT && object->SystemObject())
			ccode = S_CONTINUE;
		else if (object && ccode == L_SELECT && !object->SystemObject())
		{
			object->SetSystemObject(true);
			object->SetFocus(true);
		}
		else if (object->Next())
			object = object->Next();
		else
			object = First();
		break;

	case L_UP:
	case L_DOWN:
		if (object && object->SystemObject() && object->Event(event) != S_UNKNOWN)
			ccode = S_CONTINUE;
		else
			ccode = S_ERROR;
		break;

	default:
		ccode = S_ERROR;
		break;
	}

	// Check the current object.
	if (!object)
		return (S_ERROR);
	else if (object != current)
		object->SetFocus(true);
	return (ccode);
}

ZafWindowObject *ZafTableRecord::NotifyFocus(ZafWindowObject *focusObject, bool setFocus)
{
	// Defer to the base class.
	ZafWindowObject *object = Current();
	ZafWindowObject *invalidObject = ZafWindow::NotifyFocus(focusObject, setFocus);
	focusObject = Current();

	// Update the focus.
	if (!screenID)
		return (invalidObject);

	// Deinitialize the old record focus.
	if (object && (!setFocus || object != focusObject))
		object->SetSystemObject(false);
	if (object)
		object->Redisplay();

	// Redisplay the new record focus.
	if (setFocus && focusObject)
		focusObject->Redisplay();

	// Return the invalid object.
	return (invalidObject);
}

ZafWindowObject *ZafTableRecord::Subtract(ZafWindowObject *object)
{
	// Remove the object from the table record.
	return (ZafWindow::Subtract(object));
}

bool ZafTableRecord::SetAcceptDrop(bool )
{
	// acceptDrop is false for this class.
	acceptDrop = false;
	return (acceptDrop);
}

bool ZafTableRecord::SetBordered(bool )
{
	// bordered is false for this class.
	bordered = false;
	return (bordered);
}

bool ZafTableRecord::SetDestroyable(bool )
{
	// destroyable is false for this class.
	destroyable = false;
	return (destroyable);
}

bool ZafTableRecord::SetDisabled(bool )
{
	// disabled is false for this class.
	disabled = false;
	return (disabled);
}

bool ZafTableRecord::SetFocus(bool setFocus)
{
	// Turn on system objects.
	return (ZafWindow::SetFocus(setFocus));
}

bool ZafTableRecord::SetLocked(bool )
{
	// locked is false for this class.
	locked = false;
	return (locked);
}

bool ZafTableRecord::SetMaximized(bool )
{
	// maximized is false for this class.
	maximized = false;
	return (maximized);
}

bool ZafTableRecord::SetMinimized(bool )
{
	// minimized is false for this class.
	minimized = false;
	return (minimized);
}

bool ZafTableRecord::SetMoveable(bool )
{
	// moveable is false for this class.
	moveable = false;
	return (moveable);
}

bool ZafTableRecord::SetNoncurrent(bool )
{
	// noncurrent is false for this class.
	noncurrent = false;
	return (noncurrent);
}

ZafOffset ZafTableRecord::SetOffset(ZafOffset tOffset)
{
	// Make sure the offset has changed.
	if (offset != tOffset)
		offset = tOffset;

	// Return the current offset.
	return (offset);
}

bool ZafTableRecord::SetParentDrawFocus(bool )
{
	// parentDrawFocus is true for this class.
	parentDrawFocus = true;
	return (parentDrawFocus);
}

ZafRegionType ZafTableRecord::SetRegionType(ZafRegionType )
{
	// regionType is ZAF_INSIDE_REGION for this class.
	regionType = ZAF_INSIDE_REGION;
	return (regionType);
}

bool ZafTableRecord::SetSelected(bool setSelected)
{
	// Bypass the window level.
	return (ZafWindowObject::SetSelected(setSelected));
}

bool ZafTableRecord::SetSizeable(bool )
{
	// sizeable is false for this class.
	sizeable = false;
	return (sizeable);
}

bool ZafTableRecord::SetSupportObject(bool )
{
	// supportObject is false for this class.
	supportObject = false;
	return (supportObject);
}

bool ZafTableRecord::SetTemporary(bool )
{
	// temporary is false for this class.
	temporary = false;
	return (temporary);
}

// ----- OS Specific Functions ----------------------------------------------

void ZafTableRecord::OSMapPalette(ZafPaletteStruct &palette, ZafPaletteType type, ZafPaletteState state)
{
	// Defer to the parent table.
	parent->OSMapPalette(palette, type, state);
}

OSWindowID ZafTableRecord::OSScreenID(ZafScreenIDType type) const
{
	// Defer to the parent table.
	return (parent->OSScreenID(type));
}

void ZafTableRecord::OSRegisterObject(void)
{
	// Table record is never registered (screenID is set by RegisterObject()).
}

// ----- Persistent functions -----------------------------------------------

ZafTableRecord::ZafTableRecord(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafWindow(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	offset(-1)
{
	// Only ZafTable's virtualRecord is persisted.  Other records are created
	// at run-time as needed.

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

ZafElement *ZafTableRecord::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafTableRecord(name, persist));
}

void ZafTableRecord::Write(ZafObjectPersistence &persist)
{
	// Write the object.
	ZafWindow::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}


