//	Zinc Application Framework - Z_TABLE.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_table.hpp>
#include <z_scrll2.hpp>
#include <z_str1.hpp>
#include <z_button.hpp>
#define ZAF_TABLE_INFO
#include "gbl_def.cpp"

static ZafStringID _virtualRecordName = ZAF_ITEXT("virtualRecord");

// ----- ZafTable -----------------------------------------------------------

ZafTable::ZafTable(int left, int top, int width, int height,
	int headerWidth, int headerHeight, int rowHeight) :
	ZafWindow(left, top, width, height),
	virtualRecord(ZAF_NULLP(ZafTableRecord)),
	currentOffset(-1), maxOffset(-1), focusSize(1), grid(true),
	itemHeight(0), itemWidth(0), focusRecord(-1), focusField(-1)
{
	SetBordered(true);
	SetOSDraw(false);

	// Create the support objects.
	virtualRecord = new ZafTableRecord(width, rowHeight);
	if (headerWidth && headerHeight)
		Add(new ZafTableHeader(headerWidth, headerHeight, ZAF_CORNER_HEADER));
	if (headerHeight)
		Add(new ZafTableHeader(0, headerHeight, ZAF_COLUMN_HEADER));
	if (headerWidth)
	{
		ZafTableHeader *rowHeader = new ZafTableHeader(headerWidth, 0, ZAF_ROW_HEADER);
		ZafString *string = new ZafString(0, 0, headerWidth, ZAF_ITEXT(""), -1);
		string->SetHzJustify(ZAF_HZ_RIGHT);
		rowHeader->virtualField = string;
		Add(rowHeader);
	}
}

ZafTable::ZafTable(const ZafTable &copy) :
	ZafWindow(copy),
	virtualRecord(ZAF_NULLP(ZafTableRecord)),
	currentOffset(-1), maxOffset(copy.maxOffset), focusSize(copy.focusSize),
	grid(copy.grid), itemHeight(copy.itemHeight), itemWidth(copy.itemWidth),
	focusRecord(copy.focusRecord), focusField(copy.focusField)
{
	// ZafWindow copy constructor will have duplicated the visible records in
	// the table, so we need to get rid of those and start over.
	ZafWindowObject *object;
	while (First())
	{
		object = First();
		ZafWindow::Subtract(object);
		delete object;
	}
	SetVirtualRecord(DynamicPtrCast(copy.virtualRecord->Duplicate(), ZafTableRecord));
}

ZafTable::~ZafTable(void)
{
	// Delete the virtual record.
	if (virtualRecord)
		delete virtualRecord;
}

ZafWindowObject *ZafTable::Add(ZafWindowObject *object, ZafWindowObject *position)
{
	// Defer to the extended Add() function.
	return (Add(object, ZAF_NULLP(ZafIChar), position));
}

ZafWindowObject *ZafTable::Add(ZafWindowObject *object, ZafIChar *title, ZafWindowObject *position)
{
	// Check for a support or invalid object.
	if (object->parent)
		return (object);
	else if (object->SupportObject())
		return (ZafWindow::Add(object, position));

	// Check for a virtual record field.
	ZafTableRecord *record = DynamicPtrCast(object, ZafTableRecord);
	if (!record)
	{
		// Add the header information.
		ZafTableHeader *header = ColumnTableHeader();
		if (header)
			header->Add(new ZafString(0, 0, 1, title ? title : object->Text(), -1));

		// Check for override options.
		if (grid)
			object->SetBordered(false);
		ZafButton *button = DynamicPtrCast(object, ZafButton);
		if (button)
			button->SetAutoSize(false);

		// Add the object to the virtual record.
		return (virtualRecord->Add(object, position));
	}

	// Add the object to the table.
	object->SetSystemObject(false);
	object->SetParentPalette(true);
	object->SetParentDrawFocus(true);
	ZafWindow::Add(object, position);

	// Create a row field for the new record.
	ZafTableHeader *header = RowTableHeader();
	ZafWindowObject *rowField = ZAF_NULLP(ZafWindowObject);
	if (header && header->VirtualField())
	{
		rowField = header->VirtualField()->Duplicate();
		rowField = header->Add(rowField);
	}

	// Reposition the record if the parent is registered.
	if (screenID)
	{
		// Determine the object's list position.
		int line = record->Previous() ? record->Previous()->zafRegion.bottom + 2 : 0;
		ZafOffset newOffset = record->Previous() ? record->Previous()->Offset() + 1 : 0;
		for (ZafTableRecord *temp = record; temp; newOffset++, temp = temp->Next())
		{
			// Position the record.
			ComputeNextPosition(temp, rowField, line);
			temp->Event(N_SIZE);

			// Read the record information.
			if (temp->Offset() != newOffset)
			{
				SeekOptimized(newOffset);
				temp->SetOffset(currentOffset);
				ReadRecord(*temp, rowField);
			}

			// Determine the next row-field.
			if (rowField)
				rowField = rowField->Next();
		}
	}

	// Return the object pointer.
	return (object);
}

void ZafTable::ComputeNextPosition(ZafTableRecord *object, ZafWindowObject *rowField, int &line)
{
	// Set the object region.
	object->zafRegion.left = 0;
	object->zafRegion.right = itemWidth - 1;
	object->zafRegion.top = line;
	object->zafRegion.bottom = line = line + itemHeight - 1;
	line += 1 + focusSize;
	if (rowField)
	{
		rowField->zafRegion.left = 0;
		rowField->zafRegion.top = rowField->Previous() ? object->zafRegion.top + ZAF_HEADER_OBJECT_SHIFT : 0;
		rowField->zafRegion.right = rowField->parent->zafRegion.Width() - 1;
		rowField->zafRegion.bottom = object->zafRegion.bottom + 1 + ZAF_HEADER_OBJECT_SHIFT;
	}
}

void ZafTable::ComputePreviousPosition(ZafTableRecord *object, ZafWindowObject *rowField, int &line)
{
	// Set the object region.
	object->zafRegion.left = 0;
	object->zafRegion.right = itemWidth - 1;
	object->zafRegion.bottom = line;
	object->zafRegion.top = line = line - itemHeight + 1;
	line -= 1 + focusSize;
	if (rowField)
	{
		rowField->zafRegion.left = 0;
		rowField->zafRegion.top = rowField->Previous() ? object->zafRegion.top + ZAF_HEADER_OBJECT_SHIFT : 0;
		rowField->zafRegion.right = rowField->parent->zafRegion.Width() - 1;
		rowField->zafRegion.bottom = object->zafRegion.bottom + 1 + ZAF_HEADER_OBJECT_SHIFT;
	}
}

void ZafTable::ComputeRecordPositions(ZafOffset newOffset)
{
	// Find the row information.
	ZafTableHeader *header = RowTableHeader();
	ZafWindowObject *rowField = header ? header->First() : ZAF_NULLP(ZafWindowObject);

	// Position the existing records.
	int line = focusSize;
	int bottom = clientRegion.bottom - clientRegion.top;
	ZafTableRecord *object;
	for (object = First(); object && line + itemHeight <= bottom && newOffset <= maxOffset; object = object->Next())
	{
		// Position the record.
		ComputeNextPosition(object, rowField, line);
		if (screenID)
			object->Event(N_SIZE);

		// Check for the focus record.
		if (focusRecord == newOffset)
		{
			SetCurrent(object);
			if (focusField != -1)
				object->SetCurrent(object->Get(focusField));
		}

		// Read the record information.
		if (object->Offset() != newOffset)
		{
			SeekOptimized(newOffset);
			object->SetOffset(currentOffset);
			ReadRecord(*object, rowField);
		}
		newOffset++;

		// Determine the next row-field.
		if (rowField && line < bottom)
			rowField = rowField->Next();
	}

	// Remove excess records.
	while (object)
	{
		// Write the record information.
		if (object->Changed())
		{
			SeekOptimized(object->Offset());
			WriteRecord(*object, rowField);
			object->SetChanged(false);
		}

		// Remove the object and associated rowField.
		ZafTableRecord *nextObject = object->Next();
		if (object->zafRegion.top < clientRegion.bottom)
			object->Redisplay(); // The object area must be explicitly cleared.
		Subtract(object);
		delete object;
		object = nextObject;
		if (rowField)
		{
			ZafWindowObject *nextRowField = rowField->Next();
			if (rowField->zafRegion.top < clientRegion.bottom)
				rowField->Redisplay(); // The object area must be explicitly cleared.
			header->Subtract(rowField);
			delete rowField;
			rowField = nextRowField;
		}
	}

	// Add new records to fill the table.
	newOffset = Last() ? Last()->Offset() + 1 : 0;
	while (line + itemHeight <= bottom && newOffset <= maxOffset)
	{
		ZafTableRecord *object = DynamicPtrCast(virtualRecord->Duplicate(), ZafTableRecord);
		Add(object);
		line = object->zafRegion.bottom + 1;
		newOffset++;
	}

	// Check the scroll area.
	ScrollEvent(S_VSCROLL_COMPUTE);
}

ZafEventType ZafTable::Draw(const ZafEventStruct &event, ZafEventType ccode)
{
	// Compute the actual draw region.
	ZafRegionStruct drawRegion = BeginDraw();
	ZafRegionStruct clip = drawRegion;
	if (ccode == S_REDISPLAY_REGION)
		clip = event.region;

#	if defined(ZAF_MOTIF)
	if (ccode == E_MOTIF && event.data)
	{
		XRectangle xclip;
		XClipBox((_XRegion *)event.data, &xclip);
		clip.Assign(xclip.x, xclip.y, xclip.width, xclip.height);
	}
#	endif

	// Draw the table records.
	ZafWindowObject *object;
	ZafRegionStruct region = drawRegion; region.bottom = region.top - 1;
	for (object = First(); object; object = object->Next())
	{
		// Draw the item.
		region = object->zafRegion;
		if (region.Overlap(clip))
			object->Draw(event, ccode);
	}

	// Clear any remaining blank area.
	region.top = region.bottom + 1;
	region.bottom = drawRegion.bottom;
	if (region.Overlap(clip))
	{
		display->SetPalette(LogicalPalette(ZAF_PM_BACKGROUND, ZAF_PM_ANY_STATE));
		display->Rectangle(region, 0, true);
	}

	// Draw the grid.
//	if (grid)
		DrawGrid(drawRegion, clip);

	// Draw the region rectangle.
	object = Current();
	if (object && Focus())
	{
		drawRegion = ConvertToOSRegion(object);
		DrawFocus(drawRegion, S_CURRENT);
	}

	// Return the control code.
	EndDraw();
	return (ccode);
}

ZafEventType ZafTable::DrawFocus(ZafRegionStruct &region, ZafEventType ccode)
{
	// Clear the background area.
	if (ccode != S_CURRENT || grid)
	{
		display->SetPalette(LogicalPalette(ZAF_PM_BACKGROUND, ZAF_PM_ANY_STATE));
		display->Line(region.left, region.top - 1, region.right, region.top - 1);
		display->Line(region.left, region.bottom + 1, region.right, region.bottom + 1);
	}

	// Draw the focus area.
	if (ccode == S_CURRENT || grid)
	{
		if (ccode == S_CURRENT)
			display->SetPalette(LogicalPalette(ZAF_PM_OUTLINE, ZAF_PM_CURRENT));
		else
			display->SetPalette(LogicalPalette(ZAF_PM_DARK_SHADOW, ZAF_PM_NOT_CURRENT));
		display->Line(region.left, region.top - 1, region.right, region.top - 1);
		display->Line(region.left, region.bottom + 1, region.right, region.bottom + 1);
	}

	// Return success.
	return (ccode);
}

void ZafTable::DrawGrid(const ZafRegionStruct &drawRegion, const ZafRegionStruct &clip)
{
	ZafWindowObject *object;
	int line, column;

	// Draw the grid.
	if (grid)
		display->SetPalette(LogicalPalette(ZAF_PM_DARK_SHADOW, ZAF_PM_NOT_CURRENT));
	else
		display->SetPalette(LogicalPalette(ZAF_PM_BACKGROUND, ZAF_PM_NOT_CURRENT));
	for (object = virtualRecord->First(); object && object->Next(); object = object->Next())
	{
		column = drawRegion.left + object->zafRegion.right;
		display->Line(column, drawRegion.top, column, drawRegion.bottom);
	}
	for (line = drawRegion.top + itemHeight + 1; line < drawRegion.bottom; line += itemHeight + 1)
	{
		if (line >= clip.top && line <= clip.bottom)
			display->Line(drawRegion.left, line, drawRegion.right, line);
	}
}

ZafEventType ZafTable::Event(const ZafEventStruct &event)
{
	// Check for zinc events.
	ZafEventType ccode = LogicalEvent(event);
	ZafTableRecord *object = Current();
	switch (ccode)
	{
	case S_INITIALIZE:
		ccode = ZafWindow::Event(event);

		// Determine the record sizes.
		if (virtualRecord && virtualRecord->zafRegion.coordinateType != ZAF_PIXEL)
		{
			virtualRecord->ZafWindowObject::ConvertRegion(virtualRecord->zafRegion, ZAF_PIXEL);
			if (virtualRecord->CoordinateType() == ZAF_CELL)
				virtualRecord->zafRegion -= ZAF_SHADOW_WIDTH;
			virtualRecord->Event(event);
		}
		break;

	// ----- Position & Size messages -----
	case N_SIZE:
		virtualRecord->zafRegion.right = clientRegion.Width() - 1;

		// Compute the record positions.
		itemWidth = clientRegion.Width();
		itemHeight = virtualRecord->zafRegion.Height();
		ComputeRecordPositions(First() ? First()->Offset() : 0);

#		if defined(ZAF_MOTIF)
		// Configure the table grid area.
		XtVaSetValues(screenID, XmNwidth, clientRegion.Width(), XmNheight, clientRegion.Height(), 0);
#		endif
		break;

	// ----- Selection messages -----
	case L_BEGIN_SELECT:
		// Check for out-of-bounds.
		if (!clientRegion.Overlap(event.position, clientRegion.left, clientRegion.top))
			break;

		// Try to find an overlapping item.
		for (object = First(); object; object = object->Next())
			if (object->zafRegion.Overlap(event.position))
			{
				// Check for non-selectable overlaps.
				if (object->Disabled())
					return (S_ERROR);

				// Reset the focus.
				object->Event(event);
				break;
			}

		// Check for the current focus.
		if (!Focus())
			SetFocus(true);
		break;

	// ----- Movement messages -----
	case L_SELECT:
	case L_LEFT:
	case L_RIGHT:
	case L_BOL:
	case L_EOL:
	case L_UP:
	case L_DOWN:
	case L_FIRST:
	case L_LAST:
	case L_PGDN:
	case L_PGUP:
	case L_PREVIOUS:
		if (!object || object->MoveEvent(ccode) == S_ERROR)
			ccode = MoveEvent(ccode);
		break;

	case L_NEXT:
		// Check for movement off the table.
		if (event.key.shiftState & S_CTRL)
		{
			// Move to the next object.
			for (ZafWindowObject *focus = Next(); focus && focus != this; focus = focus->Next())
				if (!focus->Disabled() && !focus->Noncurrent())
				{
					focus->SetFocus(true);
					return (ccode);
				}
				else if (focus->Next())
					focus = focus->Next();
				else
					focus = DynamicPtrCast(parent, ZafWindow)->First();
		}
		else if (!object || object->MoveEvent(ccode) == S_ERROR)
			ccode = MoveEvent(ccode);
		break;

	// ----- Miscellaneous messages -----
	case S_HOT_KEY:
		ccode = S_UNKNOWN;
		break;

	case S_REDISPLAY:
	case S_REDISPLAY_DATA:
	case S_REDISPLAY_REGION:
		ccode = ZafWindowObject::Event(event);
		if (ccode == S_REDISPLAY_DATA)
		{
			ZafTableHeader *header = RowTableHeader();
			if (header)
				header->Event(event);
		}
		break;

	case S_DEINITIALIZE:
		ccode = ZafWindow::Event(event);

		// Write out any changed records.
		{
		ZafTableHeader *header = RowTableHeader();
		ZafWindowObject *rowField = header ? header->First() : ZAF_NULLP(ZafWindowObject);
		for (object = First(); object; object = object->Next())
			if (object->Changed())
			{
				SeekOptimized(object->Offset());
				WriteRecord(*object, rowField);
				object->SetChanged(false);
				if (rowField)
					rowField = rowField->Next();
			}
		}
		break;

	default:
		ccode = ZafWindow::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

ZafLogicalEvent ZafTable::LogicalEvent(const ZafEventStruct &event)
{
	ZafLogicalEvent ccode = MapEvent(defaultEventMap, event);
	return ((ccode == L_NONE) ? ZafWindow::LogicalEvent(event) : ccode);
}

const ZafPaletteStruct *ZafTable::MapClassPalette(ZafPaletteType type, ZafPaletteState state)
{
	const ZafPaletteStruct *palette = MapPalette(defaultPaletteMap, type, state);
	return (palette ? palette : ZafWindow::MapClassPalette(type, state));
}

ZafEventType ZafTable::MoveEvent(const ZafEventStruct &event)
{
	// Determine the new selection.
	ZafEventType ccode = event.type;
	ZafEventStruct sEvent(S_VSCROLL);
	ZafTableRecord *object = Current();
	ZafTableRecord *pObject = object ? object->Previous() : ZAF_NULLP(ZafTableRecord);
	ZafTableRecord *nObject = object ? object->Next() : ZAF_NULLP(ZafTableRecord);
//???	int height = clientRegion.bottom - clientRegion.top;
	int oldFieldIndex = (Current() && Current()->Current()) ? Current()->Current()->ListIndex() : -1;
	int fieldIndex = oldFieldIndex;

	switch (ccode)
	{
	case L_PREVIOUS:
		if (pObject)
			object = pObject;
		else if (!object)
		{
			// The current record is actually off the screen.
			if (focusField > 0)
				fieldIndex = focusField - 1;
			else
				fieldIndex = Last()->Last()->ListIndex();
			sEvent.scroll.delta = focusRecord - First()->Offset();
			ScrollEvent(sEvent);
			for (object = First(); object; object = object->Next())
				if (object->Offset() == focusRecord)
					break;
			break;
		}
		else if (object->Offset())
		{
			object->SetFocus(false);
			sEvent.scroll.delta = -1;
			ScrollEvent(sEvent);
			object = First();
		}
		else
			object = ZAF_NULLP(ZafTableRecord);
		if (object && object->Last())
			fieldIndex = object->Last()->ListIndex();
		break;

	case L_NEXT:
	case L_SELECT:
		if (nObject)
			object = nObject;
		else if (!object)
		{
			// The current record is actually off the screen.
			if (focusField < Last()->Last()->ListIndex())
				fieldIndex = focusField + 1;
			else
				fieldIndex = 0;
			sEvent.scroll.delta = focusRecord - Last()->Offset();
			ScrollEvent(sEvent);
			for (object = First(); object; object = object->Next())
				if (object->Offset() == focusRecord)
					break;
			break;
		}
		else if (object->Offset() < maxOffset)
		{
			object->SetFocus(false);
			sEvent.scroll.delta = 1;
			ScrollEvent(sEvent);
			object = Last();
		}
		else
			object = ZAF_NULLP(ZafTableRecord);
		if (object && object->First())
			fieldIndex = 0;
		break;

	case L_UP:
		if (object && object->Current())
			fieldIndex = object->Current()->ListIndex();
		if (pObject)
			object = pObject;
		else if (!object)
		{
			// The current record is actually off the screen.
			if (focusRecord > 0)
				focusRecord--;
			fieldIndex = focusField;
			sEvent.scroll.delta = focusRecord - First()->Offset();
			ScrollEvent(sEvent);
			for (object = First(); object; object = object->Next())
				if (object->Offset() == focusRecord)
					break;
		}
		else if (object->Offset())
		{
			object->SetFocus(false);
			sEvent.scroll.delta = -1;
			ScrollEvent(sEvent);
			object = First();
		}
		break;

	case L_DOWN:
		if (object && object->Current())
			fieldIndex = object->Current()->ListIndex();
		if (nObject)
			object = nObject;
		else if (!object)
		{
			// The current record is actually off the screen.
			fieldIndex = focusField;
			if (focusRecord < maxOffset)
				focusRecord++;
			sEvent.scroll.delta = focusRecord - Last()->Offset();
			ScrollEvent(sEvent);
			for (object = First(); object; object = object->Next())
				if (object->Offset() == focusRecord)
					break;
		}
		else if (object->Offset() < maxOffset)
		{
			object->SetFocus(false);
			sEvent.scroll.delta = 1;
			ScrollEvent(sEvent);
			object = Last();
		}
		break;

	case L_PGUP:
		// Check for page actions.
		if (!object)
		{
			// The current record is actually off the screen.
			focusRecord -= Count() - 1;
			if (focusRecord < 0)
				focusRecord = 0;
			fieldIndex = focusField;
			sEvent.scroll.delta = focusRecord - First()->Offset();
			ScrollEvent(sEvent);
		}
		if (object == First())
		{
			// Go to the top of the previous page.
			object->SetFocus(false);
			sEvent.scroll.delta = 1 - Count();
			if (-sEvent.scroll.delta > First()->Offset())
				sEvent.scroll.delta = -First()->Offset();
			ScrollEvent(sEvent);
		}
		object = First();
		break;

	case L_PGDN:
		// Check for page actions.
		if (!object)
		{
			// The current record is actually off the screen.
			focusRecord += Count() - 1;
			if (focusRecord > maxOffset)
				focusRecord = maxOffset;
			fieldIndex = focusField;
			sEvent.scroll.delta = focusRecord - Last()->Offset();
			ScrollEvent(sEvent);
		}
		if (object == Last())
		{
			// Go to the bottom of the next page.
			object->SetFocus(false);
			sEvent.scroll.delta = Count() - 1;
			if (sEvent.scroll.delta > maxOffset - Last()->Offset())
				sEvent.scroll.delta = maxOffset - Last()->Offset();
			ScrollEvent(sEvent);
		}
		object = Last();
		break;

	case L_FIRST:
		if (First()->Offset() != 0)
		{
			sEvent.scroll.delta = -First()->Offset();
			ScrollEvent(sEvent);
		}
		object = First();
		break;

	case L_LAST:
		if (Last()->Offset() != maxOffset)
		{
			sEvent.scroll.delta = maxOffset - Last()->Offset();
			ScrollEvent(sEvent);
		}
		object = Last();
		break;
	}

	// Check the current object.
	if (object && (object != current || oldFieldIndex != fieldIndex))
	{
		ZafWindowObject *field = (fieldIndex != -1) ? object->Get(fieldIndex) : ZAF_NULLP(ZafWindowObject);
		if (field)
			field->SetFocus(true);
		else
			object->SetFocus(true);
	}
	return (ccode);
}

ZafWindowObject *ZafTable::NotifyFocus(ZafWindowObject *focusObject, bool setFocus)
{
	// Defer to the base class.
	ZafWindowObject *object = Current();
	ZafWindowObject *invalidObject = ZafWindow::NotifyFocus(focusObject, setFocus);
	focusObject = Current();

	// Update the focus.
	if (screenID && focusObject)
	{
		ZafRegionStruct drawRegion = BeginDraw();

		// Turn off the old item focus.
		if (object && object != focusObject)
		{
			drawRegion = ConvertToOSRegion(object);
			DrawFocus(drawRegion, S_NON_CURRENT);
		}

		// Turn on the new item focus and update the focus information.
		drawRegion = ConvertToOSRegion(focusObject);
		DrawFocus(drawRegion, setFocus ? S_CURRENT : S_NON_CURRENT);

		EndDraw();
	}

	// Return the invalid object.
	return (invalidObject);
}

ZafEventType ZafTable::ScrollEvent(const ZafEventStruct &event)
{
	// Process the scrolling request.
	ZafEventType ccode = event.type;
	ZafTableRecord *record;
	switch (ccode)
	{
	case S_VSCROLL_CHECK:
		break;

	case S_VSCROLL:
	case N_VSCROLL:
		if (event.scroll.delta && First())
		{
			ZafTableHeader *header = RowTableHeader();
			ZafWindowObject *rowField = ZAF_NULLP(ZafWindowObject);
			ZafTableRecord *oldCurrent = record = Current();

			// Remove the focus rectangle.
			if (record)
			{
				ZafRegionStruct region = record->BeginDraw();
				DrawFocus(region, S_NON_CURRENT);
				record->EndDraw();

				focusRecord = record->Offset();
				if (record->Current())
					focusField = record->Index(record->Current());
			}

			// Shift the records to optimize reads and writes.
			ZafOffset delta = event.scroll.delta;
			if (-delta > First()->Offset())
				delta = -First()->Offset();
			if (delta > maxOffset - Last()->Offset())
				delta = maxOffset - Last()->Offset();
			ZafOffset newOffset = First()->Offset() + delta;
			if (delta > 0 && delta < Count())
			{
				while (delta-- > 0)
				{
					record = First();
					if (header)
						rowField = header->First();

					// Write the record information.
					if (record->Changed())
					{
						SeekOptimized(record->Offset());
						WriteRecord(*record, rowField);
						record->SetChanged(false);
					}

					// Remove the record from the table.
					if (record->Current())
					{
						ZafWindowObject::SetFocus(true);
						record->Current()->SetSystemObject(false);
					}
					ZafList::Subtract(record);
					ZafList::Add(record);
					if (rowField)
					{
						header->ZafList::Subtract(rowField);
						header->ZafList::Add(rowField);
					}
				}
			}
			else if (delta < 0 && -delta < Count())
			{
				while (delta++ < 0)
				{
					record = Last();
					if (header)
						rowField = header->Last();

					// Write the record information.
					if (record->Changed())
					{
						SeekOptimized(record->Offset());
						WriteRecord(*record, rowField);
						record->SetChanged(false);
					}

					// Remove the record from the table.
					if (record->Current())
					{
						ZafWindowObject::SetFocus(true);
						record->Current()->SetSystemObject(false);
					}
					ZafList::Subtract(record);
					ZafList::Add(record, First());
					if (rowField)
					{
						header->ZafList::Subtract(rowField);
						header->ZafList::Add(rowField, header->First());
					}
				}
			}
			else if (delta && record)
			{
				// Reset the current record.
				if (record->Current())
				{
					ZafWindowObject::SetFocus(true);
					record->Current()->SetSystemObject(false);
				}
				SetCurrent(ZAF_NULLP(ZafWindowObject));

				// Write out any changed records.
				ZafTableHeader *header = RowTableHeader();
				ZafWindowObject *rowField = header ? header->First() : ZAF_NULLP(ZafWindowObject);
				for (record = First(); record; record = record->Next())
					if (record->Changed())
					{
						SeekOptimized(record->Offset());
						WriteRecord(*record, rowField);
						record->SetChanged(false);
						if (rowField)
							rowField = rowField->Next();
					}
			}

			// Check the focus rectangle.  Notify oldCurrent that it no 
			// longer has focus.  Must do this here since its data may get
			// changed in ComputeRecordPosition(), called below.
			if (!Current() && oldCurrent)
				oldCurrent->SetFocus(false);

			// Recalculate the object regions, read new record data into 
			// records that were wrapped and set new current, if there is one.
			automaticUpdate = false;
			ComputeRecordPositions(newOffset);
			automaticUpdate = true;

			// Get pointer to "new" current record.
			record = Current();

			// Re-add the focus rectangle.
			if (record)
			{
				if (!oldCurrent)
					record->SetFocus(true);
				else
				{
					ZafRegionStruct region = record->BeginDraw();
					DrawFocus(region, S_CURRENT);
					record->EndDraw();
				}
			}

			// Check for existing repaint directives.
#			if defined(ZAF_MOTIF)
			ZafEventStruct check;
			if (screenID && Visible() && XtIsRealized(screenID) && XtWindow(screenID) &&
				(eventManager->Get(check, Q_NO_BLOCK | Q_NO_DESTROY | Q_NO_POLL | Q_END) != 0 || check.route != this))
			{
				check.route = this;
				check.type = S_REDISPLAY_DATA;
				eventManager->Put(check);
			}
#			else
			RedisplayData();
#			endif
		}
		break;

	case S_VSCROLL_COMPUTE:
		{
		// Check for a scroll-bar.
		ZafScrollBar *scrollBar = VerticalScrollBar();
		if (scrollBar && First())
		{
			// Reset the scroll values.
			ZafScrollStruct vScroll;
			vScroll.delta = 1;
			vScroll.showing = Count();
			vScroll.minimum = 0;
			vScroll.maximum = maxOffset - vScroll.showing + 1;
			vScroll.current = First()->Offset();

			// Adjust the vertical scroll-bar values.
			scrollBar->ScrollData()->SetScroll(vScroll);
		}
		}
		break;

	default:
		// No other messages apply to the table.
		return (S_ERROR);
	}

	// Return the control code.
	return (ccode);
}

ZafLogicalColor ZafTable::HeaderBackgroundColor(ZafLogicalColor *color, ZafLogicalColor *mono)
{
	// Get the background color from the header objects.
	ZafLogicalColor tColor = ZAF_CLR_NULL;
	ZafWindowObject *object = SupportFirst();
	if (object)
	{
		object->BackgroundColor(&tColor, mono);
		if (color)
			*color = tColor;
	}

	// Return the current background color.
	return (tColor);
}

ZafLogicalColor ZafTable::HeaderTextColor(ZafLogicalColor *color, ZafLogicalColor *mono)
{
	// Get the text color from the header objects.
	ZafLogicalColor tColor = ZAF_CLR_NULL;
	ZafWindowObject *object = SupportFirst();
	if (object)
	{
		object->TextColor(&tColor, mono);
		if (color)
			*color = tColor;
	}

	// Return the current text color.
	return (tColor);
}

int ZafTable::HeaderHeight(void) const
{
	const ZafTableHeader *header = ColumnTableHeader();
	if (header)
		return (header->zafRegion.Height());

	return (0);
}

int ZafTable::HeaderWidth(void) const
{
	ZafTableHeader *header = RowTableHeader();
	if (header)
		return (header->zafRegion.Width());

	return (0);
}

int ZafTable::RowHeight(void) const
{
	// Return the current value.
	return (virtualRecord->zafRegion.Height());
}

ZafCoordinateType ZafTable::SetCoordinateType(ZafCoordinateType newCoordinateType)
{
	// Make sure the attribute has changed.
	if (coordinateType != newCoordinateType && !screenID)
	{
		ZafWindowObject::SetCoordinateType(newCoordinateType);
		for (ZafWindowObject *object = SupportFirst(); object; object = object->Next())
			object->SetCoordinateType(coordinateType);
		virtualRecord->SetCoordinateType(coordinateType);
	}

	// Return the current attribute.
	return (coordinateType);
}

ZafLogicalColor ZafTable::SetHeaderBackgroundColor(ZafLogicalColor color, ZafLogicalColor mono)
{
	// Set the background color for the header objects.
	for (ZafWindowObject *object = SupportFirst(); object; object = object->Next())
		object->SetBackgroundColor(color, mono);
	return (color);
}

ZafLogicalColor ZafTable::SetHeaderTextColor(ZafLogicalColor color, ZafLogicalColor mono)
{
	// Set the text color for the header objects.
	for (ZafWindowObject *object = SupportFirst(); object; object = object->Next())
		object->SetTextColor(color, mono);
	return (color);
}

int ZafTable::SetHeaderHeight(int headerHeight)
{
	// Make sure the request is valid.
	if (!screenID)
	{
		ZafTableHeader *header = ColumnTableHeader();
		if (header)
			header->zafRegion.bottom = header->zafRegion.top + headerHeight - 1;
		header = CornerTableHeader();
		if (header)
			header->zafRegion.bottom = header->zafRegion.top + headerHeight - 1;
	}

	// Return the current value.
	return (headerHeight);
}

int ZafTable::SetHeaderWidth(int headerWidth)
{
	// Make sure the request is valid.
	if (!screenID)
	{
		ZafTableHeader *header = RowTableHeader();
		if (header)
			header->zafRegion.right = header->zafRegion.left + headerWidth - 1;
		header = CornerTableHeader();
		if (header)
			header->zafRegion.right = header->zafRegion.left + headerWidth - 1;
	}

	// Return the current value.
	return (headerWidth);
}

int ZafTable::SetRowHeight(int rowHeight)
{
	// Make sure the request is valid.
	if (!screenID)
		virtualRecord->zafRegion.bottom = virtualRecord->zafRegion.top + rowHeight - 1;

	// Return the current value.
	return (rowHeight);
}

bool ZafTable::SetAcceptDrop(bool )
{
	// acceptDrop is false for this class.
	acceptDrop = false;
	return (acceptDrop);
}

ZafOffset ZafTable::SetCurrentOffset(ZafOffset tCurrentOffset)
{
	currentOffset = tCurrentOffset;
	return (currentOffset);
}

bool ZafTable::SetDestroyable(bool )
{
	// destroyable is false for this class.
	destroyable = false;
	return (destroyable);
}

bool ZafTable::SetGrid(bool setGrid)
{
	// Check the attribute.
	if (grid != setGrid && !screenID)
		grid = setGrid;

	// Return the current attribute.
	return (grid);
}

bool ZafTable::SetLocked(bool )
{
	// locked is false for this class.
	locked = false;
	return (locked);
}

ZafOffset ZafTable::SetMaxOffset(ZafOffset tMaxOffset)
{
	maxOffset = tMaxOffset;
	return (maxOffset);
}

bool ZafTable::SetMaximized(bool )
{
	// maximized is false for this class.
	maximized = false;
	return (maximized);
}

bool ZafTable::SetMinimized(bool )
{
	// minimized is false for this class.
	minimized = false;
	return (minimized);
}

bool ZafTable::SetMoveable(bool )
{
	// moveable is false for this class.
	moveable = false;
	return (moveable);
}

bool ZafTable::SetSizeable(bool )
{
	// sizeable is false for this class.
	sizeable = false;
	return (sizeable);
}

bool ZafTable::SetTemporary(bool )
{
	// temporary is false for this class.
	temporary = false;
	return (temporary);
}

ZafTableRecord *ZafTable::SetVirtualRecord(ZafTableRecord *newVirtualRecord)
{
	// Make sure this is a valid request.
	if (!screenID && virtualRecord != newVirtualRecord)
	{
		if (virtualRecord)
			delete virtualRecord;
		virtualRecord = newVirtualRecord;
	}

	// Return a pointer to the virtual record.
	return (virtualRecord);
}

ZafWindowObject *ZafTable::Subtract(ZafWindowObject *object)
{
	// Defer to the base class.
	return (ZafWindow::Subtract(object));
}

// ----- Data Members -------------------------------------------------------

ZafError ZafTable::DeleteRecord(ZafOffset deleteOffset)
{
	if (deleteOffset <= maxOffset)
	{
		maxOffset--;

		ZafOffset firstOffset = First()->Offset();
		ZafOffset lastOffset = Last()->Offset();

		if (deleteOffset < firstOffset)
		{
			ZafOffset offset = firstOffset - 1;
			for (ZafTableRecord *record = First(); record; record = record->Next())
				record->SetOffset(offset++);
			Event(S_VSCROLL_COMPUTE);
		}
		else if (deleteOffset <= lastOffset)
		{
			for (ZafTableRecord *record = First(); record; record = record->Next())
			{
				if (record == Current())
					focusRecord = record->Offset();
				if (record->Offset() >= deleteOffset)
					record->SetOffset(-1);
			}

			ComputeRecordPositions(firstOffset);

			if (Current())
				Current()->Draw(ZafEventStruct(S_REDISPLAY), S_REDISPLAY);
		}
	}

	return (ZAF_ERROR_INVALID);
}

ZafError ZafTable::InsertRecord(ZafOffset insertOffset)
{
	if (insertOffset <= maxOffset + 1)
	{
		maxOffset++;

		ZafOffset firstOffset = First() ? First()->Offset() : 0;
		ZafOffset lastOffset = Last()? Last()->Offset() : 0;

		if (insertOffset < firstOffset)
		{
			ZafOffset offset = firstOffset + 1;
			for (ZafTableRecord *record = First(); record; record = record->Next())
				record->SetOffset(offset++);
			Event(S_VSCROLL_COMPUTE);
		}
		else if (insertOffset <= lastOffset)
		{
			for (ZafTableRecord *record = First(); record; record = record->Next())
			{
				if (record->Offset() == insertOffset)
					focusRecord = record->Offset();
				if (record->Offset() >= insertOffset)
					record->SetOffset(-1);
			}

			ComputeRecordPositions(firstOffset);

			if (maxOffset == 0)
			{
				if (focus)
					First()->SetFocus(true);
				else
					SetCurrent(First());
			}

			if (Current())
				Current()->Draw(ZafEventStruct(S_REDISPLAY), S_REDISPLAY);
		}
	}

	return (ZAF_ERROR_INVALID);
}


ZafError ZafTable::SeekNextRecord(void)
{
	if (currentOffset > maxOffset)
		return (ZAF_ERROR_INVALID);
	currentOffset++;
	return (ZAF_ERROR_NONE);
}

ZafError ZafTable::SeekOptimized(ZafOffset newOffset)
{
	// Optimize the seek.
	if (currentOffset == newOffset)
		; // no seek is necessary;
	else if (currentOffset + 1 == newOffset)
		SeekNextRecord();
	else if (currentOffset - 1 == newOffset)
		SeekPreviousRecord();
	else
		SeekRandomRecord(newOffset, ZAF_SEEK_START); // no hope!
	return (ZAF_ERROR_NONE);
}

ZafError ZafTable::SeekPreviousRecord(void)
{
	if (currentOffset == 0)
		return (ZAF_ERROR_INVALID);
	currentOffset--;
	return (ZAF_ERROR_NONE);
}

ZafError ZafTable::SeekRandomRecord(ZafOffset offset, ZafSeek location)
{
	// Reset the location.
	if (location == ZAF_SEEK_CURRENT)
		offset += currentOffset;
	else if (location == ZAF_SEEK_END)
		offset += maxOffset;

	// Check for out-of-bounds.
	if (offset < 0 || offset > maxOffset)
		return (ZAF_ERROR_INVALID);

	// Reset the current value.
	currentOffset = offset;
	return (ZAF_ERROR_NONE);
}

ZafOffset ZafTable::Tell(void)
{
	// Return the current position.
	return (currentOffset);
}

// ----- Persistent functions -----------------------------------------------

ZafTable::ZafTable(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafWindow(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	virtualRecord(ZAF_NULLP(ZafTableRecord)),
	currentOffset(-1), focusSize(1), itemHeight(0), itemWidth(0),
	focusRecord(-1), focusField(-1)
{
	// ZafWindow persistent constructor loaded the virtualRecord and added it
	// as a child, so need to remove it.
	while (First())
	{
		ZafWindowObject *object = First();
		ZafWindow::Subtract(object);
		delete object;
	}

	if (persist.Error() == ZAF_ERROR_NONE)
		ZafTable::ReadData(persist);

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafTable::ReadData(ZafObjectPersistence &persist)
{
	// Read the data.
	ZafIChar virtualRecordName[ZAF_MAXNAMELEN];
	ZafFile *file = persist.CurrentFile();
	ZafUInt16 flag1;
	ZafInt32 value32;
	*file >> flag1 >> value32 >> virtualRecordName;
	grid = (flag1 & 0x0001) ? true : false;
	maxOffset = (ZafOffset)value32;

	// Read the virtualRecord.
	if (*virtualRecordName)
		SetVirtualRecord(new ZafTableRecord(virtualRecordName, persist));
}

void ZafTable::Write(ZafObjectPersistence &persist)
{
	// Bypass ZafWindow calls to keep the children from being saved.
	ZafWindowObject::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafWindow::WriteData(persist);
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafWindow::WriteSupportChildren(persist);
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafTable::WriteData(persist);

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafTable::WriteData(ZafObjectPersistence &persist)
{
	if (virtualRecord && !virtualRecord->StringID())
	{
		virtualRecord->SetStringID(_virtualRecordName);
		virtualRecord->Event(S_INITIALIZE);
	}
	const ZafIChar *virtualRecordName = virtualRecord ? virtualRecord->StringID() : ZAF_NULLP(ZafIChar);

	ZafFile *file = persist.CurrentFile();
	ZafUInt16 flag1 = 0;
	flag1 |= grid ? 0x0001 : 0;
	ZafInt32 value32 = (ZafInt32)maxOffset;
	*file << flag1 << value32 << virtualRecordName;

	// Write the virtualRecord.
	if (virtualRecord)
		virtualRecord->Write(persist);
}

