//	Zinc Application Framework - Z_SYS.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_sys.hpp>
#include <z_app.hpp>
#define ZAF_SYSTEM_BUTTON_INFO
#include "img_def.cpp"
#include "lang_def.cpp"
#include "gbl_def.cpp"

ZafSystemButton *ZafWindow::SystemButton(void) const
{
	for (ZafWindowObject *object = SupportFirst(); object; object = object->Next())
		if (object->NumberID() == ZAF_NUMID_SYSTEM)
			return (DynamicPtrCast(object, ZafSystemButton));
	return (ZAF_NULLP(ZafSystemButton));
}

ZafPopUpMenu *ZafWindow::SystemButtonMenu(void) const
{
	for (ZafWindowObject *object = SupportFirst(); object; object = object->Next())
		if (object->NumberID() == ZAF_NUMID_SYSTEM)
		{
			ZafSystemButton *sysButton = DynamicPtrCast(object, ZafSystemButton);
			return (sysButton ? &sysButton->menu : ZAF_NULLP(ZafPopUpMenu));
		}
	return (ZAF_NULLP(ZafPopUpMenu));
}

// ----- ZafSystemButton ----------------------------------------------------

static ZafStringIDChar ZAF_FARDATA _stringID[] = ZAF_ITEXT("ZAF_NUMID_SYSTEM");
static ZafStringIDChar ZAF_FARDATA _menuStringID[] = ZAF_ITEXT("ZAF_NUMID_SYSTEM_BUTTON_MENU");

ZafSystemButton::ZafSystemButton(ZafSystemButtonType zSystemButtonType) :
	ZafButton(0, 0, 0, 0, ZAF_NULLP(ZafIChar)),
	menu(0, 0), systemButtonType((ZafSystemButtonType)-1)
{
	// Check the image and language information.
	ImageAllocate();
	LanguageAllocate();

	// Initialize the system button information.
	SetNumberID(ZAF_NUMID_SYSTEM);
	SetStringID(_stringID);
	SetSystemButtonType(zSystemButtonType);

	SetBordered(false);
	SetNoncurrent(true);
	SetSupportObject(true);
	SetSystemObject(false);
	SetRegionType(ZAF_AVAILABLE_REGION);

	SetHzJustify(ZAF_HZ_CENTER);
	SetDepth(1);
	SetButtonType(ZAF_3D_BUTTON);

	menu.parent = this;
	menu.SetTemporary(true);
	menu.SetDestroyable(false);
	menu.SetNumberID(ZAF_NUMID_SYSTEM_BUTTON_MENU);
	menu.SetStringID(_menuStringID);
}

ZafSystemButton::ZafSystemButton(const ZafSystemButton &copy) :
	ZafButton(copy), menu(copy.menu),
	systemButtonType((ZafSystemButtonType)-1)
{
	// Check the image and language information.
	ImageAllocate();
	LanguageAllocate();

	// Initialize the system-button information.
	SetSystemButtonType(copy.systemButtonType);
}

ZafSystemButton::~ZafSystemButton(void)
{
	// Check the image and language information.
	ImageFree();
	LanguageFree();
}

ZafWindowObject *ZafSystemButton::Duplicate(void)
{
	return (new ZafSystemButton(*this));
}

void ZafSystemButton::ImageAllocate(void)
{
	// Initialize the images.
	if (!sysBitmap)
	{
		ZafApplication::AddStaticModule(ImageFree);
		sysBitmap = new ZafBitmapData(_sys_ZafBitmap);
		sysBitmap->staticHandle = true;
		sysBitmap->SetDestroyable(false);
	}
}

void ZafSystemButton::ImageFree(bool globalRequest)
{
	// Destroy the static images.
	if (globalRequest && sysBitmap)
	{
		sysBitmap->staticHandle = false;
		delete sysBitmap;
		sysBitmap = ZAF_NULLP(ZafBitmapData);
	}
}

void ZafSystemButton::LanguageAllocate(void)
{
	// Check for initialization.
	if (language)
		return;
	ZafApplication::AddStaticModule(LanguageFree);

	// Try the zafDataManager.
	if (!language && zafDataManager)
		language = DynamicPtrCast(zafDataManager->AllocateData(className, ZafLanguageData::className, ZafLanguageData::classID), ZafLanguageData);

	// Default to code initialization.
	if (!language)
		language = new ZafLanguageData(defaultStrings, true);
}

void ZafSystemButton::LanguageFree(bool globalRequest)
{
	// Destroy the static language.
	if (globalRequest && language && language->Destroyable() &&
		language->NotifyCount() == 0)
	{
		delete language;
		language = ZAF_NULLP(ZafLanguageData);
	}
}

const ZafPaletteStruct *ZafSystemButton::MapClassPalette(ZafPaletteType type, ZafPaletteState state)
{
	const ZafPaletteStruct *palette = MapPalette(defaultPaletteMap, type, state);
	return (palette ? palette : ZafButton::MapClassPalette(type, state));
}

bool ZafSystemButton::SetAcceptDrop(bool )
{
	// acceptDrop is false for this class.
	acceptDrop = false;
	return (acceptDrop);
}

bool ZafSystemButton::SetAllowDefault(bool )
{
	// allowDefault is false for this class.
	allowDefault = false;
	return (allowDefault);
}

bool ZafSystemButton::SetAllowToggling(bool )
{
	// allowToggling is false for this class.
	allowToggling = false;
	return (allowToggling);
}

bool ZafSystemButton::SetAutoRepeatSelection(bool )
{
	// autoRepeatSelection is false for this class.
	autoRepeatSelection = false;
	return (autoRepeatSelection);
}

bool ZafSystemButton::SetAutoSize(bool )
{
	// autoSize is true for this class.
	autoSize = true;
	return (autoSize);
}

bool ZafSystemButton::SetBordered(bool )
{
	// bordered is false for this class.
	bordered = false;
	return (bordered);
}

ZafButtonType ZafSystemButton::SetButtonType(ZafButtonType )
{
	// buttonType is ZAF_3D_BUTTON for this class.
	buttonType = ZAF_3D_BUTTON;
	return (buttonType);
}

bool ZafSystemButton::SetCopyDraggable(bool )
{
	// copyDraggable is false for this class.
	copyDraggable = false;
	return (copyDraggable);
}

int ZafSystemButton::SetDepth(int )
{
	// depth is 1 for this class.
	depth = 1;
	return (depth);
}

bool ZafSystemButton::SetDisabled(bool )
{
	// disabled is false for this class.
	disabled = false;
	return (disabled);
}

bool ZafSystemButton::SetFocus(bool )
{
	// focus is false for this class.
	focus = false;
	return (focus);
}

const ZafIChar *ZafSystemButton::SetHelpContext(const ZafIChar *)
{
	// help context is empty for this class.
	return (ZAF_NULLP(ZafIChar));
}

const ZafIChar *ZafSystemButton::SetHelpObjectTip(const ZafIChar *)
{
	// helpObjectTip cannot be set for this class.
	return (ZAF_NULLP(ZafIChar));
}

ZafIChar ZafSystemButton::SetHotKey(ZafIChar , int )
{
	// hotKeyChar and hotKeyIndex are null for this class.
	hotKeyChar = 0;
	hotKeyIndex = -1;
	return (hotKeyChar);
}

ZafHzJustify ZafSystemButton::SetHzJustify(ZafHzJustify )
{
	// hzJustify is ZAF_HZ_CENTER for this class.
	hzJustify = ZAF_HZ_CENTER;
	return (hzJustify);
}

bool ZafSystemButton::SetLinkDraggable(bool )
{
	// linkDraggable is false for this class.
	linkDraggable = false;
	return (linkDraggable);
}

bool ZafSystemButton::SetMoveDraggable(bool )
{
	// moveDraggable is false for this class.
	moveDraggable = false;
	return (moveDraggable);
}

bool ZafSystemButton::SetNoncurrent(bool )
{
	// noncurrent is true for this class.
	noncurrent = true;
	return (noncurrent);
}

bool ZafSystemButton::SetParentDrawBorder(bool )
{
	// parentDrawBorder is false for this class.
	parentDrawBorder = false;
	return (parentDrawBorder);
}

bool ZafSystemButton::SetParentDrawFocus(bool )
{
	// parentDrawFocus is false for this class.
	parentDrawFocus = false;
	return (parentDrawFocus);
}

bool ZafSystemButton::SetParentPalette(bool )
{
	// parentPalette is false for this class.
	parentPalette = false;
	return (parentPalette);
}

const ZafIChar *ZafSystemButton::SetQuickTip(const ZafIChar *)
{
	// quickTip cannot be set for this class.
	return (ZAF_NULLP(ZafIChar));
}

ZafRegionType ZafSystemButton::SetRegionType(ZafRegionType )
{
	// regionType is ZAF_AVAILABLE_REGION for this class.
	regionType = ZAF_AVAILABLE_REGION;
	return (regionType);
}

bool ZafSystemButton::SetSelected(bool )
{
	// selected is false for this class.
	selected = false;
	return (selected);
}

bool ZafSystemButton::SetSelectOnDoubleClick(bool )
{
	// selectOnDoubleClick is false for this class.
	selectOnDoubleClick = false;
	return (selectOnDoubleClick);
}

bool ZafSystemButton::SetSelectOnDownClick(bool )
{
	// selectOnDownClick is false for this class.
	selectOnDownClick = false;
	return (selectOnDownClick);
}

bool ZafSystemButton::SetSendMessageWhenSelected(bool )
{
	// sendMessageWhenSelected is true for this class.
	sendMessageWhenSelected = false;
	return (sendMessageWhenSelected);
}

ZafEventType ZafSystemButton::SetValue(ZafEventType )
{
	// value is 0 for this class.
	value = 0;
	return (value);
}

bool ZafSystemButton::SetSupportObject(bool )
{
	// supportObject is true for this class.
	supportObject = true;
	return (supportObject);
}

ZafUserFunction ZafSystemButton::SetUserFunction(ZafUserFunction )
{
	// userFunction is null for this class.
	userFunction = ZAF_NULLF(ZafUserFunction);
	return (userFunction);
}

ZafVtJustify ZafSystemButton::SetVtJustify(ZafVtJustify )
{
	// vtJustify is ZAF_VT_CENTER for this class.
	vtJustify = ZAF_VT_CENTER;
	return (vtJustify);
}

// ----- Persistent functions -----------------------------------------------

ZafSystemButton::ZafSystemButton(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafButton(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	menu(0, 0), systemButtonType((ZafSystemButtonType)-1)
{
	// Check the image information.
	ImageAllocate();
	LanguageAllocate();

	// Set the system button information.
	SetBordered(false);
	SetNoncurrent(true);
	SetHzJustify(ZAF_HZ_CENTER);
	SetSupportObject(true);
	SetSystemObject(false);
	SetRegionType(ZAF_AVAILABLE_REGION);
	SetNumberID(ZAF_NUMID_SYSTEM);
	SetStringID(_stringID);
	SetDepth(1);

	menu.SetTemporary(true);
	menu.SetDestroyable(false);
	menu.SetNumberID(ZAF_NUMID_SYSTEM_BUTTON_MENU);
	menu.SetStringID(_menuStringID);

	// Read the object data.
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafSystemButton::ReadData(persist);

	// Read the menu information.
	if (systemButtonType != ZAF_NATIVE_SYSTEM_BUTTON)
		menu.ReadChildren(persist);

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

ZafElement *ZafSystemButton::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafSystemButton(name, persist));
}

void ZafSystemButton::ReadData(ZafObjectPersistence &persist)
{
	// Read the system button information.
	ZafUInt16 flag1;
	ZafFile *file = persist.CurrentFile();
	*file >> flag1;
	SetSystemButtonType((ZafSystemButtonType)flag1);
}

void ZafSystemButton::Write(ZafObjectPersistence &persist)
{
	// Write the object.
	ZafButton::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafSystemButton::WriteData(persist);

	// Write the menu information.
	if (systemButtonType != ZAF_NATIVE_SYSTEM_BUTTON)
		menu.WriteChildren(persist);

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafSystemButton::WriteData(ZafObjectPersistence &persist)
{
	// Write the data.
	ZafUInt16 flag1 = (ZafUInt16)systemButtonType;
	ZafFile *file = persist.CurrentFile();
	*file << flag1;
}

