//	Zinc Application Framework - Z_REGION.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_coord.hpp>
#include <z_dsp.hpp>
#include <z_pos.hpp>
#include <z_region.hpp>
#define ZAF_REGION_INFO
#include "gbl_def.cpp"

// ----- ZafCoordinateStruct ------------------------------------------------

long ZafCoordinateStruct::ConvertXValue(long value, ZafCoordinateType typeIn, ZafCoordinateType typeOut)
{
	// This function converts a value in "typeIn" coordinates to a value
	// in "typeOut" coordinates.  Rounding is used to eliminate "Drift."
	// Optimizations favor speed over size.

	// Make sure conversion is necessary.
	if (typeOut == typeIn)
		return value;

	// Convert the units.
	if (typeIn == ZAF_CELL)
	{
		if (typeOut == ZAF_MINICELL)
		{
			if (value >= 0)
				return ((value * 2 * miniDenominatorX + miniNumeratorX) / (2 * miniNumeratorX));
			else
				return ((value * 2 * miniDenominatorX - miniNumeratorX) / (2 * miniNumeratorX));
		}
		else
			// Convert to pixel coordinates.
			value *= cellWidth;
	}
	else if (typeIn == ZAF_MINICELL)
	{
		if (typeOut == ZAF_CELL)
		{
			if (value >= 0)
				return ((value * 2 * miniNumeratorX + miniDenominatorX) / (2 * miniDenominatorX));
			else
				return ((value * 2 * miniNumeratorX - miniDenominatorX) / (2 * miniDenominatorX));
		}
		else
		{
			// Convert to pixel coordinates.
			if (value >= 0)
				value = (value * 2 * cellWidth * miniNumeratorX + miniDenominatorX) /
					(2 * miniDenominatorX);
			else
				value = (value * 2 * cellWidth * miniNumeratorX - miniDenominatorX) /
					(2 * miniDenominatorX);
		}
	}
	else if (typeIn == ZAF_POINTS)
	{
		if (typeOut == ZAF_TWIPS)
			return (value * 20);
		else
		{
			// Convert to pixel coordinates.
			if (value >= 0)
				value = (value * pixelsPerInchX + 36) / 72;
			else
				value = (value * pixelsPerInchX - 36) / 72;
		}
	}
	else if (typeIn == ZAF_TWIPS)
	{
		if (typeOut == ZAF_POINTS)
		{
			if (value >= 0)
				return ((value + 10) / 20);
			else
				return ((value - 10) / 20);
		}
		else
		{
			// Convert to pixel coordinates.
			if (value >= 0)
				value = (value * pixelsPerInchX + 720) / 1440;
			else
				value = (value * pixelsPerInchX - 720) / 1440;
		}
	}

	if (typeOut == ZAF_PIXEL)
		return (value);
	else if (typeOut == ZAF_CELL)
	{
		if (value >= 0)
			return ((value * 2 + cellWidth) / (2 * cellWidth));
		else
			return ((value * 2 - cellWidth) / (2 * cellWidth));
	}
	else if (typeOut == ZAF_MINICELL)
	{
		if (value >= 0)
			return ((value * 2 * miniDenominatorX + (cellWidth * miniNumeratorX)) /
				(2 * miniNumeratorX * cellWidth));
		else
			return ((value * 2 * miniDenominatorX - (cellWidth * miniNumeratorX)) /
				(2 * miniNumeratorX * cellWidth));
	}
	else if (typeOut == ZAF_POINTS)
	{
		if (value >= 0)
			return ((value * 144 + pixelsPerInchX) / (2 * pixelsPerInchX));
		else
			return ((value * 144 - pixelsPerInchX) / (2 * pixelsPerInchX));
	}
	else if (typeOut == ZAF_TWIPS)
	{
		if (value >= 0)
			return ((value * 2880 + pixelsPerInchX) / (2 * pixelsPerInchX));
		else
			return ((value * 2880 - pixelsPerInchX) / (2 * pixelsPerInchX));
	}

	return (0); // Error condition.
}

long ZafCoordinateStruct::ConvertYValue(long value, ZafCoordinateType typeIn, ZafCoordinateType typeOut)
{
	// This function converts a value in "typeIn" coordinates to a value
	// in "typeOut" coordinates.  Rounding is used to eliminate "Drift."
	// Optimizations favor speed over size.

	// Convert the units.
	if (typeIn == ZAF_CELL)
	{
		if (typeOut == ZAF_MINICELL)
		{
			if (value >= 0)
				return ((value * 2 * miniDenominatorY + miniNumeratorY) / (2 * miniNumeratorY));
			else
				return ((value * 2 * miniDenominatorY - miniNumeratorY) / (2 * miniNumeratorY));
		}
		else
			// Convert to pixel coordinates.
			value *= cellHeight;
	}
	else if (typeIn == ZAF_MINICELL)
	{
		if (typeOut == ZAF_CELL)
		{
			if (value >= 0)
				return ((value * 2 * miniNumeratorY + miniDenominatorY) / (2 * miniDenominatorY));
			else
				return ((value * 2 * miniNumeratorY - miniDenominatorY) / (2 * miniDenominatorY));
		}
		else
		{
			// Convert to pixel coordinates.
			if (value >= 0)
				value = (value * 2 * cellHeight * miniNumeratorY + miniDenominatorY) /
					(2 * miniDenominatorY);
			else
				value = (value * 2 * cellHeight * miniNumeratorY - miniDenominatorY) /
					(2 * miniDenominatorY);
		}
	}
	else if (typeIn == ZAF_POINTS)
	{
		if (typeOut == ZAF_TWIPS)
			return (value * 20);
		else
		{
			// Convert to pixel coordinates.
			if (value >= 0)
				value = (value * pixelsPerInchY + 36) / 72;
			else
				value = (value * pixelsPerInchY - 36) / 72;
		}
	}
	else if (typeIn == ZAF_TWIPS)
	{
		if (typeOut == ZAF_POINTS)
		{
			if (value >= 0)
				return ((value + 10) / 20);
			else
				return ((value - 10) / 20);
		}
		else
		{
			// Convert to pixel coordinates.
			if (value >= 0)
				value = (value * pixelsPerInchY + 720) / 1440;
			else
				value = (value * pixelsPerInchY - 720) / 1440;
		}
	}

	if (typeOut == ZAF_PIXEL)
		return (value);
	else if (typeOut == ZAF_CELL)
	{
		if (value >= 0)
			return ((value * 2 + cellHeight) / (2 * cellHeight));
		else
			return ((value * 2 - cellHeight) / (2 * cellHeight));
	}
	else if (typeOut == ZAF_MINICELL)
	{
		if (value >= 0)
			return ((value * 2 * miniDenominatorY + (cellHeight * miniNumeratorY)) /
				(2 * miniNumeratorY * cellHeight));
		else
			return ((value * 2 * miniDenominatorY - (cellHeight * miniNumeratorY)) /
				(2 * miniNumeratorY * cellHeight));
	}
	else if (typeOut == ZAF_POINTS)
	{
		if (value >= 0)
			return ((value * 144 + pixelsPerInchY) / (2 * pixelsPerInchY));
		else
			return ((value * 144 - pixelsPerInchY) / (2 * pixelsPerInchY));
	}
	else if (typeOut == ZAF_TWIPS)
	{
		if (value >= 0)
			return ((value * 2880 + pixelsPerInchY) / (2 * pixelsPerInchY));
		else
			return ((value * 2880 - pixelsPerInchY) / (2 * pixelsPerInchY));
	}

	return (0); // Error condition.
}

// ----- ZafPositionStruct --------------------------------------------------

void ZafPositionStruct::ConvertCoordinates(ZafCoordinateType newType)
{
	// Check for a valid conversion.
	if (newType != coordinateType)
	{
		column = (int)ConvertXValue(column, coordinateType, newType);
		line = (int)ConvertYValue(line, coordinateType, newType);

		// Reset the new type.
		coordinateType = newType;
	}
}

ZafPositionStruct ZafPositionStruct::Position(ZafCoordinateType type)
{
	// Convert the position.
	ZafPositionStruct position = *this;
	position.ConvertCoordinates(type);
	return (position);
}

// ----- ZafRegionStruct ----------------------------------------------------

void ZafRegionStruct::ConvertCoordinates(ZafCoordinateType newType)
{
	// Check for a valid conversion.
	if (newType != coordinateType)
	{
		// Perform coordinate conversion.
		int width = (int)ConvertXValue(Width(), coordinateType, newType);
		int height = (int)ConvertYValue(Height(), coordinateType, newType);
		left = (int)ConvertXValue(left, coordinateType, newType);
		top = (int)ConvertYValue(top, coordinateType, newType);
		right = left + width - 1;
		bottom = top + height - 1;

		// Reset the new type.
		coordinateType = newType;
	}
}

ZafRegionStruct ZafRegionStruct::Region(ZafCoordinateType type)
{
	// Convert the region.
	ZafRegionStruct region = *this;
	region.ConvertCoordinates(type);
	return (region);
}

ZafRegionStruct ZafRegionStruct::operator++(void)
{
	--left, --top, ++right, ++bottom;
	return (*this);
}

ZafRegionStruct ZafRegionStruct::operator++(int)
{
	ZafRegionStruct tmp = *this;
	left--, top--, right++, bottom++;
	return (tmp);
}

ZafRegionStruct ZafRegionStruct::operator--(void)
{
	left++, top++, right--, bottom--;
	return (*this);
}

ZafRegionStruct ZafRegionStruct::operator--(int)
{
	ZafRegionStruct tmp = *this;
	++left, ++top, --right, --bottom;
	return (tmp);
}

// ----- ZafRegionElement & ZafRegionList -----------------------------------

ZafRegionElement::ZafRegionElement(OSWindowID _screenID, const ZafRegionStruct &_region) :
	useCount(0), screenID(_screenID)
{
	region = _region;
}

ZafRegionElement::ZafRegionElement(OSWindowID _screenID, int _left,
	int _top, int _right, int _bottom, ZafCoordinateType type) :
	useCount(0), screenID(_screenID)
{
	region.left = _left;
	region.top = _top;
	region.right = _right;
	region.bottom = _bottom;
	region.coordinateType = type;
}

ZafRegionElement::~ZafRegionElement(void)
{
}

ZafRegionList::ZafRegionList(void)
{
}

ZafRegionList::~ZafRegionList(void)
{
}

void ZafRegionList::Split(OSWindowID screenID, const ZafRegionStruct &region,
	int allocateBelow)
{
	ZafRegionStruct tRegion, sRegion;
	ZafRegionElement *dRegion, *t_dRegion;

	// Split any overlapping regions.
	for (dRegion = First(); dRegion; dRegion = t_dRegion)
	{
		// Preset the previous region element.
		t_dRegion = dRegion->Next();

		// Object encompasses the whole region.
		if (dRegion->region.Encompassed(region))
		{
			ZafList::Subtract(dRegion);
			delete dRegion;
		}

		// Object overlaps the region.
		else if (dRegion->region.Overlap(region, tRegion))
		{
			screenID = dRegion->screenID;
			tRegion = dRegion->region;
			sRegion = region;

			// Region is split at a maximum shown by the following set
			// of regions:
			//		 _________
			//		|_________|		1
			//		|__|___|__|		2,3,4
			//		|_________|		5
			//

			// Check for a top region (region 1 above).
			if (region.top > tRegion.top)
				ZafList::Add(new ZafRegionElement(screenID, tRegion.left,
					tRegion.top, tRegion.right, region.top - 1,
					region.coordinateType), ZAF_NULLP(ZafElement));
			else
				sRegion.top = tRegion.top;

			// Check for a bottom region (region 5 above).
			if (region.bottom < tRegion.bottom)
				ZafList::Add(new ZafRegionElement(screenID, tRegion.left,
					region.bottom + 1, tRegion.right, tRegion.bottom,
					region.coordinateType), ZAF_NULLP(ZafElement));
			else
				sRegion.bottom = tRegion.bottom;

			// Check for a left region (region 2 above).
			if (region.left > tRegion.left)
				ZafList::Add(new ZafRegionElement(screenID, tRegion.left,
					sRegion.top, region.left - 1, sRegion.bottom,
					region.coordinateType), ZAF_NULLP(ZafElement));

			// Check for a right region (region 4 above).
			if (region.right < tRegion.right)
				ZafList::Add(new ZafRegionElement(screenID, region.right + 1,
					sRegion.top, tRegion.right, sRegion.bottom,
					region.coordinateType), ZAF_NULLP(ZafElement));

			// Region 3 is the object's region.
			ZafList::Subtract(dRegion);
			delete dRegion;
		}
	}

	// Check for new allocation space.
	if (!First() && allocateBelow)
		ZafList::Add(new ZafRegionElement(screenID, region.left,
			region.bottom + 10, region.right, region.bottom + 20,
			region.coordinateType), ZAF_NULLP(ZafElement));
}

