//	Zinc Application Framework - Z_PRST1.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_win.hpp>
#include <z_string.hpp>
#include <z_utils.hpp>
#if defined(ZAF_PERSIST_ALL)
#	include <zaf.hpp>
#endif
#define ZAF_OBJECT_PERSISTENCE_INFO
#include "gbl_def.cpp"

ZafClassID ZafObjectPersistence::GetCurrentClassID(void)
{
	// Check for a valid level.
	if (current < 0)
		return (ID_END);

	// Find a valid entry.
	ZafClassID matchID = entry[current].classID;
	if (!matchID && entry[current].className)
	{
		// First, try the object table.
		int i = 0;
		for (i = 0; objectConstructor[i].classID != ID_END; i++)
			if (!streq(objectConstructor[i].className, entry[current].className))
			{
				matchID = objectConstructor[i].classID;
				break;
			}

		// Next, try the data table.
		if (!matchID)
			matchID = ZafDataPersistence::CurrentClassID();
	}

	// Return the class name.
	return (matchID);
}

ZafClassName ZafObjectPersistence::GetCurrentClassName(void)
{
	// Check for a valid level.
	if (current < 0)
		return (ZAF_NULLP(ZafDataNameChar));

	// Find a valid entry.
	ZafClassName matchName = entry[current].className;
	if (!matchName && entry[current].classID)
	{
		// First, try the object table.
		int i = 0;
		for (i = 0; objectConstructor[i].classID != ID_END; i++)
			if (objectConstructor[i].classID == entry[current].classID)
			{
				matchName = objectConstructor[i].className;
				break;
			}

		// Next, try the data table.
		if (!matchName)
			matchName = ZafDataPersistence::CurrentClassName();
	}

	// Return the class name.
	return (matchName);
}

// ----- ZafObjectPersistence -----------------------------------------------

ZafObjectPersistence::ZafObjectPersistence(ZafFileSystem *tFileSystem, DataConstructor *tDataConstructor,
	ObjectConstructor *tObjectConstructor, UserFunction *tUserCallback,
	CompareFunction *tCompareFunction, UserObject *tUserObject) :
	ZafDataPersistence(tFileSystem, tDataConstructor)
{
	objectConstructor = ZAF_NULLP(ObjectConstructor);
	SetObjectConstructors(tObjectConstructor);
	compareFunction = tCompareFunction;
	userFunction = tUserCallback;
	userObject = tUserObject;
}

ZafObjectPersistence::ZafObjectPersistence(const ZafObjectPersistence &copy) :
	ZafDataPersistence(copy)
{
	objectConstructor = ZAF_NULLP(ObjectConstructor);
	SetObjectConstructors(copy.objectConstructor);
	compareFunction = copy.compareFunction;
	userFunction = copy.userFunction;
	userObject = copy.userObject;
}

ZafObjectPersistence::~ZafObjectPersistence(void)
{
	// Clear the tables.
	ClearObjectConstructors();
	ClearCompareFunctions();
	ClearUserCallback();
	ClearUserObject();
}

ZafObjectPersistence::CompareFunction *ZafObjectPersistence::AddCompareFunction(
	ZafDataName name, ZafCompareFunction function)
{
	// Check for an existing function.
	int offset = 0;
	if (compareFunction)
	{
		// Try to find a matching class.
		for ( ; compareFunction[offset].name; offset++)
			if (!streq(name, compareFunction[offset].name))
			{
				// Check for invalid function.
				if (function && function != compareFunction[offset].function)
					return (&compareFunction[offset]);

				// Increment the use count.
				compareFunction[offset].useCount++;
				return (&compareFunction[offset]);
			}

		// Reallocate the array.
		if (((offset + 1) % ZAF_DYNAMIC_ARRAY_INCREMENT) == 0)
		{
			CompareFunction *tempArray = compareFunction;
			compareFunction = new CompareFunction[offset + ZAF_DYNAMIC_ARRAY_INCREMENT + 1];
			memcpy(compareFunction, tempArray, offset * sizeof(CompareFunction));
			delete []tempArray;
		}
	}
	else
		compareFunction = new CompareFunction[ZAF_DYNAMIC_ARRAY_INCREMENT];

	// Set the compare function.
	compareFunction[offset].useCount = 0;
	compareFunction[offset].name = strdup(name);
	compareFunction[offset].function = function;

	// Set the end-of-array.
	offset++;
	compareFunction[offset].useCount = 0;
	compareFunction[offset].name = ZAF_NULLP(ZafDataNameChar);
	compareFunction[offset].function = 0;

	// Return the compare function.
	return (&compareFunction[offset-1]);
}

ZafObjectPersistence::ObjectConstructor *ZafObjectPersistence::AddObjectConstructor(
	ZafClassName className, ZafClassID classID, ZafObjectConstructor constructor)
{
	// Check for an existing function.
	int offset = 0;
	if (objectConstructor)
	{
		// Try to find a matching class.
		for ( ; objectConstructor[offset].classID != ID_END; offset++)
			if (classID == objectConstructor[offset].classID)
			{
				// Check for invalid function.
				if (constructor && constructor != objectConstructor[offset].constructor)
					return (&objectConstructor[offset]);

				// Increment the use count.
				objectConstructor[offset].useCount++;
				return (&objectConstructor[offset]);
			}

		// Reallocate the array.
		if (((offset + 1) % ZAF_DYNAMIC_ARRAY_INCREMENT) == 0)
		{
			ObjectConstructor *tempArray = objectConstructor;
			objectConstructor = new ObjectConstructor[offset + ZAF_DYNAMIC_ARRAY_INCREMENT + 1];
			memcpy(objectConstructor, tempArray, offset * sizeof(ObjectConstructor));
			delete []tempArray;
		}
	}
	else
		objectConstructor = new ObjectConstructor[ZAF_DYNAMIC_ARRAY_INCREMENT];

	// Set the read function.
	objectConstructor[offset].useCount = 0;
	objectConstructor[offset].classID = classID;
	objectConstructor[offset].className = className;
	objectConstructor[offset].constructor = constructor;

	// Set the end-of-array.
	offset++;
	objectConstructor[offset].useCount = 0;
	objectConstructor[offset].classID = ID_END;
	objectConstructor[offset].className = '\0';
	objectConstructor[offset].constructor = ZAF_NULLF(ZafObjectConstructor);

	// Return the read function.
	return (&objectConstructor[offset-1]);
}

ZafObjectPersistence::UserFunction *ZafObjectPersistence::AddUserCallback(
	ZafDataName name, ZafUserFunction function)
{
	// Check for an existing function.
	int offset = 0;
	if (userFunction)
	{
		// Try to find a matching class.
		for ( ; userFunction[offset].name; offset++)
			if (!streq(name, userFunction[offset].name))
			{
				// Check for invalid function.
				if (function && function != userFunction[offset].function)
					return (&userFunction[offset]);

				// Increment the use count.
				userFunction[offset].useCount++;
				return (&userFunction[offset]);
			}

		// Reallocate the array.
		if (((offset + 1) % ZAF_DYNAMIC_ARRAY_INCREMENT) == 0)
		{
			UserFunction *tempArray = userFunction;
			userFunction = new UserFunction[offset + ZAF_DYNAMIC_ARRAY_INCREMENT + 1];
			memcpy(userFunction, tempArray, offset * sizeof(UserFunction));
			delete []tempArray;
		}
	}
	else
		userFunction = new UserFunction[ZAF_DYNAMIC_ARRAY_INCREMENT];

	// Set the user function.
	userFunction[offset].useCount = 0;
	userFunction[offset].name = strdup(name);
	userFunction[offset].function = function;

	// Set the end-of-array.
	offset++;
	userFunction[offset].useCount = 0;
	userFunction[offset].name = ZAF_NULLP(ZafDataNameChar);
	userFunction[offset].function = 0;

	// Return the user function.
	return (&userFunction[offset-1]);
}

ZafObjectPersistence::UserObject *ZafObjectPersistence::AddUserObject(
	ZafDataName name, void *object)
{
	// Check for an existing function.
	int offset = 0;
	if (userObject)
	{
		// Try to find a matching class.
		for ( ; userObject[offset].name; offset++)
			if (!streq(name, userObject[offset].name))
			{
				// Check for invalid function.
				if (object && object != userObject[offset].object)
					return (&userObject[offset]);

				// Increment the use count.
				userObject[offset].useCount++;
				return (&userObject[offset]);
			}

		// Reallocate the array.
		if (((offset + 1) % ZAF_DYNAMIC_ARRAY_INCREMENT) == 0)
		{
			UserObject *tempArray = userObject;
			userObject = new UserObject[offset + ZAF_DYNAMIC_ARRAY_INCREMENT + 1];
			memcpy(userObject, tempArray, offset * sizeof(UserFunction));
			delete []tempArray;
		}
	}
	else
		userObject = new UserObject[ZAF_DYNAMIC_ARRAY_INCREMENT];

	// Set the user object.
	userObject[offset].useCount = 0;
	userObject[offset].name = strdup(name);
	userObject[offset].object = object;

	// Set the end-of-array.
	offset++;
	userObject[offset].useCount = 0;
	userObject[offset].name = ZAF_NULLP(ZafDataNameChar);
	userObject[offset].object = ZAF_NULLP(void);

	// Return the user object.
	return (&userObject[offset-1]);
}

void ZafObjectPersistence::ClearCompareFunctions(void)
{
	// Check for a valid persist table.
	if (compareFunction)
	{
		delete []compareFunction;
		compareFunction = ZAF_NULLP(CompareFunction);
	}
}

void ZafObjectPersistence::ClearObjectConstructors(void)
{
	// Check for a valid persist table.
	if (objectConstructor)
	{
		delete []objectConstructor;
		objectConstructor = ZAF_NULLP(ObjectConstructor);
	}
}

void ZafObjectPersistence::ClearUserCallback(void)
{
	// Check for a valid persist table.
	if (userFunction)
	{
		delete []userFunction;
		userFunction = ZAF_NULLP(UserFunction);
	}
}

void ZafObjectPersistence::ClearUserObject(void)
{
	// Check for a valid persist table.
	if (userObject)
	{
		delete []userObject;
		userObject = ZAF_NULLP(UserObject);
	}
}

bool ZafObjectPersistence::GenerateCPP(const ZafIChar *name, ZafFileSystem *newFileSystem, ZafFile *newFile)
{
	return (false);
}

bool ZafObjectPersistence::GenerateHPP(const ZafIChar *name, ZafFileSystem *newFileSystem, ZafFile *newFile)
{
	return (false);
}

ZafCompareFunction ZafObjectPersistence::GetCompareFunction(ZafDataName name)
{
	// Check for a valid persist table.
	if (compareFunction)
	{
		// Try to find a matching entry.
		for (int i = 0; compareFunction[i].name; i++)
			if (name && !streq(name, compareFunction[i].name))
				return (compareFunction[i].function);
	}

	// User object not found.
	return (0);
}

ZafDataName ZafObjectPersistence::GetCompareFunctionName(ZafCompareFunction function)
{
	// Check for a valid persist table.
	if (compareFunction && function)
	{
		// Try to find a matching entry.
		for (int i = 0; compareFunction[i].name; i++)
			if (function == compareFunction[i].function)
				return (compareFunction[i].name);
	}

	// CompareFunction name not found.
	return (0);
}

ZafDataName ZafObjectPersistence::GetClassName(ZafClassID classID)
{
	// Check for a valid object table.
	if (objectConstructor)
	{
		// Try to find a matching entry.
		for (int i = 0; objectConstructor[i].className; i++)
			if (classID == objectConstructor[i].classID)
				return (objectConstructor[i].className);
	}

	// Check for a valid data table.
	if (dataConstructor)
	{
		// Try to find a matching entry.
		for (int i = 0; dataConstructor[i].className; i++)
			if (classID == dataConstructor[i].classID)
				return (dataConstructor[i].className);
	}

	// User object name not found.
	return (0);
}

ZafObjectConstructor ZafObjectPersistence::GetObjectConstructor(ZafClassID classID, ZafClassName className)
{
	// Check for a valid persist table.
	if (objectConstructor)
	{
		// Try to find a matching entry.
		for (int i = 0; objectConstructor[i].classID != ID_END; i++)
			if ((classID && classID == objectConstructor[i].classID) ||
				(className && !streq(className, objectConstructor[i].className)))
				return (objectConstructor[i].constructor);
	}

	// Read function not found.
	return (0);
}

ZafUserFunction ZafObjectPersistence::GetUserCallback(ZafDataName name)
{
	// Check for a valid persist table.
	if (userFunction)
	{
		// Try to find a matching entry.
		for (int i = 0; userFunction[i].name; i++)
			if (name && !streq(name, userFunction[i].name))
				return (userFunction[i].function);
	}

	// User object not found.
	return (0);
}

void *ZafObjectPersistence::GetUserObject(ZafDataName name)
{
	// Check for a valid persist table.
	if (userObject)
	{
		// Try to find a matching entry.
		for (int i = 0; userObject[i].name; i++)
			if (name && !streq(name, userObject[i].name))
				return (userObject[i].object);
	}

	// User object not found.
	return (0);
}

ZafDataName ZafObjectPersistence::GetUserCallbackName(ZafUserFunction function)
{
	// Check for a valid persist table.
	if (userFunction && function)
	{
		// Try to find a matching entry.
		for (int i = 0; userFunction[i].name; i++)
			if (function == userFunction[i].function)
				return (userFunction[i].name);
	}

	// User user callback name not found.
	return (0);
}

ZafDataName ZafObjectPersistence::GetUserObjectName(void *tUserObject)
{
	// Check for a valid persist table.
	if (userObject && tUserObject)
	{
		// Try to find a matching entry.
		for (int i = 0; userObject[i].name; i++)
			if (tUserObject == userObject[i].object)
				return (userObject[i].name);
	}

	// User object name not found.
	return (0);
}

ZafObjectPersistence &ZafObjectPersistence::PopLevel(void)
{
	// Defer to the base class.
	ZafDataPersistence::PopLevel();
	return (*this);
}

ZafObjectPersistence &ZafObjectPersistence::PushLevel(ZafClassName className, ZafClassID classID, ZafPersistEntryType type)
{
	ZafDataPersistence::PushLevel(className, classID, type);
	return (*this);
}

bool ZafObjectPersistence::SetObjectConstructors(ObjectConstructor *tObjectConstructor)
{
	if (objectConstructor)
	{
		delete objectConstructor;
		objectConstructor = ZAF_NULLP(ObjectConstructor);
	}

	int cnt = 0;
	while(tObjectConstructor[cnt].classID != ID_END)
	{
		AddObjectConstructor(tObjectConstructor[cnt].className, tObjectConstructor[cnt].classID,
			tObjectConstructor[cnt].constructor);
		++cnt;
	}

	return (true);
}

bool ZafObjectPersistence::SubtractCompareFunction(ZafDataName name)
{
	// Check for a valid table.
	if (!compareFunction)
		return (false);

	// Try to find a matching entry.
	for (int i = 0; compareFunction[i].name; i++)
		if (!streq(name, compareFunction[i].name))
		{
			// Clean up the current entry.
			delete [](ZafIChar *)compareFunction[i].name;

			// Check for an empty array.
			if (i == 0 && compareFunction[i+1].name == 0)
			{
				delete []compareFunction;
				compareFunction = ZAF_NULLP(CompareFunction);
			}
			else
			{
				// Remove the object from the notification array.
				for (int j = i; compareFunction[j].name; j++)
					compareFunction[j] = compareFunction[j+1];
			}

			// Entry found.
			return (true);
		}

	// Entry not found.
	return (false);
}

bool ZafObjectPersistence::SubtractObjectConstructor(ZafClassID classID, ZafClassName className)
{
	// Check for a valid table.
	if (!objectConstructor)
		return (false);

	// Try to find a matching entry.
	for (int i = 0; objectConstructor[i].classID != ID_END; i++)
		if ((classID && classID == objectConstructor[i].classID) ||
			(className && !streq(className, objectConstructor[i].className)))
		{
			// Check for an empty array.
			if (i == 0 && objectConstructor[i+1].classID == 0)
			{
				delete []objectConstructor;
				objectConstructor = ZAF_NULLP(ObjectConstructor);
			}
			else
			{
				// Remove the object from the notification array.
				for (int j = i; objectConstructor[j].classID; j++)
					objectConstructor[j] = objectConstructor[j+1];
			}

			// Entry found.
			return (true);
		}

	// Entry not found.
	return (false);
}

bool ZafObjectPersistence::SubtractUserCallback(ZafDataName name)
{
	// Check for a valid table.
	if (!userFunction)
		return (false);

	// Try to find a matching entry.
	for (int i = 0; userFunction[i].name; i++)
		if (!streq(name, userFunction[i].name))
		{
			// Clean up the current entry.
			delete [](ZafIChar *)userFunction[i].name;

			// Check for an empty array.
			if (i == 0 && userFunction[i+1].name == 0)
			{
				delete []userFunction;
				userFunction = ZAF_NULLP(UserFunction);
			}
			else
			{
				// Remove the object from the notification array.
				for (int j = i; userFunction[j].name; j++)
					userFunction[j] = userFunction[j+1];
			}

			// Entry found.
			return (true);
		}

	// Entry not found.
	return (false);
}

bool ZafObjectPersistence::SubtractUserObject(ZafDataName name)
{
	// Check for a valid table.
	if (!userObject)
		return (false);

	// Try to find a matching entry.
	for (int i = 0; userObject[i].name; i++)
		if (!streq(name, userObject[i].name))
		{
			// Clean up the current entry.
			delete [](ZafIChar *)userObject[i].name;

			// Check for an empty array.
			if (i == 0 && userObject[i+1].name == 0)
			{
				delete []userObject;
				userObject = ZAF_NULLP(UserObject);
			}
			else
			{
				// Remove the object from the notification array.
				for (int j = i; userObject[j].name; j++)
					userObject[j] = userObject[j+1];
			}

			// Entry found.
			return (true);
		}

	// Entry not found.
	return (false);
}

