//	Zinc Application Framework - Z_PRST.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_data.hpp>
#include <z_string.hpp>
#include <z_utils.hpp>
#if defined(ZAF_PERSIST_ALL)
#	include <z_datall.hpp>
#endif
#define ZAF_DATA_PERSISTENCE_INFO
#include "gbl_def.cpp"

static ZafIChar _objectDataName[] = ZAF_ITEXT("objectData");

// ----- ZafDataPersistence -------------------------------------------------

ZafDataPersistence::ZafDataPersistence(ZafFileSystem *fileSystem,
	DataConstructor *tDataConstructor) :
	current(0), error(ZAF_ERROR_NONE), subSystem()
{
	dataConstructor = ZAF_NULLP(DataConstructor);
	SetDataConstructors(tDataConstructor);
	for (int i = 0; i < ZAF_MAXFILES; i++)
	{
		entry[i].level = 0;
		entry[i].type = ZAF_PERSIST_ROOT_DIRECTORY;
		entry[i].className = 0;
		entry[i].classID = 0;
		entry[i].file = ZAF_NULLP(ZafFile);
		entry[i].fileSystem = ZAF_NULLP(ZafFileSystem);
		entry[i].restorePath[0] = '\0';
	}
//	entry[current].file = file;
	entry[current].fileSystem = fileSystem;
	if (fileSystem)
		subSystem.Add(fileSystem);
}

ZafDataPersistence::ZafDataPersistence(const ZafDataPersistence &copy) :
	current(0), error(ZAF_ERROR_NONE), subSystem()
{
	dataConstructor = ZAF_NULLP(DataConstructor);
	SetDataConstructors(copy.dataConstructor);
}

ZafDataPersistence::~ZafDataPersistence(void)
{
	// Clear the tables.
	ClearDataConstructors();
	ClearFileSystems();
}

bool ZafDataPersistence::AddDataConstructor(ZafClassName className, ZafClassID classID, ZafDataConstructor constructor)
{
	// Check for an existing function.
	int offset = 0;
	if (dataConstructor)
	{
		// Try to find a matching class.
		for ( ; dataConstructor[offset].classID != ID_END; offset++)
			if (classID == dataConstructor[offset].classID)
			{
				// Check for invalid function.
				if (constructor && constructor != dataConstructor[offset].constructor)
					return (false);

				// Increment the use count.
				dataConstructor[offset].useCount++;
				return (true);
			}

		// Reallocate the array.
		if (((offset + 1) % ZAF_DYNAMIC_ARRAY_INCREMENT) == 0)
		{
			DataConstructor *tempArray = dataConstructor;
			dataConstructor = new DataConstructor[offset + ZAF_DYNAMIC_ARRAY_INCREMENT + 1];
			memcpy(dataConstructor, tempArray, offset * sizeof(DataConstructor));
			delete []tempArray;
		}
	}
	else
		dataConstructor = new DataConstructor[ZAF_DYNAMIC_ARRAY_INCREMENT];

	// Set the read function.
	dataConstructor[offset].useCount = 0;
	dataConstructor[offset].classID = classID;
	dataConstructor[offset].className = className;
	dataConstructor[offset].constructor = constructor;

	// Set the end-of-array.
	offset++;
	dataConstructor[offset].useCount = 0;
	dataConstructor[offset].classID = ID_END;
	dataConstructor[offset].className = '\0';
	dataConstructor[offset].constructor = ZAF_NULLF(ZafDataConstructor);

	// Return the read function.
	return (true);
}

ZafFileSystem *ZafDataPersistence::AddFileSystem(ZafFileSystem *fileSystem, ZafFileSystem *position)
{
	// Add the file system.
	if (position)
		subSystem.Add(fileSystem, position);	// position insert.
	else
		subSystem.Add(fileSystem);				// compareFunction insert.

	// Return a pointer to the object.
	return (fileSystem);
}

ZafError ZafDataPersistence::AllocateFile(const ZafIChar *name, ZafFileMode mode)
{
	// Reset the file system.
	if (!subSystem.First())
	{
		entry[current+1].file = entry[current].file;
		entry[current+1].fileSystem = ZAF_NULLP(ZafFileSystem);
		entry[current].restorePath[0] = '\0';
		return (ZAF_ERROR_NONE);
	}

	// Check for a valid search name.
	entry[current].file = ZAF_NULLP(ZafFile);
	entry[current].fileSystem = ZAF_NULLP(ZafFileSystem);
	if (!name || !name[0])
		name = entry[current].className;

	// Try to find a valid path entry from the sub-systems.
	for (ZafElement *element = subSystem.First(); !entry[current].file && element; element = element->Next())
	{
		ZafFileSystem *fileSystem = DynamicPtrCast(element, ZafFileSystem);

		// Try to open a file.
		if (entry[current].type == ZAF_PERSIST_ROOT_DIRECTORY ||
			entry[current].type == ZAF_PERSIST_ROOT_FILE)
		{
			// Move to the root directory.
			fileSystem->GetCWD(entry[current].restorePath, ZAF_MAXPATHLEN);
			ZafIChar classDir[ZAF_MAXPATHLEN];
			strcpy(classDir, ZafFileSystem::rootDirectoryName);
			strcat(classDir, entry[current].className);
			if (fileSystem->ChDir(classDir))
			{
				if (mode & ZAF_FILE_READ)
					continue; // continue for read-only requests.
				fileSystem->MkDir(entry[current].className);
				fileSystem->ChDir(entry[current].className);
			}

			// Allocate a named directory.
			if (entry[current].type == ZAF_PERSIST_ROOT_DIRECTORY && fileSystem->ChDir(name))
			{
				fileSystem->MkDir(name, entry[current].className, entry[current].classID);
				fileSystem->ChDir(name);
			}

			// Move to the proper sub-directory.
			// printf("push root %s\n", name);
			entry[current].file = fileSystem->Open(_objectDataName, mode);
		}
		else if (entry[current].type == ZAF_PERSIST_DIRECTORY)
		{
			// Copy the restore path.
			strcpy(entry[current].restorePath, ZafFileSystem::parentDirectoryName);

			// Allocate a named directory.
			if (fileSystem->ChDir(name, entry[current].className, entry[current].classID))
			{
				if (mode & ZAF_FILE_READ)
					continue; // continue for read-only requests.
				else if (fileSystem->Error() == ZAF_ERROR_INVALID_ID)
					fileSystem->Rename(name, name, entry[current].className, entry[current].classID);
				else
					fileSystem->MkDir(name, entry[current].className, entry[current].classID);
				fileSystem->ChDir(name);
			}

			// Move to the proper sub-directory.
			// printf("push directory %s\n", name);
			entry[current].file = fileSystem->Open(_objectDataName, mode);
		}
		else
		{
			// Set an empty restore path.
			entry[current].restorePath[0] = '\0';

			// Move to the proper sub-directory.
			// printf("push file %s\n", name);
			entry[current].file = fileSystem->Open(name, mode, ZAF_NULLP(ZafStringIDChar), entry[current].classID);
		}

		// Check for a valid file system.
		if (entry[current].file)
			entry[current].fileSystem = fileSystem;
	}

	// Keep the error code.
	if (!entry[current].file)
		SetError(ZAF_ERROR_FILE_OPEN);
	else if (entry[current].file->Error() != ZAF_ERROR_NONE)
		SetError(entry[current].file->Error());
	else
		SetError(ZAF_ERROR_NONE);
	return (error);
}

void ZafDataPersistence::ClearDataConstructors(void)
{
	// Check for a valid persist table.
	if (dataConstructor)
	{
		delete []dataConstructor;
		dataConstructor = ZAF_NULLP(DataConstructor);
	}
}

void ZafDataPersistence::ClearFileSystems(void)
{
	// Do not delete the file systems, just clear the list.
	while (subSystem.First())
		subSystem.Subtract(subSystem.First());
}

ZafClassID ZafDataPersistence::CurrentClassID(void)
{
	// Check for a valid level.
	if (current < 0)
		return (ID_END);

	// Find a valid entry.
	ZafClassID matchID = entry[current].classID;
	if (!matchID && entry[current].className)
	{
		for (int i = 0; dataConstructor[i].classID != ID_END; i++)
			if (!streq(dataConstructor[i].className, entry[current].className))
			{
				matchID = dataConstructor[i].classID;
				break;
			}
	}

	// Return the class name.
	return (matchID);
}

ZafClassName ZafDataPersistence::CurrentClassName(void)
{
	// Check for a valid level.
	if (current < 0)
		return (ZAF_NULLP(ZafDataNameChar));

	// Find a valid entry.
	ZafClassName matchName = entry[current].className;
	if (!matchName && entry[current].classID)
	{
		for (int i = 0; dataConstructor[i].classID != ID_END; i++)
			if (dataConstructor[i].classID == entry[current].classID)
			{
				matchName = dataConstructor[i].className;
				break;
			}
	}

	// Return the class name.
	return (matchName);
}

ZafDataConstructor ZafDataPersistence::GetDataConstructor(ZafClassID classID, ZafClassName className)
{
	// Check for a valid table.
	if (!dataConstructor)
		return (0);

	// Try to find a matching entry.
	for (int i = 0; dataConstructor[i].classID != ID_END; i++)
		if ((classID && classID == dataConstructor[i].classID) ||
			(className == dataConstructor[i].className) ||
			(className && dataConstructor[i].className && !streq(className, dataConstructor[i].className)))
			return (dataConstructor[i].constructor);

	// Read function not found.
	return (0);
}

ZafDataPersistence &ZafDataPersistence::PopLevel(void)
{
	// Remove the file.
	if (--entry[current].level > 0)
		return (*this);

	// Decrement the level count.
	ZafFileSystem *fileSystem = entry[current].fileSystem;
	if (fileSystem)
	{
		// Delete the file and keep the error code.
		if (entry[current].file)
		{
			if (entry[current].file->Error() != ZAF_ERROR_NONE)
				SetError(entry[current].file->Error());
			fileSystem->Close(entry[current].file);
			entry[current].file = ZAF_NULLP(ZafFile);
		}

		// Restore the directory.
		// printf("pop %s\n", entry[current].restorePath);
		if (entry[current].restorePath[0])
			fileSystem->ChDir(entry[current].restorePath);
	}
	if (current)
		current--;
	return (*this);
}

ZafDataPersistence &ZafDataPersistence::PushLevel(ZafClassName className, ZafClassID classID, ZafPersistEntryType type)
{
	// Check for a new entry.
	if (entry[current].file)
	{
		current++;
		entry[current].level = 1;
		entry[current].file = ZAF_NULLP(ZafFile);
		entry[current].restorePath[0] = '\0';
	}
	else
		entry[current].level++;

	// Check for entry update.
	if (entry[current].level == 1)
	{
		entry[current].type = type;
		entry[current].className = className;
		entry[current].classID = classID;
	}

	return (*this);
}

bool ZafDataPersistence::SetDataConstructors(DataConstructor *tDataConstructor)
{
	if (dataConstructor)
	{
		delete dataConstructor;
		dataConstructor = ZAF_NULLP(DataConstructor);
	}

	int cnt = 0;
	while(tDataConstructor[cnt].classID != ID_END)
	{
		AddDataConstructor(tDataConstructor[cnt].className, tDataConstructor[cnt].classID,
			tDataConstructor[cnt].constructor);
		++cnt;
	}

	return (true);
}

bool ZafDataPersistence::SubtractDataConstructor(ZafClassID classID, ZafClassName className)
{
	// Check for a valid table.
	if (!dataConstructor)
		return (0);

	// Try to find a matching entry.
	for (int i = 0; dataConstructor[i].classID != ID_END; i++)
		if ((classID && classID == dataConstructor[i].classID) ||
			(className && !streq(className, dataConstructor[i].className)))
		{
			// Check for an empty array.
			if (i == 0 && dataConstructor[i+1].classID == 0)
			{
				delete []dataConstructor;
				dataConstructor = ZAF_NULLP(DataConstructor);
			}
			else
			{
				// Remove the data from the notification array.
				for (int j = i; dataConstructor[j].classID; j++)
					dataConstructor[j] = dataConstructor[j+1];
			}

			// Entry found.
			return (true);
		}

	// Entry not found.
	return (false);
}

ZafFileSystem *ZafDataPersistence::SubtractFileSystem(ZafFileSystem *fileSystem)
{
	// Make sure the object can be removed.
	if (subSystem.Index(fileSystem) == -1)
		return (ZAF_NULLP(ZafFileSystem));
	return ((ZafFileSystem *)subSystem.Subtract(fileSystem));
}
