//	Zinc Application Framework - Z_NOTEBK.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_notebk.hpp>
#include <z_utils.hpp>
#include <z_title.hpp>
#if defined(ZAF_MSDOS)
#	include "d_notebk.hpp"
#elif defined(ZAF_MACINTOSH)
#	include "a_notebk.hpp"
#endif
#define ZAF_NOTEBOOK_INFO
#include "gbl_def.cpp"

// ----- ZafNotebook --------------------------------------------------------

ZafNotebook::ZafNotebook(int left, int top, int width, int height) :
	ZafWindow(left, top, width, height),
	tabHeight(0), tabWidth(0), currentPage(-1)
{
	SetBordered(true);
	SetRegionType(ZAF_AVAILABLE_REGION);

#if defined(ZAF_MSDOS) || defined(ZAF_MACINTOSH)
	tabControl = new ZafNotebookTabControl;
	ZafWindow::Add(tabControl);
#endif
}

ZafNotebook::ZafNotebook(const ZafNotebook &copy) :
	ZafWindow(copy),
	tabHeight(copy.tabHeight), tabWidth(copy.tabWidth),
	currentPage(copy.currentPage)
{
	// The ZafWindow copy constructor correctly copies the notebook tabs
	// and pages so no additional information needs to be copied.

#if defined(ZAF_MSDOS) || defined(ZAF_MACINTOSH)
	tabControl = DynamicPtrCast(SupportFirst(), ZafNotebookTabControl);
#endif
}

ZafNotebook::~ZafNotebook(void)
{
}

ZafWindowObject *ZafNotebook::Duplicate(void)
{
	return (new ZafNotebook(*this));
}

ZafLogicalEvent ZafNotebook::LogicalEvent(const ZafEventStruct &event)
{
	ZafLogicalEvent ccode = MapEvent(defaultEventMap, event);
	return ((ccode == L_NONE) ? ZafWindow::LogicalEvent(event) : ccode);
}

const ZafPaletteStruct *ZafNotebook::MapClassPalette(ZafPaletteType type, ZafPaletteState state)
{
	const ZafPaletteStruct *palette = MapPalette(defaultPaletteMap, type, state);
	return (palette ? palette : ZafWindow::MapClassPalette(type, state));
}

bool ZafNotebook::SetAcceptDrop(bool )
{
	// acceptDrop is false for this class.
	acceptDrop = false;
	return (acceptDrop);
}

bool ZafNotebook::SetBordered(bool setBordered)
{
	// Make sure the request is valid.
	if (bordered != setBordered && !screenID)
		bordered = setBordered;

	// Return the current attribute.
	return (bordered);
}

bool ZafNotebook::SetDestroyable(bool )
{
	// destroyable is false for this class.
	destroyable = false;
	return (destroyable);
}

bool ZafNotebook::SetLocked(bool )
{
	// locked is false for this class.
	locked = false;
	return (locked);
}

bool ZafNotebook::SetMaximized(bool )
{
	// maximized is false for this class.
	maximized = false;
	return (maximized);
}

bool ZafNotebook::SetMinimized(bool )
{
	// minimized is false for this class.
	minimized = false;
	return (minimized);
}

bool ZafNotebook::SetMoveable(bool )
{
	// moveable is false for this class.
	moveable = false;
	return (moveable);
}

ZafSelectionType ZafNotebook::SetSelectionType(ZafSelectionType )
{
	// selectionType is ZAF_SINGLE_SELECTION for this class.
	selectionType = ZAF_SINGLE_SELECTION;
	return (selectionType);
}

bool ZafNotebook::SetSizeable(bool )
{
	// sizeable is false for this class.
	sizeable = false;
	return (sizeable);
}

int ZafNotebook::SetTabHeight(int tTabHeight)
{
	// Make sure the request is valid.
	if (tTabHeight != tabHeight && !screenID)
		tabHeight = tTabHeight;
	return (tabHeight);
}

int ZafNotebook::SetTabWidth(int tTabWidth)
{
	// Make sure the request is valid.
	if (tTabWidth != tabWidth && !screenID)
		tabWidth = tTabWidth;
	return (tabWidth);
}
		
bool ZafNotebook::SetTemporary(bool )
{
	// temporary is false for this class.
	temporary = false;
	return (temporary);
}

// ----- Persistent functions -----------------------------------------------

ZafNotebook::ZafNotebook(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafWindow(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	currentPage(-1)
{
	// Read the data.
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafNotebook::ReadData(persist);

#if defined(ZAF_MSDOS) || defined(ZAF_MACINTOSH)
	tabControl = new ZafNotebookTabControl;
	ZafWindow::Add(tabControl);
#endif

	// We must re-add the children to get the notebook tabs.
	ZafElement *oldCurrent = Current();
	ZafElement *oldSupportCurrent = support.Current();
	for (ZafWindowObject *object = First(); object; object = object->Next())
	{
		ZafWindowObject *nextObject = object->Next();
		Subtract(object);
		Add(object, nextObject);
	}
	SetCurrent(oldCurrent);
	support.SetCurrent(oldSupportCurrent);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

ZafElement *ZafNotebook::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafNotebook(name, persist));
}

void ZafNotebook::ReadData(ZafObjectPersistence &persist)
{
	// Read the tool bar information.
	ZafFile *file = persist.CurrentFile();
	*file >> tabHeight >> tabWidth;
}

void ZafNotebook::Write(ZafObjectPersistence &persist)
{
	// Write the object.
	ZafWindow::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafNotebook::WriteData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafNotebook::WriteChildren(ZafObjectPersistence &persist)
{
	// We must bypass ZafWindow::WriteChildren() to prevent the notebook 
	// tabs and pages from being written improperly.

	// Write the data.
	int index = 0;
	for (ZafWindowObject *object = First(); object; object = object->Next())
	{
		// Prepare the page for writing.
		ZafWindow *page = DynamicPtrCast(object, ZafWindow);
		ZafTitle *pageTitle = ZAF_NULLP(ZafTitle);
		if (page)
		{
			pageTitle = new ZafTitle(TabText(index++));
			page->support.Add(pageTitle);
		}

		// Write the page information.
		object->Write(persist.PushLevel(object->ClassName(), object->ClassID(), ZAF_PERSIST_DIRECTORY));
		persist.PopLevel();

		if (page)
		{
			page->support.Subtract(pageTitle);
			delete pageTitle;
		}
	}
}

void ZafNotebook::WriteData(ZafObjectPersistence &persist)
{
	// Write the data.
	ZafFile *file = persist.CurrentFile();
	*file << tabHeight << tabWidth;
}

void ZafNotebook::WriteSupportChildren(ZafObjectPersistence &)
{
	// Support children are not written for notebook.
}

