//	Zinc Application Framework - Z_MSGWIN.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_app.hpp>
#include <z_msgwin.hpp>
#include <z_stdarg.hpp>
#include <z_border.hpp>
#include <z_sys.hpp>
#include <z_icon1.hpp>
#include <z_text.hpp>
#include <z_title.hpp>
#include <z_utils.hpp>
#define ZAF_MESSAGE_WINDOW_INFO
#include "lang_def.cpp"
#include "gbl_def.cpp"

// ----- ZafMessageWindow ---------------------------------------------------

#if defined(ZAF_MSDOS)
static ZafPaletteMap ZAF_FARDATA textFieldPaletteMap[] =
{
	{ ZAF_PM_ANY_TYPE, ZAF_PM_ANY_STATE, { ZAF_LINE_SOLID, ZAF_PTN_SOLID_FILL, ZAF_CLR_BLACK, ZAF_CLR_LIGHTGRAY, ZAF_MONO_WHITE, ZAF_MONO_BLACK, ZAF_FNT_DIALOG } }
};
#endif

ZafMessageWindow::ZafMessageWindow(ZafStringData *zTitle,
	ZafIconImage zIconImage, ZafDialogFlags zMsgFlags,
	ZafDialogFlags zDefFlag, const ZafIChar *format, ...) :
	ZafDialogWindow(0, 0, 0, 0),
	msgFlags(0), defFlag(0), iconImage(zIconImage),
	icon(ZAF_NULLP(ZafIcon)), textField(ZAF_NULLP(ZafText)),
	title(ZAF_NULLP(ZafTitle)), firstButton(ZAF_NULLP(ZafButton))
{
	SetSizeable(false);

	// Check the language and local information.
	LanguageAllocate();

	// Get the formatted text.
	ZafIChar text[1024];
	va_list arguments;
	va_start(arguments, format);
	ZafStandardArg::vsprintf(text, format, &arguments);
	va_end(arguments);

	SetMessageFlags(zMsgFlags);
	SetDefaultMessageFlag(zDefFlag);
	SetMessage(text);
	title = new ZafTitle(zTitle);
	SetIconImage(iconImage);
}

ZafMessageWindow::ZafMessageWindow(const ZafIChar *zTitle,
	ZafIconImage zIconImage, ZafDialogFlags zMsgFlags,
	ZafDialogFlags zDefFlag, const ZafIChar *format, ...) :
	ZafDialogWindow(0, 0, 0, 0),
	msgFlags(0), defFlag(0), iconImage(zIconImage),
	icon(ZAF_NULLP(ZafIcon)), textField(ZAF_NULLP(ZafText)),
	title(ZAF_NULLP(ZafTitle)), firstButton(ZAF_NULLP(ZafButton))
{
	SetSizeable(false);

	// Check the language and local information.
	LanguageAllocate();

	// Get the formatted text.
	ZafIChar text[1024];
	va_list arguments;
	va_start(arguments, format);
	ZafStandardArg::vsprintf(text, format, &arguments);
	va_end(arguments);

	SetMessageFlags(zMsgFlags);
	SetDefaultMessageFlag(zDefFlag);
	SetMessage(text);
	title = new ZafTitle(new ZafStringData(zTitle));
	SetIconImage(iconImage);
}

ZafMessageWindow::ZafMessageWindow(const ZafMessageWindow &copy) :
	ZafDialogWindow(copy),
	icon(ZAF_NULLP(ZafIcon)), textField(ZAF_NULLP(ZafText)),
	title(ZAF_NULLP(ZafTitle)),
	firstButton(ZAF_NULLP(ZafButton))
{
	// Check the language and local information.
	LanguageAllocate();

	Destroy();

	SetMessageFlags(copy.msgFlags);
	SetDefaultMessageFlag(copy.defFlag);
	SetMessage(copy.textField ? copy.textField->Text() : ZAF_NULLP(ZafIChar));
	SetIconImage(copy.iconImage);
}

ZafMessageWindow::~ZafMessageWindow(void)
{
	// Free the language information.
	LanguageFree();

	// If the title was created but never added (because the message window
	// was never added to the window manager) delete the title.
	if (title && !title->parent)
		delete title;

	// If the textField was created but never added (because the message window
	// was never added to the window manager) delete the textField.
	if (textField && !textField->parent)
		delete textField;
}

void ZafMessageWindow::AddFields(void)
{
	// Remove any button fields in case some flags were removed.
	if (firstButton)
		SetDefaultButton(ZAF_NULLP(ZafButton));
	ZafWindowObject *object = firstButton;
	while (object)
	{
		ZafWindowObject *nextObject = object->Next();
		Subtract(object);
		delete object;
		object = nextObject;
	}

	firstButton = ZAF_NULLP(ZafButton);

	// Add border.
	if (!Border())
		Add(new ZafBorder);

	// Add the title and system button (if specified).
	if (title && !Title())
	{
		Add(new ZafSystemButton(ZAF_NATIVE_SYSTEM_BUTTON));
		Add(title);
	}

	// Add the icon (if applicable).
	// NOTE: ZAF_APPLICATION_ICON is 0 (the first in the icon array), so it may
	// not be used in this constructor, since iconImage is an unsigned value.
	if (iconImage)
	{
		if (icon)
		{
			Subtract(icon);
			delete icon;
		}
				
		icon = new ZafIcon(2, 1, ZAF_NULLP(ZafStringData));
		icon->SetIconImage(iconImage);
		icon->SetNoncurrent(true);
		Add(icon);
	}

	if (textField)
		Add(textField);

	// Add the buttons.
	int index = 0;
	ZafDialogFlags testFlag = 1;
	do
	{
		if (msgFlags & testFlag)
		{
			ZafButton *button = new ZafButton(0, 0, 0, 0, language->GetMessage(testFlag),
				ZAF_NULLP(ZafBitmapData), ZAF_NATIVE_BUTTON);
			button->SetAutoSize(true);
			button->SetSendMessageWhenSelected(true);
			button->SetValue(testFlag == ZAF_DIALOG_HELP ? S_HELP : S_DIALOG_FIRST + index);
			Add(button);
			if (testFlag == defFlag)
				SetDefaultButton(button);
			if (!firstButton)
				firstButton = button;
			if (testFlag == defFlag)
				current = button;
		}
		index++;
		testFlag <<= 1;
	} while (testFlag);
}

ZafDialogFlags ZafMessageWindow::ClearMessageFlags(ZafDialogFlags zMsgFlags)
{
	if (!screenID)
		msgFlags &= ~zMsgFlags;

	return (msgFlags);
}

ZafEventType ZafMessageWindow::Event(const ZafEventStruct &event)
{
	// Check for zinc events.
	ZafEventType ccode = event.type;
	switch (ccode)
	{
	case S_INITIALIZE:
		AddFields();
		ZafDialogWindow::Event(event);
		break;

	case S_COMPUTE_SIZE:
		{
		// Compute the text size.
		int textHeight = 0;
		int textWidth = 0;
		const ZafIChar *text = textField ? textField->Text() : ZAF_NULLP(ZafIChar);
		if (text)
		{
			// Initialize the display so that accurate text sizes can be obtained.
			BeginDraw();
			display->SetFont(textField->Font());

			int index = 0;
			const ZafIChar *linePtr = text;
			do
			{
				if (text[index] == '\n' || text[index] == '\r' || text[index] == '\0')
				{
					int isCR = (text[index] == '\r');
					ZafIChar saveChar = text[index];
					textField->StringData()->SetChar(index, '\0');

					ZafRegionStruct textSize = display->TextSize(linePtr);
					textHeight += textSize.Height();
					if (textSize.Width() > textWidth)
						textWidth = textSize.Width();

					textField->StringData()->SetChar(index, saveChar);
					if (isCR && text[index + 1] == '\n')
						index++;
					linePtr = text + index + 1;
				}
			} while (text[index++] != '\0');

			// Restore the display.
			EndDraw();
		}

		// Add a few pixels to allow for margins;
		textWidth += 2 * ZAF_MARGIN_WIDTH;
		textHeight += 2 * ZAF_MARGIN_WIDTH;

		// Compute the size and number of buttons.
		int buttons = 0;
		int maxTextSize = 0;
		ZafWindowObject *object;
		if (firstButton)
		{
			// Initialize the display so that accurate text sizes can be obtained.
			BeginDraw();
			display->SetFont(firstButton->LogicalPalette(ZAF_PM_TEXT, ZAF_PM_ANY_STATE).font);

			for (object = firstButton; object; object = object->Next())
			{
				ZafRegionStruct textSize = display->TextSize(object->Text());
				if (textSize.Width() > maxTextSize)
					maxTextSize = textSize.Width();
				buttons++;
			}

			// Restore the display.
			EndDraw();
		}
		int buttonWidth = ZafMax(maxTextSize + 2 * display->cellWidth, 10 * display->cellWidth);

		// Allow the icon to size itself.
		if (icon)
			icon->Event(event);

		// Compute the client size.
		int clientWidth = textWidth + 4 * display->cellWidth +
			(icon ? icon->zafRegion.Width() + 2 * display->cellWidth : 0);
		if (clientWidth < (buttons * (buttonWidth + display->cellWidth)) + 3 * display->cellWidth)
			clientWidth = (buttons * (buttonWidth + display->cellWidth)) + 3 * display->cellWidth;
		int clientHeight = textHeight + 4 * display->cellHeight;

		// Position the text field.
		if (textField)
		{
			ZafRegionStruct textRegion;
			textRegion.coordinateType = ZAF_PIXEL;
			textRegion.left = 2 * display->cellWidth + (icon ? icon->zafRegion.Width() + 2 * display->cellWidth : 0);
			textRegion.top = display->cellHeight;
			textRegion.right = textRegion.left + textWidth - 1;
			textRegion.bottom = textRegion.top + textHeight - 1;
			textField->SetRegion(textRegion);
		}

		// Position the buttons.
		int left = (clientWidth - buttons * (buttonWidth + display->cellWidth) + display->cellWidth) / 2;
		int top = ZafMax(textHeight, icon ? icon->zafRegion.Height() : 0) + 2 * display->cellHeight;
#if defined(ZAF_TEXT_MODE)
		top++;
#endif
		for (object = firstButton; object; object = object->Next())
		{
			object->Event(event);
			int buttonHeight = object->zafRegion.Height();

			ZafRegionStruct region;
			region.coordinateType = ZAF_PIXEL;
			region.left = left;
			region.right = region.left + buttonWidth - 1;
			region.bottom = top + display->cellHeight - display->postSpace - 1;
			region.top = region.bottom - buttonHeight + 1;
			object->SetRegion(region);

			left += buttonWidth + display->cellWidth;
		}

		// Set the client region.
		clientRegion.coordinateType = ZAF_PIXEL;
		clientRegion.left = clientRegion.top = 0;
		clientRegion.right = clientWidth  - 1;
		clientRegion.bottom = clientHeight - 1;

		// Center the window.
		zafRegion = clientRegion;
		zafRegion.right += 2 * display->cellWidth;
		zafRegion.bottom += 2 * display->cellHeight;
		windowManager->Center(this);
		}
		break;

	default:
		ccode = ZafDialogWindow::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

void ZafMessageWindow::LanguageAllocate(void)
{
	// Check for initialization.
	if (language)
		return;
	ZafApplication::AddStaticModule(LanguageFree);

	// Try the zafDataManager.
	if (!language && zafDataManager)
		language = DynamicPtrCast(zafDataManager->AllocateData(className, ZafLanguageData::className, ZafLanguageData::classID), ZafLanguageData);

	// Default to code initialization.
	if (!language)
		language = new ZafLanguageData(defaultMessageStrings, true);
}

void ZafMessageWindow::LanguageFree(bool globalRequest)
{
	// Destroy the static language.
	if (globalRequest && language && language->Destroyable() &&
		language->NotifyCount() == 0)
	{
		delete language;
		language = ZAF_NULLP(ZafLanguageData);
	}
}

bool ZafMessageWindow::SetAcceptDrop(bool )
{
	// acceptDrop is false for this class.
	acceptDrop = false;
	return (acceptDrop);
}

bool ZafMessageWindow::SetAutomaticUpdate(bool )
{
	// automaticUpdate is true for this class.
	automaticUpdate = true;
	return (automaticUpdate);
}

bool ZafMessageWindow::SetBordered(bool )
{
	// bordered is always false for this class.
	bordered = false;
	return (bordered);
}

ZafDialogFlags ZafMessageWindow::SetDefaultMessageFlag(ZafDialogFlags zDefFlag)
{
	if (!screenID)
		defFlag = zDefFlag;

	return (defFlag);
}

bool ZafMessageWindow::SetDestroyable(bool )
{
	// destroyable is false for this class.
	destroyable = false;
	return (destroyable);
}

bool ZafMessageWindow::SetDisabled(bool )
{
	// disabled is false for this class.
	disabled = false;
	return (disabled);
}

const ZafIChar *ZafMessageWindow::SetHelpObjectTip(const ZafIChar *)
{
	// helpObjectTip cannot be set for this class.
	return (ZAF_NULLP(ZafIChar));
}

ZafIconImage ZafMessageWindow::SetIconImage(ZafIconImage zIconImage)
{
	if (!screenID)
		iconImage = zIconImage;

	return (iconImage);
}
bool ZafMessageWindow::SetLocked(bool )
{
	// locked is false for this class.
	locked = false;
	return (locked);
}

bool ZafMessageWindow::SetMaximized(bool )
{
	// maximized is false for this class.
	maximized = false;
	return (maximized);
}

void ZafMessageWindow::SetMessage(const ZafIChar *message)
{
	if (!screenID)
	{
		if (!textField)
		{
			// Create the text field.
			textField = new ZafText(0, 0, 0, 0, new ZafStringData(message));
			textField->SetBordered(false);
			textField->SetNoncurrent(true);
			textField->SetViewOnly(true);
			textField->SetParentPalette(true);
			textField->SetWordWrap(false);
#if !defined(ZAF_TEXT_MODE)
			textField->SetOSDraw(false);
#endif
#if defined(ZAF_MSDOS)
			textField->SetUserPaletteData(new ZafPaletteData(textFieldPaletteMap, true));
#endif
		}
		else
			textField->SetText(message);
	}
}

ZafDialogFlags ZafMessageWindow::SetMessageFlags(ZafDialogFlags zMsgFlags)
{
	if (!screenID)
		msgFlags |= zMsgFlags;

	return (msgFlags);
}

bool ZafMessageWindow::SetMinimized(bool )
{
	// minimized is false for this class.
	minimized = false;
	return (minimized);
}

bool ZafMessageWindow::SetMoveable(bool )
{
	// moveable is true for this class.
	moveable = true;
	return (moveable);
}

ZafRegionType ZafMessageWindow::SetRegionType(ZafRegionType )
{
	// regionType is ZAF_INSIDE_REGION for this class.
	regionType = ZAF_INSIDE_REGION;
	return (regionType);
}

ZafSelectionType ZafMessageWindow::SetSelectionType(ZafSelectionType )
{
	// selectionType is ZAF_SINGLE_SELECTION for this class.
	selectionType = ZAF_SINGLE_SELECTION;
	return (selectionType);
}

bool ZafMessageWindow::SetSizeable(bool )
{
	// sizeable is false for this class.
	sizeable = false;
	return (sizeable);
}

bool ZafMessageWindow::SetTemporary(bool )
{
	// temporary is false for this class.
	temporary = false;
	return (temporary);
}

// ----- Persistent functions -----------------------------------------------

ZafMessageWindow::ZafMessageWindow(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafDialogWindow(name, persist.PushLevel(className, classID, ZAF_PERSIST_ROOT_DIRECTORY)),
	msgFlags(0), defFlag(0), icon(ZAF_NULLP(ZafIcon)),
	textField(ZAF_NULLP(ZafText)), title(ZAF_NULLP(ZafTitle)),
	firstButton(ZAF_NULLP(ZafButton))
{
	// Read the data.
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafMessageWindow::ReadData(persist);

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());

	// Check the language and local information.
	LanguageAllocate();
}

ZafElement *ZafMessageWindow::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafMessageWindow(name, persist));
}

void ZafMessageWindow::ReadData(ZafObjectPersistence &persist)
{
	// Read the data.
	ZafIChar titleName[ZAF_MAXNAMELEN], text[1024];
	ZafFile *file = persist.CurrentFile();
	*file >> iconImage >> msgFlags >> defFlag >> text >> titleName;

	if (*text)
		SetMessage(text);

	// Add the title and system button (if specified).
	if (*titleName)
		title = new ZafTitle(new ZafStringData(titleName, persist));
}

void ZafMessageWindow::Write(ZafObjectPersistence &persist)
{
	// Bypass ZafWindow calls to keep the children from being saved.
	ZafWindowObject::Write(persist.PushLevel(className, classID, ZAF_PERSIST_ROOT_DIRECTORY));
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafWindow::WriteData(persist);
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafDialogWindow::WriteData(persist);
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafMessageWindow::WriteData(persist);

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafMessageWindow::WriteData(ZafObjectPersistence &persist)
{
	// Write the data.
	ZafFile *file = persist.CurrentFile();
	*file << iconImage;
	*file << msgFlags;
	*file << defFlag;
	const ZafIChar *text = textField ? textField->Text() : ZAF_NULLP(ZafIChar);
	ZafStringData *stringData = ZAF_NULLP(ZafStringData);
	const ZafIChar *titleName = ZAF_NULLP(ZafIChar);
	if (title)
		stringData = title->StringData();
	if (stringData)
		titleName = stringData->StringID();

	*file << text << titleName;

	// Write the string data.
	if (stringData)
		stringData->Write(persist);
}

