//	Zinc Application Framework - Z_LANG.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_lang.hpp>
#include <z_string.hpp>
#include <z_utils.hpp>
#include <z_app.hpp>
#define ZAF_LANGUAGE_DATA_INFO
#include "gbl_def.cpp"
#include "lang_def.cpp"

// ----- ZafLanguageData ----------------------------------------------------

ZafLanguageData::ZafLanguageData(bool tStaticData) :
	ZafI18nData(),
	staticData(tStaticData), data(ZAF_NULLP(ZafLanguageStruct))
{
}

ZafLanguageData::ZafLanguageData(ZafLanguageStruct *tData, bool tStaticData) :
	ZafI18nData(),
	staticData(tStaticData), data(ZAF_NULLP(ZafLanguageStruct))
{
	SetLanguage(tData);
}

ZafLanguageData::ZafLanguageData(const ZafLanguageData &copy) :
	ZafI18nData(copy),
	staticData(copy.staticData), data(ZAF_NULLP(ZafLanguageStruct))
{
	// Copy the data contents.
	SetLanguage(copy.data);
}

ZafLanguageData::~ZafLanguageData(void)
{
	// Delete the language data.
	Clear();
}

ZafData *ZafLanguageData::Duplicate(void)
{
	return (new ZafLanguageData(*this));
}

ZafIChar *ZafLanguageData::GetMessage(ZafNumberID matchID, bool useDefault,
	ZafIChar *hotKeyChar, int *hotKeyIndex) const
{
	// Try to find a matching entry.
	for (int i = 0; data[i].text; i++)
		if (data[i].numberID == matchID)
		{
			if (hotKeyChar)
				*hotKeyChar = data[i].hotKeyChar;
			if (hotKeyIndex)
				*hotKeyIndex = data[i].hotKeyIndex;
			return (data[i].text);
		}
	return (useDefault ? data[0].text : ZAF_NULLP(ZafIChar));
}

void ZafLanguageData::Clear(void)
{
	// Clear the data.
	PushLevel();
	if (data && !StaticData())
	{
		// Delete the table contents
		for (int i = 0; data[i].text; i++)
		{
			if (data[i].stringID)
				delete (ZafIChar *)data[i].stringID;
			delete data[i].text;
		}
		delete []data;
	}
	data = ZAF_NULLP(ZafLanguageStruct);
	PopLevel();
}

void ZafLanguageData::LanguageAllocate(const ZafIChar *name)
{
	// Check for initialization.
	if (zafLanguage)
		return;
	ZafApplication::AddStaticModule(LanguageFree);

	// Try the zafDataManager.
	ZafError localError = ZAF_ERROR_NONE;
	if (!zafLanguage && zafDataManager)
	{
		// Read the language from the data manager.
		if (name && *name)
			zafLanguage = DynamicPtrCast(zafDataManager->AllocateData(name, ZafLanguageData::className, ZafLanguageData::classID), ZafLanguageData);

		// Make sure zafLanguage's destructor isn't called multiple times.
		if (zafLanguage)
			ZafApplication::SubtractStaticModule(LanguageFree);
		else if (zafDataPersistence)
		{
			localError = zafDataPersistence->Error();
			zafDataPersistence->SetError(ZAF_ERROR_NONE);
		}
	}

	// Reset the language name.
	if (!zafLanguage)
	{
		zafLanguage = new ZafLanguageData();
		zafLanguage->SetStringID(defaultLanguageName);
	}

	// Set/clear the zafLanguage error.
	zafLanguage->SetError(localError);
}

void ZafLanguageData::LanguageFree(bool globalRequest)
{
	// Destroy the static language.
	if (globalRequest && zafLanguage && zafLanguage->Destroyable() &&
		zafLanguage->NotifyCount() == 0)
	{
		delete zafLanguage;
		zafLanguage = ZAF_NULLP(ZafLanguageData);
	}
}

ZafError ZafLanguageData::SetMessage(ZafNumberID matchID, ZafIChar *text,
	ZafIChar hotKeyChar, int hotKeyIndex)
{
	// Try to find a matching entry.
	PushLevel();
	for (int i = 0; data[i].text; i++)
		if (data[i].numberID == matchID)
		{
			if (data[i].text)
				delete data[i].text;
			data[i].text = text ? strdup(text) : strdup(blankString);
			data[i].hotKeyChar = hotKeyChar;
			data[i].hotKeyIndex = hotKeyIndex;
			PopLevel();
			return (ZAF_ERROR_NONE);
		}
	PopLevel();
	return (ZAF_ERROR_INVALID_ID);
}

ZafError ZafLanguageData::SetLanguage(ZafLanguageStruct *srcTable)
{
	// Set the new information.
	PushLevel();
	if (StaticData())
		data = srcTable;
	else if (srcTable)
	{
		// Clear the old information.
		Clear();

		// Count the number of table objects.
		int count = 0;
		while (srcTable[count].text)
			count++;

		// Copy the table contents.
		data = new ZafLanguageStruct[count+1];
		for (int i = 0; i < count; i++)
		{
			data[i].text = srcTable[i].text ? strdup(srcTable[i].text) : strdup(blankString);
			data[i].numberID = srcTable[i].numberID;
			data[i].stringID = srcTable[i].stringID ? strdup(srcTable[i].stringID) : ZAF_NULLP(ZafIChar);
			data[i].hotKeyChar = srcTable[i].hotKeyChar;
			data[i].hotKeyIndex = srcTable[i].hotKeyIndex;
		}

		// Mark the end of the array.
		data[count].text = ZAF_NULLP(ZafIChar);
		data[count].numberID = ZAF_NUMID_NULL;
		data[count].stringID = ZAF_NULLP(ZafIChar);
		data[count].hotKeyChar = '\0';
		data[count].hotKeyIndex = 0;
	}
	PopLevel();

	// Return success.
	return (ZAF_ERROR_NONE);
}

ZafError ZafLanguageData::SetLanguage(const ZafLanguageData &copy)
{
	// Reset the language information.
	staticData = copy.staticData;
	SetLanguage(copy.data);
	return (ZAF_ERROR_NONE);
}

ZafError ZafLanguageData::SetLanguageName(const ZafIChar *newLanguageName)
{
	// Make sure language has been initialized once.
	LanguageAllocate(newLanguageName);

	// If the language is correct don't initialize it again.
	if (zafLanguage && !strcmp(newLanguageName, zafLanguage->StringID()))
		return (ZAF_ERROR_NONE);

	// Read a new language from the data manager.
	ZafLanguageData *newLanguage = ZAF_NULLP(ZafLanguageData);
	if (zafDataManager)
	{
		newLanguage = DynamicPtrCast(zafDataManager->AllocateData(newLanguageName, ZafLanguageData::className, ZafLanguageData::classID), ZafLanguageData);
		if (newLanguage && zafLanguage)
		{
			if (newLanguage && newLanguage->Destroyable() && newLanguage->NotifyCount() == 0)
				delete zafLanguage;
			zafLanguage = newLanguage;
			// LanguageAllocate sets ZafApplication to delete zafLanguage.
			ZafApplication::SubtractStaticModule(LanguageFree);
		}
		else if (zafDataPersistence)
			zafDataPersistence->SetError(ZAF_ERROR_NONE);
	}

	// Return the error status.
	return (newLanguage ? ZAF_ERROR_NONE : ZAF_ERROR_INVALID);
}

bool ZafLanguageData::SetStaticData(bool tStaticData)
{
	// If the data is marked static, the user is in charge of protecting 
	// the data.  Otherwise, we must copy the contents to a new array.
	// This is done by re-setting the variables and data pointer, then
	// by resetting the array.
	if (staticData == tStaticData)
		;
	else if (tStaticData)
		staticData = tStaticData;
	else
	{
		staticData = tStaticData;
		ZafLanguageStruct *newData = data;
		data = ZAF_NULLP(ZafLanguageStruct);
		SetLanguage(newData);
	}
	return (staticData);
}

// ----- Persistent functions -----------------------------------------------

ZafLanguageData::ZafLanguageData(const ZafIChar *name, ZafDataPersistence &persist) :
	ZafI18nData(name, persist.PushLevel(className, classID, ZAF_PERSIST_ROOT_DIRECTORY)),
	staticData(false), data(ZAF_NULLP(ZafLanguageStruct))
{
	// Read the object.
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafLanguageData::ReadData(persist);

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
	{
		SetError(persist.Error());
		persist.SetError(ZAF_ERROR_NONE);
	}
}

ZafElement *ZafLanguageData::Read(const ZafIChar *name, ZafDataPersistence &persist)
{
	return (new ZafLanguageData(name, persist));
}

void ZafLanguageData::ReadData(ZafDataPersistence &persist)
{
	// Read the data.
	int noOfElements;
	ZafFile *file = persist.CurrentFile();
	*file >> noOfElements;
	data = new ZafLanguageStruct[noOfElements+1];
	for (int i = 0; i < noOfElements; i++)
	{
		data[i].text = ZAF_NULLP(ZafIChar);
		ZafUInt16 hotKeyChar;
		*file >> data[i].numberID >> (ZafIChar **)&data[i].stringID >> &data[i].text >> hotKeyChar >> data[i].hotKeyIndex;
		data[i].hotKeyChar = (ZafIChar)hotKeyChar;
		if (!data[i].text)
			data[i].text = strdup(ZafLanguageData::blankString);
	}

	// Stub out the last entry.
	data[noOfElements].numberID = ZAF_NUMID_NULL;
	data[noOfElements].stringID = ZAF_NULLP(ZafIChar);
	data[noOfElements].text = ZAF_NULLP(ZafIChar);
	data[noOfElements].hotKeyChar = '\0';
	data[noOfElements].hotKeyIndex = 0;
}

void ZafLanguageData::Write(ZafDataPersistence &persist)
{
	// Write the object.
	ZafData::Write(persist.PushLevel(className, classID, ZAF_PERSIST_ROOT_DIRECTORY));
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafLanguageData::WriteData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafLanguageData::WriteData(ZafDataPersistence &persist)
{
	// Write the data.
	int noOfElements = 0;
	while (data[noOfElements].numberID != ZAF_NUMID_NULL)
		noOfElements++;

	ZafFile *file = persist.CurrentFile();
	*file << noOfElements;
	for (int i = 0; i < noOfElements; i++)
	{
		ZafUInt16 hotKeyChar = (ZafUInt16)data[i].hotKeyChar;
		*file << data[i].numberID << data[i].stringID << data[i].text << hotKeyChar << data[i].hotKeyIndex;
	}
}

