//	Zinc Application Framework - Z_HELP1.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_error.hpp>
#include <z_help.hpp>
#include <z_icon1.hpp>
#include <z_max.hpp>
#include <z_min.hpp>
#include <z_mouse2.hpp>
#include <z_title.hpp>
#include <z_scrll2.hpp>
#include <z_sys.hpp>
#include <z_utils.hpp>
#include <z_app.hpp>
#define ZAF_HELP_SYSTEM_INFO
#include "lang_def.cpp"
#include "img_def.cpp"
#include "gbl_def.cpp"

static ZafIChar ZAF_FARDATA _topicStringID[] = ZAF_ITEXT("TOPIC_CONTEXT");
static ZafIChar _matchAll[] = ZAF_ITEXT("*");
static ZafIChar _dataName[] = ZAF_ITEXT("objectData");

// ----- ZafHelpSystem ------------------------------------------------------

ZafHelpSystem::ZafHelpSystem(ZafFileSystem *tHelpStorage, ZafIChar *) :
	ZafHelpStub(), ZafWindow(0, 0, 66, 11),
	helpStorage(tHelpStorage), context(ZAF_NULLP(ZafIChar))
{
	// Initialize the global variable.
	if (!zafHelpSystem)
		zafHelpSystem = this;

	// Check the language and local information.
	LanguageAllocate();
	ImageAllocate();

	// Center the window.
	zafWindowManager->Center(this);

	SetDestroyable(false);

	// Create the general support fields.
	Add(borderField = new ZafBorder);
	Add(new ZafMaximizeButton);
	Add(new ZafMinimizeButton);
	Add(new ZafSystemButton(ZAF_NATIVE_SYSTEM_BUTTON));
	Add(titleField = new ZafTitle(new ZafStringData(language->GetMessage(ZAF_MSG_HELP_TOPIC))));

	ZafIcon *icon = new ZafIcon(0, 0, language->GetMessage(ZAF_MSG_ICONTITLE),
		helpIcon, ZAF_MINIMIZE_ICON);
	Add(icon);

	showButton = new ZafButton(2, 9, 14, 1, language->GetMessage(ZAF_MSG_SHOW_INDEX), 
		ZAF_NULLP(ZafBitmapData), ZAF_NATIVE_BUTTON);
	showButton->SetSendMessageWhenSelected(true);
	showButton->SetValue(S_HLP_SHOW_INDEX);
	Add(showButton);
	SetDefaultButton(showButton);

	closeButton = new ZafButton(48, 9, 14, 1, language->GetMessage(ZAF_MSG_CLOSE),
		ZAF_NULLP(ZafBitmapData), ZAF_NATIVE_BUTTON);
	closeButton->SetSendMessageWhenSelected(true);
	closeButton->SetValue(S_HLP_CLOSE);
	Add(closeButton);

	// Create the topic fields.
	topicTitle = new ZafPrompt(2, 0, 60, new ZafStringData());
	topicTitle->SetHzJustify(ZAF_HZ_CENTER);
	Add(topicTitle);
	topicTitle->ConvertCoordinates(ZAF_PIXEL);

	topicContext = new ZafText(2, 2, 60, 6, new ZafStringData(context, -1, false));
	topicContext->SetBordered(true);
	topicContext->SetViewOnly(true);
	topicContext->SetStringID(_topicStringID);
	ZafScrollBar *topicScroll = new ZafScrollBar(0, 0, 0, 0,
		ZAF_NULLP(ZafScrollData), ZAF_VERTICAL_SCROLL);
	topicScroll->SetSupportObject(true);
	topicContext->Add(topicScroll);
	Add(topicContext);
	topicContext->ConvertCoordinates(ZAF_PIXEL);

	// Create the index fields.
	indexTopic = new ZafPrompt(2, 0, 60, new ZafStringData(language->GetMessage(ZAF_MSG_MESSAGE1)));
	indexTopic->SetHzJustify(ZAF_HZ_CENTER);
	Add(indexTopic);
	indexTopic->ConvertCoordinates(ZAF_PIXEL);

	indexInstruction = new ZafPrompt(2, 1, 60, new ZafStringData(language->GetMessage(ZAF_MSG_MESSAGE2)));
	indexInstruction->SetHzJustify(ZAF_HZ_CENTER);
	Add(indexInstruction);
	indexInstruction->ConvertCoordinates(ZAF_PIXEL);

	indexTitle = new ZafString(2, 2, 60, new ZafStringData(ZafLanguageData::blankString, 100));
	Add(indexTitle);
	indexTitle->ConvertCoordinates(ZAF_PIXEL);

	indexContext = new ZafVtList(2, 3, 60, 5);
	indexContext->SetAutoSortData(true);
	ZafScrollBar *indexScroll = new ZafScrollBar(0, 0, 0, 0, ZAF_NULLP(ZafScrollData), ZAF_VERTICAL_SCROLL);
	indexScroll->SetSupportObject(true);
	indexContext->Add(indexScroll);
	Add(indexContext);
	indexContext->ConvertCoordinates(ZAF_PIXEL);
}

ZafHelpSystem::ZafHelpSystem(const ZafHelpSystem &copy) :
	ZafHelpStub(), ZafWindow(copy),
	helpStorage(copy.helpStorage), context(copy.context)
{
}

ZafHelpSystem::~ZafHelpSystem(void)
{
	// Remove and delete the support windows.
	if (windowManager)
		windowManager->Subtract(this);

	RemoveIndexFields();
	delete indexTopic;
	delete indexInstruction;
	delete indexTitle;
	delete indexContext;
	RemoveTopicFields();
	delete topicTitle;
	delete topicContext;
	if (context)
		delete []context;

	// Check the language information.
	LanguageFree();
	ImageFree();
}

void ZafHelpSystem::AddIndexFields(void)
{
	// Load the help contexts.
	if (helpStorage && !indexContext->First())
	{
		eventManager->SetDeviceImage(E_MOUSE, DM_WAIT);
		helpStorage->ChDir(ZafFileSystem::rootDirectoryName);
		helpStorage->ChDir(className);
		indexContext->Destroy();
		ZafFileInfoStruct fileInfo;
		for (int finished = helpStorage->FindFirst(fileInfo, _matchAll); 
			!finished; finished = helpStorage->FindNext(fileInfo, _matchAll))
		{
			ZafButton *button = new ZafButton(0, 0, 30, 1, fileInfo.stringID,
				ZAF_NULLP(ZafBitmapData), ZAF_NATIVE_BUTTON);
			button->SetDepth(0);
			button->SetSelectOnDoubleClick(true);
			button->SetUserFunction(ItemCallback);
			button->SetStringID(fileInfo.name);
			button->SetHelpContext(fileInfo.name);
			indexContext->Add(button);
		}
		eventManager->SetDeviceImage(E_MOUSE, DM_VIEW);
	}

	// Reset the title and show buttons.
	titleField->SetText(language->GetMessage(ZAF_MSG_HELP_INDEX));
	showButton->SetText(language->GetMessage(ZAF_MSG_SHOW_TOPIC));
	showButton->SetValue(S_HLP_SHOW_TOPIC);

	SetAutomaticUpdate(false);

	// Add the support fields.
	Add(indexTopic);
	Add(indexInstruction);
	Add(indexContext);
	Add(indexTitle);

	SetAutomaticUpdate(true);

	if (screenID)
		Event(ZafEventStruct(S_REDISPLAY_REGION, 0, clientRegion));
}

void ZafHelpSystem::AddTopicFields(ZafIChar *newHelpTitle, ZafIChar *newHelpContext)
{
	// Reset the title and context fields.
	topicTitle->SetText(newHelpTitle);
	topicContext->StringData()->SetText(newHelpContext, strlen(newHelpContext));

	// Reset the title and show buttons.
	titleField->SetText(language->GetMessage(ZAF_MSG_HELP_TOPIC));
	showButton->SetText(language->GetMessage(ZAF_MSG_SHOW_INDEX));
	showButton->SetValue(S_HLP_SHOW_INDEX);

	SetAutomaticUpdate(false);

	// Add the support fields.
	Add(topicTitle);
	Add(topicContext);

	SetAutomaticUpdate(true);

	if (screenID)
		Event(ZafEventStruct(S_REDISPLAY_REGION, 0, clientRegion));
}

void ZafHelpSystem::ComputeGeometry(void)
{
	// Check for absolute window sizes.
	const int MIN_WIDTH = 50 * display->cellWidth;
	const int MIN_HEIGHT = 8 * display->cellHeight;
	ZafRegionStruct region = Region();
	if (!Minimized() &&
		(region.Width() < MIN_WIDTH || region.Height() < MIN_HEIGHT))
	{
		if (region.Width() < MIN_WIDTH)
		{
			region.right = region.left + MIN_WIDTH - 1;
			SetRegion(region);
		}
		if (region.Height() < MIN_HEIGHT)
		{
			region.bottom = region.top + MIN_HEIGHT - 1;
			SetRegion(region);
		}
		return;
	}

	// Perform geometry management on children.
	ZafRegionStruct maxRegion = MaxRegion(this);
	maxRegion.HzShift(-maxRegion.left);
	maxRegion.VtShift(-maxRegion.top);
	int buttonHeight = closeButton->Region().Height();
	int buttonWidth = closeButton->Region().Width();
	int hzSpacing = 2 * display->cellWidth;
	int vtSpacing = 2 * display->cellWidth;

	// right edge
	indexTopic->zafRegion.right =
		indexInstruction->zafRegion.right =
		indexTitle->zafRegion.right =
		indexContext->zafRegion.right =
		topicTitle->zafRegion.right =
		topicContext->zafRegion.right = maxRegion.right - hzSpacing;

	// bottom edge
	region = showButton->Region();
	region.bottom = maxRegion.bottom - vtSpacing;
	region.top = closeButton->Region().bottom - buttonHeight + 1;
	showButton->SetRegion(region);

	region = closeButton->Region();
	region.right = maxRegion.right - hzSpacing;
	region.left = region.right - buttonWidth + 1;
	region.bottom = maxRegion.bottom - vtSpacing;
	region.top = region.bottom - buttonHeight + 1;
	closeButton->SetRegion(region);

	indexContext->zafRegion.bottom = topicContext->zafRegion.bottom =
		closeButton->Region().top - vtSpacing;
}

void ZafHelpSystem::DisplayHelp(const ZafIChar *name)
{
	// Try to find a matching help context.
	if (name && name[0])
	{
		// Move to the help directory.
		helpStorage->ChDir(ZafFileSystem::rootDirectoryName);
		helpStorage->ChDir(className);

		// See if the name is a direct search.
		ZafIChar title[256];
		ZafFileInfoStruct fileInfo;
		int findError = helpStorage->FindFirst(fileInfo, name);
		if (!findError)
		{
			ZafIChar *tContext = context; // save no-allocate data pointer.
			helpStorage->ChDir(fileInfo.name);
			ZafFile *hStorage = helpStorage->Open(_dataName, ZAF_FILE_READ);
			hStorage->Read(title, 256);
			hStorage->Read(&context);
			helpStorage->Close(hStorage);
			helpStorage->ChDir(ZafFileSystem::parentDirectoryName);
			RemoveIndexFields();
			AddTopicFields(title, context);
			if (tContext)
				delete []tContext;
		}
		else
		{
			// Try to match against the help system titles.
			int length = strlen(name);
			helpStorage->ChDir(ZafFileSystem::rootDirectoryName);
			helpStorage->ChDir(className);
			for (int finished = helpStorage->FindFirst(fileInfo, _matchAll);
			 	findError && !finished; finished = helpStorage->FindNext(fileInfo, _matchAll))
				if (streq(fileInfo.name, ZafFileSystem::currentDirectoryName) &&
					streq(fileInfo.name, ZafFileSystem::parentDirectoryName))
				{
					helpStorage->ChDir(fileInfo.name);
					ZafFile *hObject = helpStorage->Open(ZAF_ITEXT("objectData"), ZAF_FILE_READ);
					helpStorage->ChDir(ZafFileSystem::parentDirectoryName);
					hObject->Read(title, 256);
					if (!strnicmp(name, title, length))
					{
						findError = false;
						ZafIChar *tContext = context; // save no-allocate data pointer.
						hObject->Read(&context);
						RemoveIndexFields();
						AddTopicFields(title, context);
						if (tContext)
							delete []tContext;

						// Update the list field.
						for (ZafWindowObject *object = indexContext->First(); object; object = object->Next())
							if (!streq(fileInfo.name, object->StringID()))
							{
								// Make the object current and selected.
								object->SetSelected(true);
								indexContext->Add(object);
								indexTitle->StringData()->SetText(DynamicPtrCast(object, ZafButton)->Text());
								break;
							}
					}
					helpStorage->Close(hObject);
				}
		}
	}
	else
	{
		RemoveTopicFields();
		AddIndexFields();
	}

	// Invoke the index help.
	if (zafWindowManager->Index(this) == -1)
		zafWindowManager->Add(this);
	else
		SetFocus(true);
}

ZafEventType ZafHelpSystem::Event(const ZafEventStruct &event)
{
	// Switch on the event.
	ZafEventType ccode = event.type;
	switch (ccode)
	{
	case N_SIZE:
		ComputeGeometry();
		ccode = ZafWindow::Event(event);
		break;

	case S_HLP_CLOSE:
		zafWindowManager->Subtract(this);	// Remove the help topic window.
		zafEventManager->Put(S_CONTINUE);
		break;

	case S_HLP_SHOW_INDEX:
		RemoveTopicFields();
		AddIndexFields();
		break;

	case S_HLP_SHOW_TOPIC:
		// Display the help context.
		DisplayHelp(indexTitle->Text());
		break;

	case S_HLP_SELECT_TOPIC:
		// Display the help context.
		DisplayHelp(event.windowObject->HelpContext());
		break;

	case S_HLP_UPDATE_NAME:
		// Reset the title field.  This message is sent from ItemCallback().
		indexTitle->StringData()->SetText(event.windowObject->Text());
		break;

	default:
		ccode = ZafWindow::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

ZafEventType ZafHelpSystem::ItemCallback(ZafWindowObject *object, ZafEventStruct &event, ZafEventType ccode)
{
	// Check for selection of the field.
	if (ccode == L_DOUBLE_CLICK || ccode == L_SELECT)
	{
		// Update the help context selection.
		event.type = S_HLP_UPDATE_NAME;
		event.windowObject = object;
		object->eventManager->Put(event);
		// Show the associated help.
		if (ccode == L_DOUBLE_CLICK)
		{
			event.type = S_HLP_SELECT_TOPIC;
			object->eventManager->Put(event);
		}
	}
	return (0);
}

void ZafHelpSystem::RemoveIndexFields(void)
{
	// Remove the support fields.
	Subtract(indexTopic);
	Subtract(indexInstruction);
	Subtract(indexTitle);
	Subtract(indexContext);
}

void ZafHelpSystem::RemoveTopicFields(void)
{
	// Remove the support fields.
	Subtract(topicTitle);
	Subtract(topicContext);
}

void ZafHelpSystem::ResetStorage(ZafFileSystem *_helpStorage)
{
	// Reset the storage and update the index fields.
	if (_helpStorage == helpStorage)
		return;

	helpStorage = _helpStorage;
	indexContext->Destroy();
}

void ZafHelpSystem::ImageAllocate(void)
{
	// Initialize the images.
	if (!helpIcon)
	{
		ZafApplication::AddStaticModule(ImageFree);
		helpIcon = new ZafIconData(_help_ZafIcon);
		helpIcon->SetDestroyable(false);
	}
}

void ZafHelpSystem::ImageFree(bool globalRequest)
{
	// Destroy the static images.
	if (globalRequest && helpIcon)
	{
		delete helpIcon;
		helpIcon = ZAF_NULLP(ZafIconData);
	}
}

void ZafHelpSystem::LanguageAllocate(void)
{
	// Check for initialization.
	if (language)
		return;
	ZafApplication::AddStaticModule(LanguageFree);

	// Try the zafDataManager.
	if (!language && zafDataManager)
		language = DynamicPtrCast(zafDataManager->AllocateData(className, ZafLanguageData::className, ZafLanguageData::classID), ZafLanguageData);

	// Default to code initialization.
	if (!language)
		language = new ZafLanguageData(defaultHelpStrings, true);
}

void ZafHelpSystem::LanguageFree(bool globalRequest)
{
	// Destroy the static language.
	if (globalRequest && language && language->Destroyable() &&
		language->NotifyCount() == 0)
	{
		delete language;
		language = ZAF_NULLP(ZafLanguageData);
	}
}

