//	Zinc Application Framework - Z_GROUP.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_group.hpp>
#include <z_utils.hpp>
#include <z_ctype.hpp>
#define ZAF_GROUP_INFO
#include "gbl_def.cpp"

static ZafStringID _stringDataName = ZAF_ITEXT("stringData");

// ----- ZafGroup -----------------------------------------------------------

ZafGroup::ZafGroup(int left, int top, int width, int height, ZafStringData *zStringData) :
	ZafWindow(left, top, width, height),
	autoSelect(false), hotKeyChar(0), hotKeyIndex(-1), hzSpacing(0),
	stringData(ZAF_NULLP(ZafStringData))
{
	// Initialize the group information.
	SetBordered(true);

	// Initialize the group data.
	if (zStringData)
		SetStringData(zStringData);
}

ZafGroup::ZafGroup(int left, int top, int width, int height, const ZafIChar *text) :
	ZafWindow(left, top, width, height),
	autoSelect(false), hotKeyChar(0), hotKeyIndex(-1), hzSpacing(0), 
	stringData(ZAF_NULLP(ZafStringData))
{
	// Initialize the group information.
	SetBordered(true);

	// Initialize the group data.
	SetStringData(new ZafStringData(text));
}

ZafGroup::ZafGroup(const ZafGroup &copy) :
	ZafWindow(copy),
	autoSelect(copy.autoSelect), hotKeyChar(copy.hotKeyChar),
	hotKeyIndex(copy.hotKeyIndex), hzSpacing(copy.hzSpacing),
	stringData(ZAF_NULLP(ZafStringData))
{
	// Initialize the group information.
	if (copy.stringData)
		SetStringData(new ZafStringData(*copy.StringData()));
}

ZafGroup::~ZafGroup(void)
{
	// Restore the group information.
	if (stringData && stringData->Destroyable() && stringData->NotifyCount() == 0)
		delete stringData;
}

ZafWindowObject *ZafGroup::Duplicate(void)
{
	return (new ZafGroup(*this));
}

ZafLogicalEvent ZafGroup::LogicalEvent(const ZafEventStruct &event)
{
	ZafLogicalEvent ccode = MapEvent(defaultEventMap, event);
	return ((ccode == L_NONE) ? ZafWindow::LogicalEvent(event) : ccode);
}

const ZafPaletteStruct *ZafGroup::MapClassPalette(ZafPaletteType type, ZafPaletteState state)
{
	const ZafPaletteStruct *palette = MapPalette(defaultPaletteMap, type, state);
	return (palette ? palette : ZafWindowObject::MapClassPalette(type, state));
}

bool ZafGroup::SetAcceptDrop(bool )
{
	// acceptDrop is false for this class.
	acceptDrop = false;
	return (acceptDrop);
}

bool ZafGroup::SetAutoSelect(bool tAutoSelect)
{
	// Make sure the attribute has changed.
	if (autoSelect != tAutoSelect && !screenID)
		autoSelect = tAutoSelect;

	// Return the current attribute.
	return (autoSelect);
}

bool ZafGroup::SetDestroyable(bool )
{
	// destroyable is false for this class.
	destroyable = false;
	return (destroyable);
}

ZafIChar ZafGroup::SetHotKey(ZafIChar setHotKeyChar, int setHotKeyIndex)
{
	// Make sure the request is valid.
	if (screenID)
		return (hotKeyChar);

	// hotKeyChar affects only the action (keyStroke), and keyKeyIndex
	// affects only the appearance (underline).
	ZafIChar upperHotKeyChar = setHotKeyChar ? ToUpper(setHotKeyChar) : setHotKeyChar;
	if (hotKeyChar != upperHotKeyChar)
		hotKeyChar = upperHotKeyChar;
	if (hotKeyIndex != setHotKeyIndex)
		hotKeyIndex = setHotKeyIndex;

	// Return the current hot key.
	return (hotKeyChar);
}

bool ZafGroup::SetLocked(bool )
{
	// locked is false for this class.
	locked = false;
	return (locked);
}

bool ZafGroup::SetMaximized(bool )
{
	// maximized is false for this class.
	maximized = false;
	return (maximized);
}

bool ZafGroup::SetMinimized(bool )
{
	// minimized is false for this class.
	minimized = false;
	return (minimized);
}

bool ZafGroup::SetMoveable(bool )
{
	// moveable is false for this class.
	moveable = false;
	return (moveable);
}

bool ZafGroup::SetSizeable(bool )
{
	// sizeable is false for this class.
	sizeable = false;
	return (sizeable);
}

ZafError ZafGroup::SetStringData(ZafStringData *tStringData)
{
	// Remove the data notification.
	if (stringData)
		stringData->SubtractNotification(this, (ZafUpdateFunction)Update);

	// Reset the string data.
	if (stringData && stringData != tStringData &&
		stringData->Destroyable() && stringData->NotifyCount() == 0)
		delete stringData;
	stringData = tStringData;

	// Add the data notification.
	if (stringData)
	{
		stringData->AddNotification(this, (ZafUpdateFunction)Update);
		if (!stringData->StringID())
			stringData->SetStringID(_stringDataName);
	}
	return (ZAF_ERROR_NONE);
}

bool ZafGroup::SetTemporary(bool )
{
	// temporary is false for this class.
	temporary = false;
	return (temporary);
}

ZafError ZafGroup::SetText(const ZafIChar *text)
{
	// Reset the group text.
	if (stringData)
		stringData->SetText(text);
	else
		SetStringData(new ZafStringData(text));

	// Update the button text with the environment.
	return (OSSetText());
}

const ZafIChar *ZafGroup::Text(void)
{
	return (stringData ? stringData->Text() : ZAF_NULLP(ZafIChar));
}

// ----- Persistent functions -----------------------------------------------

ZafGroup::ZafGroup(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafWindow(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	stringData(ZAF_NULLP(ZafStringData))
{
	// Read the data.
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafGroup::ReadData(persist);

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

ZafElement *ZafGroup::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafGroup(name, persist));
}

void ZafGroup::ReadData(ZafObjectPersistence &persist)
{
	// Read the data.
	ZafUInt16 flag1, flag2;
	ZafIChar stringName[ZAF_MAXNAMELEN];
	ZafFile *file = persist.CurrentFile();
	*file >> hzSpacing >> flag1 >> flag2 >> hotKeyIndex >> stringName;
	autoSelect = (flag1 & 0x0001) ? true : false;
	hotKeyChar = (ZafIChar)flag2;

	if (*stringName)
		stringData = new ZafStringData(stringName, persist);
}

void ZafGroup::Write(ZafObjectPersistence &persist)
{
	// Write the object.
	ZafWindow::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafGroup::WriteData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafGroup::WriteData(ZafObjectPersistence &persist)
{
	// Write the data.
	ZafUInt16 flag1 = 0;
	flag1 |= autoSelect ? 0x0001 : 0;
	ZafUInt16 flag2 = (ZafUInt16)hotKeyChar;
	const ZafIChar *stringName = stringData ? stringData->StringID() : ZAF_NULLP(ZafIChar);
	ZafFile *file = persist.CurrentFile();
	*file << hzSpacing << flag1 << flag2 << hotKeyIndex << stringName;

	// Write the string data.
	if (stringData)
		stringData->Write(persist);
}

