//	Zinc Application Framework - Z_GMGR.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_gmgr.hpp>
#define ZAF_GEOMETRY_INFO
#include "gbl_def.cpp"

ZafGeometryManager *ZafWindow::GeometryManager(void) const
{
	for (ZafWindowObject *object = SupportFirst(); object; object = object->Next())
		if (object->NumberID() == ZAF_NUMID_GEOMETRY)
			return (DynamicPtrCast(object, ZafGeometryManager));
	return (ZAF_NULLP(ZafGeometryManager));
}

// ----- ZafGeometryManager -------------------------------------------------

static ZafStringIDChar ZAF_FARDATA _stringID[] = ZAF_ITEXT("ZAF_NUMID_GEOMETRY");

ZafGeometryManager::ZafGeometryManager(void) :
	ZafWindowObject(0, 0, 0, 0),
	ZafList()
{
	// Reset default window object flags.
	SetSupportObject(true);
	SetDisabled(true);
	SetRegionType(ZAF_AVAILABLE_REGION);
	SetNumberID(ZAF_NUMID_GEOMETRY);
	SetStringID(_stringID);
}

ZafGeometryManager::ZafGeometryManager(const ZafGeometryManager &copy) :
	ZafWindowObject(copy)
{
	for (ZafConstraint *constraint = copy.First(); constraint; constraint = constraint->Next())
		Add(constraint->Duplicate());
}

ZafGeometryManager::~ZafGeometryManager(void)
{
}

ZafConstraint *ZafGeometryManager::Add(ZafConstraint *object, ZafConstraint *position)
{
	// Make sure add is allowed.
	if (object->parent)
		return (object);

	// Add the object to the list.
	if (position)
		ZafList::Add(object, position);	// position insert.
	else
		ZafList::Add(object);				// compareFunction insert.
	object->parent = this;

	// Return a pointer to the object.
	return (object);
}

ZafEventType ZafGeometryManager::Draw(const ZafEventStruct &, ZafEventType ccode)
{
	// Return the control code.
	return (ccode);
}

ZafWindowObject *ZafGeometryManager::Duplicate(void)
{
	return (new ZafGeometryManager(*this));
}

ZafEventType ZafGeometryManager::Event(const ZafEventStruct &event)
{
	// Check for zinc events.
	ZafEventType ccode = event.type;

	// Switch on the event type.
	ZafConstraint *constraint;
	switch (ccode)
	{
	case S_COMPUTE_SIZE:
		zafRegion.left = zafRegion.top = zafRegion.right = zafRegion.bottom = -1;
		for (constraint = First(); constraint; constraint = constraint->Next())
			constraint->Event(event);
		break;

	case S_INITIALIZE:
		ccode = ZafWindowObject::Event(event);
		for (constraint = First(); constraint; constraint = constraint->Next())
			constraint->Event(event);
		break;

	case S_ADD_OBJECT:
		Add((ZafConstraint *)event.data);
		break;

	case S_SUBTRACT_OBJECT:
		Subtract((ZafConstraint *)event.data);
		break;

	default:
		ccode = S_UNKNOWN;
		break;
	}

	// Return the control code.
	return (ccode);
}

void ZafGeometryManager::OSSize(void)
{
	// Prevent sizing from the base class.
}

bool ZafGeometryManager::SetAcceptDrop(bool )
{
	// acceptDrop is false for this class.
	acceptDrop = false;
	return (acceptDrop);
}

bool ZafGeometryManager::SetAutomaticUpdate(bool )
{
	// automaticUpdate is true for this class.
	automaticUpdate = true;
	return (automaticUpdate);
}

ZafLogicalColor ZafGeometryManager::SetBackgroundColor(ZafLogicalColor , ZafLogicalColor )
{
	// This class does not permit background color changes.
	return (ZAF_CLR_NULL);
}

bool ZafGeometryManager::SetBordered(bool )
{
	// bordered is false for this class.
	bordered = false;
	return (bordered);
}

ZafCoordinateType ZafGeometryManager::SetCoordinateType(ZafCoordinateType )
{
	// coordinateType is ZAF_PIXEL for this class.
	coordinateType = ZAF_PIXEL;
	return (coordinateType);
}

bool ZafGeometryManager::SetCopyDraggable(bool )
{
	// copyDraggable is false for this class.
	copyDraggable = false;
	return (copyDraggable);
}

bool ZafGeometryManager::SetFocus(bool )
{
	// focus is false for this class.
	focus = false;
	return (focus);
}

ZafLogicalFont ZafGeometryManager::SetFont(ZafLogicalFont )
{
	// This class does not permit font changes.
	return (ZAF_FNT_NULL);
}

const ZafIChar *ZafGeometryManager::SetHelpContext(const ZafIChar *)
{
	// help context is empty for this class.
	return (ZAF_NULLP(ZafIChar));
}

const ZafIChar *ZafGeometryManager::SetHelpObjectTip(const ZafIChar *)
{
	// helpObjectTip cannot be set for this class.
	return (ZAF_NULLP(ZafIChar));
}

bool ZafGeometryManager::SetLinkDraggable(bool )
{
	// linkDraggable is false for this class.
	linkDraggable = false;
	return (linkDraggable);
}

bool ZafGeometryManager::SetMoveDraggable(bool )
{
	// moveDraggable is false for this class.
	moveDraggable = false;
	return (moveDraggable);
}

bool ZafGeometryManager::SetNoncurrent(bool )
{
	// noncurrent is true for this class.
	noncurrent = true;
	return (noncurrent);
}

bool ZafGeometryManager::SetDisabled(bool )
{
	// disabled is true for this class.
	disabled = true;
	return (disabled);
}

bool ZafGeometryManager::SetOSDraw(bool )
{
	// osDraw is automatically false for geometry manager.
	osDraw = false;
	return (osDraw);
}

bool ZafGeometryManager::SetParentDrawBorder(bool )
{
	// parentDrawBorder is false for this class.
	parentDrawBorder = false;
	return (parentDrawBorder);
}

bool ZafGeometryManager::SetParentDrawFocus(bool )
{
	// parentDrawFocus is false for this class.
	parentDrawFocus = false;
	return (parentDrawFocus);
}

bool ZafGeometryManager::SetParentPalette(bool )
{
	// parentPalette is false for this class.
	parentPalette = false;
	return (parentPalette);
}

const ZafIChar *ZafGeometryManager::SetQuickTip(const ZafIChar *)
{
	// quickTip cannot be set for this class.
	return (ZAF_NULLP(ZafIChar));
}

ZafRegionType ZafGeometryManager::SetRegionType(ZafRegionType )
{
	// regionType is ZAF_AVAILABLE_REGION for this class.
	regionType = ZAF_AVAILABLE_REGION;
	return (regionType);
}

bool ZafGeometryManager::SetSelected(bool )
{
	// selected is false for this class.
	selected = false;
	return (selected);
}

bool ZafGeometryManager::SetSupportObject(bool )
{
	// supportObject is true for this class.
	supportObject = true;
	return (supportObject);
}

ZafLogicalColor ZafGeometryManager::SetTextColor(ZafLogicalColor , ZafLogicalColor )
{
	// This class does not permit text color changes.
	return (ZAF_CLR_NULL);
}

ZafUserFunction ZafGeometryManager::SetUserFunction(ZafUserFunction )
{
	// userFunction is null for this class.
	userFunction = ZAF_NULLF(ZafUserFunction);
	return (userFunction);
}

ZafPaletteMap *ZafGeometryManager::SetUserPalette(ZafPaletteMap *, bool )
{
	// Setting user palette not allowed in this class.
	return (ZAF_NULLP(ZafPaletteMap));
}

bool ZafGeometryManager::SetVisible(bool )
{
	// visible is false for this class.
	visible = false;
	return (visible);
}

ZafConstraint *ZafGeometryManager::Subtract(ZafConstraint *object)
{
	// Check for a null.
	if (!object)
		return (ZAF_NULLP(ZafConstraint));
	return ((ZafConstraint *)ZafList::Subtract(object));
}

// ----- Persistent functions -----------------------------------------------

ZafGeometryManager::ZafGeometryManager(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafWindowObject(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	ZafList()
{
	// Reset default window object flags.
	SetSupportObject(true);
	SetDisabled(true);
	SetRegionType(ZAF_AVAILABLE_REGION);
	SetNumberID(ZAF_NUMID_GEOMETRY);
	SetStringID(_stringID);

	// Read the sub-object information.
	ZafList list;

	// Read the sub-object information.
	ZafFileSystem *fileSystem = persist.CurrentFileSystem();
	ZafFileInfoStruct fileInfo;
	int finished = fileSystem->FindFirst(fileInfo, ZAF_ITEXT("*"));
	while (!finished)
	{
		// Try to find a matching object constructor.
		if (fileInfo.numberID)
		{
			ZafElement *element = new ZafElement;
			element->SetNumberID(fileInfo.numberID);
			element->SetStringID(fileInfo.name);
			list.Add(element);
		}

		// Get the next entry.
		finished = fileSystem->FindNext(fileInfo, ZAF_ITEXT("*"));
	}

	for (ZafElement *element = list.First(); element; element = element->Next())
	{
		ZafObjectConstructor constructor = persist.GetObjectConstructor(element->NumberID());
		if (constructor)
		{
			Add((ZafConstraint *)(constructor)(element->StringID(),
				persist.PushLevel(element->ClassName(), element->ClassID(), ZAF_PERSIST_FILE)));
			persist.PopLevel();
		}
	}
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

ZafElement *ZafGeometryManager::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafGeometryManager(name, persist));
}

void ZafGeometryManager::Write(ZafObjectPersistence &persist)
{
	ZafWindowObject::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));

	// Write the sub-object information.
	for (ZafConstraint *object = First(); object; object = object->Next())
	{
		// Write the object.
		object->Write(persist.PushLevel(object->ClassName(), object->ClassID(), ZAF_PERSIST_FILE));
		persist.PopLevel();
	}

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

