//	Zinc Application Framework - Z_COMBO.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_combo.hpp>
#include <z_str1.hpp>
#if defined(ZAF_MSDOS)
#	include "d_combo.hpp"
#endif
#define ZAF_COMBO_BOX_INFO
#include "gbl_def.cpp"

static ZafStringID _listName = ZAF_ITEXT("list");

// ----- ZafComboBox --------------------------------------------------------

ZafComboBox::ZafComboBox(int left, int top, int width, int height) :
	ZafWindow(left, top, width, 1), viewOnly(false)
{
	// Initialize the combo-box information.
	SetBordered(true);
	list = new ZafVtList(0, 0, width, height);
	list->parent = this;
	list->SetDestroyable(false);
	list->SetTemporary(true);
	list->SetParentPalette(true);
	list->SetStringID(_listName);
	SetSelectionType(ZAF_SINGLE_SELECTION);	// Also affects the list.

#if defined(ZAF_MOTIF)
	comboItem = ZAF_NULLP(ZafWindowObject);
#elif defined(ZAF_MSDOS) || defined(ZAF_CURSES)
	comboString = comboButton = ZAF_NULLP(ZafWindowObject);
#elif defined(ZAF_MACINTOSH)
	comboString = ZAF_NULLP(ZafWindowObject);
#endif
}

ZafComboBox::ZafComboBox(const ZafComboBox &copy) :
	ZafWindow(copy), viewOnly(copy.viewOnly)
{
	// Initialize the combo-box information.
	list = new ZafVtList(*copy.list);
	list->parent = this;

	// Remove any combo-item information.
	ZafList::Destroy();
#if defined(ZAF_MOTIF)
	comboItem = ZAF_NULLP(ZafWindowObject);
#elif defined(ZAF_MSDOS) || defined(ZAF_CURSES)
	comboString = comboButton = ZAF_NULLP(ZafWindowObject);
#elif defined(ZAF_MACINTOSH)
	comboString = ZAF_NULLP(ZafWindowObject);
#endif
}

ZafComboBox::~ZafComboBox(void)
{
	delete list;
}

void ZafComboBox::BroadcastEvent(const ZafEventStruct &event)
{
	// Broadcast the event to all children.
	ZafWindowObject *object;
	for (object = SupportFirst(); object; object = object->Next())
		object->Event(event);
	for (object = ComboFirst(); object; object = object->Next())
		object->Event(event);
}

void ZafComboBox::Destroy(void)
{
	// Destroy the list elements (high level).
	OSDestroy();

	// Destroy the list elements (low level).
	list->Destroy();
}

ZafWindowObject *ZafComboBox::Duplicate(void)
{
	return (new ZafComboBox(*this));
}

ZafWindowObject *ZafComboBox::FocusObject(void) const
{
	// Return a pointer to self.
	if (list->Focus())
		return (list->Current() ? list->Current() : list);
	else if (focus)
		return ((ZafWindowObject *)this);
	return (ZAF_NULLP(ZafWindowObject));
}

ZafWindowObject *ZafComboBox::GetObject(ZafNumberID zNumberID)
{
	// Try to match on the current object.
	ZafWindowObject *match = ZafWindowObject::GetObject(zNumberID);

	if (!match && list)
		match = list->GetObject(zNumberID);

	return (match);
}

ZafWindowObject *ZafComboBox::GetObject(const ZafIChar *zStringID)
{
	// Try to match on the current object.
	ZafWindowObject *match = ZafWindowObject::GetObject(zStringID);

	if (!match && list)
		match = list->GetObject(zStringID);

	return (match);
}

ZafLogicalEvent ZafComboBox::LogicalEvent(const ZafEventStruct &event)
{
	ZafLogicalEvent ccode = MapEvent(defaultEventMap, event);
	return ((ccode == L_NONE) ? ZafWindow::LogicalEvent(event) : ccode);
}

const ZafPaletteStruct *ZafComboBox::MapClassPalette(ZafPaletteType type, ZafPaletteState state)
{
	const ZafPaletteStruct *palette = MapPalette(defaultPaletteMap, type, state);
	return (palette ? palette : ZafWindow::MapClassPalette(type, state));
}

bool ZafComboBox::SetAutomaticUpdate(bool setAutomaticUpdate)
{
	// Defer the first part to the base class.
	ZafWindowObject::SetAutomaticUpdate(setAutomaticUpdate);

	// Now mirror the list.
	return (list->SetAutomaticUpdate(automaticUpdate));
}

ZafLogicalColor ZafComboBox::SetBackgroundColor(ZafLogicalColor color, ZafLogicalColor mono)
{
	// Make sure there is a userPalette.
	if (!userPaletteData)
		SetUserPaletteData(new ZafPaletteData());

	// Add the background color entry.
	ZafPaletteStruct backgroundPalette = userPaletteData->GetPalette(ZAF_PM_BACKGROUND, ZAF_PM_NOT_SELECTED);
	backgroundPalette.colorBackground = color;
	backgroundPalette.monoBackground = mono;
	userPaletteData->AddPalette(ZAF_PM_BACKGROUND, ZAF_PM_NOT_SELECTED, backgroundPalette);

	// Add the selected text color entry.
	backgroundPalette = userPaletteData->GetPalette(ZAF_PM_TEXT, ZAF_PM_SELECTED | ZAF_PM_ENABLED);
	backgroundPalette.colorForeground = color;
	backgroundPalette.monoForeground = mono;
	userPaletteData->AddPalette(ZAF_PM_TEXT, ZAF_PM_SELECTED | ZAF_PM_ENABLED, backgroundPalette);

	// Return the current color.
	return (color);
}

bool ZafComboBox::SetBordered(bool )
{
	// bordered is true for this class.
	bordered = true;
	return (bordered);
}

ZafCompareFunction ZafComboBox::SetCompareFunction(ZafCompareFunction newCompareFunction)
{
	// Defer to the list.
	return (list->SetCompareFunction(newCompareFunction));
}

bool ZafComboBox::SetCopyDraggable(bool setCopyDraggable)
{
	// Defer the first part to the base class.
	if (ComboFirst())
		copyDraggable = ComboFirst()->SetCopyDraggable(setCopyDraggable);
	else
		ZafWindowObject::SetCopyDraggable(setCopyDraggable);

	// Return the current attribute.
	return (copyDraggable);
}

bool ZafComboBox::SetDestroyable(bool )
{
	// destroyable is false for this class.
	destroyable = false;
	return (destroyable);
}

bool ZafComboBox::SetDisabled(bool setDisabled)
{
	// Defer to the base class.
	return (ZafWindow::SetDisabled(setDisabled));
}

ZafLogicalFont ZafComboBox::SetFont(ZafLogicalFont font)
{
	// Make sure there is a userPalette.
	if (!userPaletteData)
		SetUserPaletteData(new ZafPaletteData());

	// Add the new entries.
	ZafPaletteStruct textPalette = userPaletteData->GetPalette(ZAF_PM_TEXT, ZAF_PM_NOT_SELECTED | ZAF_PM_ENABLED);
	textPalette.font = font;
	userPaletteData->AddPalette(ZAF_PM_TEXT, ZAF_PM_NOT_SELECTED | ZAF_PM_ENABLED, textPalette);
	textPalette = userPaletteData->GetPalette(ZAF_PM_TEXT, ZAF_PM_SELECTED | ZAF_PM_ENABLED);
	textPalette.font = font;
	userPaletteData->AddPalette(ZAF_PM_TEXT, ZAF_PM_SELECTED | ZAF_PM_ENABLED, textPalette);
	textPalette = userPaletteData->GetPalette(ZAF_PM_TEXT, ZAF_PM_ANY_STATE);
	textPalette.font = font;
	userPaletteData->AddPalette(ZAF_PM_TEXT, ZAF_PM_ANY_STATE, textPalette);

	// Return the current font.
	return (font);
}

const ZafIChar *ZafComboBox::SetHelpObjectTip(const ZafIChar *newHelpObjectTip)
{
	// Defer the first part to the base class.
	ZafWindowObject::SetHelpObjectTip(newHelpObjectTip);

	// Now mirror the combo sub-object.
	if (ComboFirst())
		ComboFirst()->SetHelpObjectTip(helpObjectTip);

	// Return the current help object tip.
	return (helpObjectTip);
}

const ZafIChar *ZafComboBox::SetHelpContext(const ZafIChar *newHelpContext)
{
	// Defer the first part to the base class.
	ZafWindowObject::SetHelpContext(newHelpContext);

	// Now mirror the combo sub-object.
	if (ComboFirst())
		ComboFirst()->SetHelpContext(helpContext);

	// Return the current help object tip.
	return (helpContext);
}

bool ZafComboBox::SetLinkDraggable(bool setLinkDraggable)
{
	// Defer the first part to the base class.
	if (ComboFirst())
		linkDraggable = ComboFirst()->SetLinkDraggable(setLinkDraggable);
	else
		ZafWindowObject::SetLinkDraggable(setLinkDraggable);

	// Return the current attribute.
	return (linkDraggable);
}

bool ZafComboBox::SetLocked(bool )
{
	// locked is false for this class.
	locked = false;
	return (locked);
}

bool ZafComboBox::SetMaximized(bool )
{
	// maximized is false for this class.
	maximized = false;
	return (maximized);
}

bool ZafComboBox::SetMinimized(bool )
{
	// minimized is false for this class.
	minimized = false;
	return (minimized);
}

bool ZafComboBox::SetMoveable(bool )
{
	// moveable is false for this class.
	moveable = false;
	return (moveable);
}

bool ZafComboBox::SetMoveDraggable(bool setMoveDraggable)
{
	// Defer the first part to the base class.
	if (ComboFirst())
		moveDraggable = ComboFirst()->SetMoveDraggable(setMoveDraggable);
	else
		ZafWindowObject::SetMoveDraggable(setMoveDraggable);

	// Return the current attribute.
	return (moveDraggable);
}

bool ZafComboBox::SetNoncurrent(bool setNoncurrent)
{
	// Defer the first part to the base class.
	ZafWindowObject::SetNoncurrent(setNoncurrent);

	// Now mirror the combo sub-object and the list.
	if (ComboFirst())
		ComboFirst()->SetNoncurrent(noncurrent);

	// Return the current attribute.
	return (noncurrent);
}

bool ZafComboBox::SetOSDraw(bool setOSDraw)
{
	// Make sure the attribute has changed.
	if (osDraw != setOSDraw && !screenID)
		osDraw = list->SetOSDraw(setOSDraw);

	// Return the current attribute.
	return (osDraw);
}

const ZafIChar *ZafComboBox::SetQuickTip(const ZafIChar *newQuickTip)
{
	// Defer the first part to the base class.
	ZafWindowObject::SetQuickTip(newQuickTip);

	// Now mirror the combo sub-object.
	if (ComboFirst())
		ComboFirst()->SetQuickTip(quickTip);

	// Return the current help object tip.
	return (quickTip);
}

ZafSelectionType ZafComboBox::SetSelectionType(ZafSelectionType )
{
	// selectionType is ZAF_SINGLE_SELECTION for this class.
	selectionType = ZAF_SINGLE_SELECTION;

	// Reflect changes to the list.
	list->SetSelectionType(ZAF_SINGLE_SELECTION);

	// Return the current attribute.
	return (selectionType);
}

bool ZafComboBox::SetSizeable(bool )
{
	// sizeable is false for this class.
	sizeable = false;
	return (sizeable);
}

bool ZafComboBox::SetTemporary(bool )
{
	// temporary is false for this class.
	temporary = false;
	return (temporary);
}

ZafLogicalColor ZafComboBox::SetTextColor(ZafLogicalColor color, ZafLogicalColor mono)
{
	// Make sure there is a userPalette.
	if (!userPaletteData)
		SetUserPaletteData(new ZafPaletteData());

	// Add the text color entry.
	ZafPaletteStruct textPalette = userPaletteData->GetPalette(ZAF_PM_TEXT, ZAF_PM_NOT_SELECTED | ZAF_PM_ENABLED);
	textPalette.colorForeground = color;
	textPalette.monoForeground = mono;
	userPaletteData->AddPalette(ZAF_PM_TEXT, ZAF_PM_NOT_SELECTED | ZAF_PM_ENABLED, textPalette);

	// Add the selected text color entry.
	textPalette = userPaletteData->GetPalette(ZAF_PM_BACKGROUND, ZAF_PM_SELECTED);
	textPalette.colorBackground = color;
	textPalette.monoBackground = mono;
	userPaletteData->AddPalette(ZAF_PM_BACKGROUND, ZAF_PM_SELECTED, textPalette);

	// Return the current color.
	return (color);
}

bool ZafComboBox::SetViewOnly(bool setViewOnly)
{
	// Check for equivalent flag.
	if (viewOnly != setViewOnly && !screenID)
	{
		// Reset the value and match the sub-object.
		ZafString *string = DynamicPtrCast(ComboFirst(), ZafString);
		if (string)
			viewOnly = string->SetViewOnly(setViewOnly);
		else
			viewOnly = setViewOnly;
	}

	// Return the current status.
	return (viewOnly);
}

void ZafComboBox::Sort(void)
{
	// Sort the list elements (low level).
	list->ZafList::Sort();

	// Sort the list elements (high level).
	OSSort();
}

// ----- Persistent functions -----------------------------------------------

ZafComboBox::ZafComboBox(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafWindow(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	list(ZAF_NULLP(ZafVtList))
{
	// Read the data.
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafComboBox::ReadData(persist);

	// Read the list data.
	for (ZafElement *element = first; !list && element; element= element->Next())
		list = DynamicPtrCast(element, ZafVtList);
	if (list)
		ZafList::Subtract(list);
	else
		list = new ZafVtList(0, 0, zafRegion.right - zafRegion.left, zafRegion.bottom - zafRegion.top);
	list->parent = this;
	list->SetDestroyable(false);
	list->SetTemporary(true);
	list->SetParentPalette(true);
	SetSelectionType(ZAF_SINGLE_SELECTION);	// Also affects the list.

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());

	// Initialize the combo-box information.
#if defined(ZAF_MOTIF)
	comboItem = ZAF_NULLP(ZafWindowObject);
#elif defined(ZAF_MSDOS) || defined(ZAF_CURSES)
	comboString = comboButton = ZAF_NULLP(ZafWindowObject);
#elif defined(ZAF_MACINTOSH)
	comboString = ZAF_NULLP(ZafWindowObject);
#endif
}

ZafElement *ZafComboBox::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafComboBox(name, persist));
}

void ZafComboBox::ReadData(ZafObjectPersistence &persist)
{
	// Read the data.
	ZafUInt16 flag1;
	ZafFile *file = persist.CurrentFile();
	*file >> flag1;
	viewOnly = (flag1 & 0x0001) ? true : false;
}

void ZafComboBox::Write(ZafObjectPersistence &persist)
{
	// Bypass ZafWindow calls to keep the children from being saved.
	ZafWindowObject::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafWindow::WriteData(persist);
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafComboBox::WriteData(persist);

	// Write the list data.
	list->Write(persist.PushLevel(list->ClassName(), list->ClassID(), ZAF_PERSIST_DIRECTORY));
	persist.PopLevel();

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafComboBox::WriteData(ZafObjectPersistence &persist)
{
	// Write the data.
	ZafUInt16 flag1 = 0;
	flag1 |= viewOnly ? 0x0001 : 0;
	ZafFile *file = persist.CurrentFile();
	*file << flag1;
}

