//	Zinc Application Framework - Z_BNUM2.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_stdio.hpp>
#include <z_bnum1.hpp>
#include <z_string.hpp>
#include <z_utils.hpp>
#include <z_error.hpp>
#include <z_app.hpp>
#define ZAF_BIGNUM_INFO
#include "lang_def.cpp"
#include "gbl_def.cpp"

static ZafIChar numberSeparator[3] = { '.','.',0 };
static ZafIChar rangeSeparator[3] = { ',',' ',0 };

// ----- ZafBignum ----------------------------------------------------------

ZafBignum::ZafBignum(int left, int top, int width, ZafBignumData *zBignumData) :
	ZafString(left, top, width, new ZafStringData),
	bignumData(ZAF_NULLP(ZafBignumData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Initialize the bignum data.
	SetBignumData(zBignumData ? zBignumData : new ZafBignumData);
}

ZafBignum::ZafBignum(int left, int top, int width, long value) :
	ZafString(left, top, width, new ZafStringData),
	bignumData(ZAF_NULLP(ZafBignumData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Initialize the bignum data.
	SetBignumData(new ZafBignumData(value));
}

ZafBignum::ZafBignum(int left, int top, int width, double value) :
	ZafString(left, top, width, new ZafStringData),
	bignumData(ZAF_NULLP(ZafBignumData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Initialize the bignum data.
	SetBignumData(new ZafBignumData(value));
}

ZafBignum::ZafBignum(const ZafBignum &copy) :
	ZafString(copy), bignumData(ZAF_NULLP(ZafBignumData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Copy the bignum data.
	if (copy.BignumData()->Destroyable())
		SetBignumData(new ZafBignumData(*copy.BignumData()));
	else
		SetBignumData(copy.BignumData());
}

ZafBignum::~ZafBignum(void)
{
	// Remove the data notification.
	bignumData->SubtractNotification(this, (ZafUpdateFunction)Update);

	// Check the language and local information.
	LanguageFree();

	// Restore the bignum information.
	if (bignumData->Destroyable() && bignumData->NotifyCount() == 0)
		delete bignumData;
}

ZafError ZafBignum::Decrement(ZafData *data)
{
	// Check for a valid object.
	ZafBignumData *bignum = DynamicPtrCast(data, ZafBignumData);
	if (!bignum)
		return (ZAF_ERROR_INVALID_CLASS);

	// Decrement the value.
	ZafBignumData _bignum(*bignumData);
	OSGetBignum();
	*bignumData -= *bignum; // update is automatic.

	// Return success.
	return (ZAF_ERROR_NONE);
}

ZafWindowObject *ZafBignum::Duplicate(void)
{
	return (new ZafBignum(*this));
}

ZafEventType ZafBignum::Event(const ZafEventStruct &event)
{
	// Process OS specific messages.
	ZafEventType ccode = event.type;
	if (event.InputType() == E_KEY)
		ccode = LogicalEvent(event);

	// Check for zinc events.
	switch (ccode)
	{
	// ----- Create messages -----
	case S_INITIALIZE:
		// Check for an unanswered field.
		if (!Unanswered())
			OSSetBignum();
		ccode = ZafString::Event(event);
		break;

	// ----- Action messages -----
	case L_SELECT:
	case N_NON_CURRENT:
		{
		// Keep a temporary number in case of error.
		ZafBignumData oldValue(*bignumData);
		OSGetBignum();

		// Call the associated user function.
		ccode = ZafString::Event(event);

		// Check for errors.
		if (ccode != 0 && ccode != ZAF_ERROR_LEAVE_INVALID)
			bignumData->SetBignum(oldValue);

		// Set the formatted bignum text into the field.
		OSSetBignum();
		}
		break;

	case S_COPY_DATA:
		{
		ZafWindowObject *object = event.windowObject;
		ZafBignum *bignum = DynamicPtrCast(object, ZafBignum);
		if (bignum)
			bignumData->SetBignum(*bignum->BignumData());
		}
		break;

	case S_SET_DATA:
		if (event.windowObject)
		{
			ZafWindowObject *object = event.windowObject;
			ZafBignum *bignum = DynamicPtrCast(object, ZafBignum);
			if (bignum)
				SetBignumData(bignum->BignumData());
		}
		else
			SetBignumData(new ZafBignumData(*bignumData));
		break;

	// ----- Default or unknown messages -----
	default:
		ccode = ZafString::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

ZafError ZafBignum::Increment(ZafData *data)
{
	// Check for a valid object.
	ZafBignumData *bignum = DynamicPtrCast(data, ZafBignumData);
	if (!bignum)
		return (ZAF_ERROR_INVALID_CLASS);

	// Increment the value.
	ZafBignumData _bignum(*bignumData);
	OSGetBignum();
	*bignumData += *bignum; // update is automatic.

	// Return success.
	return (ZAF_ERROR_NONE);
}

void ZafBignum::LanguageAllocate(void)
{
	// Check for initialization.
	if (errorStrings)
		return;
	ZafApplication::AddStaticModule(LanguageFree);

	// Try the zafDataManager.
	if (!errorStrings && zafDataManager)
		errorStrings = DynamicPtrCast(zafDataManager->AllocateData(className, ZafLanguageData::className, ZafLanguageData::classID), ZafLanguageData);

	// Default to code initialization.
	if (!errorStrings)
		errorStrings = new ZafLanguageData(defaultErrorStrings);
}

void ZafBignum::LanguageFree(bool globalRequest)
{
	// Destroy the static language.
	if (globalRequest && errorStrings && errorStrings->Destroyable() &&
		errorStrings->NotifyCount() == 0)
	{
		delete errorStrings;
		errorStrings = ZAF_NULLP(ZafLanguageData);
	}
}

ZafError ZafBignum::OSGetBignum(void)
{
	// Disable notification.
	ZafUpdateType update = bignumData->Update(this);
	bignumData->SetUpdate(this, ZAF_UPDATE_NONE);

	// Set the bignum based on the string data.
	OSGetText();
	error = bignumData->SetBignum(stringData->Text(), InputFormatText());

	// Restore notification.
	bignumData->SetUpdate(this, update);
	return (error);
}

ZafError ZafBignum::OSSetBignum(void)
{
	// Set the string based on the bignum data.
	ZafIChar text[ZAF_STRING_DATA_LENGTH];
	bignumData->FormattedText(text, ZAF_STRING_DATA_LENGTH, OutputFormatText());
	stringData->SetText(text);

	// Return the current status.
	return (ZAF_ERROR_NONE);
}

ZafError ZafBignum::SetBignumData(ZafBignumData *tBignumData)
{
	// Remove the data notification.
	if (bignumData)
		bignumData->SubtractNotification(this, (ZafUpdateFunction)Update);

	// Reset the bignum data.
	if (bignumData && bignumData != tBignumData &&
		bignumData->Destroyable() && bignumData->NotifyCount() == 0)
		delete bignumData;
	bignumData = tBignumData ? tBignumData : new ZafBignumData;

	// Add the data notification.
	bignumData->AddNotification(this, (ZafUpdateFunction)Update);
	return (OSSetBignum());
}

bool ZafBignum::SetLowerCase(bool )
{
	// lowerCase is set to false by this class.
	lowerCase = false;
	return (lowerCase);
}

bool ZafBignum::SetPassword(bool )
{
	// password is set to false by this class.
	password = false;
	return (password);
}

ZafError ZafBignum::SetStringData(ZafStringData *)
{
	// String data cannot be set for this class.
	return (ZAF_ERROR_INVALID);
}

bool ZafBignum::SetUpperCase(bool )
{
	// upperCase is set to false by this class.
	upperCase = false;
	return (upperCase);
}

ZafError ZafBignum::Validate(bool processError)
{
	// Check for an absolute number error. Don't set the error member to ZAF_ERROR_NONE.
	ZafIChar *stringNumber = stringData->DynamicText();
	ZafBignumData currentNumber;
	ZafError tError = currentNumber.SetBignum(stringNumber, InputFormatText());

	// Check for a range error.
	if (RangeText() && tError == ZAF_ERROR_NONE)
		tError = ZAF_ERROR_OUT_OF_RANGE;
	int numRanges = 0;
	ZafBignumData low, high;
	ZafIChar minNumber[ZAF_STRING_DATA_LENGTH];
	ZafIChar maxNumber[ZAF_STRING_DATA_LENGTH];
	ZafIChar rBuffer[1024];				// Localized range string for error messages.
	rBuffer[0] = '\0';
	for (const ZafIChar *tRange = RangeText(); tRange && tError == ZAF_ERROR_OUT_OF_RANGE; numRanges++)
	{
		tRange = ParseRange(tRange, minNumber, maxNumber);
		low.SetBignum(minNumber, InputFormatText());
		high.SetBignum(maxNumber, InputFormatText());
		if ((!minNumber[0] || currentNumber >= low) && (!maxNumber[0] || currentNumber <= high))
			tError = ZAF_ERROR_NONE;
		else if (processError)
		{
			if (rBuffer[0])
				strcat(rBuffer, rangeSeparator);
			if (minNumber[0])
				low.FormattedText(&rBuffer[strlen(rBuffer)], ZAF_STRING_DATA_LENGTH, OutputFormatText());
			strcat(rBuffer, numberSeparator);
			if (maxNumber[0])
				high.FormattedText(&rBuffer[strlen(rBuffer)], ZAF_STRING_DATA_LENGTH, OutputFormatText());
		}
	}

	// Process the error code.
	SetInvalid(false);
	if (tError == ZAF_ERROR_NONE)
	{
		// Set up the new number.
		currentNumber.FormattedText(stringNumber, ZAF_STRING_DATA_LENGTH, OutputFormatText());
		SetText(stringNumber);
		delete []stringNumber;
		return (ZAF_ERROR_NONE);
	}
	else
	{
		// Keep the error code.
		error = tError;
		if (!zafErrorSystem)
		{
			// Restore the original number.
			currentNumber.FormattedText(stringNumber, ZAF_STRING_DATA_LENGTH, OutputFormatText());
			SetText(stringNumber);
			delete []stringNumber;
			return (error);
		}
		else if (!processError)
		{
			delete []stringNumber;
			return (error);
		}
	}

	// Check for open-ended range errors.
	if (error == ZAF_ERROR_OUT_OF_RANGE && numRanges == 1)
	{
		if (minNumber[0] && !maxNumber[0])
		{
			error = ZAF_ERROR_LESS_THAN_RANGE;
			low.FormattedText(rBuffer, ZAF_STRING_DATA_LENGTH, OutputFormatText());
		}
		else if (!minNumber[0] && maxNumber[0])
		{
			error = ZAF_ERROR_GREATER_THAN_RANGE;
			high.FormattedText(rBuffer, ZAF_STRING_DATA_LENGTH, OutputFormatText());
		}
	}

	// Generate the error message and wait for a response.
	ZafIChar *title = ZafLanguageData::blankString;
	ZafIChar *errorString = errorStrings->GetMessage((ZafNumberID)error, true);
	if (errorString &&
		zafErrorSystem->ReportError(windowManager, title, ZAF_DIALOG_OK | ZAF_DIALOG_CANCEL, errorString, stringNumber, rBuffer) == S_DLG_OK)
		error = ZAF_ERROR_LEAVE_INVALID; // Keep the new value.
	delete []stringNumber;
	return (error);
}

bool ZafBignum::SetVariableName(bool )
{
	// variableName is false for this class.
	variableName = false;
	return (variableName);
}

// ----- Persistent functions -----------------------------------------------

ZafBignum::ZafBignum(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafString(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	bignumData(ZAF_NULLP(ZafBignumData))
{
	// Check the language and local information.
	LanguageAllocate();

	if (persist.Error() == ZAF_ERROR_NONE)
		ZafBignum::ReadData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

ZafElement *ZafBignum::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafBignum(name, persist));
}

void ZafBignum::ReadData(ZafObjectPersistence &persist)
{
	ZafFile *file = persist.CurrentFile();

	// Read the bignum data.
	ZafIChar bignumName[ZAF_MAXNAMELEN];
	*file >> bignumName;
	SetBignumData(bignumName[0] ? new ZafBignumData(bignumName, persist) : new ZafBignumData);
}

void ZafBignum::Write(ZafObjectPersistence &persist)
{
	// Write the object.
	ZafString::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafBignum::WriteData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafBignum::WriteData(ZafObjectPersistence &persist)
{
	// Write the data.
	const ZafIChar *bignumName = bignumData ? bignumData->StringID() : ZAF_NULLP(ZafIChar);
	ZafFile *file = persist.CurrentFile();
	*file << bignumName;
	if (bignumData)
		bignumData->Write(persist);
}

