//	Zinc Application Framework - Z_BITMAP.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_bitmap.hpp>
#include <z_lang.hpp>
#define ZAF_BITMAP_DATA_INFO
#include "gbl_def.cpp"

// ----- ZafBitmapStruct ----------------------------------------------------

ZafBitmapStruct::ZafBitmapStruct(void)
{
	width = height = 0, array = ZAF_NULLP(ZafLogicalColor), staticArray = staticHandle = false;
#	if defined(ZAF_MSWINDOWS)
	handle = mask = 0;
#	elif defined(ZAF_MOTIF)
	handle = mask = 0;
#	elif defined(ZAF_MACINTOSH)
	colorBitmap = ZAF_NULLH(PixMapHandle);
	monoBitmap = mask = ZAF_NULLH(BitMapHandle);
#	elif defined(ZAF_MSDOS)
	handle = mask = 0;
#	endif
}

ZafBitmapStruct::ZafBitmapStruct(const ZafImageStruct &data)
{
	width = data.width, height = data.height, array = data.array, staticHandle = false, staticArray = data.staticArray;
#	if defined(ZAF_MSWINDOWS)
	handle = mask = 0;
#	elif defined(ZAF_MOTIF)
	handle = mask = 0;
#	elif defined(ZAF_MACINTOSH)
	colorBitmap = ZAF_NULLH(PixMapHandle);
	monoBitmap = mask = ZAF_NULLH(BitMapHandle);
#	elif defined(ZAF_MSDOS)
	handle = mask = 0;
#	endif
}

ZafBitmapStruct::ZafBitmapStruct(int tWidth, int tHeight, ZafLogicalColor *tArray, bool tStaticArray)
{
	width = tWidth, height = tHeight, array = tArray, staticHandle = false, staticArray = tStaticArray;
#	if defined(ZAF_MSWINDOWS)
	handle = mask = 0;
#	elif defined(ZAF_MOTIF)
	handle = mask = 0;
#	elif defined(ZAF_MACINTOSH)
	colorBitmap = ZAF_NULLH(PixMapHandle);
	monoBitmap = mask = ZAF_NULLH(BitMapHandle);
#	elif defined(ZAF_MSDOS)
	handle = mask = 0;
#	endif
}

// ----- ZafBitmapData ------------------------------------------------------

ZafBitmapData::ZafBitmapData(const ZafImageStruct &data) :
	ZafImageData()
{
	// Initialize the bitmap information.
	array = ZAF_NULLP(ZafLogicalColor);
	SetImage(*this, data);

	staticHandle = false;
	ZafDisplay::InitializeOSBitmap(*this);
}

ZafBitmapData::ZafBitmapData(const ZafBitmapStruct &copy) :
	ZafImageData()
{
	// Initialize the information.
	array = ZAF_NULLP(ZafLogicalColor);
	SetImage(*this, copy);

	staticHandle = copy.staticHandle;
	ZafDisplay::InitializeOSBitmap(*this);
}

ZafBitmapData::ZafBitmapData(const ZafBitmapData &copy) :
	ZafImageData(copy)
{
	// Initialize the bitmap information.
	array = ZAF_NULLP(ZafLogicalColor);
	SetBitmap(copy);
}

ZafBitmapData::~ZafBitmapData(void)
{
	// Destroy the bitmap information.
	if (zafDisplay)
	{
		if (!StaticHandle())
			zafDisplay->DestroyOSBitmap(*this);
		if (!StaticArray())
			zafDisplay->DestroyZafBitmap(*this);
	}
	else if (array && !StaticArray())
		delete []array;
}

void ZafBitmapData::Clear(void)
{
	// Destroy the color handle and mask.
	PushLevel();
	if (zafDisplay)
	{
		zafDisplay->DestroyOSBitmap(*this);
		zafDisplay->DestroyZafBitmap(*this);
	}
	PopLevel();
}

ZafData *ZafBitmapData::Duplicate(void)
{
	return (new ZafBitmapData(*this));
}

ZafError ZafBitmapData::SetBitmap(int tWidth, int tHeight,
	ZafLogicalColor *tArray)
{
	// Clear the current os handles.
	if (zafDisplay)
		zafDisplay->DestroyOSBitmap(*this);

	// Reset the icon information.
	PushLevel();
	SetImage(*this, tWidth, tHeight, tArray);
	PopLevel();

	// Return success.
	return (ZAF_ERROR_NONE);
}

ZafError ZafBitmapData::SetBitmap(const ZafBitmapStruct &copy)
{
	// Clear the current os handles.
	if (zafDisplay)
		zafDisplay->DestroyOSBitmap(*this);

	// Reset the bitmap information.
	PushLevel();
	SetImage(*this, copy);

	// Update the os handles.
	staticHandle = copy.staticHandle;
	if (staticHandle)
		ZafDisplay::ResetOSBitmap(*this, copy);
	PopLevel();

	// Return success.
	return (ZAF_ERROR_NONE);
}

// ----- Persistent functions -----------------------------------------------

ZafBitmapData::ZafBitmapData(const ZafIChar *name, ZafDataPersistence &persist) :
	ZafImageData(name, persist.PushLevel(className, classID, ZAF_PERSIST_ROOT_DIRECTORY))
{
	// Set the default bitmap information.
	InitializeImage(*this);

	staticHandle = false;
	ZafDisplay::InitializeOSBitmap(*this);

	// Read the data.
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafBitmapData::ReadData(persist);

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

ZafElement *ZafBitmapData::Read(const ZafIChar *name, ZafDataPersistence &persist)
{
	return (new ZafBitmapData(name, persist));
}

void ZafBitmapData::ReadData(ZafDataPersistence &persist)
{
	// Read the data.
	ZafFile *file = persist.CurrentFile();
	*file >> width >> height;
	array = new ZafLogicalColor[width * height];
	file->Read(array, sizeof(ZafLogicalColor), width * height);
}

void ZafBitmapData::Write(ZafDataPersistence &persist)
{
	// Write the object.
	ZafImageData::Write(persist.PushLevel(className, classID, ZAF_PERSIST_ROOT_DIRECTORY));
	if (persist.Error() == ZAF_ERROR_NONE)
		ZafBitmapData::WriteData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafBitmapData::WriteData(ZafDataPersistence &persist)
{
	// Write the image information.
	ZafFile *file = persist.CurrentFile();
	*file << width << height;
	if (array)
		file->Write(array, sizeof(ZafLogicalColor), width * height);
}

