//	Zinc Application Framework - C_STORER.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <errno.h>
#include <fcntl.h>
#if !defined(macintosh)
extern "C"
{
#	include <sys/types.h>
#	include <sys/stat.h>
}
#endif
#if defined(__BCPLUSPLUS__) || defined(__TCPLUSPLUS__)
#	include <mem.h>
#	include <dir.h>
#	include <direct.h>
#elif (defined(__ZTC__) && !defined(macintosh)) || defined(_MSC_VER) || (defined(__WATCOMC__) && !defined(__QNX__)) || defined(__HIGHC__) || defined(__IBMCPP__) || defined(__GLOCKENSPIEL__)
#	include <direct.h>
#endif
#if defined(__GNUC__)
#	define SEEK_SET	0
#	define SEEK_CUR	1
#	define SEEK_END	2
#endif
#if defined(__ZTC__) && !defined(macintosh)
#	define _dos_getdrive dos_getdrive
#	define _dos_setdrive dos_setdrive
#endif
#if defined(_MSC_VER) && defined(DOS386)
#	include <pldos32.h>
#endif
#include <z_env.hpp>
#if defined(ZAF_MSDOS) || defined(ZAF_MSWINDOWS) || defined(ZAF_OS2) || defined(__DVX__)
#	if !defined(__IBMCPP__) && !defined(__GLOCKENSPIEL__)
#		include <dos.h>
#	endif
#endif
#if defined(_SUNOS4) || defined(SCO_UNIX)
#       include <z_stdlib.hpp>
#endif
#include <z_stdio.hpp>
#if (!defined(sun) && !defined(__sgi)) || defined(_SUNOS4) || defined(ZAF_UNICODE)
#	include <z_unistd.hpp>
#endif
#if !defined(_SUNOS4) && !defined(SCO_UNIX)
#       include <z_stdlib.hpp>
#endif
#include <z_string.hpp>
#include <z_ctype.hpp>
#include "c_storer.hpp"
#if defined(ZAF_POSIX)
extern "C"
{
#	if defined(ZAF_NEXTSTEP)
#		include <libc.h>
#	endif
#	if !defined(__QNX__)
#		include <sys/param.h>
#		define O_BINARY	0
#	endif
}
#elif defined(ZAF_MACINTOSH)
#	include "a_utils.hpp"

	OSType OldStorageReadOnly::fileCreator = 'ZDes';
	OSType OldStorageReadOnly::fileType = 'POST';
#endif
#include <z_utils.hpp>
#include "c_path.hpp"

struct ZafSignature zafSignature =
{
#	if defined(ZAF_UNICODE)
//	"Zinc Data File Version 4.5\032",
	"Zinc Data File Version 5.5\032",
#	else
//	"Zinc Data File Version 4.0\032",
	"Zinc Data File Version 5.0\032",
#	endif
	ZAF_MAJOR_FILE_VERSION, ZAF_MINOR_FILE_VERSION,
	ZAF_MAGIC_NUMBER
};

// --- support ----------------------------------------------------------------

#if defined(ZAF_BIGENDIAN)
// Data is always stored as if on a little endian machine (ala Intel x86
// family).  This was done since the slowest machines we support are based
// on the 4.77 mhz 8088.  Big endian machines swap all their data around.
// These routines are macroed out for little endian machines.

// Prototypes
void SwapDiskAddr(ZafDiskAddress *datum)
{
	union
	{
		ZafDiskAddress a;
		ZafUInt8 b[sizeof(ZafDiskAddress)];
	} tmp;
	ZafUInt8 tmpb;

	tmp.a = *datum;
	for (int j=0; j < sizeof(*datum) / 2; j++)
	{
		tmpb = tmp.b[j];
		tmp.b[j] = tmp.b[sizeof(*datum)-1-j];
		tmp.b[sizeof(*datum)-1-j] = tmpb;
	}
	*datum = tmp.a;
}

void SwapInumt(ZafINodeNumber *datum)
{
	union
	{
		ZafINodeNumber a;
		ZafUInt8 b[sizeof(ZafINodeNumber)];
	} tmp;
	ZafUInt8 tmpb;

	tmp.a = *datum;
	for (int j=0; j < sizeof(*datum) / 2; j++)
	{
		tmpb = tmp.b[j];
		tmp.b[j] = tmp.b[sizeof(*datum)-1-j];
		tmp.b[sizeof(*datum)-1-j] = tmpb;
	}
	*datum = tmp.a;
}

void SwapInode(ZafInode *di)
{
	SwapInt32(&di->size);
	SwapUInt32(&di->createTime);
	SwapUInt32(&di->modifyTime);
	for (int i=0; i < LENGTHOF(di->direct); i++)
		SwapDiskAddr(&di->direct[i]);
	SwapDiskAddr(&di->sIndirect);
	SwapDiskAddr(&di->dIndirect);
	SwapUInt16(&di->useCount);
	SwapDiskAddr(&di->fragmentBlock);
	SwapUInt16(&di->fragmentIndex);
}

void SwapSuperBlock(ZafSuperBlock *ds)
{
	SwapDiskAddr(&ds->signature.magicNumber);
	SwapUInt32(&ds->createTime);
	SwapUInt32(&ds->modifyTime);
	SwapUInt16(&ds->revision);
	SwapUInt16(&ds->blockSize);
	SwapUInt16(&ds->indirectBlockSize);
	SwapUInt16(&ds->freeInodeListHi);
	SwapUInt16(&ds->freeInodeListLo);
	SwapDiskAddr(&ds->freeDataList);
	for (int i=0; i < LENGTHOF(ds->inodeDirect); i++)
		SwapDiskAddr(&ds->inodeDirect[i]);
	SwapDiskAddr(&ds->inodeSIndirect);
	SwapDiskAddr(&ds->inodeDIndirect);
}
#endif

// Takes a possible partial path name in tmppath and converts it to a
// fully qualified path name starting from root and including a drive.
// It doesn't strip out redundant ".."s or "."s.
void OldStorageReadOnly::MakeFullPath(ZafIChar *tmppath)
{
	// Check for a null string;
	if (!tmppath)
		return;

#if defined(ZAF_POSIX)
	if (tmppath[0] != zafCodeSet->dirSepStr[0])
	{
		ZafIChar save[ZAF_MAXPATHLEN];
		strcpy(save, tmppath);
		getcwd(tmppath, ZAF_MAXPATHLEN);
		AppendFullPath(tmppath, tmppath, save);
	}
#elif defined(ZAF_MACINTOSH)
	// A full path may be a qualified partial path, still usable by the Toolbox.
	if (!strchr(tmppath, zafCodeSet->dirSepStr[0]))
		strcat(tmppath, zafCodeSet->dirSepStr);
#else
	if (tmppath[0] == zafCodeSet->dirSepStr[0])
	{
		memmove((char *)&tmppath[2], (const char *)&tmppath[0], (strlen(tmppath)+1) * sizeof(tmppath[0]));
		tmppath[1] = ':';
#if defined(ZAF_WIN32)
		unsigned drive;
		char currentDrive[MAX_PATH];
		GetCurrentDirectoryA(MAX_PATH, currentDrive);
		drive = ToUpper(currentDrive[0]) - 'A' + 1;
		tmppath[0] = drive - 1 + 'A';
#elif defined(ZAF_OS2)
		ULONG pDrive, lDrive;
		DosQueryCurrentDisk(&pDrive, &lDrive);
		tmppath[0] = pDrive - 1 + 'A';
#elif defined(ZAF_MSDOS) || defined(ZAF_MSWINDOWS) || defined(__DVX__)
		unsigned drive;
		_dos_getdrive(&drive);
		tmppath[0] = drive - 1 + 'A';
#else
		????;   This is an error;
#endif
	}
	else if (tmppath[0] && tmppath[1] == ':' && tmppath[2] != zafCodeSet->dirSepStr[0])
	{
		ZafIChar save[ZAF_MAXPATHLEN];
		strcpy(save, &tmppath[2]);
#if defined(ZAF_WIN32)
#	if defined(ZAF_UNICODE)
		char currentDrive[MAX_PATH], newDrive[MAX_PATH];
		GetCurrentDirectoryA(MAX_PATH, currentDrive);
		newDrive[0] = tmppath[0];
		newDrive[1] = ':';
		newDrive[2] = 0;
		SetCurrentDirectoryA(newDrive);
		GetCurrentDirectoryA(ZAF_MAXPATHLEN, newDrive);
		MultiByteToWideChar(CP_OEMCP, MB_PRECOMPOSED, newDrive, -1, &tmppath[2], ZAF_MAXPATHLEN-2);
		SetCurrentDirectoryA(currentDrive);
#	else
		char currentDrive[MAX_PATH];
		GetCurrentDirectory(MAX_PATH, currentDrive);
		SetCurrentDirectory(tmppath);
		GetCurrentDirectory(ZAF_MAXPATHLEN, tmppath);
		SetCurrentDirectory(currentDrive);
#	endif
#elif defined(ZAF_OS2)
		ULONG drive, xxx;
		DosQueryCurrentDisk(&drive, &xxx);
		DosSetDefaultDisk(tmppath[0] - 'a' + 1);
		getcwd(&tmppath[2], CCHMAXPATH);
		DosSetDefaultDisk(drive);
#elif defined(ZAF_MSDOS) || defined(ZAF_MSWINDOWS) || defined(__DVX__)
		unsigned drive, xxx;
		_dos_getdrive(&drive);
		_dos_setdrive(tmppath[0] - 'A' + 1, &xxx);
		getcwd(tmppath, ZAF_MAXPATHLEN);
		_dos_setdrive(drive, &xxx);
#else
		????; This is an error;
#endif
		AppendFullPath(tmppath, tmppath, save);
	}
	else if (!(tmppath[0] && tmppath[1] == ':' && tmppath[2] == zafCodeSet->dirSepStr[0]))
	{
		ZafIChar save[ZAF_MAXPATHLEN];
		strcpy(save, tmppath);
		getcwd(tmppath, ZAF_MAXPATHLEN);
		AppendFullPath(tmppath, tmppath, save);
	}
#endif
}

// --- OldStorageReadOnly statics -------------------------------------------

ZafIChar *OldStorageReadOnly::bakName = ZAF_ITEXT(".bk?");

#if defined(ZAF_DOSEXTENDED)
int OldStorageReadOnly::cacheSize = 64;
#else
int OldStorageReadOnly::cacheSize = 32;
#endif
ZafPath *OldStorageReadOnly::searchPath = ZAF_NULLP(ZafPath);

// Build a pathname from parts (hiding operating system specifics).
void OldStorageReadOnly::AppendFullPath(ZafIChar *fullPath,
	const ZafIChar *pathName, const ZafIChar *fileName,
	const ZafIChar *extension)
{
	if (pathName != fullPath)
		strcpy(fullPath, pathName);
	int i = strlen(fullPath);
#if defined(ZAF_MSDOS) || defined(ZAF_MSWINDOWS) || defined(ZAF_OS2) || defined(__DVX__)
	if (i && fullPath[i-1] != zafCodeSet->dirSepStr[0] && fullPath[i-1] != ':')
#else
	if (i && fullPath[i-1] != zafCodeSet->dirSepStr[0])
#endif
		strcat(fullPath, zafCodeSet->dirSepStr);
	if (fileName)
		strcat(fullPath, fileName);
	if (extension)
		ChangeExtension(fullPath, extension);
#if defined(ZAF_MSDOS) || defined(ZAF_MSWINDOWS)
	strupr(fullPath);
#endif
}

ZafUInt16 OldStorageReadOnly::BlockSize()
{
	ZafUInt16 returnValue;
	if (version < 500)
		returnValue = BYTES_PER_DATA_BLOCK_4X;
	else
		returnValue = sb->blockSize;

	return returnValue;
}

// Replace an extension.  Assumes the extension is separated from the rest
// of the filename with a single period.
void OldStorageReadOnly::ChangeExtension(ZafIChar *pathName,
	const ZafIChar *newExtension)
{
	ZafIChar *oldExtension = strrchr(pathName, '.');
	if (oldExtension)
		*oldExtension = '\0';
	if (newExtension)
		strcat(pathName, newExtension);
}

long OldStorageReadOnly::GetBlockPos(ZafDiskAddress block)
{
	long returnValue;

	if (version < 500)
	{
		// This is the 4.x formula.
		returnValue = block*(long)BYTES_PER_BLOCK_4X;
	}
	else
	{
		returnValue = ((long)block - 1) * sb->blockSize + (long)sizeof(ZafSuperBlock);
	}

	return returnValue;
}

ZafUInt16 OldStorageReadOnly::IndirectBlockSize()
{
	ZafUInt16 returnValue;

	if (version < 500)
		returnValue = INODE_INDIRECT_SIZE_4X;
	else
		returnValue = sb->indirectBlockSize;

	return returnValue;
}

// Implements the following truth table:
// createStorage =			true	false
// invalid path				false	false
// valid path/no file		true	false
// valid path/file RW		true	true
// valid path/file RO		false	true
// valid path/protected		false	false
// without destroying the file possibly referenced by name.
bool OldStorageReadOnly::ValidName(const ZafIChar *name, bool createStorage)
{
#if defined(ZAF_MACINTOSH)
	// Try to create the file.
	Str255 fName;
//#	if defined(ZAF_UNICODE) || defined(ZAF_ISO8859_1)
//	MapText(name, (char *)fName, false);
//#	else
	strcpy((char *)fName, name);
//#	endif
	c2pstr((char *)fName);
	errno = ::Create(fName, 0, fileCreator, fileType);
	if (errno == noErr)
	{
		FSDelete(fName, 0);
		return (createStorage);		// valid path/no file
	}
	else if (errno == dupFNErr)
	{
		// The file exists, so try for RW access.
		HParamBlockRec paramBlock;
		paramBlock.fileParam.ioNamePtr = fName;
		paramBlock.fileParam.ioVRefNum = 0;
		paramBlock.ioParam.ioPermssn = fsRdWrPerm;
		paramBlock.ioParam.ioMisc = nil;
		paramBlock.fileParam.ioDirID = 0;
		errno = PBHOpen(&paramBlock, false);
		if (errno == noErr)
		{
			FSClose(paramBlock.ioParam.ioRefNum);
			return (true);			// valid path/file RW
		}

		// Denied RW access, so try for RO access.
		paramBlock.ioParam.ioPermssn = fsRdPerm;
		errno = PBHOpen(&paramBlock, false);
		if (errno != noErr)
			return (false);			// valid path/file protected
		FSClose(paramBlock.ioParam.ioRefNum);
		return (!createStorage);	// valid path/file RO
	}
	else
		return (false);				// invalid path
#else
	FILE *fd = fopen(name, ZAF_ITEXT("r+b"));
	if (fd)
	{
		fclose(fd);
		return (true);
	}
	fd = fopen(name, ZAF_ITEXT("rb"));
	if (fd)
	{
		fclose(fd);
		return (!createStorage);
	}
	if (!createStorage)
		return (false);
	fd = fopen(name, ZAF_ITEXT("w+b"));
	if (!fd)
		return (errno == EEXIST);
	fclose(fd);
	remove(name);
	return (true);
#endif
}

// Split a pathname into parts (hiding operating specific parts).
void OldStorageReadOnly::StripFullPath(const ZafIChar *fullPath,
	ZafIChar *pathName, ZafIChar *fileName, ZafIChar *objectName,
	ZafIChar *objectPathName)
{
	// Determine the path/file split area.
	const ZafIChar *endPath = strrchr(fullPath, zafCodeSet->dirSepStr[0]);
#if defined(ZAF_MSDOS) || defined(ZAF_MSWINDOWS) || defined(ZAF_OS2)
	if (!endPath)
		endPath = strrchr(fullPath, ':');
#endif
	if (!endPath)
		endPath = fullPath;
#if defined(ZAF_MSDOS) || defined(ZAF_MSWINDOWS) || defined(ZAF_OS2)
	if (*endPath == ':')
		endPath++;
	if (*endPath == zafCodeSet->dirSepStr[0] && endPath-fullPath == 2 && endPath[-1] == ':')
		endPath++;
#endif
	// Determine the file/object split area.
	const ZafIChar *endFile = strchr(endPath, ZAF_DIRECTORY_SEPARATOR);
	if (!endFile)
		endFile = &endPath[strlen(endPath)];

	// find the end of the object path name
	const ZafIChar *endOPath = strrchr(endFile, ZAF_DIRECTORY_SEPARATOR);
	if (!endOPath)
		endOPath = &endFile[strlen(endFile)];

	// Construct the path name.
	if (pathName)
	{
		int i = (int)(endPath - fullPath);
		strncpy(pathName, fullPath, i);
		pathName[i] = '\0';
#if defined(ZAF_MSDOS) || defined(ZAF_MSWINDOWS)
		strupr(pathName);
#endif
	}
	if (*endPath == zafCodeSet->dirSepStr[0])
		endPath++;

	// Construct the file name.
	if (fileName)
	{
		int i = (int)(endFile - endPath);
		strncpy(fileName, endPath, i);
		fileName[i] = '\0';
#if defined(ZAF_MSDOS) || defined(ZAF_MSWINDOWS)
		strupr(fileName);
#endif
	}

	// Construct the object path name.
	if (objectPathName)
	{
		int i = (int)(endOPath - endFile);
		strncpy(objectPathName, endFile, i);
		objectPathName[i] = '\0';
	}
	if (*endOPath == ZAF_DIRECTORY_SEPARATOR)
		endOPath++;

	// Construct the object name.
	if (objectName)
		strcpy(objectName, endOPath);
}

// Create a name for a temporary file that is readable, writable, and doesn't
// destroy any other file.
void OldStorageReadOnly::TempName(ZafIChar *tempname)
{
#if defined(ZAF_POSIX) || (defined(ZAF_OS2) && defined(__IBMCPP__))
	tmpnam(tempname);
#else
	ZafIChar tmppath[ZAF_MAXPATHLEN];
	tmpnam(tmppath);
	MakeFullPath(tempname);
	AppendFullPath(tempname, tempname, tmppath);
#endif
}

// --- FindFirst/FindNext ---------------------------------------------------
// This allows only one Findfirst per file.  Use OpenDir/ReadDir.

static ZafIChar ZAF_FARDATA searchInfo[ZAF_MAXNAMELEN];
static const ZafIChar *searchPat;
static ZafClassID searchID;
static long patPosition;

ZafIChar *OldStorageReadOnly::FindFirst(const ZafIChar *pattern, ZafClassID *returnID)
{
	CheckStorageError(0);
	patPosition = 0;
	searchPat = pattern;
	searchID = 0;
	return (FindNext(returnID));
}

ZafIChar *OldStorageReadOnly::FindFirst(ZafClassID classID)
{
	CheckStorageError(0);
	patPosition = 0;
	searchPat = ZAF_NULLP(ZafIChar);
	searchID = classID;
	return (FindNext());
}

ZafIChar *OldStorageReadOnly::FindNext(ZafClassID *returnID)
{
	// Reset the searchID.
	if (returnID)
		*returnID = 0;

	OldDirectoryEntry dentry;
	CheckStorageError(0);
	currentDirectory->position = patPosition;
	while (true)
	{
		// Check for out-of-bounds.
		if (currentDirectory->position >= openObjects[currentDirectory->inodeIndex].inode.size)
			return ZAF_NULLP(ZafIChar);

		// Check the entry for a match.
		currentDirectory->Read(&dentry);
		if ((searchID && dentry.classID == searchID) ||
			(searchPat && !WildStrcmp(dentry.stringID, searchPat)))
		{
			// Reset the searchID.
			if (returnID)
				*returnID = dentry.classID;
			break;
		}
	}
	patPosition = currentDirectory->position;

	// Return the matching entry's name.
	strcpy(searchInfo, dentry.stringID);
	return (searchInfo);
}

// --- OldStorageReadOnly privates ------------------------------------------

ZafDiskAddress OldStorageReadOnly::AllocData(void)
{
	// Allocating here is an error.
	abort();
	return (0);
}

ZafDiskAddress OldStorageReadOnly::AppendInode(ZafINodeNumber)
{
	// Allocating here is an error.
	abort();
	return (0);
}

// Check to see if an inode (inum) is already in the open object list.
int OldStorageReadOnly::CheckOpen(ZafINodeNumber inum)
{
	// Try to find a matching inode.
	for (int i = 0; i < openLen; i++)
		if (openObjects[i].openCount && openObjects[i].inum == inum)
			return (i);

#if defined(ZAF_MACINTOSH)
	SetError(permErr);
#else
	SetError(EACCES);
#endif
	return (-1);
}

// Read or Write an inode depending on the direction.
long OldStorageReadOnly::FindInode(ZafINodeNumber inum)
{
	ZafDiskAddress inumblk;
	long pos;

	long iindex;
	long inumbyt;
	long returnValue;
	int indirectSize = IndirectBlockSize();
	int blockSize = BlockSize();
	int inodesPerBlock = blockSize / sizeof(ZafInode);
	ZafDiskAddress index;

	if (version < 500)
	{
		iindex = (inum / INODES_PER_INODE_BLOCK_4X);
		inumbyt = (inum % INODES_PER_INODE_BLOCK_4X);
	}
	else
	{
		if (blockSize > sizeof(ZafInode))
		{
			iindex = inum / inodesPerBlock;
			inumbyt = inum % inodesPerBlock;
		}
		else
		{
			iindex = inum;
			inumbyt = 0;
		}
	}

	// First look through the direct blocks.
	if (iindex < LENGTHOF(sb->inodeDirect))
	{
		index =  (int)iindex;
		if (sb->inodeDirect[index] == 0)
			sb->inodeDirect[index] = AppendInode(inum);
		inumblk = sb->inodeDirect[index];
	}
	else
	{
		// Now look in the indirect blocks
		iindex -= LENGTHOF(sb->inodeDirect);
		int diskAddrPerInodeBlock = IndirectBlockSize() / sizeof(ZafDiskAddress);
		// try to do the first stage of double indirect blocks
		ZafDiskAddress *iblk = new ZafDiskAddress[diskAddrPerInodeBlock];

		if (iindex >= diskAddrPerInodeBlock)
		{
			iindex -= diskAddrPerInodeBlock;
			// Find out if the inode doesn fit in any of our double indirect blocks.
			if (iindex / diskAddrPerInodeBlock >= diskAddrPerInodeBlock)
			{
				delete []iblk;
#if defined(ZAF_MACINTOSH)
				SetError(rfNumErr);
#else
				SetError(EINVAL);
#endif
				return (-1);
			}
			// Allocate the first level of double indirect pointers.
			if (sb->inodeDIndirect == 0)
				sb->inodeDIndirect = AppendInode(0xFFFFFFFFL);
			pos = GetBlockPos(sb->inodeDIndirect);
			SetError(ReadAt(pos, iblk, indirectSize));
			ZafDiskAddress blkind;
			ZafDiskAddress blknum;

			if (version < 500)
			{
				blkind = (ZafUInt16)(iindex % DISK_ADDR_PER_INODE_BLOCK_4X);
				blknum = (ZafUInt16)(iindex / DISK_ADDR_PER_INODE_BLOCK_4X);
			}
			else
			{
				blkind = (ZafUInt16)(iindex % diskAddrPerInodeBlock);
				blknum = (ZafUInt16)(iindex / diskAddrPerInodeBlock);

			}

			iindex = iblk[blknum];
			if (Error() == 0 && iindex == 0)
			{
				// Allocate the second level of indirect pointers.
				iindex = iblk[blknum] = AppendInode(0xFFFFFFFFL);
				SwapDiskAddr(&iblk[blknum]);
				SetError(WriteAt(pos, iblk, indirectSize));
			}
			else
			{
				index = (int)iindex;
				SwapDiskAddr(&index);
				iindex = index;
			}

			index = (int)iindex;
			pos = GetBlockPos(index);
			iindex = blkind;
			if (Error() != 0)
			{
				delete []iblk;
				return (-1);
			}
		}
		else
		{
			// Allocate an single indirect block.
			if (sb->inodeSIndirect == 0)
				sb->inodeSIndirect = AppendInode(0xFFFFFFFFL);
			pos = GetBlockPos(sb->inodeSIndirect);
		}
		// We are either reading a single indirect block or the
		// second stage of a double indirect block.
		SetError(ReadAt(pos, iblk, indirectSize));
		index = (int)iindex;
		inumblk = iblk[index];
		if (Error() == 0 && inumblk == 0)
		{
			inumblk = iblk[index] = AppendInode(inum);
			SwapDiskAddr(&iblk[index]);
			SetError(WriteAt(pos, iblk, IndirectBlockSize()));
		}
		else
			SwapDiskAddr(&inumblk);
		delete []iblk;
		if (Error() != 0)
			return (-1);
	}
	// Actually perform the read or write.
	if (version < 500)
		returnValue = (long)inumblk*BlockSize() + inumbyt*sizeof(ZafInode);
	else
		returnValue = sizeof(ZafSuperBlock) + ((long)inumblk - 1) * BlockSize()
		+ inumbyt * sizeof(ZafInode);

	return (returnValue);
}

// In our list of open objects, try to find an empty hole.  Allocate more
// space if we can't.
int OldStorageReadOnly::FindSlot(void)
{
	int i = 0;

	// Try to find an open spot in the array.
	for (; i < openLen; i++)
		if (openObjects[i].openCount <= 0)
			break;

	// No room, allocate more space.
	if (i >= openLen)
	{
		ZafOpenObject *nopen = new ZafOpenObject[openLen+NEW_INC];
		for (i = 0; i < openLen; i++)
			nopen[i] = openObjects[i];
		for (i = openLen; i < openLen+NEW_INC; i++)
			nopen[i].openCount = 0;
		delete []openObjects;
		openObjects = nopen;
		i = openLen;
		openLen += NEW_INC;
	}

	// Return the index.
	return (i);
}

// See if a data block is already in the cache.
bool OldStorageReadOnly::InMemory(ZafDiskAddress blknum)
{
	if (blknum == 0)
		return (false);
	for (int i = 0; i < cacheLen; i++)
		if (cd[i].blknum == blknum)
			return (true);
	return (false);
}

OldStorageReadOnly::~OldStorageReadOnly(void)
{
	// Check for a current directory.
	if (currentDirectory)
		delete currentDirectory;

	// If the file has been really opened.
#if defined(ZAF_MACINTOSH)
	if (fd)
		FSClose(fd);
#else
	if (fd)
		fclose(fd);
#endif

	// Deleting currentDirectory depends on openObjects existing.
	delete []openObjects;
	delete sb;
	delete []cd;
#if defined(ZAF_DOSEXTENDED16)
	if (cache)
		delete [](cache - 2);
#else
	delete []cache;
#endif
}

// Read some data from a certain position.  This is here mainly to link
// the seek and read together.
int OldStorageReadOnly::ReadAt(long pos, void *buf, int len)
{
#if defined(ZAF_MACINTOSH)
	errno = SetFPos(fd, fsFromStart, pos);
	if (errno != noErr)
		return (errno);
	long count = len;
	errno = FSRead(fd, &count, (BUFFTYPE *)buf);
	if (errno != noErr)
		return (errno);
#else
	if (fseek(fd, pos, SEEK_SET))
		return (errno);
	if (fread((BUFFTYPE *)buf, 1, len, fd) != len)
		return (errno);
#endif
	return (0);
}

// Read a data block.  First check to see if it is currently in the cache.
// If so, just return a pointer to that block.  Otherwise, find an unused
// cache block, flush it if necessary, read the data into it and return it.
// This function should only be used to read blocks of indirect pointers.
void *OldStorageReadOnly::ReadData(ZafDiskAddress blknum)
{
	// Check for a valid block number.
	if (blknum == 0)
		return (ZAF_NULLP(void));

	ZafUInt16 indirectSize = IndirectBlockSize();

	// Initialize the cache block.
	int i = 0;
	for (; i < cacheLen; i++)
		if (cd[i].blknum == blknum)
		{
			// Copy all previous cache data structures.
			ZafCacheData tmpcd = cd[i];
			tmpcd.used++;
			for (int j=i; j > 0; j--)
				cd[j] = cd[j-1];
			// Insert the first record.
			cd[0].pos = tmpcd.pos;
			cd[0].dirty = tmpcd.dirty;
			cd[0].blknum = blknum;
			cd[0].used = 1;
			// Return the cache position.
			return (&cache[tmpcd.pos*indirectSize]);
		}

	// Find the first unused cache.
	for (i=0; i < cacheLen; i++)
		if (cd[i].used == 0)
			break;
	if (i >= cacheLen)
	{
		// This is a fatal error, ran out of room.
#if defined(ZAF_MACINTOSH)
		SetError(ioErr);
#else
		SetError(ERANGE);
#endif
		abort();
	}

	// Copy, then insert a new cache.
	ZafCacheData tmpcd = cd[i];
	for (int j = i; j > 0; j--)
		cd[j] = cd[j-1];
	int cpos = (int) (tmpcd.pos * indirectSize);
	if (tmpcd.dirty)
	{
		long dpos = GetBlockPos(tmpcd.blknum);
		SetError(WriteAt(dpos, &cache[cpos], indirectSize));
		if (Error() != 0)
			return (ZAF_NULLP(void));
		tmpcd.dirty = 0;
	}

	// Insert the first record.
	cd[0].pos = tmpcd.pos;
	cd[0].dirty = tmpcd.dirty;
	cd[0].blknum = blknum;
	cd[0].used = 1;

	// Determine the new cache position.
	long dpos = GetBlockPos(blknum);
	SetError(ReadAt(dpos, &cache[cpos], indirectSize));
	if (Error() != 0)
		return (ZAF_NULLP(void));
	return (&cache[cpos]);
}

int OldStorageReadOnly::RWInode(ZafINodeNumber inum, ZafInode *ientry, int)
{
	long pos = FindInode(inum);
	SetError(ReadAt(pos, ientry, sizeof(*ientry)));
	if (Error() != 0)
		return -1;
	SwapInode(ientry);
	return (0);
}

// Walk a path, strip the last name, return both end directory and name
// inside that directory to operate on.
void OldStorageReadOnly::WalkPartialPath(const ZafIChar *pathname,
	OldStorageObjectReadOnly **parentdir, const ZafIChar **filename)
{
	*parentdir = WalkPath(pathname, true);
	*filename = pathname + strlen(pathname);
	while (*filename != pathname && **filename != ZAF_DIRECTORY_SEPARATOR)
		(*filename)--;
	if (**filename == ZAF_DIRECTORY_SEPARATOR)
		(*filename)++;
	if (!**filename)
		*filename = currentDirectoryName;
	if (*parentdir == ZAF_NULLP(OldStorageObjectReadOnly))
		*parentdir = currentDirectory;
}

// Given a path name into a STORAGE file, walk down it.  Return errors
// on bogus directory names, and return a NULL.  If stripLast is TRUE then
// we don't try to move to the last piece of the path (it might be a file
// or we might want to manipulate it).
OldStorageObjectReadOnly *OldStorageReadOnly::WalkPath(const ZafIChar *name, bool stripLast)
{
	if (!name || ! *name)
		return ZAF_NULLP(OldStorageObjectReadOnly);
	OldStorageObjectReadOnly *startdir = currentDirectory;
	ZafIChar *tmpname = new ZafIChar[strlen(name)+2];
	ZafIChar *strt = tmpname;
	ZafIChar *stop = tmpname;
	strcpy(tmpname, name);
	tmpname[strlen(name)+1] = '\0';	// hard mark the end

	// if (stripLast) then don't try to get into the last part of the path
	// Strip it off.
	if (stripLast)
	{
		int i = strlen(tmpname);
		while (i && tmpname[i] != ZAF_DIRECTORY_SEPARATOR)
			i--;
		if (i == 0 && tmpname[0] == ZAF_DIRECTORY_SEPARATOR)
			i++;
		tmpname[i] = '\0';
		tmpname[i+1] = '\0';	// hard mark the end
	}

	// There was no path to traverse.
	if (!*strt)
	{
		delete []tmpname;
		return ZAF_NULLP(OldStorageObjectReadOnly);
	}
	for (;;)
	{
		// strip out a directory name
		while (*stop && *stop != ZAF_DIRECTORY_SEPARATOR)
			stop++;
		*stop = '\0';
		if (stop == strt)
		{
			// empty name, just a ~, start from root
			if (openObjects[startdir->inodeIndex].inum != 0)
			{
				static OldDirectoryEntry dentry = { 0, 0, 0, 0, { 0 } };
				ZafInode ientry;

				if (RWInode((ZafINodeNumber)0, &ientry, ZAF_IREAD) < 0)
					break;
				if (startdir != currentDirectory)
					delete startdir;
				startdir = (OldStorageObjectReadOnly *)AllocateFile(ZAF_NULLP(ZafIChar), 0, 0);
				startdir->mode = currentDirectory->mode;
				startdir->OpenTheObject(*this, &dentry, &ientry, false);
			}
			stop++;
			// That's all folks.
			if (! *stop)
			{
				delete []tmpname;
				return (startdir);
			}
			strt = stop;
		}
		// Empty name (~~ ???) error.
		else if (! *strt)
			break;
		// No directory, error.
		else if (!startdir->FindName(strt))
			break;
		// Have a name, and it exists.  Let's move there.
		else
		{
			OldStorageObjectReadOnly *dir = (OldStorageObjectReadOnly *)AllocateFile(ZAF_NULLP(ZafIChar), 0, 0);
			OldDirectoryEntry dentry;
			ZafInode ientry;

			startdir->Read(&dentry);
			dir->classID = dentry.classID;
			dir->stringID = strdup(dentry.stringID);
			dir->mode = startdir->mode;
			int i = CheckOpen(dentry.inum);
			if (i < 0)
			{
				SetError(0);	// not an error yet
				if (RWInode(dentry.inum, &ientry, ZAF_IREAD) < 0)
				{
					delete dir;
					break;
				}
			}
			else
				ientry = openObjects[i].inode;
			dir->OpenTheObject(*this, &dentry, &ientry, false);
			if (!(openObjects[dir->inodeIndex].inode.useCount &
			      ZAF_DIRECTORY_TAG))
			{
				delete dir;
				break;
			}
			if (currentDirectory != startdir)
				delete startdir;
			startdir = dir;
			stop++;
			// That's all folks.
			if (! *stop)
			{
				delete []tmpname;
				return (startdir);
			}
			strt = stop;
		}
	}

	// All errors come through here.
	if (currentDirectory != startdir)
		delete startdir;
	delete []tmpname;
	return ZAF_NULLP(OldStorageObjectReadOnly);
}

int OldStorageReadOnly::WriteAt(long, void *, int)
{
	abort();
	return (0);
}

// This function should only be used to write blocks of indirect pointers.
void OldStorageReadOnly::WriteData(void *data, bool tModified)
{
	if ((char *)data < cache || (char *)data >= &cache[cacheLen*IndirectBlockSize()])
	{
#if defined(ZAF_MACINTOSH)
		SetError(ioErr);
#else
		SetError(ERANGE);
#endif
		abort();
	}
	ZafUInt16 i = (ZafUInt16)(((char *)data - cache) / IndirectBlockSize());
	// This is probably always 0 (or 1)
	for (int j=0; j < cacheLen; j++)
		if (cd[j].pos == i)
		{
			// FlagSet is probably overkill.
			cd[j].dirty |= ReadWrite() && tModified;
			cd[j].used = 0;
			return;
		}
#if !defined(ZAF_MSWINDOWS) && !defined(ZAF_MACINTOSH)
	puts("Fatal internal OldStorage error.\n");
#endif
	abort();
}

bool OldStorageReadOnly::SearchPath(ZafIChar *tmp)
{
	const ZafIChar *path = ZafLanguageData::blankString;
	bool first = true;
	while (path != ZAF_NULLP(ZafIChar))
	{
		AppendFullPath(tmp, path, pname);
		AppendFullPath(tmp, tmp, fname);
		if (ValidName(tmp, false))
			break;
		path = searchPath ? (first ? searchPath->FirstPathName() : searchPath->NextPathName()) : ZAF_NULLP(ZafIChar);
		first = false;
	}
	if (path != ZAF_NULLP(ZafIChar))
	{
		AppendFullPath(tmp, path, pname);
		strcpy(pname, tmp);
	}
	MakeFullPath(pname);
	return (path ? false : true);
}

void OldStorageReadOnly::ValidateZincFile(void)
{
#if defined(ZAF_3x_COMPAT)
	SetError(0);
	if (sb->signature.majorVersion == 3 &&
	    sb->signature.minorVersion == 0)
	{
		sb->freeInodeListHi = 0;
		sb->signature.minorVersion = 2;
		sb->signature.copyrightNotice[25] = '2';
	}
	if (sb->signature.magicNumber != ZAF_MAGIC_NUMBER ||
	    sb->signature.majorVersion < 3)
	{
#if defined(ZAF_MACINTOSH)
		SetError(rfNumErr);
#else
		SetError(EINVAL);
#endif
		return;
	}
#	if defined(ZAF_UNICODE)
	if (sb->signature.majorVersion == 3)
	{
		sb->signature.minorVersion = 6;
		sb->signature.copyrightNotice[25] = '6';
	}
	else
	{
		sb->signature = zafSignature;
	}
#	endif
#else
	if (sb->signature.magicNumber != ZAF_MAGIC_NUMBER ||
	    sb->signature.majorVersion < 4)
	{
#if defined(ZAF_MACINTOSH)
		SetError(rfNumErr);
#else
		SetError(EINVAL);
#endif
		return;
	}
#	if defined(ZAF_UNICODE)
		sb->signature = zafSignature;
#	endif
#endif
}

void OldStorageReadOnly::SetUpLocalData(void)
{
	sb->revision++;
	cacheLen = cacheSize;
	ZafUInt16 indirectBlockSize = IndirectBlockSize();
#if defined(ZAF_DOSEXTENDED16)
	// The DOS Extender can butt this up against the end of a segment
	// thereby creating a non-ANSI program (we can_t compare a pointer
	// against the address following "cache" because it wraps around
	// to 0x0000).  This will fix it!!!
	cache = new char[cacheLen * indirectBlockSize + 4] + 2;
#else
	cache = new char[cacheLen * indirectBlockSize];
#endif
	cd = new ZafCacheData[cacheLen];
	if (cache == ZAF_NULLP(char) || cd == ZAF_NULLP(ZafCacheData))
	{
#if defined(ZAF_MACINTOSH)
		SetError(ioErr);
#else
		SetError(ENOMEM);
#endif
		return;
	}
	// Initialize the cache.
	int i = 0;
	for (; i < cacheLen; i++)
	{
		cd[i].pos = i;
		cd[i].blknum = 0;
		cd[i].dirty = 0;
		cd[i].used = 0;
	}
	// initialize the open file table to all unused
	openLen = NEW_INC;
	openObjects = new ZafOpenObject[openLen];
	for (i=0; i < openLen; i++)
		openObjects[i].openCount = 0;
}

void OldStorageReadOnly::ChDirRoot(void)
{
	// or start up access in the root (chdir("~")).
	if (currentDirectory)
		delete currentDirectory;
	currentDirectory = (OldStorageObjectReadOnly *)AllocateFile(ZAF_NULLP(ZafIChar), 0, 0);
	currentDirectory->file = this;
	currentDirectory->inodeIndex = 0;
	currentDirectory->position = 0;
	currentDirectory->mode = mode;
	ZafInode ientry;
	if (RWInode((ZafINodeNumber)0, &ientry, ZAF_IREAD) < 0)
		return;
	OldDirectoryEntry dentry;
	dentry.inum = 0;
	currentDirectory->OpenTheObject(*this, &dentry, &ientry, false);
}

int OldStorageReadOnly::MkDir(const ZafIChar *, const ZafClassName, ZafClassID)
{
	return (-1);
}

// --- OldStorageReadOnly publics -------------------------------------------

OldStorageReadOnly::OldStorageReadOnly(void) :
	OldDirectory(ZAF_NULLP(ZafIChar), ZAF_FILE_READ | ZAF_FILE_BINARY),
	openLen(0), openObjects(ZAF_NULLP(ZafOpenObject)),
	currentDirectory(ZAF_NULLP(OldStorageObjectReadOnly)), fd(NULL),
	sb(ZAF_NULLP(ZafSuperBlock)), cacheLen(0), cd(ZAF_NULLP(ZafCacheData)),
	cache(ZAF_NULLP(char))
{
	// Update the version number.
	SetVersion(100 * ZAF_MAJOR_FILE_VERSION + ZAF_MINOR_FILE_VERSION);
}

OldStorageReadOnly::OldStorageReadOnly(const ZafIChar *name) :
	OldDirectory(ZAF_NULLP(ZafIChar), ZAF_FILE_READ | ZAF_FILE_BINARY),
	openLen(0), openObjects(ZAF_NULLP(ZafOpenObject)),
	currentDirectory(ZAF_NULLP(OldStorageObjectReadOnly)), fd(NULL),
	sb(ZAF_NULLP(ZafSuperBlock)), cacheLen(0), cd(ZAF_NULLP(ZafCacheData)),
	cache(ZAF_NULLP(char))
{
	StripFullPath(name, pname, fname);
	// If this is an old file, find it in the searchPath.
	ZafIChar tmp[ZAF_MAXPATHLEN];
	SearchPath(tmp);
	// open the file
	// The real file for read-only.
	AppendFullPath(tmp, pname, fname);
#if defined(ZAF_MACINTOSH)
	fd = MacOpen(tmp, fileCreator, fileType, mode);
	if (!fd)
	{
		SetError(errno);
		return;
	}
#else
	fd = fopen(tmp, ZAF_ITEXT("rb"));
	if (!fd)
	{
		SetError(errno);
		return;
	}
#endif
	sb = new ZafSuperBlock;
	SetError(ReadAt(0L, sb, sizeof(*sb)));
	SwapSuperBlock(sb);
	SetVersion(100 * sb->signature.majorVersion + sb->signature.minorVersion);

	// Validate that this is a Zinc file.
	ValidateZincFile();
	CheckStorageErrorNull();
	SetUpLocalData();
	CheckStorageErrorNull();
	ChDirRoot();
	CheckStorageErrorNull();
	SetError(0);
}

OldFile *OldStorageReadOnly::AllocateFile(const ZafIChar *name,
	const ZafClassName className, ZafClassID classID)
{
	// Move to the sub-directory.
	if (!name)
		name = className;

	// Create the sub-object.
	OldStorageObjectReadOnly *object;
	if (name)
		object = new OldStorageObjectReadOnly(*this, name, classID);
	else
		object = new OldStorageObjectReadOnly;

	// Return the new object.
	if (object->Error())
	{
		delete object;
		return (ZAF_NULLP(OldFile));
	}
	return (object);
}

int OldStorageReadOnly::ChDir(const ZafIChar *name)
{
	CheckStorageError(-1);
	OldStorageObjectReadOnly *startdir = WalkPath(name, false);
	if (startdir == ZAF_NULLP(OldStorageObjectReadOnly))
		return (-1);
	if (startdir != currentDirectory)
		delete currentDirectory;
	currentDirectory = startdir;
	return (0);
}

int OldStorageReadOnly::GetCWD(ZafIChar *path, int pathLen)
{
	// To make the loop code easier, short circuit if we are already
	// in the root directory.
	if (openObjects[currentDirectory->inodeIndex].inum == 0)
	{
		if (pathLen < strlen(rootDirectoryName))
		{
#if defined(ZAF_MACINTOSH)
			SetError(ioErr);
#else
			SetError(ERANGE);
#endif
			return (-1);
		}
		strcpy(path, rootDirectoryName);
		return (0);
	}

#define DOEXIT(x)	if (oldCurrentDirectory != currentDirectory)	\
				delete currentDirectory;		\
			currentDirectory = oldCurrentDirectory;		\
			return (x)

	OldStorageObjectReadOnly *oldCurrentDirectory = currentDirectory;

	path[pathLen-1] = 0;
	int currentPathLen = pathLen - 1;

	// The alogrithm goes as such: Save our Inum, change to our parent,
	// search for our Inum in our parent's data, loop until we hit the
	// root directory (Inum == 0).  On the way, append our name and a
	// separator to the front of the path.
	ZafINodeNumber searchInum;
	while ((searchInum = openObjects[currentDirectory->inodeIndex].inum) != 0)
	{
		OldStorageObjectReadOnly *parentDir = WalkPath(parentDirectoryName, false);
		if (oldCurrentDirectory != currentDirectory)
			delete currentDirectory;
		currentDirectory = parentDir;

		OldDirectoryEntry dentry;
		currentDirectory->position = 0;
		long currentSize = openObjects[currentDirectory->inodeIndex].inode.size;
		while (currentDirectory->position < currentSize)
		{
			if (currentDirectory->Read(&dentry) < 0)
			{
#if defined(ZAF_MACINTOSH)
				SetError(rfNumErr);
#else
				SetError(EINVAL);
#endif
				DOEXIT(-1);
			}
			if (dentry.inum == searchInum)
			{
				currentDirectory->position = 0;
				break;
			}
		}
		if (currentDirectory->position >= currentSize)
		{
#if defined(ZAF_MACINTOSH)
			SetError(rfNumErr);
#else
			SetError(EINVAL);
#endif
			DOEXIT(-1);
		}
		currentPathLen -= strlen(dentry.stringID) + strlen(rootDirectoryName);
		if (currentPathLen < 0)
		{
#if defined(ZAF_MACINTOSH)
			SetError(ioErr);
#else
			SetError(ERANGE);
#endif
			DOEXIT(-1);
		}
		memcpy(&path[currentPathLen], rootDirectoryName, strlen(rootDirectoryName)*sizeof(ZafIChar));
		memcpy(&path[currentPathLen+strlen(rootDirectoryName)], dentry.stringID, strlen(dentry.stringID)*sizeof(ZafIChar));
	}
	memmove((char *)path, (char *)&path[currentPathLen], (pathLen-currentPathLen)*sizeof(ZafIChar));
	DOEXIT(0);
#undef DOEXIT
}

OldStorageDirectory *OldStorageReadOnly::OpenDir(const ZafIChar *name)
{
	OldStorageObjectReadOnly *tmp = new OldStorageObjectReadOnly(*this, name, 0);
	if (tmp->Error() != 0 ||
	    (!(tmp->file->openObjects[tmp->inodeIndex].inode.useCount & ZAF_DIRECTORY_TAG)))
	{
		delete tmp;
		return (ZAF_NULLP(OldStorageDirectory));
	}
	return (new OldStorageDirectory(tmp));
}

ZafStatsStruct *OldStorageReadOnly::Stats(void)
{
	static ZafStatsStruct stats;
	stats.createTime = sb->createTime;
	stats.modifyTime = sb->modifyTime;
#if defined(ZAF_MACINTOSH)
	// Fix for time value since Apple uses 1904.
	stats.createTime += 31554100L * 66L + 249000L;
	stats.modifyTime += 31554100L * 66L + 249000L;

	GetEOF(fd, &stats.size);
#else
	fseek(fd, 0, SEEK_END);
	stats.size = ftell(fd);
#endif
	stats.useCount = 1;		// ??? Unix will change this
	stats.revision = sb->revision;
	stats.countryID = 0;
	stats.inum = 0;			// ??? Unix will change this
	return (&stats);
}

void OldStorageReadOnly::FreeFile(OldFile *storageObject)
{
	if (storageObject)
		delete storageObject;
}

// --- OldStorageObjectReadOnly publics -------------------------------------

OldStorageObjectReadOnly::OldStorageObjectReadOnly(void) :
	ZafBufferObject(),
	file(ZAF_NULLP(OldStorageReadOnly)), stringID(ZAF_NULLP(ZafIChar)),
	inodeIndex(-1), cachedBlknum(ZAF_NULLP(ZafDiskAddress)),
	cachedBlkptr(ZAF_NULLP(ZafDiskAddress))
{
	mode = ZAF_FILE_READ;
}

OldStorageObjectReadOnly::OldStorageObjectReadOnly(OldStorageReadOnly &pfile, const ZafIChar *name, ZafClassID ) :
	ZafBufferObject(),
	file(ZAF_NULLP(OldStorageReadOnly)), stringID(ZAF_NULLP(ZafIChar)),
	inodeIndex(-1), cachedBlknum(ZAF_NULLP(ZafDiskAddress)),
	cachedBlkptr(ZAF_NULLP(ZafDiskAddress))
{
	mode = ZAF_FILE_READ;

#define DOEXIT	if (parentdir != pfile.currentDirectory)	\
			delete parentdir

	if (name == ZAF_NULLP(ZafIChar))
	{
#if defined(ZAF_MACINTOSH)
		SetError(rfNumErr);
#else
		SetError(EINVAL);
#endif
		return;
	}

	OldStorageObjectReadOnly *parentdir;
	const ZafIChar *fname;
	pfile.WalkPartialPath(name, &parentdir, &fname);
	bool found = parentdir->FindName(fname);

	OldDirectoryEntry dentry;
	ZafInode ientry;
	if (!found)
	{
		// If the object doesn't exist
#if defined(ZAF_MACINTOSH)
		SetError(fnfErr);
#else
		SetError(ENOENT);
#endif
		DOEXIT;
		return;
	}

	// If the object does exist
	parentdir->Read(&dentry);
	classID = dentry.classID;
	stringID = strdup(dentry.stringID);
	int i = pfile.CheckOpen(dentry.inum);
	if (i < 0)
	{
		pfile.SetError(0);		// not an error yet
		if (pfile.RWInode(dentry.inum, &ientry, ZAF_IREAD) < 0)
		{
			SetError(pfile.Error());
			pfile.SetError(0);
			DOEXIT;
			return;
		}
	}
	else
		ientry = pfile.openObjects[i].inode;
	if (ientry.useCount == 0)
	{
#if defined(ZAF_MACINTOSH)
		SetError(fnfErr);
#else
		SetError(ENOENT);
#endif
		DOEXIT;
		return;
	}
	OpenTheObject(pfile, &dentry, &ientry, false);
	DOEXIT;
#undef DOEXIT
}

OldStorageObjectReadOnly::~OldStorageObjectReadOnly(void)
{
	if (stringID)
		delete []stringID;
	ZafOpenObject *o = CheckObject();
	if (o)
		o->openCount--;
}

ZafStatsStruct *OldStorageObjectReadOnly::Stats(void)
{
	ZafOpenObject *o = CheckObject();
	if (o == ZAF_NULLP(ZafOpenObject))
		return (ZAF_NULLP(ZafStatsStruct));

	static ZafStatsStruct stats;
	stats.createTime = o->inode.createTime;
	stats.modifyTime = o->inode.modifyTime;
#if defined(ZAF_MACINTOSH)
	// Fix for time value since Apple uses 1904.
	stats.createTime += 31554100L * 66L + 249000L;
	stats.modifyTime += 31554100L * 66L + 249000L;
#endif
	stats.size = length;
	stats.useCount = o->inode.useCount;
	stats.revision = o->revision;
	stats.countryID = o->country;
	stats.inum = o->inum;
	return (&stats);
}

// --- OldStorageObjectReadOnly privates -----------------------------------

// Find an object's information in its files open-object array.
// Return errors if it doesn't exist.
ZafOpenObject *OldStorageObjectReadOnly::CheckObject(void)
{
	if (file == ZAF_NULLP(OldStorageReadOnly))
	{
#if defined(ZAF_MACINTOSH)
		SetError(fnfErr);
#else
		SetError(ENOENT);
#endif
		return (ZAF_NULLP(ZafOpenObject));
	}
	if (inodeIndex < 0 || inodeIndex >= file->openLen)
	{
#if defined(ZAF_MACINTOSH)
		SetError(rfNumErr);
#else
		SetError(EBADF);
#endif
		return (ZAF_NULLP(ZafOpenObject));
	}
	ZafOpenObject *o = &file->openObjects[inodeIndex];
	if (o->openCount < 0)
	{
#if defined(ZAF_MACINTOSH)
		SetError(rfNumErr);
#else
		SetError(EBADF);
#endif
		return (ZAF_NULLP(ZafOpenObject));
	}
	return (o);
}

// Find a name in a directory object.
bool OldStorageObjectReadOnly::FindName(const ZafIChar *name)
{
	// Assume this is a directory
	position = 0;
	while (position < length)
	{
		long i = position;
		OldDirectoryEntry dentry;
		if (Read(&dentry) < 0)
			return (false);
		if (streq(name, dentry.stringID) == 0)
		{
			position = i;
			return (true);
		}
	}
	return (false);
}

// Find the position in the file for the specified block.
// Given a (relative) block number in an object, return the (absolute)
// block number in the file.
ZafDiskAddress OldStorageObjectReadOnly::GetBlockPtr(ZafDiskAddress blknum)
{
	// Find direct blocks.  (Maybe we shouldn't cache them...)
	ZafInode *inode = &file->openObjects[inodeIndex].inode;
	if (blknum < LENGTHOF(inode->direct))
	{
		if (inode->direct[blknum] == 0)
			inode->direct[blknum] = file->AllocData();
		return (inode->direct[blknum]);
	}
	blknum -= LENGTHOF(inode->direct);
	// single indirect blocks
	ZafUInt16 indirectSize = file->IndirectBlockSize();
	ZafUInt16 addrPerBlock = indirectSize/ sizeof(ZafDiskAddress);
	if (blknum < addrPerBlock)
	{
		if (inode->sIndirect == 0)
		{
			if (version < 500)
				inode->sIndirect = file->AllocData();
			else
				inode->sIndirect = file->AppendInode(0xFFFFFFFFL);
		}
		ZafDiskAddress *iblk = (ZafDiskAddress *)file->ReadData(inode->sIndirect);
		if (iblk == ZAF_NULLP(ZafDiskAddress))
			return (0);
		ZafDiskAddress retval = iblk[blknum];
		bool tModified = (retval == 0);

		// If there wasn't an inode in the list, allocate one and store it out.
		if (tModified)
		{
			retval = iblk[blknum] = file->AllocData();
			SwapDiskAddr(&iblk[blknum]);
			file->WriteAt(file->GetBlockPos(inode->sIndirect), iblk,
				indirectSize);
			tModified = false;
		}
		else
			SwapDiskAddr(&retval);
		file->WriteData(iblk, tModified);
		return (retval);
	}
	blknum -= addrPerBlock;
	// double indirect blocks
	unsigned int blkptr = blknum % addrPerBlock;
	blknum /= addrPerBlock;
	if (blknum < addrPerBlock)
	{
		if (inode->dIndirect == 0)
		{
			if (version < 500)
				inode->dIndirect = file->AllocData();
			else
				inode->dIndirect = file->AppendInode(0xFFFFFFFFL);
		}
		ZafDiskAddress *iblk = (ZafDiskAddress *)file->ReadData(inode->dIndirect);
		if (iblk == ZAF_NULLP(ZafDiskAddress))
			return (0);
		ZafDiskAddress retval = iblk[blknum];
		bool tModified = (retval == 0);

		// If there wasn't an inode in the list allocate one and store it out.
		if (tModified)
		{
			if (version < 500)
				retval = iblk[blknum] = file->AllocData();
			else
				retval = iblk[blknum] = file->AppendInode(0xFFFFFFFFL);

			SwapDiskAddr(&iblk[blknum]);
			file->WriteAt(file->GetBlockPos(inode->dIndirect), iblk, indirectSize);
			tModified = false;
		}
		else
			SwapDiskAddr(&retval);
		file->WriteData(iblk, tModified);

		iblk = (ZafDiskAddress *)file->ReadData(retval);
		if (iblk == ZAF_NULLP(ZafDiskAddress))
			return (0);

		//Save the address of the block of adresses
		ZafDiskAddress tempAddr;
		tempAddr = retval;
		retval = iblk[blkptr];
		
		tModified = (retval == 0);
		// If there wasn't an inode in the list allocate one and store it out.
		if (tModified)
		{
			retval = iblk[blkptr] = file->AllocData();
			SwapDiskAddr(&iblk[blkptr]);
			file->WriteAt(file->GetBlockPos(tempAddr), iblk, indirectSize);
			tModified = false;
		}
		else
			SwapDiskAddr(&retval);
		file->WriteData(iblk, tModified);
		return (retval);
	}
	return (0);		// error
}

// Utility routine for Storage to use.
// Don't change this without changing OldStorageDirectory::ReadDir().
int OldStorageObjectReadOnly::Read(OldDirectoryEntry *dirent)
{
	CheckObjectError(-1);
	ZafUInt16 sinum;
	int i = Read(sinum);
	if (i < 0)
		return (-1);
	int j;
	if (sinum == 0xffff)
	{
		if ((j = Read(dirent->inum)) < 0)
			return (-1);
		i += j;
	}
	else
		dirent->inum = sinum;
	if ((j = Read(dirent->classID)) < 0)
		return (-1);
	i += j;
	if ((j = Read(dirent->revision)) < 0)
		return (-1);
	i += j;
	if ((j = Read(dirent->country)) < 0)
		return (-1);
	i += j;
	if ((j = Read(dirent->stringID, sizeof(dirent->stringID))) < 0)
		return (-1);
	return (i + j);
}

// Do the common work to open up an object.
void OldStorageObjectReadOnly::OpenTheObject(OldStorageReadOnly &pfile,
	OldDirectoryEntry *dentry, ZafInode *ientry, bool truncate)
{
	// if it is already open, dup the object
	int i = pfile.CheckOpen(dentry->inum);
	if (i >= 0)
		pfile.openObjects[i].openCount++;
	else
	{
		pfile.SetError(0);		// not an error yet
		// find an empty slot
		if ((i = pfile.FindSlot()) < 0)
		{
#if defined(ZAF_MACINTOSH)
			SetError(ioErr);
#else
			SetError(ENOMEM);
#endif
			return;
		}
		pfile.openObjects[i].inum = dentry->inum;
		pfile.openObjects[i].openCount = 1;
		pfile.openObjects[i].inode = *ientry;
		pfile.openObjects[i].modified = false;
		pfile.openObjects[i].revision = dentry->revision;
		pfile.openObjects[i].country = dentry->country;
	}
	// This next line may be changed sometime to get rid of space used.
	if (truncate)
		pfile.openObjects[i].inode.size = 0;
	inodeIndex = i;
	position = 0;
	file = &pfile;
	delete []buffer;

	int blockSize = file->BlockSize();

	// Find the version of the object
	int versionOffset = (file->Version() >= 500) ? sizeof(ZafVersion) : 0;

	// Don't read version information for directory entries or new objects
	if (pfile.openObjects[i].inode.useCount & ZAF_DIRECTORY_TAG)
		versionOffset = 0;

	length = ientry->size;
	ZafDiskAddress diskAddr;

	if (versionOffset && length)
	{
		char versionBuffer[sizeof(ZafVersion)];
		buffer = versionBuffer;

		// Get the position of the first block.
		diskAddr = GetBlockPtr(0); 
		file->SetError(file->ReadAt(file->GetBlockPos(diskAddr), buffer, sizeof(ZafVersion)));

		// Use the built in functionality to take care of byte swapping
		*this>>version;

		// rewind the position after reading in the version.
		position = 0;
	}
	else
		version = file->Version();

	size = length + (blockSize - length % blockSize);

	//Modifiy size to take into account the version offset
	if (versionOffset && length + versionOffset > size )
//	if (versionOffset && length)
		size += blockSize;

	buffer = new char[(ZafUInt16)size];
	int blockNum = 0;

//	int tempOffset = versionOffset;

//	for (i=0; i < length + tempOffset; i += blockSize)
	for (i=0; i < length; i += blockSize)
	{
		diskAddr = GetBlockPtr(blockNum++);

		long diskPos = file->GetBlockPos(diskAddr);
		file->SetError(file->ReadAt(diskPos + versionOffset, &buffer[i],
			blockSize - versionOffset));

		// We only need to read version information the first time.
		if (versionOffset)
		{
			i -= versionOffset;
			versionOffset = 0;
		}

		if (file->Error() != 0)
			return;
	}
}

int OldStorageObjectReadOnly::Write(const OldDirectoryEntry *)
{
	// Writing here is an error.
	abort();
	return (0);
}

// ---- OldStorageDirectory --------------------------------------------------

OldStorageDirectory::OldStorageDirectory(OldStorageObjectReadOnly *_directory)
{
	PartialConstruct(_directory);
}

OldStorageDirectory::~OldStorageDirectory(void)
{
	delete []directoryBuffer;
	delete directory;
}

OldDirectoryEntry *OldStorageDirectory::ReadDir(void)
{
	int j;
	if (position >= size)
		return (ZAF_NULLP(OldDirectoryEntry));

	ZafUInt16 oldInum = 0;
	for (j=0; j < sizeof(oldInum); j++)
		oldInum |= (directoryBuffer[position++] << 8*j);
	if (oldInum == 0xFFFF)
	{
		current.inum = 0;
		for (j=0; j < sizeof(current.inum); j++)
			current.inum |= (directoryBuffer[position++] << 8*j);
	}
	else
		current.inum = oldInum;

	current.classID = 0;
	for (j=0; j < sizeof(current.classID); j++)
		current.classID |= (directoryBuffer[position++] << 8*j);

	current.revision = 0;
	for (j=0; j < sizeof(current.revision); j++)
		current.revision |= (directoryBuffer[position++] << 8*j);

	current.country = 0;
	for (j=0; j < sizeof(current.country); j++)
		current.country |= (directoryBuffer[position++] << 8*j);

	ZafUInt16 stringSize = 0;
	for (j=0; j < sizeof(stringSize); j++)
		stringSize |= (directoryBuffer[position++] << 8*j);

	int doUni = (stringSize & ZAF_UNICODE_FLAG);
	stringSize &= ~ZAF_UNICODE_FLAG;
	if (doUni)
	{
		for (int i = 0; i < stringSize; i++)
		{
#if defined(ZAF_UNICODE)
			current.stringID[i] = 0;
			for (j=0; j < sizeof(current.stringID[i]); j++)
				current.stringID[i] |= ((ZafUInt16)directoryBuffer[position++] << 8*j);
#else
			current.stringID[i] = (ZafUInt8)directoryBuffer[position++];
			position++;
#endif
		}
	}
	else
	{
		for (int i = 0; i < stringSize; i++)
			current.stringID[i] = (ZafUInt8)directoryBuffer[position++];
	}
	current.stringID[stringSize] = 0;

	return (&current);
}

void OldStorageDirectory::RewindDir(void)
{
	delete []directoryBuffer;
	PartialConstruct(directory);
}

void OldStorageDirectory::PartialConstruct(OldStorageObjectReadOnly *_directory)
{
	directory = _directory;
	ZafOpenObject *o = directory->CheckObject();
	size = 0;
	if (o == ZAF_NULLP(ZafOpenObject))
		return;
	size = (ZafUInt16)o->inode.size;
	directoryBuffer = new ZafUInt8[size];
	directory->TransferData(directoryBuffer, size, ZafBufferObject::ZAF_IREAD);
	position = 0;
}
