//	Zinc Application Framework - C_STORE.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include "c_store.hpp"
#include <z_string.hpp>
#include <z_utils.hpp>

ZafBufferObject::ZafBufferObject(int blocks) :
	OldFile(ZAF_NULLP(ZafIChar), ZAF_FILE_READWRITE | ZAF_FILE_BINARY | ZAF_FILE_DERIVED_ACCESS),
	position(0), length(0), size(BYTES_PER_BLOCK * blocks)
{
	// Allocate the buffer.
	buffer = new char[(unsigned int)size];

	// Update the version number.
	SetVersion(100 * ZAF_MAJOR_FILE_VERSION + ZAF_MINOR_FILE_VERSION);
}

ZafBufferObject::~ZafBufferObject(void)
{
	// Destroy the buffer.
	delete []buffer;
}

long ZafBufferObject::GetLength(void) const
{
	// Return the current length.
	return (length);
}

ZafError ZafBufferObject::Rename(const ZafIChar *newPathName)
{
	// Rename the buffer.
	if (pathName)
		delete []pathName;
	pathName = newPathName ? strdup(newPathName) : ZAF_NULLP(ZafIChar);
	return (ZAF_ERROR_NONE);
}

ZafError ZafBufferObject::Seek(ZafOffset offset, ZafSeek location)
{
	// Determine the actual buffer offset.
	if (location == ZAF_SEEK_START)
		position = offset;
	else if (location == ZAF_SEEK_CURRENT)
		position += offset;
	else if (location == ZAF_SEEK_END)
		position = Tell() + offset;

	// Check against absolute constraints.
	if (position > length)
		position = length;
	else if (position < 0)
		position = 0;
	return (0);
}

ZafOffset ZafBufferObject::Tell(void) const
{
	// Return the current offset.
	return (position);
}

int ZafBufferObject::TransferData(void *tdata, unsigned N, ZafDirection direction)
{
	char *data = (char *)tdata;
	for (int i = 0; i < N; i++)
	{
		if (position == size)
		{
			if (direction == ZAF_IREAD)
				return (i);
			size += 256;
			char *tmp = new char[(unsigned int)size];
			memmove(tmp, buffer, (unsigned int)(size-256));
			delete []buffer;
			buffer = tmp;
		}
		if (direction == ZAF_IWRITE)
			buffer[(unsigned int)position++] = data[i];
		else
			data[i] = buffer[(unsigned int)position++];
	}
	if (length < position)
		length = position;
	return N;
}

// ----- Read Functions -----------------------------------------------------

int ZafBufferObject::Read(ZafInt8 &value)
{
	return (TransferData(&value, sizeof(value), ZAF_IREAD));
}

int ZafBufferObject::Read(ZafUInt8 &value)
{
	return (TransferData(&value, sizeof(value), ZAF_IREAD));
}

#if (ZAF_WORD_SIZE != 64)
int ZafBufferObject::Read(int &value)
{
	ZafInt32 temp;
	int ccode = ZafBufferObject::Read(temp);
	value = (int)temp;
	return (ccode);
}
#endif

#if defined(ZAF_BIGENDIAN)
int ZafBufferObject::Read(ZafInt16 &value)
{
	ZafUInt8 buff[sizeof(value)];

	int i = TransferData(buff, sizeof(value), ZAF_IREAD);
	if (i > 0)
	{
		value = 0;
		for (int j=0; j < sizeof(value); j++)
			value |= (buff[j] << 8*j);
	}
	return (i);
}

int ZafBufferObject::Read(ZafUInt16 &value)
{
	ZafUInt8 buff[sizeof(value)];

	int i = TransferData(buff, sizeof(value), ZAF_IREAD);
	if (i > 0)
	{
		value = 0;
		for (int j=0; j < sizeof(value); j++)
			value |= (buff[j] << 8*j);
	}
	return (i);
}

int ZafBufferObject::Read(ZafInt32 &value)
{
	ZafUInt8 buff[sizeof(value)];

	int i = TransferData(buff, sizeof(value), ZAF_IREAD);
	if (i > 0)
	{
		value = 0;
		for (int j=0; j < sizeof(value); j++)
			value |= (buff[j] << 8*j);
	}
	return (i);
}

int ZafBufferObject::Read(ZafUInt32 &value)
{
	ZafUInt8 buff[sizeof(value)];

	int i = TransferData(buff, sizeof(value), ZAF_IREAD);
	if (i  > 0)
	{
		value = 0;
		for (int j=0; j < sizeof(value); j++)
			value |= (buff[j] << 8*j);
	}
	return (i);
}
#else
int ZafBufferObject::Read(ZafInt16 &value)
{
	return (TransferData(&value, sizeof(value), ZAF_IREAD));
}

int ZafBufferObject::Read(ZafUInt16 &value)
{
	return (TransferData(&value, sizeof(value), ZAF_IREAD));
}

int ZafBufferObject::Read(ZafInt32 &value)
{
	return (TransferData(&value, sizeof(value), ZAF_IREAD));
}

int ZafBufferObject::Read(ZafUInt32 &value)
{
	return (TransferData(&value, sizeof(value), ZAF_IREAD));
}
#endif

int ZafBufferObject::Read(ZafIChar *string, int len)
{
	*string = '\0';
	ZafUInt16 readSize;
	int cnt1 = ZafBufferObject::Read(readSize);
	if (cnt1 != sizeof(readSize))
		return (cnt1);
	// Unicode support can load both Unicode and ISO 8859-1 strings
	// without data loss.  The upper 128 characters of OEM strings
	// are bogus in Unicode mode.  ISO and OEM support can load the
	// first 256 or 128 Unicode characters respectivly.  Both loose
	// information outside this range.
	int doUni = (readSize & ZAF_UNICODE_FLAG);
	readSize &= ~ZAF_UNICODE_FLAG;
	if (len != -1 && readSize > len)
	{
		position -= cnt1;
		return (-1);
	}
	int cnt2 = 0;
	if (readSize)
#if defined(ZAF_UNICODE)
		if (doUni)		// Byte Swap each short
		{
			for (int i = 0; i < readSize; i++)
			{
				ZafUInt16 c;
				cnt2 += ZafBufferObject::Read(c);
				string[i] = c;
			}
		}
		else
		{
			for (int i = 0; i < readSize; i++)
			{
				ZafUInt8 tmp;
				cnt2 += ZafBufferObject::Read(tmp);
				string[i] = tmp;
			}
		}
#else
		if (doUni)
			for (int i = 0; i < readSize; i++)
			{
				ZafUInt16 tmp;
				cnt2 += ZafBufferObject::Read(tmp);
				string[i] = (char)tmp;
			}
		else
			cnt2 += TransferData(string, readSize, ZAF_IREAD);
#endif
	string[readSize] = '\0';
	return (cnt2 != (readSize * sizeof(string[0])) ? cnt2 : cnt1 + cnt2);
}

int ZafBufferObject::Read(ZafIChar **string)
{
	ZafUInt16 readSize;
	int cnt1 = ZafBufferObject::Read(readSize);
	int cnt2 = 0;
	int doUni = (readSize & ZAF_UNICODE_FLAG);
	readSize &= ~ZAF_UNICODE_FLAG;
	if (cnt1 == sizeof(readSize) && readSize)
	{
		// See Unicode related comments in previous function.
		*string = new ZafIChar[readSize + 1];
#if defined(ZAF_UNICODE)
		if (doUni)		// Byte Swap each short
		{
			for (int i = 0; i < readSize; i++)
			{
				ZafUInt16 c;
				cnt2 += ZafBufferObject::Read(c);
				(*string)[i] = c;
			}
		}
		else
		{
			for (int i = 0; i < readSize; i++)
			{
				ZafUInt8 tmp;
				cnt2 += ZafBufferObject::Read(tmp);
				(*string)[i] = tmp;
			}
		}
#else
		if (doUni)
			for (int i = 0; i < readSize; i++)
			{
				ZafUInt16 tmp;
				cnt2 += ZafBufferObject::Read(tmp);
				(*string)[i] = (char)tmp;
			}
		else
			cnt2 += TransferData(*string, readSize, ZAF_IREAD);
#endif
		(*string)[readSize] = '\0';
	}
	else
		*string = ZAF_NULLP(ZafIChar);
	return (cnt2 != (readSize * sizeof((*string)[0])) ? cnt2 : cnt1 + cnt2);
}

int ZafBufferObject::Read(void *readBuffer, int readSize, int readLength)
{
	return (TransferData(readBuffer, readSize * readLength, ZAF_IREAD));
}

// ----- Write Functions ----------------------------------------------------

int ZafBufferObject::Write(ZafInt8 value)
{
	return (TransferData(&value, sizeof(value), ZAF_IWRITE));
}

int ZafBufferObject::Write(ZafUInt8 value)
{
	return (TransferData(&value, sizeof(value), ZAF_IWRITE));
}

#if (ZAF_WORD_SIZE != 64)
int ZafBufferObject::Write(int value)
{
	ZafInt32 temp = (ZafInt16)value;
	return (ZafBufferObject::Write(temp));
}
#endif

#if defined(ZAF_BIGENDIAN)
int ZafBufferObject::Write(ZafInt16 value)
{
	ZafUInt8 buff[sizeof(value)];

	for (int j=0; j < sizeof(value); j++)
		buff[j] = (ZafUInt8)((value >> 8*j) & 0xff);
	return (TransferData(buff, sizeof(value), ZAF_IWRITE));
}

int ZafBufferObject::Write(ZafUInt16 value)
{
	ZafUInt8 buff[sizeof(value)];

	for (int j=0; j < sizeof(value); j++)
		buff[j] = (ZafUInt8)((value >> 8*j) & 0xff);
	return (TransferData(buff, sizeof(value), ZAF_IWRITE));
}

int ZafBufferObject::Write(ZafInt32 value)
{
	ZafUInt8 buff[sizeof(value)];

	for (int j=0; j < sizeof(value); j++)
		buff[j] = (ZafUInt8)((value >> 8*j) & 0xff);
	return (TransferData(buff, sizeof(value), ZAF_IWRITE));
}

int ZafBufferObject::Write(ZafUInt32 value)
{
	ZafUInt8 buff[sizeof(value)];

	for (int j=0; j < sizeof(value); j++)
		buff[j] = (ZafUInt8)((value >> 8*j) & 0xff);
	return (TransferData(buff, sizeof(value), ZAF_IWRITE));
}
#else
int ZafBufferObject::Write(ZafInt16 value)
{
	return (TransferData(&value, sizeof(value), ZAF_IWRITE));
}

int ZafBufferObject::Write(ZafUInt16 value)
{
	return (TransferData(&value, sizeof(value), ZAF_IWRITE));
}

int ZafBufferObject::Write(ZafInt32 value)
{
	return (TransferData(&value, sizeof(value), ZAF_IWRITE));
}

int ZafBufferObject::Write(ZafUInt32 value)
{
	return (TransferData(&value, sizeof(value), ZAF_IWRITE));
}
#endif

int ZafBufferObject::Write(const ZafIChar *string)
{
	ZafUInt16 tSize = (string == ZAF_NULLP(ZafIChar) ? 0 : strlen(string));
	// See Unicode comments in previous functions.
#if defined(ZAF_UNICODE)
	int i;
	int isISO = 1;
	// Let's try to store the string as an ISO string (save lots of space).
	for (i=0; i < tSize; i++)
		if (string[i] > 0x00ff)
			isISO = 0;
	int cnt1 = ZafBufferObject::Write((ZafUInt16)(tSize | (isISO ? 0 : ZAF_UNICODE_FLAG)));
	if (cnt1 != sizeof(tSize))
		return cnt1;
	int cnt2 = 0;
	// Must (possibly) byte swap the file
	if (isISO)
		for (i = 0; i < tSize; i++)
		{	
			ZafUInt8 c = string[i];
			cnt2 += ZafBufferObject::Write(c);
		}
	else
		for (i = 0; i < tSize; i++)
		{	
			ZafUInt16 c = string[i];
			cnt2 += ZafBufferObject::Write(c);
		}
#else
	int cnt1 = ZafBufferObject::Write(tSize);
	if (cnt1 != sizeof(tSize))
		return cnt1;
	int cnt2 = 0;
	cnt2 += TransferData((void *)string, tSize, ZAF_IWRITE);
#endif
	return (cnt2 != (tSize * sizeof(string[0])) ? cnt2 : cnt1 + cnt2);
}

int ZafBufferObject::Write(void *writeBuffer, int writeSize, int writeLength)
{
	return (TransferData(writeBuffer, writeSize * writeLength, ZAF_IWRITE));
}

