//	Zinc Application Framework - CONVERT.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <zaf.hpp>
#include "c_storer.hpp"
#include "convert.hpp"
#include <z_prst.hpp>

// ----- ZafConvert ---------------------------------------------------------

#define BUFFER_READ_LENGTH 2048

class ZafConvert
{
public:
	ZafConvert(const ZafIChar *srcName, const ZafIChar *dstName);
	~ZafConvert(void);
	void ConvertBitmaps(void);
	void ConvertDeriveTable(void);
	void ConvertFile(void);
	void ConvertHelpContexts(void);
	void ConvertIcons(void);
	void ConvertLanguage(void);
	void ConvertLocale(void);
	void ConvertMessages(void);
	void ConvertWindows(void);
	void CopyDirectory(const ZafIChar *srcName, const ZafIChar *dstName,
		const ZafClassName className, ZafClassID classID,
		ZafPersistEntryType entryType);
	ZafError Error(void) const { return (error); }
	void Save(void);

private:
	ZafError error;
	OldStorageReadOnly *srcStorage;
	ZafStorage *dstStorage;
	ZafObjectPersistence *persist;
};

ZafConvert::ZafConvert(const ZafIChar *srcName, const ZafIChar *dstName) :
	error(ZAF_ERROR_NONE), srcStorage(0), dstStorage(0), persist(0)
{
	// Open the source file.
	if (srcName)
		srcStorage = new OldStorageReadOnly(srcName);
	if (!srcStorage || srcStorage->Error())
	{
		char *source = zafCodeSet->ConvertToOSString(srcName);
		printf("Invalid source file %s\n", srcName ? source : "<null>");
		delete []source;
		error = ZAF_ERROR_INVALID_SOURCE;
		return;
	}

	// Open the destination file.
	if (dstName)
		dstStorage = new ZafStorage(dstName, ZAF_FILE_OPENCREATE | ZAF_FILE_READWRITE);
	if (!dstStorage || dstStorage->Error())
	{
		char *dest = zafCodeSet->ConvertToOSString(dstName);
		printf("Invalid destination file %s\n", dstName ? dest : "<null>");
		delete []dest;
		error = ZAF_ERROR_INVALID_TARGET;
		return;
	}

	// Create the persist table.
	persist = new ZafObjectPersistence(dstStorage, zafDefaultDataConstructor, zafDefaultObjectConstructor);

	// Print status.
	if (!Error())
		printf("Starting file conversion...\n");
	
}

ZafConvert::~ZafConvert(void)
{
	// Clean up the variables.
	if (srcStorage)
		delete srcStorage;
	if (dstStorage)
		delete dstStorage;
	if (persist)
		delete persist;

	// Print status.
	if (!Error())
		printf("...Done!\n");
	
}

void ZafConvert::ConvertFile(void)
{
	// Convert the file.
	ConvertBitmaps();
	ConvertIcons();
	ConvertHelpContexts();
	ConvertMessages();
	ConvertLanguage();
	ConvertLocale();
	ConvertDeriveTable();
	ConvertWindows();
}

void ZafConvert::ConvertBitmaps(void)
{
	// Convert the bitmaps.
//	CopyDirectory("UI_BITMAP", ZafBitmapData::className,
//		ZafBitmapData::className, ZafBitmapData::classID,
//		ZAF_PERSIST_ROOT_DIRECTORY);
	ZafIChar *srcName = ZAF_ITEXT("UI_BITMAP");
	ZafIChar *dstName = ZafBitmapData::className;

	// Make sure the destination exists.
	dstStorage->ChDir(ZafFileSystem::rootDirectoryName);
	if (dstStorage->ChDir(dstName))
	{
		dstStorage->MkDir(dstName);
		dstStorage->ChDir(dstName);
	}

	// Copy the directory.
	printf("Copying %s to %s:\n", srcName, dstName);
	
	srcStorage->ChDir(ZafFileSystem::rootDirectoryName);
	if (srcStorage->ChDir(srcName))
		return;
	for (ZafIChar *entry = srcStorage->FindFirst(ZAF_ITEXT("*")); entry; entry = srcStorage->FindNext())
		if (strcmp(entry, ZafFileSystem::currentDirectoryName) &&
			strcmp(entry, ZafFileSystem::parentDirectoryName))
		{
			printf("\t%s\n", entry);

			OldStorageObjectReadOnly srcObject(*srcStorage, entry, 0);
			ZafImageStruct image;
			ZafInt16 value;
			srcObject >> value;
			image.width = value;
			srcObject >> value;
			image.height = value;
			image.staticArray = false;
			image.array = new ZafLogicalColor[image.width * image.height];
			srcObject.Read(image.array, sizeof(ZafLogicalColor), image.width * image.height);
			ZafBitmapData imageData(image);
			imageData.SetStringID(entry);
			imageData.Write(*persist);
			delete []image.array;
		}
}

void ZafConvert::ConvertDeriveTable(void)
{
	static ZafIChar srcName[] = ZAF_ITEXT("DERIVE_TABLE");
	static ZafIChar dstName[] = ZAF_ITEXT("ZafDerivedObject");

	// Make sure the destination exists.
	dstStorage->ChDir(ZafFileSystem::rootDirectoryName);
	dstStorage->MkDir(dstName);
	dstStorage->ChDir(dstName);

	// Convert the derive table entries.
	printf("Converting Derived Objects:\n");
	
	srcStorage->ChDir(ZafFileSystem::rootDirectoryName);
	OldStorageObjectReadOnly srcObject(*srcStorage, srcName, 0);

	ZafClassNameChar deriveClassName[ZAF_MAXNAMELEN];
	ZafClassID deriveClassID;
	ZafClassID baseClassID;
	while (srcObject.Read(deriveClassID) && deriveClassID != ID_END)
	{
		srcObject >> baseClassID >> deriveClassName;
		printf("\t%s\n", deriveClassName);
		

//		dstStorage->MkDir(deriveClassName);
//		dstStorage->ChDir(deriveClassName);
		ZafFile *dstFile = dstStorage->Open(deriveClassName, ZAF_FILE_CREATE | ZAF_FILE_READWRITE,
			ZAF_NULLP(ZafStringIDChar), deriveClassID);
		int noOfFields = 0;
		*dstFile << baseClassID << noOfFields;
		dstStorage->Close(dstFile);

//		dstStorage->ChDir(ZafFileSystem::parentDirectoryName);
	}
}

void ZafConvert::ConvertHelpContexts(void)
{
	// Convert the help contexts.
	ZafIChar title[256];
	ZafIChar *helpText;

	ZafIChar *srcName = ZAF_ITEXT("UI_HELP");
	ZafIChar *dstName = ZafHelpSystem::className;
	// Make sure the destination exists.
	dstStorage->ChDir(ZafFileSystem::rootDirectoryName);
	if (dstStorage->ChDir(dstName))
	{
		dstStorage->MkDir(dstName);
		dstStorage->ChDir(dstName);
	}

	// Copy the directory.
	printf("Copying %s to %s:\n", srcName, dstName);
	
	srcStorage->ChDir(ZafFileSystem::rootDirectoryName);
	if (srcStorage->ChDir(srcName))
		return;

	for (ZafIChar *entry = srcStorage->FindFirst(ZAF_ITEXT("*")); entry; entry = srcStorage->FindNext())
		if (strcmp(entry, ZafFileSystem::currentDirectoryName) &&
			strcmp(entry, ZafFileSystem::parentDirectoryName))
		{
			printf("\t%s\n", entry);

			OldStorageObjectReadOnly srcObject(*srcStorage, entry, 0);
			srcObject.Read(title, 256);
			dstStorage->MkDir(entry, title, ID_ZAF_HELP_DATA);
			dstStorage->ChDir(entry);

			// Build a table of help contexts for later use
			ZafElement *element = new ZafElement;
			element->SetNumberID(srcObject.classID);
			element->SetStringID(entry);
			ZafConvertWindow::helpList.Add(element);

			// Read in the data.
			ZafIChar *objectName = ZAF_ITEXT("objectData");
			ZafFile *dstFile = dstStorage->Open(objectName, ZAF_FILE_CREATE | ZAF_FILE_READWRITE);
			dstFile->Write(title);
			srcObject.Read(&helpText);
			dstFile->Write(helpText);
			delete []helpText;

			dstStorage->Close(dstFile);

			dstStorage->ChDir(ZafFileSystem::parentDirectoryName);
		}
}

void ZafConvert::ConvertIcons(void)
{
	// Convert the icons.
	ZafIChar *srcName = ZAF_ITEXT("UI_ICON");
	ZafIChar *dstName = ZafIconData::className;

	// Make sure the destination exists.
	dstStorage->ChDir(ZafFileSystem::rootDirectoryName);
	if (dstStorage->ChDir(dstName))
	{
		dstStorage->MkDir(dstName);
		dstStorage->ChDir(dstName);
	}

	// Copy the directory.
	printf("Copying %s to %s:\n", srcName, dstName);
	
	srcStorage->ChDir(ZafFileSystem::rootDirectoryName);
	if (srcStorage->ChDir(srcName))
		return;
	for (ZafIChar *entry = srcStorage->FindFirst(ZAF_ITEXT("*")); entry; entry = srcStorage->FindNext())
		if (strcmp(entry, ZafFileSystem::currentDirectoryName) &&
			strcmp(entry, ZafFileSystem::parentDirectoryName))
		{
			printf("\t%s\n", entry);

			OldStorageObjectReadOnly srcObject(*srcStorage, entry, 0);
			ZafImageStruct image;
			ZafInt16 value;
			srcObject >> value;
			image.width = value;
			srcObject >> value;
			image.height = value;
			image.staticArray = false;
			image.array = new ZafLogicalColor[image.width * image.height];
			srcObject.Read(image.array, sizeof(ZafLogicalColor), image.width * image.height);
			ZafIconData imageData(image);
			imageData.SetStringID(entry);
			imageData.Write(*persist);
			delete []image.array;
		}
}

void ZafConvert::ConvertLanguage(void)
{
	// Convert the language.
}

void ZafConvert::ConvertLocale(void)
{
	// Convert the locale.
}

void ZafConvert::ConvertMessages(void)
{
	static ZafIChar srcName[] = ZAF_ITEXT("UI_MESSAGE");
	static ZafIChar *dstName = ZafLanguageData::className;

	// Make sure the destination exists.
	dstStorage->ChDir(ZafFileSystem::rootDirectoryName);
	dstStorage->MkDir(dstName);
	dstStorage->ChDir(dstName);

	// Copy the directory.
	printf("Converting Messages:\n");
	
	srcStorage->ChDir(ZafFileSystem::rootDirectoryName);
	if (srcStorage->ChDir(srcName))
		return;
	for (ZafIChar *entry = srcStorage->FindFirst(ZAF_ITEXT("*")); entry; entry = srcStorage->FindNext())
		if (strcmp(entry, ZafFileSystem::currentDirectoryName) &&
			strcmp(entry, ZafFileSystem::parentDirectoryName))
		{
			printf("\t%s\n", entry);
			
			dstStorage->MkDir(entry, ZAF_NULLP(ZafStringIDChar), ZafLanguageData::classID);
			dstStorage->ChDir(entry);

			OldStorageObjectReadOnly srcObject(*srcStorage, entry, 0);
			ZafFile *dstFile = dstStorage->Open(ZAF_ITEXT("objectData"), ZAF_FILE_CREATE | ZAF_FILE_READWRITE);

			// Convert the entries.
			ZafInt16 noOfElements;
			srcObject >> noOfElements;
			int nOfElements = noOfElements;
			*dstFile << nOfElements;
			for (int i = 0; i < noOfElements; i++)
			{
				ZafLanguageStruct data;
				srcObject >> data.numberID >> &data.stringID >> &data.text;
				ZafConvertWindow::StripHotkey(data.text, data.hotKeyChar, data.hotKeyIndex);
				*dstFile << data.numberID << data.stringID << data.text << (ZafUInt16)data.hotKeyChar << data.hotKeyIndex;
				if (data.stringID)
					delete data.stringID;
				if (data.text)
					delete data.text;
			}

			dstStorage->Close(dstFile);
			dstStorage->ChDir(ZafFileSystem::parentDirectoryName);
		}
}

void ZafConvert::ConvertWindows(void)
{
	static ZafIChar srcName[] = ZAF_ITEXT("UIW_WINDOW");
	static ZafIChar *dstName = ZafWindow::className;

	// Make sure the destination exists.
	dstStorage->ChDir(ZafFileSystem::rootDirectoryName);
	dstStorage->MkDir(dstName);
	dstStorage->ChDir(dstName);

	// Convert the windows.
	printf("Converting Windows:\n");
	ZafWindowObject *window;
	
	srcStorage->ChDir(ZafFileSystem::rootDirectoryName);
	OldStorageDirectory *srcDirectory = srcStorage->OpenDir(srcName);
	if (!srcDirectory)
		return;
	for (OldDirectoryEntry *entry = srcDirectory->ReadDir(); entry; entry = srcDirectory->ReadDir())
		if (strcmp(entry->stringID, ZafFileSystem::currentDirectoryName) &&
			strcmp(entry->stringID, ZafFileSystem::parentDirectoryName))
		{
			printf("\t%s\n", entry->stringID);
			
			window = ZafConvertWindow::ConvertWindow(entry->stringID, srcStorage, persist);

			if (window)
			{
				window->Write(*persist);
				delete window;
			}
		}
	delete srcDirectory;
}

void ZafConvert::CopyDirectory(const ZafIChar *srcName,
	const ZafIChar *dstName, const ZafClassName,
	ZafClassID classID, ZafPersistEntryType entryType)
{
	// Make sure the destination exists.
	dstStorage->ChDir(ZafFileSystem::rootDirectoryName);
	if (dstStorage->ChDir(dstName))
	{
		dstStorage->MkDir(dstName);
		dstStorage->ChDir(dstName);
	}

	// Copy the directory.
	printf("Copying %s to %s:\n", srcName, dstName);
	
	srcStorage->ChDir(ZafFileSystem::rootDirectoryName);
	srcStorage->ChDir(srcName);
	for (ZafIChar *entry = srcStorage->FindFirst(ZAF_ITEXT("*")); entry; entry = srcStorage->FindNext())
		if (strcmp(entry, ZafFileSystem::currentDirectoryName) &&
			strcmp(entry, ZafFileSystem::parentDirectoryName))
		{
			printf("\t%s\n", entry);
			
			ZafIChar *objectName = entry;
			if (entryType == ZAF_PERSIST_ROOT_DIRECTORY)
			{
				dstStorage->MkDir(entry, ZAF_NULLP(ZafStringIDChar), classID);
				dstStorage->ChDir(entry);
				objectName = ZAF_ITEXT("objectData");
			}

			OldStorageObjectReadOnly srcObject(*srcStorage, entry, 0);
			ZafFile *dstFile = dstStorage->Open(objectName, ZAF_FILE_CREATE | ZAF_FILE_READWRITE);
			char buffer[BUFFER_READ_LENGTH];
			int length;
			while ((length = srcObject.Read(buffer, BUFFER_READ_LENGTH, 1)) > 0)
				dstFile->Write(buffer, length, 1);
			dstStorage->Close(dstFile);

			if (entryType == ZAF_PERSIST_ROOT_DIRECTORY)
				dstStorage->ChDir(ZafFileSystem::parentDirectoryName);
		}
}

void ZafConvert::Save(void)
{
	// Save the file.
	if (dstStorage)
		dstStorage->Save();
}

// ----- Main() -------------------------------------------------------------

int ZafApplication::Main(void)
{
	// Ensure main() is linked properly.
	LinkMain();

#if defined(ZAF_MSWINDOWS)
	freopen("convert.out", "wt", stdout);
#endif

	// Check the arguments
	if (argc < 3)
	{
		printf("Usage: convert <source> <destination>\n");
		
		return (0);
	}
	else if (!strcmp(argv[1], argv[2]))
	{
		printf("Cannot have <source> <destination> be same name.\n");
		
		return (0);
	}

	// Convert the file.
	ZafConvert convert(argv[1], argv[2]);
	if (!convert.Error())
	{
		convert.ConvertFile();
		convert.Save();
	}
#if defined(ZAF_MSWINDOWS)
	fclose(stdout);
#endif

	// Return success.
	return (0);
}

