//	Zinc Application Framework - W_WIN1.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

// Give ZafMSWindowsApp access to protected data.
#define OSWindow ZafMSWindowsApp
#include "w_app.hpp"
#include <z_button.hpp>
#include <z_ctype.hpp>
#include <z_htips.hpp>
#include <z_keymap.hpp>
#include <z_scrll2.hpp>
#include <z_win.hpp>


// ----- ZafWindow ---------------------------------------------------------

ZafEventMap ZAF_FARDATA ZafWindow::defaultEventMap[] =
{
	{ L_DOWN,	 	E_KEY, 		GRAY_DOWN_ARROW,	S_KEYDOWN },
	{ L_DOWN,	 	E_KEY, 		WHITE_DOWN_ARROW,	S_KEYDOWN },
	{ L_LEFT,	 	E_KEY, 		GRAY_LEFT_ARROW,	S_KEYDOWN },
	{ L_LEFT,	 	E_KEY, 		WHITE_LEFT_ARROW,	S_KEYDOWN },
	{ L_NEXT,	 	E_KEY, 		TAB,		   		S_KEYDOWN },
	{ L_PGDN, 		E_KEY, 		GRAY_PGDN, 			S_KEYDOWN },
	{ L_PGDN, 		E_KEY, 		WHITE_PGDN, 		S_KEYDOWN },
	{ L_PGUP, 		E_KEY, 		GRAY_PGUP, 			S_KEYDOWN },
	{ L_PGUP, 		E_KEY, 		WHITE_PGUP, 		S_KEYDOWN },
	{ L_PREVIOUS,	E_KEY, 		TAB,		   		S_KEYDOWN | S_SHIFT },
	{ L_RIGHT,	 	E_KEY, 		GRAY_RIGHT_ARROW,	S_KEYDOWN },
	{ L_RIGHT,	 	E_KEY, 		WHITE_RIGHT_ARROW,	S_KEYDOWN },
	{ L_UP,		 	E_KEY, 		GRAY_UP_ARROW,		S_KEYDOWN },
	{ L_UP,		 	E_KEY, 		WHITE_UP_ARROW,		S_KEYDOWN },
	{ L_NONE,		0,	  		0,	 				0 }
};

ZafPaletteMap ZAF_FARDATA ZafWindow::defaultPaletteMap[] =
{
	{ ZAF_PM_NONE, 0 }
};

ZafWindowObject *ZafWindow::Add(ZafWindowObject *object, ZafWindowObject *position)
{
	// Make sure add is allowed.
	if (object->parent)
		return (object);

	// Add the object to the list.
	if (object->SupportObject())
	{
		if (position)
			support.Add(object, position);	// position insert.
		else
			support.Add(object);			// compareFunction insert.
	}
	else
	{
		if (position)
			ZafList::Add(object, position);	// position insert.
		else
			ZafList::Add(object);			// compareFunction insert.
	}

	// Set object's parent.
	object->parent = this;

	// See if object should be the "current" object.
//	bool setCurrent = (object->Focus() ||
//		(!current && !object->Noncurrent() && !object->Disabled())) ?
//		true : false;
	bool setCurrent = object->Focus() ? true : false;

	// Clear the focus flag. (The object can't really have focus until it
	// is displayed in the window.)
	object->SetFocus(false);

	// See if the window (this) is already on the screen.
	if (screenID)
	{
		if (object->SupportObject())
		{
			if (object->SystemObject())
			{
				bool saveVisible = object->Visible();
				object->SetVisible(false);
				object->Event(S_CREATE);

				RECT rect;
				WINDOWPOS windowPos;
				OSWindowID frameID = OSScreenID(ZAF_FRAMEID);
				GetWindowRect(frameID, &rect);
				if (parent)
					MapWindowPoints(NULL, parent->screenID, (POINT *)&rect, 2);
				windowPos.hwnd = frameID;
				windowPos.x = rect.left;
				windowPos.y = rect.top;
				windowPos.cx = rect.right - rect.left;
				windowPos.cy = rect.bottom - rect.top;
				windowPos.flags = SWP_NOACTIVATE | SWP_NOMOVE | SWP_NOOWNERZORDER |
					SWP_NOREDRAW | SWP_NOSENDCHANGING | SWP_NOZORDER;

				SendMessage(frameID, WM_WINDOWPOSCHANGED, (WPARAM)0, (LPARAM)&windowPos);
				object->SetVisible(saveVisible);
			}
			else
			{
				//???
				Event(S_DESTROY);
				Event(S_CREATE);
			}
		}
		else
			// Create the object (make it appear in the window).
			object->Event(S_CREATE);
	}

	// Update the focus.
	if (setCurrent)
	{
		if (Focus())
			object->SetFocus(true);
		else
			ZafList::SetCurrent(object);
	}

	return (object);
}

ZafRegionStruct ZafWindow::BeginDraw(void)
{
	// Prepare the display for drawing.
	if (editMode && parent)
	{
//???
//		HDC hDC = GetDCEx(OSScreenID(ZAF_FRAMEID), NULL, DCX_PARENTCLIP);
		HRGN hRegion = CreateRectRgn(0, 0, 800, 600);
		HDC hDC = GetDCEx(OSScreenID(ZAF_FRAMEID), hRegion, DCX_PARENTCLIP);
		DeleteObject(hRegion);

		display->BeginDraw(hDC, screenID, zafRegion, zafRegion);

	}
	else
		display->BeginDraw(0, screenID, zafRegion, zafRegion);

	if (systemObject)
	{
		if (editMode)
		{
			ZafRegionStruct drawRegion = zafRegion;
			drawRegion.right -= drawRegion.left;
			drawRegion.bottom -= drawRegion.top;
			drawRegion.left = drawRegion.top = 0;
			return (drawRegion);
		}
		else
		{
			RECT clientRect;
			GetClientRect(OSScreenID(ZAF_FRAMEID), &clientRect);
			ZafRegionStruct drawRegion;
			drawRegion.ImportPoint(clientRect);
			return (drawRegion);
		}
	}
	else
		return zafRegion;
}

bool ZafWindow::ConvertToZafEvent(ZafEventStruct &event)
{
	if (event.InputType() == E_MOUSE && !EditMode())
	{
		if (event.osEvent.message >= WM_MOUSEFIRST && event.osEvent.message <= WM_MOUSELAST)
		{
			ZafWindowObject::ConvertToZafEvent(event);

			if (systemObject)
			{
				RECT rect;
//				POINT point;
				GetWindowRect(screenID, &rect);
//				point.x = rect.left;
//				point.y = rect.top;
//				ScreenToClient(screenID, &point);
				ScreenToClient(screenID, (POINT *)&rect);
//				event.position.column += point.x;
//				event.position.lint += point.y;
				event.position.column += rect.left;
				event.position.line += rect.top;
			}

			return true;
		}
		else
			// Don't convert non-client mouse messages.
			return false;
	}
	else
		return ZafWindowObject::ConvertToZafEvent(event);
};

ZafEventType ZafWindow::Draw(const ZafEventStruct &event, ZafEventType ccode)
{
	// See if a background ("wallpaper") image exists.
//??? Is there a better way to get the image?
	ZafWindowObject *backgroundImage;
	for (backgroundImage = SupportFirst(); backgroundImage; backgroundImage = backgroundImage->Next())
		if (backgroundImage->IsA(ID_ZAF_IMAGE))
			break;

	if (backgroundImage)
	{
		backgroundImage->SetRegion(clientRegion);
		ccode = backgroundImage->Draw(event, ccode);
	}
	else
	{
		// Fill the background.
		ZafRegionStruct drawRegion = BeginDraw();
		DrawBackground(drawRegion, ccode);
		EndDraw();
	}

	return ccode;
}

ZafEventType ZafWindow::Event(const ZafEventStruct &event)
{
	ZafEventType ccode = event.type;

	if (ccode == E_OSEVENT)
		return ZafWindow::OSEvent(event);

	// Process ZAF Events.
	switch (ccode)
	{
	case S_ADD_OBJECT:
		Add(event.windowObject);
		break;

	case S_DESTROY:
		if (!SystemObject())
		{
			ZafWindowObject::Event(ZafEventStruct(S_REDISPLAY));
			BroadcastEvent(event);
			ZafWindowObject::Event(ZafEventStruct(S_DEINITIALIZE));
		}
		else
			ccode = ZafWindowObject::Event(event);
		break;

	case S_DEINITIALIZE:
		// Check for a deleted help object.
		if (!parent && windowManager->helpObject && windowManager->helpObject->RootObject() == this)
		{
			windowManager->helpObject = ZAF_NULLP(ZafWindowManager);
			ZafEventStruct tEvent;
			tEvent.type = DH_SET_HELP_OBJECT;
			tEvent.windowObject = ZAF_NULLP(ZafWindowObject);
			eventManager->Event(tEvent, E_HELPTIPS);
		}

		// Broadcast to all of the children.
		BroadcastEvent(event);

		// Default to the base class.
		ccode = ZafWindowObject::Event(event);
		break;

	case S_HOT_KEY:
		{
		ccode = S_UNKNOWN;

		// If the window has focus then its children have already been
		// tested for hot keys.
		if (!Focus())
		{
			// Give all objects a chance to process the hot key.
			ZafWindowObject *hotKeyObject = SupportFirst();
			while (hotKeyObject && ccode == S_UNKNOWN)
			{
				ccode = hotKeyObject->Event(event);
				hotKeyObject = hotKeyObject->Next();
			}
			hotKeyObject = First();
			while (hotKeyObject && ccode == S_UNKNOWN)
			{
				ccode = hotKeyObject->Event(event);
				hotKeyObject = hotKeyObject->Next();
			}
		}
		}
		break;

	case S_INITIALIZE:
		// Default to the base class.
		ccode = ZafWindowObject::Event(event);

		//??? This should be handled in ConvertCoordinates.
		ConvertRegion(clientRegion, ZAF_PIXEL);

		// Broadcast to all of the children.
		BroadcastEvent(event);
		break;

	case S_MAXIMIZE:
		SetMaximized(true);
		break;

	case S_MINIMIZE:
		SetMinimized(true);
		break;

	case S_REDISPLAY_DATA:
		BroadcastEvent(event);
		break;

	case S_REGISTER_OBJECT:
		{
		// Register the object.
		ccode = ZafWindowObject::Event(event);

		// Register all of the children.
		BroadcastEvent(event);
		}
		break;

	case S_RESTORE:
		if (Maximized())
			SetMaximized(false);
		else if (Minimized())
			SetMinimized(false);
		break;

	case S_SIZE:
		if (screenID && SupportObject())
		{
			zafRegion = event.region;
			ConvertCoordinates(ZAF_PIXEL);

			//??? Duplicate code;
			RECT rect;
			WINDOWPOS windowPos;
			OSWindowID frameID = parent->OSScreenID(ZAF_FRAMEID);
			GetWindowRect(frameID, &rect);
			if (parent)
				MapWindowPoints(NULL, parent->screenID, (POINT *)&rect, 2);
			windowPos.hwnd = frameID;
			windowPos.x = rect.left;
			windowPos.y = rect.top;
			windowPos.cx = rect.right - rect.left;
			windowPos.cy = rect.bottom - rect.top;
			windowPos.flags = SWP_NOACTIVATE | SWP_NOMOVE | SWP_NOOWNERZORDER |
				SWP_NOREDRAW | SWP_NOSENDCHANGING | SWP_NOZORDER;
			SendMessage(frameID, WM_WINDOWPOSCHANGED, (WPARAM)0, (LPARAM)&windowPos);
		}
		else
			ccode = ZafWindowObject::Event(event);
		break;

	case S_SUBTRACT_OBJECT:
		Subtract(event.windowObject);
		break;

	// ----- Notification Events -----

	case N_MOVE:
		if (!systemObject)
		{
			// System object children must be manually moved.
			for (ZafWindowObject *object = First(); object; object = object->Next())
				if (object->SystemObject())
				{
					// Convert zafRegion to OS coordinates.
					ZafRegionStruct osRegion = ConvertToOSRegion(object);

					// Size/move the object with the OS.
					MoveWindow(object->OSScreenID(ZAF_FRAMEID), osRegion.left, osRegion.top, osRegion.Width(),
						osRegion.Height(), TRUE);
				}
				else
					// Trickle the N_MOVE in case there are system objects at
					// a lower level.
					object->Event(N_MOVE);
		}
		break;

	case N_SIZE:
		{
		//??? Is this necessary?
		if (systemObject)
		{
			UpdateWindow(screenID);

			//??? Is there a better way to get the image?
			ZafWindowObject *backgroundImage;
			for (backgroundImage = SupportFirst(); backgroundImage; backgroundImage = backgroundImage->Next())
				if (backgroundImage->IsA(ID_ZAF_IMAGE))
					break;
			if (backgroundImage)
				backgroundImage->Event(event);
		}

		// Let children adjust their size based on the new window size.
		ZafWindowObject *object;
		for (object = First(); object; object = object->Next())
			object->Event(ZafEventStruct(S_COMPUTE_SIZE));

		// Perform geometry management based on the new window size.
		ZafWindowObject *geometryManager = (ZafWindowObject *)GeometryManager();
		if (geometryManager)
//??? S_COMPUTE_GEOMETRY.
			geometryManager->Event(ZafEventStruct(S_COMPUTE_SIZE));

		// "Ripple" sizing operation to the next level.
		for (object = First(); object; object = object->Next())
			object->OSSize();
		}
		break;

 	case N_CLOSE:
		if (Locked())
			// Prevent the window from being closed.
			ccode = S_ERROR;
		else
			// Allow the window to be closed.
			ccode = 0;
		break;

	// ----- Scroll messages -----
	case S_VSCROLL:
	case N_VSCROLL:
	case S_VSCROLL_CHECK:
	case S_VSCROLL_COMPUTE:
		ccode = ScrollEvent(event);
		break;

	default:
		ccode = ZafWindowObject::Event(event);
		if (ccode == S_UNKNOWN && current)
			ccode = Current()->Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

//??? Why both horizontal and vertical weights?
ZafRegionStruct ZafWindow::MaxRegion(ZafWindowObject *object, ZafVtJustify vtJustify, ZafHzJustify)
{
	// If "object" is equal to "this," then the region to be occupied
	// by the client window is returned.
	if (object == this)
		object = ZAF_NULLP(ZafWindowObject);

	ZafRegionStruct region;
	ZafWindowObject *firstObject;
	if (!object || object->SupportObject())
	{
		// Object is inside the frame window.
		RECT clientRect;
		GetClientRect(OSScreenID(ZAF_FRAMEID), &clientRect);
		region.left = clientRect.left; region.top = clientRect.top;
		region.right = clientRect.right - 1; region.bottom = clientRect.bottom - 1;
		region.coordinateType = ZAF_PIXEL;
		firstObject = SupportFirst();
	}
	else
	{
		// Object is inside the client window.
		region = clientRegion;
		if (object->RegionType() != ZAF_AVAILABLE_REGION)
			return (region);
		firstObject = First();
	}

	// Reserve all regions occupied by objects prior to "object" in
	// the window's list.
	ZafRegionList regionList;
	regionList.Add(new ZafRegionElement(screenID, region));
	for (ZafWindowObject *element = firstObject; element && element != object;
		element = element->Next())
	{
		if (element->RegionType() == ZAF_AVAILABLE_REGION &&
			(element->SystemObject() || !element->SupportObject()))
				regionList.Split(screenID, element->zafRegion, true);
	}

	// If the available region in the window is non-rectangular, then 
	// the vertical justification determines whethor the top-most or the
	// bottom-most rectangular portion of the available region is returned.
	if (vtJustify == ZAF_VT_BOTTOM)
		region = regionList.First()->region;
	else
		region = regionList.Last()->region;

	return (region);
}

ZafWindowObject *ZafWindow::Subtract(ZafWindowObject *object)
{
	if (!object)
		return object;

	// See if object has been created.
	if (object->screenID)		   
		object->Event(ZafEventStruct(S_DESTROY));

	// Remove the object from the window's list.
	if (object->SupportObject())
	{
		support.Subtract(object);
		if (screenID)
		{
			if (object->SystemObject())
			{
				RECT rect;
				WINDOWPOS windowPos;
				OSWindowID frameID = OSScreenID(ZAF_FRAMEID);
				GetWindowRect(frameID, &rect);
				if (parent)
					MapWindowPoints(NULL, parent->screenID, (POINT *)&rect, 2);
				windowPos.hwnd = frameID;
				windowPos.x = rect.left;
				windowPos.y = rect.top;
				windowPos.cx = rect.right - rect.left;
				windowPos.cy = rect.bottom - rect.top;
				windowPos.flags = SWP_NOACTIVATE | SWP_NOMOVE | SWP_NOOWNERZORDER |
					SWP_NOREDRAW | SWP_NOSENDCHANGING | SWP_NOZORDER;

				SendMessage(frameID, WM_WINDOWPOSCHANGED, (WPARAM)0, (LPARAM)&windowPos);
			}
			else
			{
				//???
				Event(S_DESTROY);
				Event(S_CREATE);
			}
		}
	}
	else
		ZafList::Subtract(object);
	object->parent = ZAF_NULLP(ZafWindow);

	return (object);
}

// ----- Attributes ---------------------------------------------------------

bool ZafWindow::SetDestroyable(bool setDestroyable)
{
	destroyable = setDestroyable;
	return destroyable;
}

bool ZafWindow::SetDisabled(bool setDisabled)
{
	if (setDisabled != disabled)
	{
		ZafWindowObject::SetDisabled(setDisabled);
		Redisplay();
	}

	return disabled;
}

bool ZafWindow::SetFocus(bool setFocus)
{
//	if (screenID && SystemObject())
	if (screenID)
	{
		if (!current)
		{
			for (ZafWindowObject *object = First(); object; object = object->Next())
				if (!object->Disabled() && !object->Noncurrent())
				{
					SetCurrent(object);
					break;
				}
		}

		// Set the focus with the OS.
//		if (setFocus && current && Current()->SystemObject())
		if (setFocus && current)
			Current()->SetFocus(setFocus);
		else if (!setFocus && !parent)
			SetActiveWindow(GetDesktopWindow());
		else
			ZafWindowObject::SetFocus(setFocus);
	}
	else
		ZafWindowObject::SetFocus(setFocus);

	return (focus);
}

bool ZafWindow::SetMaximized(bool setMaximized)
{
	if (setMaximized != maximized)
	{
		// Maximize or restore the window.
		maximized = setMaximized;

		// Perform the operation with the OS.
		if (screenID)
			ShowWindow(OSScreenID(ZAF_FRAMEID), maximized ? SW_SHOWMAXIMIZED : SW_RESTORE);
	}

	return maximized;
}

bool ZafWindow::SetMinimized(bool setMinimized)
{
	if (setMinimized != minimized)
	{
		// Minimize or restore the window.
		minimized = setMinimized;

		// Perform the operation with the OS.
		if (screenID)
			ShowWindow(OSScreenID(ZAF_FRAMEID), minimized ? SW_SHOWMINIMIZED : SW_RESTORE);
	}

	return minimized;
}

ZafSelectionType ZafWindow::SetSelectionType(ZafSelectionType setSelectionType)
{
	if (selectionType != setSelectionType && !screenID)
	{
		selectionType = setSelectionType;

		if (selectionType == ZAF_SINGLE_SELECTION)
		{
			// If the current object is selected, let it remain selected.
			ZafWindowObject *selectedObject = (Current() && Current()->Selected()) ?
				Current() : ZAF_NULLP(ZafWindowObject);

			// Clear all selections except for one.
			for (ZafWindowObject *object = First(); object; object = object->Next())
				if (object->Selected())
				{
					if (!selectedObject)
						selectedObject = object;
					else if (object != selectedObject)
						object->SetSelected(false);
				}
		}
	}

	return selectionType;
}

// ----- OS Specific Functions ----------------------------------------------

void ZafWindow::OSDestroy(void)
{
	for (ZafWindowObject *object = First(); object; object = object->Next())
		object->Event(ZafEventStruct(S_DESTROY));
}

ZafEventType ZafWindow::OSEvent(const ZafEventStruct &event)
{
	// "top-down" processing.
	ZafWindowObject *routeObject = event.route;
	if (routeObject && routeObject != this)
	{
		// Find correct path to "event.route."
		while (routeObject && routeObject->parent != this)
			routeObject = routeObject->parent;
	}
	else
		routeObject = ZAF_NULLP(ZafWindowObject);

	ZafEventType ccode = routeObject ? routeObject->Event(event) : S_UNKNOWN;

	// Process default button selection.
	if (defaultButton && event.osEvent.message == WM_KEYDOWN &&
		event.osEvent.wParam == VK_RETURN &&
		(ccode == S_UNKNOWN || ccode == 0))
			defaultButton->Event(L_SELECT);

	if (ccode == S_UNKNOWN)
	{
		// Process hot keys.
		if (event.osEvent.message == WM_SYSKEYDOWN)
		{
			// Generate a hot-key event.
			ZafEventStruct hotKeyEvent = event;
			ConvertToZafEvent(hotKeyEvent);
			hotKeyEvent.type = S_HOT_KEY;
			hotKeyEvent.key.value = ToUpper(hotKeyEvent.key.value);

			// Make sure current object/window has first chance to
			// handle hot keys.
			if (current)
				ccode = Current()->Event(hotKeyEvent);

			// Give all objects a chance to process the hot key.
			ZafWindowObject *hotKeyObject = SupportFirst() ? SupportFirst() : First();
			while (ccode == S_UNKNOWN && hotKeyObject)
			{
				if (hotKeyObject != current)
					ccode = hotKeyObject->Event(hotKeyEvent);
				hotKeyObject = (hotKeyObject == SupportLast()) ?
					First() : hotKeyObject->Next();
			}

			// Return if a hot-key was processed.
			if (hotKeyObject)
				return (ccode);
			else if (ccode == S_UNKNOWN && !parent)
			{
				TranslateMessage(&event.osEvent);
				return (DefaultCallback(event));
			}
		}
		else if (!parent && event.osEvent.message == WM_SYSKEYUP)
		{
			TranslateMessage(&event.osEvent);
			return (DefaultCallback(event));
		}

		// Handle movement keys (tab / arrows).
		if (event.osEvent.message == WM_KEYDOWN)
		{
			int keyCode = (int)event.osEvent.wParam;
			switch (keyCode)
			{
			case VK_DOWN:
			case VK_LEFT:
			case VK_RIGHT:
			case VK_UP:
				if (Current())
				{
					// Move focus to the nearest object in the specified direction.
					ZafWindowObject *nextObject = ZAF_NULLP(ZafWindowObject);
					ZafWindowObject *wrapObject = ZAF_NULLP(ZafWindowObject);
					for (ZafWindowObject *object = First(); object; object = object->Next())
					{
						// Is it an object that can get focus?
						if (object == Current() || object->Disabled() ||
							object->Noncurrent())
								continue;

						// See if the object is the closest object in the specified direction.
						if (keyCode == VK_LEFT && !(object->zafRegion.top > Current()->zafRegion.bottom ||
							object->zafRegion.bottom < Current()->zafRegion.top))
						{
							if (object->zafRegion.right < Current()->zafRegion.right)
							{
								if (!nextObject || object->zafRegion.right > nextObject->zafRegion.right)
									nextObject = object;
							}
							else if (!wrapObject || object->zafRegion.right > wrapObject->zafRegion.right)
								wrapObject = object;
						}
						else if (keyCode == VK_RIGHT && !(object->zafRegion.top > Current()->zafRegion.bottom ||
							object->zafRegion.bottom < Current()->zafRegion.top))
						{
							if (object->zafRegion.left > Current()->zafRegion.left)
							{
								if (!nextObject || object->zafRegion.left < nextObject->zafRegion.left)
									nextObject = object;
							}
							else if (!wrapObject || object->zafRegion.left < wrapObject->zafRegion.left)
								wrapObject = object;
						}
						else if (keyCode == VK_UP && !(object->zafRegion.left > Current()->zafRegion.right ||
							object->zafRegion.right < Current()->zafRegion.left))
						{
							if (object->zafRegion.bottom < Current()->zafRegion.bottom)
							{
								if (!nextObject || object->zafRegion.bottom > nextObject->zafRegion.bottom)
									nextObject = object;
							}
							else if (!wrapObject || object->zafRegion.bottom > wrapObject->zafRegion.bottom)
								wrapObject = object;
						}
						else if (keyCode == VK_DOWN && !(object->zafRegion.left > Current()->zafRegion.right ||
							object->zafRegion.right < Current()->zafRegion.left))
						{
							if (object->zafRegion.top > Current()->zafRegion.top)
							{
								if (!nextObject || object->zafRegion.top < nextObject->zafRegion.top)
									nextObject = object;
							}
							else if (!wrapObject || object->zafRegion.top < wrapObject->zafRegion.top)
								wrapObject = object;
						}
					}

					if (nextObject || wrapObject)
					{
						(nextObject ? nextObject : wrapObject)->SetFocus(true);
						ccode = 0;
					}
					else
						ccode = S_UNKNOWN;
				}
				break;

			case VK_TAB:
				if (GetKeyState(VK_CONTROL) & 0x8000)
					ccode = S_UNKNOWN;
				else if (GetKeyState(VK_SHIFT) & 0x8000)
				{
					// Change the focus to the previous object.
					ZafWindowObject *prevObject = Current() ? Current()->Previous() : Last();
					while (prevObject != Current())
					{
						// Skip fields that can't become current.
						if (prevObject && !prevObject->Disabled() &&
							!prevObject->Noncurrent())
						{
							prevObject->SetFocus(true);
							ccode = 0;
							break;
						}

						if (prevObject)
							prevObject = prevObject->Previous();
						else
							prevObject = Last();
					}
				}
				else
				{
					// Change the focus to the next object.
					ZafWindowObject *nextObject = Current() ? Current()->Next() : First();
					while (nextObject != Current())
					{
						// Skip fields that can't become current.
						if (nextObject && !nextObject->Disabled() &&
							!nextObject->Noncurrent())
						{
							nextObject->SetFocus(true);
							ccode = 0;
							break;
						}

						if (nextObject)
							nextObject = nextObject->Next();
						else
							nextObject = First();
					}
				}
				break;
			}
		}
	}

	// Don't process OS events not directed to this object.
	if (routeObject)
		return ccode;

	// Handle OS events meant for this object.
	switch (event.osEvent.message)
	{
	case WM_COMMAND:
		{
#if defined(ZAF_WIN32)
		WORD wID = LOWORD(event.osEvent.wParam);
		HWND hwndCtrl = (HWND)event.osEvent.lParam;
#else
		WORD wID = event.osEvent.wParam;
		HWND hwndCtrl = (HWND)LOWORD(event.osEvent.lParam);
#endif
		// Send the message to the child object for processing.
		if (hwndCtrl)
			// Child object is a control.
			ccode = SendMessage(hwndCtrl, event.osEvent.message, event.osEvent.wParam, event.osEvent.lParam);
		else
		{
			// Child object is a menu item.
			ZafWindowObject *object = GetObject(wID);
			ccode = object->Event(event);
		}
		}
		break;

#if defined(ZAF_WIN32)
	case WM_CTLCOLORMSGBOX:
	case WM_CTLCOLOREDIT:
	case WM_CTLCOLORLISTBOX:
	case WM_CTLCOLORBTN:
	case WM_CTLCOLORDLG:
	case WM_CTLCOLORSCROLLBAR:
	case WM_CTLCOLORSTATIC:
		{
		HWND hwndChild = (HWND)event.osEvent.lParam;
#else
	case WM_CTLCOLOR:
		{
		HWND hwndChild = (HWND)LOWORD(event.osEvent.lParam);
#endif
		// Send the WM_CTLCOLOR message to the child object.
		// (ZAF objects handle their own WM_CTLCOLOR messages instead
		// of having their owners control the colors.)
		if (hwndChild == screenID)
			ccode = ZafWindowObject::OSEvent(event);
		else
			ccode = SendMessage(hwndChild, event.osEvent.message,
				event.osEvent.wParam, event.osEvent.lParam);
#if defined(ZAF_WIN32)
		}
#else
		}
#endif
		break;

 	case WM_DRAWITEM:
		{
		DRAWITEMSTRUCT *draw = (DRAWITEMSTRUCT *)event.osEvent.lParam;

		// Send the WM_DRAWITEM message to the object to be drawn.
		// (ZAF "items" draw themselves instead of having their
		// owners draw them.)
		ccode = SendMessage(draw->hwndItem, event.osEvent.message,
			event.osEvent.wParam, event.osEvent.lParam);
		}
		break;

	case WM_ERASEBKGND:
		if (OSDraw())
		{
			ZafWindowObject *backgroundImage;
			for (backgroundImage = SupportFirst(); backgroundImage; backgroundImage = backgroundImage->Next())
				if (backgroundImage->IsA(ID_ZAF_IMAGE))
					break;

			if (backgroundImage)
			{
				HDC hDC = (HDC)event.osEvent.wParam;
				display->BeginDraw(hDC, 0, zafRegion, zafRegion);
				backgroundImage->Draw(event, S_REDISPLAY);
				display->EndDraw();
				ccode = 0;
			}
			else
			{
				// Get the device context.
				HDC hDC = (HDC)event.osEvent.wParam;

				// Get the background palette.
				ZafPaletteStruct palette = LogicalPalette(ZAF_PM_BACKGROUND, PaletteState());

				// Get the OS colors from the background palette.
				COLORREF colorForeground = (palette.colorForeground == ZAF_CLR_DEFAULT) ?
					palette.osPalette.colorForeground :
					display->colorTable[palette.colorForeground];
				COLORREF colorBackground = (palette.colorBackground == ZAF_CLR_DEFAULT) ?
					palette.osPalette.colorBackground :
					display->colorTable[palette.colorBackground];

				// Set the OS colors.
				SelectPalette(hDC, display->logicalPalette, FALSE);
				::SetTextColor(hDC, colorForeground);
				SetBkColor(hDC, colorBackground);

				// Get the background region.
				RECT rect;
				GetClientRect(screenID, &rect);

				// Draw the background.
				HBRUSH hBrush = display->GetGdiCacheBrush(palette);
				HBRUSH hOldBrush = (HBRUSH)SelectObject(hDC, hBrush);
				PatBlt(hDC, rect.left, rect.top, rect.right - rect.left,
					rect.bottom - rect.top, PATCOPY);
				SelectObject(hDC, hOldBrush);
				display->ReleaseGdiCacheObject(hBrush);
				ccode = 1;
			}
		}
		else
			ccode = ZafWindowObject::Event(event);
		break;

	case WM_HSCROLL:
	case WM_VSCROLL:
		{
#if defined (ZAF_WIN32)
		HWND hwndScrollBar = (HWND)event.osEvent.lParam;
#else
		HWND hwndScrollBar = (HWND)HIWORD(event.osEvent.lParam);
#endif

		// Send the messages to the scroll bar for handling.
		if (hwndScrollBar)
			// Scroll bar is a child control.
			ccode = SendMessage(hwndScrollBar, event.osEvent.message,
				event.osEvent.wParam, event.osEvent.lParam);
		else
		{
			// Scroll bar is a support object.
			ZafWindowObject *hScroll = HorizontalScrollBar();
			ZafWindowObject *vScroll = VerticalScrollBar();
			if (event.osEvent.message == WM_HSCROLL && hScroll)
				ccode = hScroll->Event(event);
			else if (event.osEvent.message == WM_VSCROLL && vScroll)
				ccode = vScroll->Event(event);
			else
				ccode = DefaultCallback(event);
		}
		}
		break;

	case WM_MEASUREITEM:
		ccode = GetObject((ZafNumberID)((MEASUREITEMSTRUCT *)event.osEvent.lParam)->CtlID)->Event(event);
		break;

	case WM_NOTIFY:
		{
		NMHDR *notificationHeader = (NMHDR *)event.osEvent.lParam;

		// Send the notification to the child object.
		SendMessage(notificationHeader->hwndFrom, event.osEvent.message, event.osEvent.wParam, event.osEvent.lParam);

//??? Is this call needed?
//		ccode = DefaultCallback(event);
		}
		break;

	case WM_PAINT:
		if (OSDraw())
			ccode = DefaultCallback(event);
		else
			ccode = ZafWindowObject::OSEvent(event);
		break;

	case WM_PALETTECHANGED:
		{
		// Allow children to update/realize palettes.
		BroadcastEvent(event);

		// Realize the display's logical palette.
		HWND hwnd = OSScreenID(ZAF_FRAMEID);
		HDC hDC = GetDC(hwnd);
		SelectPalette(hDC, display->logicalPalette, TRUE);
		if (RealizePalette(hDC))
			InvalidateRect(screenID, NULL, TRUE);
		ReleaseDC(hwnd, hDC);
		}
		break;

	case WM_QUERYNEWPALETTE:
		{
		ZafEventStruct paletteEvent = event;

		// Give the current object the first chance to realize a palette.
		if (Current())
			paletteEvent.rawCode = (ZafRawCode)Current()->Event(paletteEvent);

		// Give all other children a chance to realize palettes.
		ZafWindowObject *object;
		for (object = First(); object; object = object->Next())
			if (object != current)
				paletteEvent.rawCode = (ZafRawCode)object->Event(paletteEvent);
		for (object = SupportFirst(); object; object = object->Next())
			if (object != current)
				paletteEvent.rawCode = (ZafRawCode)object->Event(paletteEvent);

		if (!parent)
		{
			// Realize the display's logical palette.
			HWND hwnd = OSScreenID(ZAF_FRAMEID);
			HDC hDC = GetDC(hwnd);
			SelectPalette(hDC, display->logicalPalette, (BOOL)paletteEvent.data);
			if (RealizePalette(hDC))
				InvalidateRect(screenID, NULL, TRUE);
			ReleaseDC(hwnd, hDC);
			ccode = TRUE;
		}
		else 
			ccode = (ZafEventType)paletteEvent.data;
		}
		break;

	case WM_WINDOWPOSCHANGED:
		{
		ccode = DefaultCallback(event);

		WINDOWPOS *windowPos = (WINDOWPOS *)event.osEvent.lParam;

		// Update zafRegion.
		if (event.osEvent.hwnd == OSScreenID(ZAF_FRAMEID) &&
			(windowPos->flags & (SWP_NOSIZE | SWP_NOMOVE)) !=
			(SWP_NOSIZE | SWP_NOMOVE))
		{
			// Get the OS region.
			ZafRegionStruct osRegion;
			osRegion.coordinateType = ZAF_PIXEL;
			osRegion.left = windowPos->x;
			osRegion.right = osRegion.left + windowPos->cx - 1;
			osRegion.top = windowPos->y;
			osRegion.bottom = osRegion.top + windowPos->cy - 1;

			// Update zafRegion by converting the OS region.
			zafRegion = parent ? parent->ConvertToZafRegion(this, &osRegion) :
				windowManager->ConvertToZafRegion(this, &osRegion);
		}

		// Update clientRegion.
		ZafRegionStruct oldClientRegion = clientRegion;
		RECT clientRect;
		GetClientRect(screenID, &clientRect);
		if (clientRect.right - clientRect.left > 0 &&
			clientRect.bottom - clientRect.top > 0)
			clientRegion.ImportPoint(clientRect);

		// Send Notifications.
		if (event.osEvent.hwnd == OSScreenID(ZAF_FRAMEID) &&
			!(windowPos->flags & SWP_NOMOVE))
		{
			ZafEventStruct notifyEvent(N_MOVE);
			notifyEvent.region = oldRegion;
			Event(notifyEvent);
		}

		if (!(windowPos->flags & SWP_NOSIZE) ||
			clientRegion.Width() != oldClientRegion.Width() ||
			clientRegion.Height() != oldClientRegion.Height())
		{
			ZafEventStruct notifyEvent(N_SIZE);
			notifyEvent.region = oldRegion;
			Event(notifyEvent);
		}
		}
		break;

	default:
		ccode = ZafWindowObject::OSEvent(event);
		if (ccode == S_UNKNOWN && current)
			ccode = Current()->Event(event);
		break;
	}

	return (ccode);
}

void ZafWindow::OSMapPalette(ZafPaletteStruct &palette, ZafPaletteType type, ZafPaletteState state)
{
	ZafWindowObject::OSMapPalette(palette, type, state);
}

void ZafWindow::OSRegisterObject(void)
{
	// Find decorations and set style bits accordingly.
	bool hasFrame = parent ? false : true;
	DWORD dwFrameStyle = WS_CLIPSIBLINGS | WS_CLIPCHILDREN;
	DWORD dwFrameExStyle = 0;
	DWORD dwClientStyle = WS_VISIBLE | WS_CHILD;
	for (ZafWindowObject *object = SupportFirst(); object; object = object->Next())
	{
		if (object->SystemObject())
			hasFrame = true;
		else if (object->NumberID() == ZAF_NUMID_BORDER)
		{
			if (Sizeable())
				dwFrameStyle |= WS_SIZEBOX;
			else
				dwFrameStyle |= WS_BORDER;
		}
		else if (object->NumberID() == ZAF_NUMID_MAXIMIZE)
			dwFrameStyle |= WS_MAXIMIZEBOX;
		else if (object->NumberID() == ZAF_NUMID_MINIMIZE)
			dwFrameStyle |= WS_MINIMIZEBOX;
		else if (object->NumberID() == ZAF_NUMID_TITLE)
			dwFrameStyle |= WS_CAPTION;
		else if (object->NumberID() == ZAF_NUMID_HZ_SCROLL)
			dwClientStyle |= WS_HSCROLL;
		else if (object->NumberID() == ZAF_NUMID_VT_SCROLL)
			dwClientStyle |= WS_VSCROLL;
		else if (object->NumberID() == ZAF_NUMID_SYSTEM)
			dwFrameStyle |= WS_SYSMENU;
	}

	if (Bordered() && !(dwFrameStyle & (WS_SIZEBOX | WS_BORDER)) &&
		(SupportFirst() || OSDraw()))
	{
#if defined(ZAF_MSWINDOWS_3D)
		if (parent)
			dwFrameExStyle |= WS_EX_CLIENTEDGE;
		else
			dwFrameStyle |= WS_BORDER;
#else
		dwFrameStyle |= WS_BORDER;
#endif
	}

	if (!parent && !(dwFrameStyle & WS_CAPTION))
		dwFrameStyle |= WS_POPUP;

	if (visible)
		dwFrameStyle |= WS_VISIBLE; 
	if (parent)
		dwFrameStyle |= WS_CHILD;

	if (hasFrame)
	{
		// Create the frame window.
		OSWindowID frameID = ZafMSWindowsApp::CreateSubclassedWindow(this, NULL, dwFrameStyle, ZAF_NULLP(ZafIChar), NULL, dwFrameExStyle);

		// Re-direct the frame window's "window procedure."
		SetWindowLong(frameID, GWL_WNDPROC, (LONG)ZafMSWindowsApp::FrameJumpProc);

		// Create the client window.
		screenID = ZafMSWindowsApp::CreateSubclassedWindow(this, NULL, dwClientStyle, ZAF_NULLP(ZafIChar), frameID);
	}
	else
		// Create the client window.
		screenID = ZafMSWindowsApp::CreateSubclassedWindow(this, NULL, dwFrameStyle | dwClientStyle, ZAF_NULLP(ZafIChar), NULL, dwFrameExStyle);
}

OSWindowID ZafWindow::OSScreenID(ZafScreenIDType type) const
{
	if (type == ZAF_FRAMEID && screenID && systemObject)
	{
		OSWindowID frameID = GetParent(screenID);
		if (frameID && (!parent || frameID != (supportObject ?
			parent->OSScreenID(ZAF_FRAMEID) : parent->screenID)))
				return (frameID);
	}
	return (screenID);
}

// ----- Frame Window -------------------------------------------------------

LRESULT CallDefWindowProc(HWND hwnd, UINT wMsg, WPARAM wParam, LPARAM lParam)
{
	return CallWindowProc((WINDOWSPROC)DefWindowProc, hwnd, wMsg, wParam, lParam);
}

LRESULT CALLBACK ZafMSWindowsApp::FrameJumpProc(HWND hwnd, UINT wMsg, WPARAM wParam, LPARAM lParam)
{
	return FrameProc(CallDefWindowProc, hwnd, wMsg, wParam, lParam);
}

LRESULT ZafMSWindowsApp::FrameProc(LRESULT (DefaultProc)(HWND, UINT, WPARAM, LPARAM),
	HWND hwnd, UINT wMsg, WPARAM wParam, LPARAM lParam)
{
	// Get a pointer to the Zinc window.
	ZafWindow *window = DynamicPtrCast(ObjectFromHandle(hwnd), ZafWindow);

	// Create a ZAF event.
  	ZafEventStruct event(E_MSWINDOWS);
	event.osEvent.hwnd = hwnd;
	event.osEvent.message = wMsg;
	event.osEvent.wParam = wParam;
	event.osEvent.lParam = lParam;

	// Process the windows message.
	switch (wMsg)
	{
	case WM_KEYDOWN:
	case WM_KEYUP:
	case WM_SYSKEYDOWN:
	case WM_SYSKEYUP:
		TranslateMessage(&event.osEvent);
		break;

	case WM_ACTIVATE:
		if ((BOOL)wParam)
			window->SetFocus(true);
		return 0;

	case WM_ACTIVATEAPP:
//??? Should the window manager gain/lose focus when the application becomes active/inactive?
		if (!(BOOL)wParam)
		{
			// Application is losing focus.
			if (window->Focus())
				window->NotifyFocus(ZAF_NULLP(ZafWindowObject), false);
		}
		break;

	case WM_CLOSE:
		{
//??? An S_CLOSE_WINDOW event is needed.
		// Prevent the Windows close operation and post an S_CLOSE event.
		ZafEventStruct closeEvent(S_CLOSE);
		closeEvent.windowObject = window;
		window->eventManager->Put(closeEvent);
		}
		return (0);
 
	case WM_COMMAND:
		return (SendMessage(window->screenID, wMsg, wParam, lParam));

	case WM_MENUSELECT:
		{
#if defined(ZAF_WIN32)
		UINT flags = (UINT)HIWORD(wParam);
#else
		WORD flags = LOWORD(lParam);
#endif
		ZafWindowObject *pullDownMenu = (ZafWindowObject *)window->PullDownMenu();
		ZafWindowObject *systemButton = (ZafWindowObject *)window->SystemButton();

		if (flags == 0xFFFF)
		{
			if (pullDownMenu && pullDownMenu->Focus())
				pullDownMenu->Event(ZafEventStruct(S_NON_CURRENT));
			else if (systemButton && systemButton->Focus())
				systemButton->Event(ZafEventStruct(S_NON_CURRENT));
		}
		else if (flags & MF_HILITE)
		{
			if (flags & MF_SYSMENU)
				systemButton->Event(event);
			else if (pullDownMenu)
				pullDownMenu->Event(event);
		}
		}
		return 0;

	case WM_NCDESTROY:
		// Reset the original window procedure.
		SetWindowLong(hwnd, GWL_WNDPROC, (LONG)DefWindowProc);

#if !defined(ZAF_WIN32)
		// Remove the window properties.
		RemoveProp(hwnd, (LPCSTR)ZafMSWindowsApp::objectLowWord);
		RemoveProp(hwnd, (LPCSTR)ZafMSWindowsApp::objectHighWord);
#endif
		break;

//??? should this be handled by system button.
// *** ZafSystemButton
/* START BLOCK COMMENT
**		case WM_INITMENUPOPUP:
**			// Check the system-button for invalid options.
**			{
**	MessageBeep(0);
**			HMENU nmenu = (HMENU)wParam;
**	//		if (nmenu != menu && HIWORD(lParam))
**			{
**	//			EnableMenuItem(nmenu, SC_RESTORE, MF_BYCOMMAND |
**	//				(!FlagSet(woStatus, WOS_MAXIMIZED | WOS_MINIMIZED) ?
**	//				MF_GRAYED : MF_ENABLED));
**	//			EnableMenuItem(nmenu, SC_MOVE, MF_BYCOMMAND |
**	//				((FlagSet(woAdvancedFlags, WOAF_NO_MOVE) || FlagSet(woStatus, WOS_MAXIMIZED)) ?
**	//				MF_GRAYED : MF_ENABLED));
**	//			EnableMenuItem(nmenu, SC_SIZE, MF_BYCOMMAND |
**	//				((FlagSet(woAdvancedFlags, WOAF_NO_SIZE) || FlagSet(woStatus, WOS_MAXIMIZED | WOS_MINIMIZED)) ?
**	//				MF_GRAYED : MF_ENABLED));
**				EnableMenuItem(nmenu, SC_CLOSE, MF_BYCOMMAND |
**					(window->Locked()) ? MF_GRAYED : MF_ENABLED);
**			}
**			}
**			break;
END BLOCK COMMENT */

	case WM_PALETTECHANGED:
 		if ((HWND)wParam == hwnd)
			return (0);
		// else continue.
	case WM_QUERYNEWPALETTE:
		{
		event.rawCode = (ZafRawCode)(wMsg == WM_PALETTECHANGED) ? TRUE : FALSE;
		return window->Event(event);
		}

	case WM_QUERYDRAGICON:
//??? Should test for minimize icon.
		return (0);

	case WM_SIZE:
		//??? Should always have a window?
		if (window)
		{
			WORD fwSizeType = wParam;
			window->minimized = window->maximized = false;
			if (fwSizeType == SIZE_MAXIMIZED)
				window->maximized = true;
			else if (fwSizeType == SIZE_MINIMIZED)
				window->minimized = true;
		}
		break;

	case WM_SYSCOMMAND:
		{
		WORD command = wParam & 0xFFF0;

		switch (command)
		{
		case SC_MOVE:
			if (!window->Moveable())
				return (FALSE);
			break;

		case SC_SIZE:
			if (!window->Sizeable())
				return (FALSE);
			break;
		}
		}		
		break;

	case WM_WINDOWPOSCHANGED:
		{
		WINDOWPOS *windowPos = (WINDOWPOS *)event.osEvent.lParam;

		// See if the object has been moved or sized.
		if ((windowPos->flags & (SWP_NOSIZE | SWP_NOMOVE)) != (SWP_NOSIZE | SWP_NOMOVE))
		{
			// Get the OS region.
			ZafRegionStruct osRegion;
			osRegion.left = windowPos->x;
			osRegion.right = osRegion.left + windowPos->cx - 1;
			osRegion.top = windowPos->y;
			osRegion.bottom = osRegion.top + windowPos->cy - 1;

			// Update zafRegion by converting the OS region.
			window->zafRegion = window->parent ? window->parent->ConvertToZafRegion(window, &osRegion) :
				window->windowManager->ConvertToZafRegion(window, &osRegion);

			// Send move notification.
			if (!(windowPos->flags & SWP_NOMOVE))
			{
				ZafEventStruct notifyEvent(N_MOVE);
				notifyEvent.region = window->oldRegion;
				window->Event(notifyEvent);
			}

			// See if the window's size has changed.
			if (!(windowPos->flags & SWP_NOSIZE))
			{
				// Resize the support objects.
				for (ZafWindowObject *object = window->SupportFirst(); object; object = object->Next())
					if (object->SystemObject())
					{
						object->Event(ZafEventStruct(S_COMPUTE_SIZE));
						object->OSSize();
					}

				// Resize the client window.
				ZafRegionStruct client = window->MaxRegion(window);
  				MoveWindow(window->screenID, client.left, client.top,
  					client.Width(), client.Height(), TRUE);
			}
		}

//??? Dont wan't to call DefaultProc for MDI.
//???	return DefaultProc(hwnd, wMsg, wParam, lParam);
		return (0);
		}
	}

	return DefaultProc(hwnd, wMsg, wParam, lParam);
}

// ----- Stubs --------------------------------------------------------------

ZafPositionStruct ZafWindow::ConvertToOSPosition(const ZafWindowObject *object,
	const ZafPositionStruct *position) const
{
	return (ZafWindowObject::ConvertToOSPosition(object, position));
}

ZafPositionStruct ZafWindow::ConvertToZafPosition(const ZafWindowObject *object,
	const ZafPositionStruct *position) const
{
	return (ZafWindowObject::ConvertToZafPosition(object, position));
}

ZafEventType ZafWindow::DragDropEvent(const ZafEventStruct &event)
{
//??? How does window handle drag-drop...ZafText must work also.
	return ZafWindowObject::DragDropEvent(event);
}

ZafEventType ZafWindow::DrawFocus(ZafRegionStruct &region, ZafEventType ccode)
{
	return (ZafWindowObject::DrawFocus(region, ccode));
}

void ZafWindow::GetClip(ZafWindowObject *, ZafRegionStruct &, ZafRegionStruct &)
{
	// Handled automatically by Windows.
}

ZafEventType ZafWindow::Modify(const ZafEventStruct &event)
{
	return (ZafWindowObject::Modify(event));
}

void ZafWindow::OSSize(void)
{
	if (systemObject)
	{
		if (parent)
			ZafWindowObject::OSSize();
		else
		{
			int targetClientWidth = clientRegion.Width();
			int targetClientHeight = clientRegion.Height();

			ZafWindowObject::OSSize();

			int clientWidth = clientRegion.Width();
			int clientHeight = clientRegion.Height();

			// Set the client region size.
			while (clientWidth != targetClientWidth || clientHeight != targetClientHeight)
			{
				zafRegion.right += targetClientWidth - clientWidth;
				zafRegion.bottom += targetClientHeight - clientHeight;

				ZafWindowObject::OSSize();

				if (clientRegion.Width() == clientWidth && clientRegion.Height() == clientHeight)
				{
					RECT rect;
					GetWindowRect(screenID, &rect);
					zafRegion.ImportPoint(rect);
					return;
				}
				else
				{
					clientWidth = clientRegion.Width();
					clientHeight = clientRegion.Height();
				}
			}
		}
	}
	else
	{
		clientRegion.coordinateType = ZAF_PIXEL;
		clientRegion.left = clientRegion.top = 0;
		clientRegion.right = zafRegion.Width() - 1;
		clientRegion.bottom = zafRegion.Height() - 1;
		ZafWindowObject::OSSize();
	}
}

ZafError ZafWindow::OSUpdatePalettes(ZafPaletteType zafTypes, ZafPaletteType osTypes)
{
	if (screenID)
	{
		SendMessage(screenID, WM_SETREDRAW, (WPARAM)FALSE, 0);
		ZafWindowObject *object;
		for (object = SupportFirst(); object; object = object->Next())
			if (object->SystemObject())
				object->OSUpdatePalettes(zafTypes, osTypes);
		for (object = First(); object; object = object->Next())
			if (object->SystemObject())
				object->OSUpdatePalettes(zafTypes, osTypes);
		ZafWindowObject::OSUpdatePalettes(zafTypes, osTypes);
	}

	return (ZAF_ERROR_NONE);
}

ZafEventType ZafWindow::ScrollEvent(const ZafEventStruct &event)
{
	return (event.type);
}

bool ZafWindow::SetAutomaticUpdate(bool tAutomaticUpdate)
{
	return ZafWindowObject::SetAutomaticUpdate(tAutomaticUpdate);
}

bool ZafWindow::SetVisible(bool tVisible)
{
	return (ZafWindowObject::SetVisible(tVisible));
}
