//	Zinc Application Framework - W_VLIST.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include "w_app.hpp"
#include <z_keymap.hpp>
#include <z_str1.hpp>
#include <z_vlist.hpp>

// ----- ZafVtList --------------------------------------------------------

ZafEventMap ZAF_FARDATA ZafVtList::defaultEventMap[] =
{
	{ L_FIRST,		E_KEY,	 	GRAY_HOME,	  		S_KEYDOWN },
	{ L_FIRST,		E_KEY,	 	WHITE_HOME,	  		S_KEYDOWN },
	{ L_PGDN, 		E_KEY, 		GRAY_PGDN,			S_KEYDOWN },
	{ L_PGDN, 		E_KEY, 		WHITE_PGDN,			S_KEYDOWN },
	{ L_PGUP, 		E_KEY, 		GRAY_PGUP, 			S_KEYDOWN },
	{ L_PGUP, 		E_KEY, 		WHITE_PGUP,			S_KEYDOWN },
	{ L_SELECT,	 	E_KEY, 		SPACE,				S_KEYDOWN },
	{ L_NONE,		0,			0,					0 }
};

ZafPaletteMap ZAF_FARDATA ZafVtList::defaultPaletteMap[] =
{
	{ ZAF_PM_TEXT, ZAF_PM_SELECTED, { ZAF_LINE_SOLID, ZAF_PTN_SOLID_FILL, ZAF_CLR_DEFAULT, ZAF_CLR_DEFAULT, ZAF_MONO_BLACK, ZAF_MONO_WHITE, ZAF_FNT_DIALOG } },
	{ ZAF_PM_BACKGROUND, ZAF_PM_ANY_STATE, { ZAF_LINE_SOLID, ZAF_PTN_SOLID_FILL, ZAF_CLR_DEFAULT, ZAF_CLR_DEFAULT, ZAF_MONO_BLACK, ZAF_MONO_WHITE, ZAF_FNT_SYSTEM } },
	{ ZAF_PM_FOCUS, ZAF_PM_ANY_STATE, { ZAF_LINE_SOLID, ZAF_PTN_SOLID_FILL, ZAF_CLR_WHITE, ZAF_CLR_BLACK, ZAF_MONO_WHITE, ZAF_MONO_BLACK, ZAF_FNT_SYSTEM } },
	{ ZAF_PM_NONE, 0 }
};

ZafWindowObject *ZafVtList::Add(ZafWindowObject *object, ZafWindowObject *position)
{
	if (object->parent)
		return (object);

	object->SetSystemObject(false);
	object->SetParentDrawBorder(true);
	object->SetParentDrawFocus(true);
	object->SetParentPalette(true);

	// Add the object.
	ZafWindow::Add(object, position);

	// Insert the list item (if the list is already on the screen).
	if (screenID)
		SendMessage(screenID, LB_INSERTSTRING, (WPARAM)Index(object), OSDraw() ? (LPARAM)object->Text() : (LPARAM)object);

	// Update the list selection.
	if (object->Selected())
		NotifySelection(object, true);

	return object;
}

ZafEventType ZafVtList::DrawBorder(ZafRegionStruct &, ZafEventType ccode)
{
	return ccode;
}

ZafEventType ZafVtList::Draw(const ZafEventStruct &, ZafEventType ccode)
{
	// List is always drawn by windows.
	// (Ownerdraw means children draw themselves insetead of being drawn by
	// the list.)
	return ccode;
}

ZafEventType ZafVtList::DragDropEvent(const ZafEventStruct &event)
{
	ZafEventType ccode = event.type;

	switch (ccode)
	{
	case S_DROP_DEFAULT:
		if (windowManager->dragObject->MoveDraggable())
			ccode = S_DROP_MOVE;
		else if (windowManager->dragObject->CopyDraggable())
			ccode = S_DROP_COPY;
		else
		{
			ccode = S_ERROR;
			break;
		}
		//Continue.
	case S_DROP_COPY:
	case S_DROP_MOVE:
		if (acceptDrop && this != windowManager->dragObject)
		{
			if (windowManager->dragObject->IsA(ID_ZAF_LIST))
			{
				// Move the selected objects.
				ZafWindow *dragList = DynamicPtrCast(windowManager->dragObject, ZafWindow);
				ZafWindowObject *object = dragList->First();
				while (object)
				{
					ZafWindowObject *nextObject = object->Next();
					if (object->Selected())
					{
						if (ccode == S_DROP_COPY)
							Add(object->Duplicate());
						else
						{
							dragList->Subtract(object);
							Add(object);
						}
					}
					object = nextObject;
				}
			}
			else
			{
				const ZafIChar *dropText = windowManager->dragObject->Text();
				if (dropText && *dropText)
				{
					ZafString *newString = new ZafString(0, 0, 0, new ZafStringData(dropText));
					newString->SetBordered(false);
					Add(newString);
					if (ccode != S_DROP_COPY)
						windowManager->dragObject->SetText(ZafLanguageData::blankString);
				}
				else
					ccode = S_ERROR;
			}
		}
		else
			ccode = S_ERROR;
		break;

	default:
		ccode = ZafWindowObject::DragDropEvent(event);
		break;
	}

	return (ccode);
}

ZafEventType ZafVtList::Event(const ZafEventStruct &event)
{
	ZafEventType ccode = event.type;

	if (ccode == E_OSEVENT)
		return ZafVtList::OSEvent(event);

	// Switch on the event type.
	switch (ccode)
	{
	case S_HOT_KEY:
		ccode = S_UNKNOWN;
		break;

	case S_REDISPLAY_DATA:
		ZafWindowObject::Event(ZafEventStruct(S_REDISPLAY));
		break;

	case S_REGISTER_OBJECT:
		{
		ccode = ZafWindow::Event(event);

		// Add the items to the list.
		int index = 0;
		for (ZafWindowObject *object = First(); object; object = object->Next())
		{
			SendMessage(screenID, LB_INSERTSTRING, (WPARAM)-1, OSDraw() ? (LPARAM)object->Text() : (LPARAM)object);
			if (object->Selected())
			{
				if (SelectionType() != ZAF_SINGLE_SELECTION)
					SendMessage(screenID, LB_SETSEL, (WPARAM)TRUE, (LPARAM)index);
				else
				{
					ZafList::SetCurrent(object);
					SendMessage(screenID, LB_SETCURSEL, (WPARAM)index, (LPARAM)0);
				}
			}
			index++;
		}

		// Set the current item in the list.
		if (SelectionType() != ZAF_SINGLE_SELECTION)
		{
			if (!current && first)
				ZafList::SetCurrent(first);
			if (current)
				SendMessage(screenID, LB_SETCARETINDEX, (WPARAM)Index(current), (LPARAM)0);
		}
		}
		break;

	case N_SIZE:
		break;

	default:
		ccode = ZafWindow::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

ZafWindowObject *ZafVtList::NotifyFocus(ZafWindowObject *object, bool setFocus)
{
	if (systemObject)
	{
		ZafWindowObject *invalidObject = ZafWindow::NotifyFocus(object, setFocus);
		if (object != this && setFocus && !invalidObject)
		{
			int index = Index(object);
			if (SendMessage(screenID, LB_GETCARETINDEX, (WPARAM)0, (LPARAM)0) != index)
				SendMessage(screenID, LB_SETCARETINDEX, (WPARAM)index, (LPARAM)0);
		}

		return invalidObject;
	}
	else
		// Let the combo box process the focus.
		return parent->NotifyFocus(object, setFocus);
}

ZafWindowObject *ZafVtList::NotifySelection(ZafWindowObject *object, bool setSelected)
{
	ZafWindow::NotifySelection(object, setSelected);
	if (screenID && object->Selected() == setSelected)
	{
		if (SystemObject())
		{
			int index = Index(object);
			if ((SendMessage(screenID, LB_GETSEL, (WPARAM)index, (LPARAM)0) ? true : false) != setSelected)
			{
				if (SelectionType() != ZAF_SINGLE_SELECTION)
				{
					SendMessage(screenID, LB_SETSEL, (WPARAM)setSelected, (LPARAM)index);
//??? LB_SETSEL makes the list item current.
//???					SendMessage(screenID, LB_SETCARETINDEX, (WPARAM)Index(current), (LPARAM)0);
				}
				else if (setSelected)
					SendMessage(screenID, LB_SETCURSEL, (WPARAM)index, (LPARAM)0);
			}
		}
		else
			parent->NotifySelection(object, setSelected);
	}

	return object;
}

ZafWindowObject *ZafVtList::Subtract(ZafWindowObject *object)
{
	if (systemObject && !object->SupportObject())
		SendMessage(screenID, LB_DELETESTRING, (WPARAM)Index(object), (LPARAM)0);
	return ZafWindow::Subtract(object);
}

// ----- OS Specific Functions ----------------------------------------------

void ZafVtList::OSDestroy(void)
{
	SendMessage(screenID, LB_RESETCONTENT, 0, 0);
	for (ZafWindowObject *object = First(); object; object = object->Next())
		object->Event(S_DEINITIALIZE);
}

ZafEventType ZafVtList::OSEvent(const ZafEventStruct &event)
{
	ZafEventType ccode = event.type;

	switch (event.osEvent.message)
	{
	case WM_COMMAND:
		{
//???		ccode = DefaultCallback(event);

#if defined(ZAF_WIN32)
		WORD wNotifyCode = HIWORD(event.osEvent.wParam);
#else
		WORD wNotifyCode = HIWORD(event.osEvent.lParam);
#endif
		if (wNotifyCode == LBN_SELCHANGE)
		{
			// Get index of the current item.
			int currentIndex = (int)SendMessage(screenID, LB_GETCARETINDEX, 0, 0);

			// Update the selected and current items.
			if (SelectionType() != ZAF_SINGLE_SELECTION)
			{
				// Build an array of selected item indexes.
				int selectedItems = (int)SendMessage(screenID, LB_GETSELCOUNT, 0, 0);
				int *selectionArray = new int[selectedItems];
				SendMessage(screenID, LB_GETSELITEMS, (WPARAM)selectedItems, (LPARAM)selectionArray);

				int index = 0;
				int selectionIndex = 0;
				for (ZafWindowObject *object = First(); object; object = object->Next())
				{
					// Update current list item.
					if (index == currentIndex && object != current)
						object->NotifyFocus(object, true);

					// Update list items selections.
					if (!object->Disabled())
					{
						if (selectionIndex < selectedItems && index == selectionArray[selectionIndex])
						{
							if (!object->Selected())
							{
								// Select the item.
								object->SetSelected(true);
								(object->*(object->memberUserFunction))(event, L_SELECT);
							}
						}
						else
						{
							if (object->Selected())
							{
								// Un-select the item.
								object->SetSelected(false);
								(object->*(object->memberUserFunction))(event, L_SELECT);
							}
						}
					}

					// Increment the indexes.
					if (selectionIndex < selectedItems && selectionArray[selectionIndex] <= index)
						selectionIndex++;
					index++;

					// Stop if finished.
					if (index > currentIndex && selectionIndex > selectedItems)
						break;
				}

				// Clean up.
				delete []selectionArray;
			}
			else
			{
				// Get the new current/selected object.
				ZafWindowObject *object = DynamicPtrCast(ZafList::Get(currentIndex), ZafWindowObject);

				// Set the focus and selection.
				if (object)
				{
					object->NotifyFocus(object, true);
					object->SetSelected(true);
					(object->*(object->memberUserFunction))(event, L_SELECT);
				}
			}
		}
		}
		break;

#if defined(ZAF_WIN32)
	case WM_CTLCOLORLISTBOX:
#else
	case WM_CTLCOLOR:
#endif
		{
		// Get the palettes.
		ZafPaletteState state = PaletteState();
		ZafPaletteStruct textPalette = LogicalPalette(ZAF_PM_TEXT, state);
		ZafPaletteStruct backgroundPalette = LogicalPalette(ZAF_PM_BACKGROUND, state);

		// Get the OS colors from the text palette.
		COLORREF colorForeground = (textPalette.colorForeground == ZAF_CLR_DEFAULT) ?
			textPalette.osPalette.colorForeground :
			display->colorTable[textPalette.colorForeground];
		COLORREF colorBackground = (backgroundPalette.colorBackground == ZAF_CLR_DEFAULT) ?
			backgroundPalette.osPalette.colorBackground :
			display->colorTable[backgroundPalette.colorBackground];

		// Set the OS colors.
		SelectPalette((HDC)event.osEvent.wParam, display->logicalPalette, FALSE);
		::SetTextColor((HDC)event.osEvent.wParam, colorForeground);
		SetBkColor((HDC)event.osEvent.wParam, colorBackground);

		// Get a background brush.
		HBRUSH backgroundBrush = display->GetGdiCacheBrush(backgroundPalette);
		display->ReleaseGdiCacheObject(backgroundBrush);
		ccode = (ZafEventType)backgroundBrush;
		}
		break;

	case WM_DRAWITEM:
		{
		DRAWITEMSTRUCT *drawItemStruct = (DRAWITEMSTRUCT *)event.osEvent.lParam;

		// Begin drawing operation.
		display->BeginDraw(drawItemStruct->hDC, 0, zafRegion, zafRegion);

		if (drawItemStruct->itemID == (UINT)-1)
		{
			ZafRegionStruct focusRegion;
			focusRegion.ImportPoint(drawItemStruct->rcItem);

			if (drawItemStruct->itemState & ODS_FOCUS)
				DrawFocus(focusRegion, S_CURRENT);
			else
				DrawFocus(focusRegion, S_NON_CURRENT);
		}
		else
		{
			ZafWindowObject *object = (ZafWindowObject *)drawItemStruct->itemData;
			object->zafRegion.ImportPoint(drawItemStruct->rcItem);
			object->Event(event);
		}

		// End drawing operation.
		display->EndDraw();

		ccode = TRUE;
		}
		break;

	case WM_KEYDOWN:
		switch ((int)event.osEvent.wParam)
		{
		case VK_UP:
		case VK_DOWN:
		case VK_LEFT:
		case VK_RIGHT:
		case VK_PRIOR:
		case VK_NEXT:
		case VK_SPACE:
			// Allow arrowing.
			ccode = DefaultCallback(event);
			break;

		default:
			ccode = ZafWindowObject::OSEvent(event);
			break;
		}
		break;

	case WM_MEASUREITEM:
		if (OSDraw())
			ccode = ZafWindowObject::OSEvent(event);
		else
		{
			MEASUREITEMSTRUCT *measureItemStruct = (MEASUREITEMSTRUCT *)event.osEvent.lParam;
			ZafWindowObject *object = (ZafWindowObject *)measureItemStruct->itemData;
			object->Event(ZafEventStruct(S_COMPUTE_SIZE));
			measureItemStruct->itemWidth = object->zafRegion.Width();
			measureItemStruct->itemHeight = object->zafRegion.Height();
			ccode = TRUE;
		}
		break;

	case WM_PAINT:
	case WM_ERASEBKGND:
		// Don't call Draw().
		ccode = DefaultCallback(event);
		break;

	case WM_WINDOWPOSCHANGED:
		// Set the client region.
		RECT clientRect;
		GetClientRect(screenID, &clientRect);
		clientRegion.ImportPoint(clientRect);
		ccode = ZafWindowObject::Event(event);
		break;

	default:
		ccode = ZafWindowObject::OSEvent(event);
	}

	return (ccode);
}

void ZafVtList::OSMapPalette(ZafPaletteStruct &palette, ZafPaletteType type, ZafPaletteState state)
{
	// Get default colors from the operating system.
	if (type == ZAF_PM_TEXT || type == ZAF_PM_FOREGROUND)
	{
		if (palette.colorForeground == ZAF_CLR_DEFAULT && palette.osPalette.colorForeground == 0xFFFFFFFFL)
		{
			if (state & ZAF_PM_SELECTED)
				palette.osPalette.colorForeground = GetSysColor(COLOR_HIGHLIGHTTEXT);
			else
				palette.osPalette.colorForeground = GetSysColor(COLOR_WINDOWTEXT);
		}
		if (palette.colorBackground == ZAF_CLR_DEFAULT && palette.osPalette.colorBackground == 0xFFFFFFFFL)
		{
			if (state & ZAF_PM_SELECTED)
				palette.osPalette.colorBackground = GetSysColor(COLOR_HIGHLIGHT);
			else
				palette.osPalette.colorBackground = GetSysColor(COLOR_WINDOW);
		}
	}
	else if (type == ZAF_PM_BACKGROUND)
	{
		if (palette.colorForeground == ZAF_CLR_DEFAULT && palette.osPalette.colorForeground == 0xFFFFFFFFL)
		{
			if (state & ZAF_PM_SELECTED)
				palette.osPalette.colorForeground = GetSysColor(COLOR_HIGHLIGHT);
			else
				palette.osPalette.colorForeground = GetSysColor(COLOR_WINDOW);
		}
		if (palette.colorBackground == ZAF_CLR_DEFAULT && palette.osPalette.colorBackground == 0xFFFFFFFFL)
		{
			if (state & ZAF_PM_SELECTED)
				palette.osPalette.colorBackground = GetSysColor(COLOR_HIGHLIGHT);
			else
				palette.osPalette.colorBackground = GetSysColor(COLOR_WINDOW);
		}
	}
	else
		ZafWindow::OSMapPalette(palette, type, state);
}

void ZafVtList::OSRegisterObject(void)
{
	DWORD dwStyle = LBS_NOTIFY | WS_CHILD | WS_CLIPSIBLINGS;
	DWORD dwExStyle = 0;

	if (SelectionType() == ZAF_MULTIPLE_SELECTION)
		dwStyle |= LBS_MULTIPLESEL;
	else if (SelectionType() == ZAF_EXTENDED_SELECTION)
		dwStyle |= (LBS_MULTIPLESEL | LBS_EXTENDEDSEL);

	if (visible)
		dwStyle |= WS_VISIBLE;

	if (bordered)
	{
#if defined(ZAF_MSWINDOWS_3D)
		if (ZafMSWindowsApp::native3D)
			dwExStyle |= WS_EX_CLIENTEDGE;
		else
			dwStyle |= WS_BORDER;
#else
		dwStyle |= WS_BORDER;
#endif
	}

	if (disabled && !editMode)
		dwStyle |= WS_DISABLED;

	if (!osDraw)
		dwStyle |= LBS_OWNERDRAWVARIABLE;

	if (VerticalScrollBar())
		dwStyle |= WS_VSCROLL;

	// Create the list.
	screenID = ZafMSWindowsApp::CreateSubclassedWindow(this, ZAF_ITEXT("LISTBOX"), dwStyle, ZAF_NULLP(ZafIChar), 0, dwExStyle);

	// Set the font.
	ZafPaletteStruct fontPalette = LogicalPalette(ZAF_PM_TEXT, ZAF_PM_ACTIVE | ZAF_PM_ENABLED);
	SendMessage(screenID, WM_SETFONT, (WPARAM)display->fontTable[fontPalette.font], (LPARAM)FALSE);
}

ZafError ZafVtList::OSUpdatePalettes(ZafPaletteType, ZafPaletteType)
{
	if (screenID)
	{
		ZafPaletteStruct fontPalette = LogicalPalette(ZAF_PM_TEXT, ZAF_PM_ACTIVE | ZAF_PM_ENABLED);
		SendMessage(screenID, WM_SETREDRAW, (WPARAM)FALSE, 0);
		SendMessage(screenID, WM_SETFONT, (WPARAM)display->fontTable[fontPalette.font], (LPARAM)FALSE);
		if (automaticUpdate && visible)
			SendMessage(screenID, WM_SETREDRAW, (WPARAM)TRUE, 0);
		Redisplay();
	}

	return (ZAF_ERROR_NONE);
}

// ----- Stubs --------------------------------------------------------------

ZafRegionStruct ZafVtList::ConvertToDrawRegion(const ZafWindowObject *object,
	const ZafRegionStruct *region) const
{
	return ZafWindowObject::ConvertToDrawRegion(object, region);
}

ZafEventType ZafVtList::DrawFocus(ZafRegionStruct &region, ZafEventType ccode)
{
	return ZafWindowObject::DrawFocus(region, ccode);
}

void ZafVtList::GetClip(ZafWindowObject *, ZafRegionStruct &, ZafRegionStruct &)
{
	// Handled automatically by Windows.
}

ZafEventType ZafVtList::MoveEvent(const ZafEventStruct &event)
{
	// Handled automatically by Windows.
	return event.type;
}

OSWindowID ZafVtList::OSScreenID(ZafScreenIDType) const
{
	return screenID;
}

void ZafVtList::OSSize(void)
{
	ZafWindowObject::OSSize();
}


void ZafVtList::OSSort(void)
{
	if (screenID)
	{
		SendMessage(screenID, LB_RESETCONTENT, (WPARAM)0, (LPARAM)0);

		//??? Duplicate code.
		//??? Optimize Refresh.
		int index = 0;
		for (ZafWindowObject *object = First(); object; object = object->Next())
		{
			SendMessage(screenID, LB_INSERTSTRING, (WPARAM)-1, OSDraw() ? (LPARAM)object->Text() : (LPARAM)object);
			if (object->Selected())
			{
				if (SelectionType() != ZAF_SINGLE_SELECTION)
					SendMessage(screenID, LB_SETSEL, (WPARAM)TRUE, (LPARAM)index);
				else
				{
					ZafList::SetCurrent(object);
					SendMessage(screenID, LB_SETCURSEL, (WPARAM)index, (LPARAM)0);
				}
			}
			index++;
		}
	}
}

ZafEventType ZafVtList::ScrollEvent(const ZafEventStruct &event)
{
	// Scrolling is handled automatically by Windows.
	return event.type;
}

bool ZafVtList::SetAutomaticUpdate(bool setAutomaticUpdate)
{
	// Defer to the base class.
	return (ZafWindow::SetAutomaticUpdate(setAutomaticUpdate));
}

bool ZafVtList::SetVisible(bool setVisible)
{
	return ZafWindowObject::SetVisible(setVisible);
}


