//	Zinc Application Framework - W_MDIWIN.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#define OSWindow ZafMSWindowsApp
#include "w_app.hpp"
#include <z_mdiwin.hpp>

ZafPaletteMap ZAF_FARDATA ZafMDIWindow::defaultPaletteMap[] =
{
	{ ZAF_PM_BACKGROUND, ZAF_PM_ANY_STATE, { ZAF_LINE_SOLID, ZAF_PTN_SOLID_FILL, ZAF_CLR_DEFAULT, ZAF_CLR_DEFAULT, ZAF_MONO_BLACK, ZAF_MONO_WHITE, ZAF_FNT_DIALOG } },
	{ ZAF_PM_NONE, 0 }
};

LRESULT CallDefFrameProc(HWND hwnd, UINT wMsg, WPARAM wParam, LPARAM lParam)
{
	// Get a pointer to the Zinc window.
	ZafWindow *window = (ZafWindow *)ZafMSWindowsApp::ObjectFromHandle(hwnd);
	return DefFrameProc(hwnd, window ? window->screenID : 0, wMsg, wParam, lParam);
}

LRESULT CALLBACK ZafMSWindowsApp::MDIFrameJumpProc(HWND hwnd, UINT wMsg, WPARAM wParam, LPARAM lParam)
{
	switch (wMsg)
	{
	case WM_COMMAND:
		{
#if defined(ZAF_WIN32)
		WORD wID = LOWORD(wParam);
#else
		WORD wID = wParam;
#endif
		if (wID >= 0xF000)
			return (CallDefFrameProc(hwnd, wMsg, wParam, lParam));
		else
			return FrameProc(CallDefFrameProc, hwnd, wMsg, wParam, lParam);
		}

	default:
		return FrameProc(CallDefFrameProc, hwnd, wMsg, wParam, lParam);
	}
}

LRESULT CallDefMDIChildProc(HWND hwnd, UINT wMsg, WPARAM wParam, LPARAM lParam)
{
	return DefMDIChildProc(hwnd, wMsg, wParam, lParam);
}

LRESULT CALLBACK ZafMSWindowsApp::MDIChildJumpProc(HWND hwnd, UINT wMsg, WPARAM wParam, LPARAM lParam)
{
	switch (wMsg)
	{
	case WM_CLOSE:
		{
		ZafWindow *window = DynamicPtrCast(ObjectFromHandle(hwnd), ZafWindow);

		// Prevent the close operation and post an S_MDI_CLOSE event.
		ZafEventStruct closeEvent(S_MDI_CLOSE);
		closeEvent.windowObject = window;
		window->eventManager->Put(closeEvent);
		}
		return (0);

	case WM_MDIACTIVATE:
		if ((BOOL)wParam)
		{
			ZafWindow *window = DynamicPtrCast(ObjectFromHandle(hwnd), ZafWindow);
			if (window)
				window->SetFocus(true);
		}
		return 0;
	}

	return FrameProc(CallDefMDIChildProc, hwnd, wMsg, wParam, lParam);
}

// ----- ZafMDIWindow -----------------------------------------------------

ZafEventMap ZAF_FARDATA ZafMDIWindow::defaultEventMap[] =
{
	{ L_NONE, 0, 0, 0 }
};

ZafEventType ZafMDIWindow::Draw(const ZafEventStruct &event, ZafEventType ccode)
{
	return ZafWindow::Draw(event, ccode);
}

bool ZafMDIWindow::SetVisible(bool visible)
{
	return ZafWindow::SetVisible(visible);
}

ZafWindowObject *ZafMDIWindow::Add(ZafWindowObject *object, ZafWindowObject *position)
{
	if (mdiType == ZAF_MDI_PARENT)
		return (ZafWindow::Add(object, position ? position : First()));
	else
		return ZafWindow::Add(object, position);
}

ZafEventType ZafMDIWindow::Event(const ZafEventStruct &event)
{
	ZafEventType ccode = event.type;

	MSG msg = event.osEvent;
	if (TranslateMDISysAccel(screenID, &msg))
		return (0);

	// Process Windows Messages.
	if (ccode == E_OSEVENT)
		return ZafMDIWindow::OSEvent(event);

	switch (ccode)
	{
	case S_DESTROY:
		if (mdiType == ZAF_MDI_PARENT)
			ccode = ZafWindow::Event(event);
		else if (screenID)
		{
			// Remove focus before destruction to allow data to be saved.
			if (focus)
				NotifyFocus(ZAF_NULLP(ZafWindowObject), false);

			// Destroy the object's OS counterpart. (Destroys entire tree.)
			SendMessage(parent->screenID, WM_MDIDESTROY, (WPARAM)OSScreenID(ZAF_FRAMEID), (LPARAM)0);

			// Deinitialize the object.
			Event(ZafEventStruct(S_DEINITIALIZE));
		}
		break;

	case S_REGISTER_OBJECT:
		// Register the object.
		ccode = ZafWindowObject::Event(event);

		// Register all of the children.
		if (mdiType == ZAF_MDI_PARENT)
		{
			// Broadcast the event to all children.
			ZafWindowObject *object;
			for (object = SupportFirst(); object; object = object->Next())
				object->Event(event);
			for (object = Last(); object; object = object->Previous())
				object->Event(event);
		}
		else
			BroadcastEvent(event);
		break;

	case S_MDI_CLOSE:
		// Get a pointer to the window to be closed.
		ZafWindow *closeWindow;
		if (event.windowObject)
			// The window to be closed was provided with the event.
			closeWindow = DynamicPtrCast(event.windowObject, ZafWindow);
		else
			// Get the first window.
			closeWindow = DynamicPtrCast(First(), ZafWindow);

		// Make sure it is ok to close the window.
		if (closeWindow && !closeWindow->Locked() &&
			(closeWindow->Event(ZafEventStruct(N_CLOSE)) == 0))
		{
			Subtract(closeWindow);
			if (closeWindow->Destroyable())
				delete closeWindow;
		}
		break;

	default:
		ccode = ZafWindow::Event(event);
	}

	// Return the control code.
	return (ccode);
}

ZafEventType ZafMDIWindow::MoveEvent(const ZafEventStruct &)
{
	return S_UNKNOWN;
}

// ----- OS Specific Functions ----------------------------------------------

ZafEventType ZafMDIWindow::OSEvent(const ZafEventStruct &event)
{
	ZafEventType ccode = event.osEvent.message;

	switch (ccode)
	{
	default:
		ccode = ZafWindow::OSEvent(event);
	}

	return ccode;
}

void ZafMDIWindow::OSRegisterObject(void)
{
	DWORD dwFrameStyle = WS_CLIPSIBLINGS | WS_CLIPCHILDREN;
	DWORD dwFrameExStyle = 0;
	DWORD dwClientStyle = WS_VISIBLE | WS_CHILD;
	for (ZafWindowObject *object = SupportFirst(); object; object = object->Next())
	{
		if (object->NumberID() == ZAF_NUMID_BORDER)
		{
			if (Sizeable())
				dwFrameStyle |= WS_SIZEBOX;
			else
				dwFrameStyle |= WS_BORDER;
		}
		else if (object->NumberID() == ZAF_NUMID_MAXIMIZE)
			dwFrameStyle |= WS_MAXIMIZEBOX;
		else if (object->NumberID() == ZAF_NUMID_MINIMIZE)
			dwFrameStyle |= WS_MINIMIZEBOX;
		else if (object->NumberID() == ZAF_NUMID_TITLE)
			dwFrameStyle |= WS_CAPTION;
		else if (object->NumberID() == ZAF_NUMID_HZ_SCROLL)
			dwClientStyle |= WS_HSCROLL;
		else if (object->NumberID() == ZAF_NUMID_VT_SCROLL)
			dwClientStyle |= WS_VSCROLL;
		else if (object->NumberID() == ZAF_NUMID_SYSTEM)
			dwFrameStyle |= WS_SYSMENU;
	}

	if (Bordered() && !(dwFrameStyle & (WS_SIZEBOX | WS_BORDER)))
	{
#if defined(ZAF_MSWINDOWS_3D)
		if (parent)
			dwFrameExStyle |= WS_EX_CLIENTEDGE;
		else
			dwFrameStyle |= WS_BORDER;
#else
		dwFrameStyle |= WS_BORDER;
#endif
	}

	if (!parent && !(dwFrameStyle & WS_CAPTION))
		dwFrameStyle |= WS_POPUP;

	if (!parent)
	{
		// Create the frame window.
		OSWindowID frameID = ZafMSWindowsApp::CreateSubclassedWindow(this, ZAF_MDI_FRAME_CLASS, dwFrameStyle, NULL, 0, dwFrameExStyle);

		// Re-direct the frame window's "window procedure."
		SetWindowLong(frameID, GWL_WNDPROC, (LONG)ZafMSWindowsApp::MDIFrameJumpProc);

		// Create the client window.
		CLIENTCREATESTRUCT clientCreateStruct;
		clientCreateStruct.hWindowMenu = 0;
		clientCreateStruct.idFirstChild = 0x1000;
		screenID = ZafMSWindowsApp::CreateSubclassedWindow(this, ZAF_ITEXT("MDICLIENT"), dwClientStyle, ZAF_NULLP(ZafIChar), frameID,
			WS_EX_NOPARENTNOTIFY, 0, &clientCreateStruct);
	}
	else
	{
#if defined(ZAF_WIN32) && defined(ZAF_UNICODE)
		MDICREATESTRUCTA mdiCreateStruct;
#else
		MDICREATESTRUCT mdiCreateStruct;
#endif
		mdiCreateStruct.szClass = ZAF_MDI_CHILD_CLASSA;
		mdiCreateStruct.szTitle = 0;
		mdiCreateStruct.hOwner = ZafMSWindowsApp::hInstance;
		mdiCreateStruct.x = 0;
		mdiCreateStruct.y = 0;
		mdiCreateStruct.cx = 0;
		mdiCreateStruct.cy = 0;
		mdiCreateStruct.style = dwFrameStyle;
		mdiCreateStruct.lParam = 0;

#if defined(ZAF_WIN32) && defined(ZAF_UNICODE)
		OSWindowID frameID = (HWND)SendMessageA(parent->screenID, WM_MDICREATE, (WPARAM)0, (LPARAM)&mdiCreateStruct);
#else
		OSWindowID frameID = (HWND)SendMessage(parent->screenID, WM_MDICREATE, (WPARAM)0, (LPARAM)&mdiCreateStruct);
#endif

		// Set the "z-order."
		if (previous)
		{
			HWND hwndPrevious = Previous()->OSScreenID(ZAF_FRAMEID);
			if (hwndPrevious)
				SetWindowPos(frameID, hwndPrevious, 0, 0, 0, 0,
					SWP_NOMOVE | SWP_NOSIZE | SWP_NOREDRAW);
		}

		// Update Active child.
		HWND hwndFirst = GetWindow(parent->screenID, GW_CHILD);
#if defined(ZAF_WIN32)
		HWND hwndActive = (HWND)SendMessage(parent->screenID, WM_MDIGETACTIVE, (WPARAM)0, (LPARAM)0);
#else
		HWND hwndActive = (HWND)LOWORD(SendMessage(parent->screenID, WM_MDIGETACTIVE, (WPARAM)0, (LPARAM)0));
#endif
		if (hwndActive != hwndFirst)
			SendMessage(parent->screenID, WM_MDIACTIVATE, (WPARAM)hwndFirst, (LPARAM)0);

		// Re-direct the frame window's "window procedure."
#if defined(ZAF_WIN32)
		SetWindowLong(frameID, GWL_USERDATA, (LONG)this);
#else
		SetProp(frameID, (LPCSTR)ZafMSWindowsApp::objectLowWord, (HANDLE)LOWORD(this));
		SetProp(frameID, (LPCSTR)ZafMSWindowsApp::objectHighWord, (HANDLE)HIWORD(this));
#endif

		screenID = ZafMSWindowsApp::CreateSubclassedWindow(this, NULL, dwClientStyle, ZAF_NULLP(ZafIChar), frameID);
	}
}

void ZafMDIWindow::OSMapPalette(ZafPaletteStruct &palette, ZafPaletteType type, ZafPaletteState state)
{
	// Get default colors from the operating system.
	if (type == ZAF_PM_BACKGROUND)
	{
		if (mdiType == ZAF_MDI_PARENT)
		{
			if (palette.colorForeground == ZAF_CLR_DEFAULT && palette.osPalette.colorForeground == 0xFFFFFFFFL)
				palette.osPalette.colorForeground = GetSysColor(COLOR_APPWORKSPACE);
			if (palette.colorBackground == ZAF_CLR_DEFAULT && palette.osPalette.colorBackground == 0xFFFFFFFFL)
				palette.osPalette.colorBackground = GetSysColor(COLOR_APPWORKSPACE);
		}
		else
		{
			if (palette.colorForeground == ZAF_CLR_DEFAULT && palette.osPalette.colorForeground == 0xFFFFFFFFL)
#if defined(ZAF_MSWINDOWS_3D)
				palette.osPalette.colorForeground = GetSysColor(COLOR_3DFACE);
#else
				palette.osPalette.colorForeground = GetSysColor(COLOR_WINDOW);
#endif
			if (palette.colorBackground == ZAF_CLR_DEFAULT && palette.osPalette.colorBackground == 0xFFFFFFFFL)
#if defined(ZAF_MSWINDOWS_3D)
				palette.osPalette.colorBackground = GetSysColor(COLOR_3DFACE);
#else
				palette.osPalette.colorBackground = GetSysColor(COLOR_WINDOW);
#endif
		}
	}
	else
		ZafWindow::OSMapPalette(palette, type, state);
}

ZafWindowObject *ZafMDIWindow::NotifyFocus(ZafWindowObject *object, bool focus)
{
	return (ZafWindow::NotifyFocus(object, focus));
}
