// COPYRIGHT (C) 1990-1997.  All Rights Reserved.
// Zinc Software Incorporated.  Pleasant Grove, Utah  USA
// May be freely copied, used and distributed.

#include <zaf.hpp>
#include "bitmap2.hpp"
#include "bitmap4.hpp"

// Constructor for the animated bitmap class.
ZafAnimatedBitmap::ZafAnimatedBitmap(int left, int top, int width, int height,
 	ZafIChar *bitmapSeriesName, int tNumberOfBitmaps, int tTimerInterval) :
 	ZafBitmap(left, top, width, height, ZAF_NULLP(ZafIChar)),
	bitmapSeriesName(ZAF_NULLP(ZafIChar))
{
	// Set the initial information and general attributes.
	SetBitmapIndex(0);
	SetNoncurrent(TRUE);

	// Create and add the timer to the event manager.
	bitmapTimer = new ZafTimer(D_ON, TimerInterval());
	bitmapTimer->SetQueueEvents(FALSE);
	bitmapTimer->Add(this);

	// Initialize the series name and number of bitmaps.
	SetBitmapSeriesName(bitmapSeriesName);
	SetNumberOfBitmaps(tNumberOfBitmaps);

	// Give the object an initial bitmap to start with.
	ZafIChar firstBitmap[50];
	sprintf(firstBitmap, ZAF_ITEXT("%s%i"), bitmapSeriesName, bitmapIndex % numberOfBitmaps);
	SetBitmapName(firstBitmap);

	// Set various other attributes.
	SetAnimated(TRUE);
	SetAutoSize(TRUE);
	SetTimerInterval(tTimerInterval);

	*zafEventManager + bitmapTimer;
}

// Destructor for the animated bitmap class is used for cleanup.
ZafAnimatedBitmap::~ZafAnimatedBitmap()
{
	if (bitmapTimer)
	{
		bitmapTimer->Subtract(this);
		*zafEventManager - bitmapTimer;
		delete bitmapTimer;
	}

	if (bitmapSeriesName)
		delete [] bitmapSeriesName;

	// Make sure this object isn't on the notification list. 
	zafDataManager->SubtractNotification(this);
}

// All event processing is handled in the Event function.
ZafEventType ZafAnimatedBitmap::Event(const ZafEventStruct &event)
{
	ZafEventType ccode = event.type;
	switch(ccode)
	{
	// The N_TIMER event is used to increment the counter and add the next bitmap.
	case N_TIMER:
		// Get the next bitmap if animated.
		if (Animated())
			SetBitmapData();
		break;

	default:
		ccode = ZafBitmap::Event(event);
	}
	return (ccode);
}

// Get the next bitmap in the series to be displayed.
void ZafAnimatedBitmap::SetBitmapData()
{
	// Set up the next bitmap name in the series.
	bitmapIndex++;
	sprintf(bitmapName, ZAF_ITEXT("%s%i"), bitmapSeriesName, bitmapIndex % numberOfBitmaps);
	if (bitmapIndex >= numberOfBitmaps)
		bitmapIndex = 0;

	// The zafDataManager can tranparently load the next bitmap
	// from the persistence file using AllocateData.
	bitmapData =
		DynamicPtrCast(zafDataManager->AllocateData(
			bitmapName, 
			ZafBitmapData::className, 
			ZafBitmapData::classID), 
			ZafBitmapData);

	zafDataManager->AddNotification(this);
	RedisplayData();
}

// Set the current bitmap index.
void ZafAnimatedBitmap::SetBitmapIndex(int tBitmapIndex)
{
	bitmapIndex = tBitmapIndex;
}

// Set the animation, TRUE = on, FALSE = off.
bool ZafAnimatedBitmap::SetAnimated(bool tAnimated)
{
	animated = tAnimated;
	return (animated);
}

// Set the bitmap series name.
void ZafAnimatedBitmap::SetBitmapSeriesName(ZafIChar *tBitmapSeriesName)
{
	bitmapSeriesName = strdup(tBitmapSeriesName);
}

// Set the number of bitmaps.
void ZafAnimatedBitmap::SetNumberOfBitmaps(int tNumberOfBitmaps)
{
	numberOfBitmaps = tNumberOfBitmaps;
}

// Set the timer interval.
void ZafAnimatedBitmap::SetTimerInterval(int tInterval)
{
	timerInterval = tInterval;
	bitmapTimer->SetInterval(timerInterval);
}
