source debug.tcl

set XDCC_CFG(ON) 0
set XDCC_CFG(Quiet) 1
set XDCC_CFG(PackList) ""
set XDCC_CFG(Queue) ""
set XDCC_CFG(QueueMax) 10
set XDCC_CFG(SlotsMax) 10
set XDCC_CFG(SlotsUsed) 0

proc createPack {mask} {
    set files ""
    if {[catch {set files [glob $mask]}]} {
	echo "*** XDCC: No files matching \"$mask\" were found"
	return ""
    }
    set size 0
    foreach f $files {
	incr size [file size $f]
    }
    return [linsert $files 0 $size]
}

proc deletePack {xdccCfgName packNo} {
    upvar $xdccCfgName xdcc_cfg
    if {$packNo > [llength $xdcc_cfg(PackList)]} {
	error "Pack \#$packNo does not exist."
    }
    set idx [expr $packNo - 1]
    set xdcc_cfg(PackList) [lreplace $xdcc_cfg(PackList) $idx $idx]
}

proc addPack {xdccCfgName} {
    upvar $xdccCfgName xdcc_cfg
    set desc [prompt "Pack Description" "Desc: " "New Pack"]
    set mask [file join [prompt "Files to Add" "File Mask: " ""]]
    set pack [createPack $mask]
    if {$mask == "" || $pack == ""} {
	echo "*** XDCC: Pack creation cancelled"
	return
    }
    set pack [concat \{$desc\} $pack]
    lappend xdcc_cfg(PackList) $pack
    echo "*** XDCC: Pack successfully added, new xdcc list is:"
    foreach line [makePackStringList xdcc_cfg] {
	echo $line
    }
}

proc makePackStringList {xdccCfgName} {
    upvar $xdccCfgName xdcc_cfg
    set stringList ""
    set packs [llength $xdcc_cfg(PackList)]
    if {$packs <= 0} {return [list "No Packs Offered"]}
    if {$packs > 1} {
	set packs "$packs packs"
    } else {
	set packs "$packs pack"
    }
    set header "** $packs **"
    if {$xdcc_cfg(QueueMax)} {
	append header " $xdcc_cfg(SlotsUsed) of $xdcc_cfg(SlotsMax) slots used"
    } else {
	append header " No limit on open DCCs"
    }
    if {$xdcc_cfg(QueueMax)}  {
	append header ", Queue: [llength $xdcc_cfg(Queue)]/$xdcc_cfg(QueueMax)"
    } else {
	append header "."
    }
    set count 1
    set sizeTotal 0
    lappend stringList $header
    foreach pack $xdcc_cfg(PackList) {
	set desc [lindex $pack 0]
	set size [lindex $pack 1]
	set filecount [fileCount $pack]
	incr sizeTotal $size
	set size [parseFileSize $size]
	set pstring "\#${count} \[${filecount}/${size}\] $desc"
	lappend stringList $pstring
	incr count
    }
    return $stringList
}

proc fileList {pack} {
    return [lrange $pack 2 end]
}

proc fileCount {pack} {
    return [llength [lrange $pack 2 end]]
}

proc parseFileSize {size} {
    if {$size > 1048576} {
	set size "[format "%.1f" [expr $size / 1048576.0]]M"
    } elseif {$size > 1024} {
	set size "[format "%.1f" [expr $size / 1024.0]]K"
    } else {
	set size "[format "%.0f" ${size}]B"
    }
    return $size
}
    
alias xdcc {
    set cmd [string tolower [lindex [args] 0]]
    switch $cmd {
	"on" {
	    set XDCC_CFG(ON) 1
	    echo "*** XDCC Is now on"
	}
	
	"off" {
	    set XDCC_CFG(ON) 0
	    echo "*** XDCC is now off"
	}

	"addpack" {
	    addPack XDCC_CFG
	}

	"delpack" {
	    deletePack XDCC_CFG [lindex [args] 1]
	    echo "*** Deleted Pack \#[lindex [args] 1]"
	}

	"plist" {
	    echo "*** Sending XDCC list to current window"
	    foreach line [makePackStringList XDCC_CFG] {
		say $line
	    }
	}
	
	"queuemax" {
	    set XDCC_CFG(QueueMax) [lindex [args] 1] 
	}
	
	"slotmax" {
	    set XDCC_CFG(SlotsMax) [lindex [args] 1]
	}

	"quiet" {
	    echo "*** XDCC now in Quiet mode"
	    set XDCC_CFG(Quiet) 1
	}
	
	"verbose" {
	    echo "*** XDCC now in Verbose mode"
	    set XDCC_CFG(Quiet) 0
	}
	
	"help" {
	    echo "Usage: /xdcc <command> \[options\]"
	    echo "Where <command> is one of:"
	    echo "    ON"
	    echo "    OFF"
	    echo "    ADDPACK"
	    echo "    DELPACK"
	    echo "    PLIST"
	    echo "    QUEUEMAX"
	    echo "    SLOTMAX"
	    echo "    QUIET"
	    echo "    VERBOSE"
	    echo "    HELP"
	}
	
	default {
	    echo "Unknown XDCC command - \"$cmd\" - try /xdcc help"
	}
    
    } 
    complete
}
    
proc handleXDCCRequest {xdccCfgName arg1 args} {
    upvar $xdccCfgName xdcc_cfg
    if {[lindex $args 0] == ""} {
	set temp [split $arg1]
	set arg1 [lindex $temp 0]
	set args [lrange $temp 1 end]
    }
    set ctcpcmd [string tolower $arg1]
    if {[regexp {.dcc} $ctcpcmd]} {
	if {!$xdcc_cfg(ON)} {
	    /raw notice [nick] :No Packs Offered
	    return
	}
	if {$ctcpcmd == "tdcc"} {
	    set cmd tdcc
	} else {
	    set cmd dcc
	}
	set arg2 [join $args]
	set xcmd ""
	regexp {([a-z]+)( *\#([0-9]+))?} $arg2 a xcmd b num
	switch $xcmd {
	    "list" {
		foreach line [makePackStringList xdcc_cfg] {
		    /raw notice [nick] :$line
		}
	    }
	    "send" {
		if {$num != ""} {
		    set pack [lindex $xdcc_cfg(PackList) [expr $num - 1]]
		    if { [expr $xdcc_cfg(SlotsUsed) + [llength [fileList $pack]]] > $xdcc_cfg(SlotsMax) } {
			if { [llength $xdcc_cfg(Queue)] >= $xdcc_cfg(QueueMax) } {
			    /raw notice [nick] :Queue is full, try again later
			    return
			} else {
			    lappend xdcc_cfg(Queue) [list [nick] $num $cmd]
			    if {!$xdcc_cfg(Quiet)} {
				echo "*** XDCC: Adding [nick]/\#$num to Queue"
			    }
			    /raw notice [nick] :All DCC Slots in use
			    /raw notice [nick] :You are position [llength $xdcc_cfg(Queue)]/$xdcc_cfg(QueueMax) in the Queue
			    return
			}
		    }
		    if {!$xdcc_cfg(Quiet)} {
			echo "*** XDCC: Sending [nick] pack \#$num"
		    }
		    foreach file [fileList $pack] {
			regsub -all {/} $file {\\} filen
			/$cmd send [nick] $filen
		    }
		}
	    }
	    default {
		/raw notice [nick] :Invalid XDCC Command
	    }
	}
	return
    }
}

proc checkQueue {xdccCfgName} {
    upvar $xdccCfgName xdcc_cfg
    if {$xdcc_cfg(Queue) == ""} {return}
    set qhead [lindex $xdcc_cfg(Queue) 0]
    set nick [lindex $qhead 0]
    set packNo [lindex $qhead 1]
    set cmd [lindex $qhead 2]
    set pack [lindex $xdcc_cfg(PackList) [expr $packNo - 1]]
    if {[expr $xdcc_cfg(SlotsUsed) + [llength [fileList $pack]]] <= $xdcc_cfg(SlotsMax)} {
	set xdcc_cfg(Queue) [lreplace $xdcc_cfg(Queue) 0 0]
	if {!$xdcc_cfg(Quiet)} {
	    echo "*** XDCC: Sending $nick pack \#$packNo"
	}
	/raw notice $nick :Sending you pack \#$packNo
	foreach file [fileList $pack] {
	    regsub -all {/} $file {\\} filen
	    /$cmd send $nick $filen
	}
    }
}

on ctcp {
    handleXDCCRequest XDCC_CFG [lindex [args] 1] [lrange [args] 2 end]
}

on privmsg {
    handleXDCCRequest XDCC_CFG [lindex [args] 1] [lrange [args] 2 end]
}

on dcc_begin {
    incr XDCC_CFG(SlotsUsed) 1
}

on dcc_complete {
    incr XDCC_CFG(SlotsUsed) -1
    if {$XDCC_CFG(SlotsUsed) < 0} {
	set XDCC_CFG(SlotsUsed) 0
    } 
    checkQueue XDCC_CFG
}

on dcc_error {
    incr XDCC_CFG(SlotsUsed) -1
}
