/*++

Copyright (c) 1997 Microsoft Corporation

Module Name:

    parscan.c

Abstract:

    This module contains the code for a sample parallel
    scanner class driver.

Author:

    Timothy T. Wells (WestTek, L.L.C.) 20-January-1997

Environment:

    Kernel mode

Revision History :

--*/

#ifdef MEMPHIS
#include "wdm.h"
#define WANTVXDWRAPS
#include <basedef.h>
#include <vmm.h>
#include <ntkern.h>
#include <vxdwraps.h>
#include <vtd.h>
#else
#include "ntddk.h"
#endif
#include "ntddser.h"
#include "parallel.h"
#include "parscan.h"
#include "parlog.h"

#if DBG
extern ULONG ParDebugLevel = -1;
#endif

extern const PHYSICAL_ADDRESS PhysicalZero = {0};

NTSTATUS
DriverEntry(
    IN  PDRIVER_OBJECT  DriverObject,
    IN  PUNICODE_STRING RegistryPath
    );

NTSTATUS
ParScanPnpAddDevice(
    IN PDRIVER_OBJECT pDriverObject,
    IN PDEVICE_OBJECT pPhysicalDeviceObject
    );
    
NTSTATUS 
ParScanParallelPnp (
    IN PDEVICE_OBJECT pDeviceObject,
    IN PIRP           pIrp
   );
   
BOOLEAN
ParScanMakeNames(
    IN  ULONG           ParallelPortNumber,
    OUT PUNICODE_STRING ClassName,
    OUT PUNICODE_STRING LinkName
    );
    
VOID
ParScanLogError(
    IN  PDRIVER_OBJECT      DriverObject,
    IN  PDEVICE_OBJECT      DeviceObject OPTIONAL,
    IN  PHYSICAL_ADDRESS    P1,
    IN  PHYSICAL_ADDRESS    P2,
    IN  ULONG               SequenceNumber,
    IN  UCHAR               MajorFunctionCode,
    IN  UCHAR               RetryCount,
    IN  ULONG               UniqueErrorValue,
    IN  NTSTATUS            FinalStatus,
    IN  NTSTATUS            SpecificIOStatus
    );

NTSTATUS
ParScanGetPortInfoFromParClass(
    IN OUT  PDEVICE_EXTENSION   Extension
    );
    
NTSTATUS
ParScanSynchCompletionRoutine(
    IN PDEVICE_OBJECT   DeviceObject,
    IN PIRP             Irp,
    IN PKEVENT          Event
    );
    
NTSTATUS
ParScanCompleteIrp(
    IN PDEVICE_OBJECT   DeviceObject,
    IN PIRP             Irp,
    IN PVOID            Context
    );
    
#define WAIT        1
#define NO_WAIT     0

NTSTATUS
ParScanCallParent(
    IN PDEVICE_EXTENSION        Extension,
    IN PIRP                     Irp,
    IN BOOLEAN                  Wait,
    IN PIO_COMPLETION_ROUTINE   CompletionRoutine
    );
    
NTSTATUS
ParScanQueueIORequest(
    IN PDEVICE_EXTENSION Extension,
    IN PIRP              Irp
    );
    
NTSTATUS
ParScanExecuteEscape(
    IN PDEVICE_EXTENSION Extension,
    IN PIRP              Irp
    );
    
//
// Keep track of the number of parallel port devices created...
//
ULONG g_NumPorts = 0;

//
// Definition of OpenCloseMutex.
//
extern ULONG OpenCloseReferenceCount = 1;
extern PFAST_MUTEX OpenCloseMutex = NULL;


NTSTATUS
DriverEntry(
    IN PDRIVER_OBJECT DriverObject,
    IN PUNICODE_STRING RegistryPath
    )

/*++

Routine Description:

    This routine is called at system initialization time to initialize
    this driver.

Arguments:

    DriverObject    - Supplies the driver object.

    RegistryPath    - Supplies the registry path for this driver.

Return Value:

    STATUS_SUCCESS          - We could initialize at least one device.
    STATUS_NO_SUCH_DEVICE   - We could not in itialize even one device.

--*/

{

    //
    // Initialize the Driver Object with driver's entry points
    //

    DriverObject->MajorFunction[IRP_MJ_CREATE]            = ParScanCreateOpen;
    DriverObject->MajorFunction[IRP_MJ_CLOSE]             = ParScanClose;
    DriverObject->MajorFunction[IRP_MJ_CLEANUP]           = ParScanCleanup;
    DriverObject->MajorFunction[IRP_MJ_READ]              = ParScanRead;
    DriverObject->MajorFunction[IRP_MJ_WRITE]             = ParScanWrite;
    DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL]    = ParScanDeviceControl;
    DriverObject->MajorFunction[IRP_MJ_QUERY_INFORMATION] = ParScanQueryInformationFile;
    DriverObject->MajorFunction[IRP_MJ_SET_INFORMATION]   = ParScanSetInformationFile;
    DriverObject->MajorFunction[IRP_MJ_PNP]               = ParScanParallelPnp;    
    DriverObject->DriverExtension->AddDevice              = ParScanPnpAddDevice;
    DriverObject->DriverUnload = ParScanUnload;

    return STATUS_SUCCESS;
}

NTSTATUS
ParScanPnpAddDevice(
    IN PDRIVER_OBJECT pDriverObject,
    IN PDEVICE_OBJECT pPhysicalDeviceObject
    )
/*++

Routine Description:

    This routine is called to create a new instance of the device.

Arguments:

    pDriverObject           - pointer to the driver object for this instance of parport.

    pPhysicalDeviceObject   - pointer to the device object that represents the port.

Return Value:

    STATUS_SUCCESS          - if successful.
    STATUS_UNSUCCESSFUL     - otherwise.

--*/
{
    UNICODE_STRING      ClassName;
    UNICODE_STRING      LinkName;
    NTSTATUS            Status;
    PDEVICE_EXTENSION   Extension;
    PDEVICE_OBJECT      pDeviceObject;

    //
    // Get the Class and Link names.
    //

    if (!ParScanMakeNames (g_NumPorts, &ClassName, &LinkName)) {

        ParScanLogError(pDriverObject, 
                        NULL, 
                        PhysicalZero, 
                        PhysicalZero, 
                        0, 
                        0, 
                        0, 
                        1,
                        STATUS_SUCCESS, 
                        PAR_INSUFFICIENT_RESOURCES);

        DebugDump(PARERRORS,("PARSCAN: Could not form Unicode name strings.\n"));

        return STATUS_INSUFFICIENT_RESOURCES;
    }
    
    //
    // Create the device object for this device.
    //

    Status = IoCreateDevice(pDriverObject, 
                            sizeof(DEVICE_EXTENSION),
                            &ClassName,
                            FILE_DEVICE_PARALLEL_PORT,
                            0, 
                            FALSE, 
                            &pDeviceObject);

    
    if (!NT_SUCCESS(Status)) {

        ExFreePool(ClassName.Buffer);
        ExFreePool(LinkName.Buffer);
    
        ParScanLogError(pDriverObject, 
                        NULL, 
                        PhysicalZero, 
                        PhysicalZero,
                        0, 
                        0, 
                        0, 
                        9, 
                        STATUS_SUCCESS, 
                        PAR_INSUFFICIENT_RESOURCES);

        DebugDump(PARERRORS, ("PARPORT:  Could not create a device for %d\n", g_NumPorts));

        return Status;
    }

    //
    // The device object has a pointer to an area of non-paged
    // pool allocated for this device.  This will be the device
    // extension.
    //

    Extension = pDeviceObject->DeviceExtension;

    //
    // Zero all of the memory associated with the device
    // extension.
    //

    RtlZeroMemory(Extension, sizeof(DEVICE_EXTENSION));

    //
    // Get a "back pointer" to the device object.
    //

    Extension->DeviceObject = pDeviceObject;

    //
    // Setup buffered I/O
    //
    pDeviceObject->Flags |= DO_BUFFERED_IO;
    
    //
    // Attach our new Device to our parents stack.
    //
    Extension->PortDeviceObject = IoAttachDeviceToDeviceStack(
                                  pDeviceObject, 
                                  pPhysicalDeviceObject);
                                    
    if (NULL == Extension->PortDeviceObject) {
    
        ExFreePool(ClassName.Buffer);
        ExFreePool(LinkName.Buffer);
        
        IoDeleteDevice(pDeviceObject);
        
        return STATUS_NOT_SUPPORTED;
    }


    Extension->ClassName        = ClassName;
    Extension->SymbolicLinkName = LinkName;
    
    //
    // We have created the device, so increment the counter in the
    // IO system that keeps track.  Anyplace that we do an IoDeleteDevice
    // we need to decrement.
    //

    g_NumPorts++;

    return STATUS_SUCCESS;

}

NTSTATUS 
ParScanParallelPnp (
    IN PDEVICE_OBJECT pDeviceObject,
    IN PIRP           pIrp
   )
/*++

Routine Description:

    This routine handles all PNP IRPs.

Arguments:

    pDeviceObject           - represents a parallel device
    
    pIrp                    - PNP Irp 
    
Return Value:

    STATUS_SUCCESS          - if successful.
    STATUS_UNSUCCESSFUL     - otherwise.

--*/
{
    NTSTATUS                        Status = STATUS_NOT_SUPPORTED;
    PDEVICE_EXTENSION               Extension;
    PIO_STACK_LOCATION              pIrpStack;
    
    pIrpStack = IoGetCurrentIrpStackLocation( pIrp );
    
    Extension = pDeviceObject->DeviceExtension;
    
    switch (pIrpStack->MinorFunction) {
        
        case IRP_MN_START_DEVICE:
        
            //
            // Call down to our parent
            //

            Status = ParScanCallParent(Extension, pIrp, WAIT, NULL);
            
            if (!NT_SUCCESS(Status)) {

                pIrp->IoStatus.Status      = Status;
                pIrp->IoStatus.Information = 0;

                IoCompleteRequest(pIrp, IO_NO_INCREMENT);
                return (Status);

            }

            //
            // The parent's initialization is complete (PARCLASS).
            //

            Status = ParScanGetPortInfoFromParClass(Extension);

            if (!NT_SUCCESS(Status)) {

                ParScanLogError(pDeviceObject->DriverObject, 
                                pDeviceObject, 
                                PhysicalZero, 
                                PhysicalZero, 
                                0, 
                                0, 
                                0, 
                                4,
                                Status, 
                                PAR_CANT_FIND_PORT_DRIVER);

                DebugDump(
                    PARERRORS,("PARSCAN: Can't get port info from parclass device.\n"));

                pIrp->IoStatus.Status      = Status;
                pIrp->IoStatus.Information = 0;

                IoCompleteRequest(pIrp, IO_NO_INCREMENT);
                return (Status);
            }

            //
            // Now setup the symbolic link for windows.
            //
            
            Status = IoCreateUnprotectedSymbolicLink(&Extension->SymbolicLinkName, &Extension->ClassName);

            if (NT_SUCCESS(Status)) {

                // We were able to create the symbolic link, so record this
                // value in the extension for cleanup at unload time.

                Extension->CreatedSymbolicLink = TRUE;

                // Write out the result of the symbolic link to the registry.
    
                Status = RtlWriteRegistryValue(RTL_REGISTRY_DEVICEMAP,
                                               L"Parallel Scanners",
                                               Extension->ClassName.Buffer,
                                               REG_SZ,
                                               Extension->SymbolicLinkName.Buffer,
                                               Extension->SymbolicLinkName.Length + sizeof(WCHAR));
                if (!NT_SUCCESS(Status)) {

                    //
                    // Oh well, it didn't work.  Just go to cleanup.
                    //

                    DebugDump(PARERRORS,
                              ("PARSCAN: Couldn't create the device map entry\n"
                               "--------  for port %wZ\n",
                               &Extension->ClassName));

                    ParScanLogError(pDeviceObject->DriverObject, 
                                    pDeviceObject, 
                                    PhysicalZero, 
                                    PhysicalZero, 
                                    0, 
                                    0, 
                                    0, 
                                    6, 
                                    Status,
                                    PAR_NO_DEVICE_MAP_CREATED);
                }
    
            } else {

                //
                // Oh well, couldn't create the symbolic link.
                //

                Extension->CreatedSymbolicLink = FALSE;
                
                ExFreePool(Extension->SymbolicLinkName.Buffer);

                DebugDump(PARERRORS,
                          ("PARSCAN: Couldn't create the symbolic link\n"
                           "--------  for port %wZ\n",
                           &Extension->ClassName));

                ParScanLogError(pDeviceObject->DriverObject, 
                                pDeviceObject, 
                                PhysicalZero, 
                                PhysicalZero, 
                                0, 
                                0, 
                                0, 
                                5, 
                                Status, 
                                PAR_NO_SYMLINK_CREATED);

            }


            ExFreePool(Extension->ClassName.Buffer);

            Status = STATUS_SUCCESS;
            break;

        case IRP_MN_REMOVE_DEVICE:
            Status = STATUS_SUCCESS;
            break;            

        case IRP_MN_STOP_DEVICE:
            Status = STATUS_SUCCESS;
            break;
            
        default:
            break;            
    }
    
    //
    // Complete the IRP...
    //
    
    pIrp->IoStatus.Status      = Status;
    pIrp->IoStatus.Information = 0;

    IoCompleteRequest(pIrp, IO_NO_INCREMENT);
                
    return( Status );
    
}

BOOLEAN
ParScanMakeNames(
    IN  ULONG           ParallelPortNumber,
    OUT PUNICODE_STRING ClassName,
    OUT PUNICODE_STRING LinkName
    )

/*++

Routine Description:

    This routine generates the names \Device\ParScanN, 
    and \DosDevices\LPTSCAN(N+1) where N is 'ParallelPortNumber'.
    This routine will allocate pool so that the buffers of 
    these unicode strings need to be eventually freed.

Arguments:

    ParallelPortNumber  - Supplies the port number.

    ClassName           - Returns the class name.

    LinkName            - Returns the link name.

Return Value:

    FALSE   - Failure.
    TRUE    - Success.

--*/
{
    UNICODE_STRING  Prefix;
    UNICODE_STRING  Digits;
    UNICODE_STRING  LinkPrefix;
    UNICODE_STRING  LinkDigits;
    WCHAR           DigitsBuffer[10];
    WCHAR           LinkDigitsBuffer[10];
    UNICODE_STRING  ClassSuffix; 
    UNICODE_STRING  LinkSuffix;
    NTSTATUS        Status;

    //
    // Put together local variables for constructing names.
    //
    
    RtlInitUnicodeString(&Prefix, L"\\Device\\");
    RtlInitUnicodeString(&LinkPrefix, L"\\DosDevices\\");
    
    //
    // WORKWORK: Change the name to be device specific.
    //
    RtlInitUnicodeString(&ClassSuffix, PARSCAN_NT_SUFFIX);
    RtlInitUnicodeString(&LinkSuffix, PARSCAN_LINK_NAME);
    
    Digits.Length        = 0;
    Digits.MaximumLength = 20;
    Digits.Buffer        = DigitsBuffer;
    
    LinkDigits.Length        = 0;
    LinkDigits.MaximumLength = 20;
    LinkDigits.Buffer        = LinkDigitsBuffer;
    
    Status = RtlIntegerToUnicodeString(ParallelPortNumber, 10, &Digits);
    if (!NT_SUCCESS(Status)) {
        return FALSE;
    }
    
    Status = RtlIntegerToUnicodeString(ParallelPortNumber + 1, 10, &LinkDigits);
    if (!NT_SUCCESS(Status)) {
        return FALSE;
    }

    //
    // Make the class name.
    //
    
    ClassName->Length = 0;
    ClassName->MaximumLength = Prefix.Length + ClassSuffix.Length +
                               Digits.Length + sizeof(WCHAR);
                               
    ClassName->Buffer = ExAllocatePool(PagedPool, ClassName->MaximumLength);
    if (!ClassName->Buffer) {
        return FALSE;
    }
    
    RtlZeroMemory(ClassName->Buffer, ClassName->MaximumLength);
    RtlAppendUnicodeStringToString(ClassName, &Prefix);
    RtlAppendUnicodeStringToString(ClassName, &ClassSuffix);
    RtlAppendUnicodeStringToString(ClassName, &Digits);

    //
    // Make the link name.
    //
    
    LinkName->Length = 0;
    LinkName->MaximumLength = LinkPrefix.Length + LinkSuffix.Length +
                              LinkDigits.Length + sizeof(WCHAR);
                             
    LinkName->Buffer = ExAllocatePool(PagedPool, LinkName->MaximumLength);
    if (!LinkName->Buffer) {
        ExFreePool(ClassName->Buffer);
        return FALSE;
    }
    
    RtlZeroMemory(LinkName->Buffer, LinkName->MaximumLength);
    RtlAppendUnicodeStringToString(LinkName, &LinkPrefix);
    RtlAppendUnicodeStringToString(LinkName, &LinkSuffix);
    RtlAppendUnicodeStringToString(LinkName, &LinkDigits);

    return TRUE;
}

VOID
ParScanLogError(
    IN  PDRIVER_OBJECT      DriverObject,
    IN  PDEVICE_OBJECT      DeviceObject OPTIONAL,
    IN  PHYSICAL_ADDRESS    P1,
    IN  PHYSICAL_ADDRESS    P2,
    IN  ULONG               SequenceNumber,
    IN  UCHAR               MajorFunctionCode,
    IN  UCHAR               RetryCount,
    IN  ULONG               UniqueErrorValue,
    IN  NTSTATUS            FinalStatus,
    IN  NTSTATUS            SpecificIOStatus
    )

/*++

Routine Description:

    This routine allocates an error log entry, copies the supplied data
    to it, and requests that it be written to the error log file.

Arguments:

    DriverObject        - Supplies a pointer to the driver object for the
                            device.

    DeviceObject        - Supplies a pointer to the device object associated
                            with the device that had the error, early in
                            initialization, one may not yet exist.

    P1,P2               - Supplies the physical addresses for the controller
                            ports involved with the error if they are available
                            and puts them through as dump data.

    SequenceNumber      - Supplies a ulong value that is unique to an IRP over
                            the life of the irp in this driver - 0 generally
                            means an error not associated with an irp.

    MajorFunctionCode   - Supplies the major function code of the irp if there
                            is an error associated with it.

    RetryCount          - Supplies the number of times a particular operation
                            has been retried.

    UniqueErrorValue    - Supplies a unique long word that identifies the
                            particular call to this function.

    FinalStatus         - Supplies the final status given to the irp that was
                            associated with this error.  If this log entry is
                            being made during one of the retries this value
                            will be STATUS_SUCCESS.

    SpecificIOStatus    - Supplies the IO status for this particular error.

Return Value:

    None.

--*/

{
    PIO_ERROR_LOG_PACKET    ErrorLogEntry;
    PVOID                   ObjectToUse;
    SHORT                   DumpToAllocate;

    if (ARGUMENT_PRESENT(DeviceObject)) {
    
        ObjectToUse = DeviceObject;
        
    } else {
    
        ObjectToUse = DriverObject;
        
    }

    DumpToAllocate = 0;

    if (P1.LowPart != 0 || P1.HighPart != 0) {
        DumpToAllocate = (SHORT) sizeof(PHYSICAL_ADDRESS);
    }

    if (P2.LowPart != 0 || P2.HighPart != 0) {
        DumpToAllocate += (SHORT) sizeof(PHYSICAL_ADDRESS);
    }

    ErrorLogEntry = IoAllocateErrorLogEntry(ObjectToUse,
            (UCHAR) (sizeof(IO_ERROR_LOG_PACKET) + DumpToAllocate));

    if (!ErrorLogEntry) {
        return;
    }

    ErrorLogEntry->ErrorCode         = SpecificIOStatus;
    ErrorLogEntry->SequenceNumber    = SequenceNumber;
    ErrorLogEntry->MajorFunctionCode = MajorFunctionCode;
    ErrorLogEntry->RetryCount        = RetryCount;
    ErrorLogEntry->UniqueErrorValue  = UniqueErrorValue;
    ErrorLogEntry->FinalStatus       = FinalStatus;
    ErrorLogEntry->DumpDataSize      = DumpToAllocate;

    if (DumpToAllocate) {

        RtlCopyMemory(ErrorLogEntry->DumpData, &P1, sizeof(PHYSICAL_ADDRESS));

        if (DumpToAllocate > sizeof(PHYSICAL_ADDRESS)) {

            RtlCopyMemory(((PUCHAR) ErrorLogEntry->DumpData) +
                          sizeof(PHYSICAL_ADDRESS), &P2,
                          sizeof(PHYSICAL_ADDRESS));
        }
    }

    IoWriteErrorLogEntry(ErrorLogEntry);
}

NTSTATUS
ParScanGetPortInfoFromParClass(
    IN OUT  PDEVICE_EXTENSION   Extension
    )

/*++

Routine Description:

    This routine will request the parclass information from parclas
    and fill it in the device extension.

Arguments:

    Extension   - Supplies the device extension.

Return Value:

    STATUS_SUCCESS  - Success.
    !STATUS_SUCCESS - Failure.

--*/

{
    KEVENT                      Event;
    PIRP                        Irp;
    PARCLASS_INFORMATION        ParclassInfo;
    IO_STATUS_BLOCK             IoStatus;
    NTSTATUS                    Status;

    KeInitializeEvent(&Event, NotificationEvent, FALSE);

    Irp = IoBuildDeviceIoControlRequest(IOCTL_INTERNAL_PARCLASS_CONNECT,
                                        Extension->PortDeviceObject,
                                        NULL, 0, &ParclassInfo,
                                        sizeof(PARCLASS_INFORMATION),
                                        TRUE, &Event, &IoStatus);

    if (!Irp) {
    
        return STATUS_INSUFFICIENT_RESOURCES;
    }

    Status = IoCallDriver(Extension->PortDeviceObject, Irp);

    if (!NT_SUCCESS(Status)) {
        return Status;
    }

    Status = KeWaitForSingleObject(&Event, Executive, KernelMode, FALSE, NULL);

    if (!NT_SUCCESS(Status)) {
        return Status;
    }

    Status = IoStatus.Status;

    if (!NT_SUCCESS(Status)) {
        return(Status);
    }

    Extension->Controller           = ParclassInfo.Controller;
    Extension->SpanOfController     = ParclassInfo.SpanOfController;
    Extension->SetupDriverCallback  = ParclassInfo.SetupDriverCallback;
    Extension->PortContext          = ParclassInfo.Context;

    if (Extension->SpanOfController < PARALLEL_REGISTER_SPAN) {
        return STATUS_INSUFFICIENT_RESOURCES;
    }

    return Status;
}

NTSTATUS
ParScanSynchCompletionRoutine(
    IN PDEVICE_OBJECT   DeviceObject,
    IN PIRP             Irp,
    IN PKEVENT          Event
    )

/*++

Routine Description:

    This routine is for use with synchronous IRP processing.  
    All it does is signal an event, so the driver knows it
    can continue.

Arguments:

    DriverObject - Pointer to driver object created by system.

    Irp          - Irp that just completed

    Event        - Event we'll signal to say Irp is done

Return Value:

    None.

--*/

{

    KeSetEvent((PKEVENT) Event, 0, FALSE);
    
    return (STATUS_MORE_PROCESSING_REQUIRED);

}

NTSTATUS
ParScanCompleteIrp(
    IN PDEVICE_OBJECT   DeviceObject,
    IN PIRP             Irp,
    IN PVOID            Context
    )

/*++

Routine Description:

    This routine is for use with synchronous IRP processing.  
    All it does is signal an event, so the driver knows it
    can continue.

Arguments:

    DriverObject - Pointer to driver object created by system.

    Irp          - Irp that just completed

    Event        - Event we'll signal to say Irp is done

Return Value:

    None.

--*/

{

    NTSTATUS    Status;
    
    //
    // WORKWORK  Do any post IO processing here...
    //
    
    //
    // If parclass marked this IRP as pending, update our stack location
    // to show the same
    //
    if (Irp->PendingReturned) {
    
        IoMarkIrpPending (Irp);
    }    
        
    Status = Irp->IoStatus.Status;
    
    return (Status);

}

NTSTATUS
ParScanCallParent(
    IN PDEVICE_EXTENSION        Extension,
    IN PIRP                     pIrp,
    IN BOOLEAN                  Wait,
    IN PIO_COMPLETION_ROUTINE   CompletionRoutine
    )

/*++

Routine Description:

    This routine will call the next driver in the WDM chain
    (which should be PARCLASS).

Arguments:

    Extension    - Device Extension.

    Irp          - Irp to call parent with.

Return Value:

    NTSTATUS.

--*/

{
    PIO_STACK_LOCATION              pIrpStack;
    PIO_STACK_LOCATION              pNextIrpStack;
    KEVENT                          Event;
    PVOID                           Context;
    NTSTATUS                        Status;
    
    pIrpStack = IoGetCurrentIrpStackLocation(pIrp);
    Context = NULL;
    
    //
    // Prepare to call down to the parent with the PNP Request...
    //
        
    pNextIrpStack = IoGetNextIrpStackLocation(pIrp);
    pNextIrpStack->MajorFunction = pIrpStack->MajorFunction;
    pNextIrpStack->MinorFunction = pIrpStack->MinorFunction;
                
    RtlCopyMemory(&pNextIrpStack->Parameters, 
                  &pIrpStack->Parameters, 
                  sizeof(pIrpStack->Parameters.Others));

    if (Wait) {
    
        KeInitializeEvent(&Event, NotificationEvent, FALSE);

        CompletionRoutine = ParScanSynchCompletionRoutine;
        Context = (PVOID)&Event;
            
    } 
     
    IoSetCompletionRoutine(
        pIrp,
        CompletionRoutine,
        Context,
        TRUE,
        TRUE,
        TRUE
        );
        
    //
    // Call down to our parent
    //

    Status = IoCallDriver(Extension->PortDeviceObject, pIrp);

    if (Wait && Status == STATUS_PENDING) {

        //
        // Still pending, wait for the IRP to complete
        //

        KeWaitForSingleObject(&Event, Suspended, KernelMode, FALSE, NULL);
        
        Status = pIrp->IoStatus.Status;
            
    }
     
    return Status;
}         

NTSTATUS
ParScanSetupDriverCallback(
    IN PDEVICE_EXTENSION Extension,
    IN PIRP              Irp
    )

/*++

Routine Description:

    This routine will call the next driver in the WDM chain
    (which should be PARCLASS).

Arguments:

    Extension    - Device Extension.

    Irp          - Irp to call parent with.

Return Value:

    NTSTATUS.

--*/

{
    NTSTATUS        Status;

    //
    // Call down to our parent
    //
    Status = Extension->SetupDriverCallback (
                Extension->PortContext, 
                Irp, 
                (PCALLBACK)ParScanExecuteEscape, 
                Extension);
                
    return (Status);
     
}

NTSTATUS
ParScanExecuteEscape(
    IN PDEVICE_EXTENSION Extension,
    IN PIRP              Irp
    )

/*++

Routine Description:

    This routine is called by the parclass dispatch routine just
    after the port has been allocated but prior to starting the
    I/O operation.  Now is the time for us to turn on the dongle.

Arguments:

    DeviceObject    - Supplies the device object.

    Irp             - Supplies the I/O request packet.

Return Value:

    STATUS_MORE_PROCESSING_REQUIRED     - Success.
   !STATUS_MORE_PROCESSING_REQUIRED     - Failure.

--*/

{

    // 
    // WORKWORK  Implementation specific code goes here.
    // To have parclass continue processing the request you must return
    // STATUS_MORE_PROCESSING_REQUIRED.  You can also choose to handle this
    // Io request and return STATUS_SUCCESS to have parclass complete the IRP
    // and return.
    //
    
    return (STATUS_MORE_PROCESSING_REQUIRED);
    
}

NTSTATUS
ParScanCreateOpen(
    IN  PDEVICE_OBJECT  DeviceObject,
    IN  PIRP            Irp
    )

/*++

Routine Description:

    This routine is the dispatch for a create requests.

Arguments:

    DeviceObject    - Supplies the device object.

    Irp             - Supplies the I/O request packet.

Return Value:

    STATUS_SUCCESS  - Success.
    !STATUS_SUCCESS - Failure.

--*/

{
    NTSTATUS            Status;
    PDEVICE_EXTENSION   Extension;
    
    Extension = DeviceObject->DeviceExtension;

    Status = ParScanCallParent(Extension, Irp, WAIT, NULL);
    
    DebugDump(PARIRPPATH, 
              ("PARSCAN: [CreateOpen] After CallParent Status = %x\n", Status));

    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    return Status;
}

NTSTATUS
ParScanClose(
    IN  PDEVICE_OBJECT  DeviceObject,
    IN  PIRP            Irp
    )

/*++

Routine Description:

    This routine is the dispatch for a close requests.

Arguments:

    DeviceObject    - Supplies the device object.

    Irp             - Supplies the I/O request packet.

Return Value:

    STATUS_SUCCESS  - Success.

--*/

{
    NTSTATUS            Status;
    PDEVICE_EXTENSION   Extension;
    
    Extension = DeviceObject->DeviceExtension;

    Status = ParScanCallParent(Extension, Irp, WAIT, NULL);
    
    DebugDump(PARIRPPATH, 
              ("PARSCAN: [Close] After CallParent Status = %x\n", Status));

    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    return Status;
}

NTSTATUS
ParScanCleanup(
    IN  PDEVICE_OBJECT  DeviceObject,
    IN  PIRP            Irp
    )

/*++

Routine Description:

    This routine is the dispatch for a cleanup requests.

Arguments:

    DeviceObject    - Supplies the device object.

    Irp             - Supplies the I/O request packet.

Return Value:

    STATUS_SUCCESS  - Success.

--*/

{
    NTSTATUS            Status;
    PDEVICE_EXTENSION   Extension;
    
    Extension = DeviceObject->DeviceExtension;

    Status = ParScanCallParent(Extension, Irp, WAIT, NULL);
    
    DebugDump(PARIRPPATH, 
              ("PARSCAN: [Cleanup] After CallParent Status = %x\n", Status));

    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    return Status;
}

VOID
ParScanCancelRequest(
    PDEVICE_OBJECT DeviceObject,
    PIRP Irp
    )

/*++

Routine Description:

    This routine is used to cancel any request in the parallel driver.

Arguments:

    DeviceObject - Pointer to the device object for this device

    Irp - Pointer to the IRP to be canceled.

Return Value:

    None.

--*/

{
    NTSTATUS            Status;
    PDEVICE_EXTENSION   Extension;
    
    Extension = DeviceObject->DeviceExtension;

    Status = ParScanCallParent(Extension, Irp, WAIT, NULL);
    
    DebugDump(PARIRPPATH, 
              ("PARSCAN: [Cleanup] After CallParent Status = %x\n", Status));

    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    return;
}

NTSTATUS
ParScanRead(
    IN  PDEVICE_OBJECT  DeviceObject,
    IN  PIRP            Irp
    )

/*++

Routine Description:

    This routine is the dispatch for read and write requests.

Arguments:

    DeviceObject    - Supplies the device object.

    Irp             - Supplies the I/O request packet.

Return Value:

    STATUS_SUCCESS              - Success.
    STATUS_PENDING              - Request pending.
    STATUS_INVALID_PARAMETER    - Invalid parameter.

--*/

{
    NTSTATUS            Status;
    PDEVICE_EXTENSION   Extension;
    
    Extension = DeviceObject->DeviceExtension;

    ParScanSetupDriverCallback(Extension, Irp);
    
    Status = ParScanCallParent(Extension, Irp, NO_WAIT, ParScanCompleteIrp);
    
    DebugDump(PARIRPPATH, 
              ("PARSCAN: [Read] After CallParent Status = %x\n", Status));

    return Status;
}

NTSTATUS
ParScanWrite(
    IN  PDEVICE_OBJECT  DeviceObject,
    IN  PIRP            Irp
    )

/*++

Routine Description:

    This routine is the dispatch for read and write requests.

Arguments:

    DeviceObject    - Supplies the device object.

    Irp             - Supplies the I/O request packet.

Return Value:

    STATUS_SUCCESS              - Success.
    STATUS_PENDING              - Request pending.
    STATUS_INVALID_PARAMETER    - Invalid parameter.

--*/

{
    NTSTATUS            Status;
    PDEVICE_EXTENSION   Extension;
    
    Extension = DeviceObject->DeviceExtension;

    ParScanSetupDriverCallback(Extension, Irp);
    
    Status = ParScanCallParent(Extension, Irp, NO_WAIT, ParScanCompleteIrp);
    
    DebugDump(PARIRPPATH, 
              ("PARSCAN: [Write] After CallParent Status = %x\n", Status));

    return Status;
}

NTSTATUS
ParScanDeviceControl(
    IN  PDEVICE_OBJECT  DeviceObject,
    IN  PIRP            Irp
    )

/*++

Routine Description:

    This routine is the dispatch for device control requests.

Arguments:

    DeviceObject    - Supplies the device object.

    Irp             - Supplies the I/O request packet.

Return Value:

    STATUS_SUCCESS              - Success.
    STATUS_PENDING              - Request pending.
    STATUS_BUFFER_TOO_SMALL     - Buffer too small.
    STATUS_INVALID_PARAMETER    - Invalid io control request.

--*/

{
    NTSTATUS            Status;
    PDEVICE_EXTENSION   Extension;
    
    Extension = DeviceObject->DeviceExtension;

    ParScanSetupDriverCallback(Extension, Irp);
    
    Status = ParScanCallParent(Extension, Irp, WAIT, ParScanCompleteIrp);
    
    DebugDump(PARIRPPATH, 
              ("PARSCAN: [Cleanup] After CallParent Status = %x\n", Status));

    return Status;
}

NTSTATUS
ParScanQueryInformationFile(
    IN  PDEVICE_OBJECT  DeviceObject,
    IN  PIRP            Irp
    )

/*++

Routine Description:

    This routine is used to query the end of file information on
    the opened parallel port.  Any other file information request
    is retured with an invalid parameter.

    This routine always returns an end of file of 0.

Arguments:

    DeviceObject    - Supplies the device object.

    Irp             - Supplies the I/O request packet.

Return Value:

    STATUS_SUCCESS              - Success.
    STATUS_INVALID_PARAMETER    - Invalid file information request.
    STATUS_BUFFER_TOO_SMALL     - Buffer too small.

--*/

{
    NTSTATUS            Status;
    PDEVICE_EXTENSION   Extension;
    
    Extension = DeviceObject->DeviceExtension;

    Status = ParScanCallParent(Extension, Irp, WAIT, NULL);
    
    DebugDump(PARIRPPATH, 
              ("PARSCAN: [Cleanup] After CallParent Status = %x\n", Status));

    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    return Status;
}

NTSTATUS
ParScanSetInformationFile(
    IN  PDEVICE_OBJECT  DeviceObject,
    IN  PIRP            Irp
    )

/*++

Routine Description:

    This routine is used to set the end of file information on
    the opened parallel port.  Any other file information request
    is retured with an invalid parameter.

    This routine always ignores the actual end of file since
    the query information code always returns an end of file of 0.

Arguments:

    DeviceObject    - Supplies the device object.

    Irp             - Supplies the I/O request packet.

Return Value:

    STATUS_SUCCESS              - Success.
    STATUS_INVALID_PARAMETER    - Invalid file information request.

--*/

{
    NTSTATUS            Status;
    PDEVICE_EXTENSION   Extension;
    
    Extension = DeviceObject->DeviceExtension;

    Status = ParScanCallParent(Extension, Irp, WAIT, NULL);
    
    DebugDump(PARIRPPATH, 
              ("PARSCAN: [Cleanup] After CallParent Status = %x\n", Status));

    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    return Status;
}

VOID
ParScanUnload(
    IN  PDRIVER_OBJECT  DriverObject
    )

/*++

Routine Description:

    This routine loops through the device list and cleans up after
    each of the devices.

Arguments:

    DriverObject    - Supplies the driver object.

Return Value:

    None.

--*/

{
    PDEVICE_OBJECT      CurrentDevice;
    PDEVICE_EXTENSION   Extension;

    DebugDump(PARUNLOAD,
              ("PARSCAN: In ParUnload\n"));

    while (CurrentDevice = DriverObject->DeviceObject) {

        Extension = CurrentDevice->DeviceExtension;

        if (Extension->CreatedSymbolicLink) {
        
            IoDeleteSymbolicLink(&Extension->SymbolicLinkName);
            
            RtlDeleteRegistryValue(RTL_REGISTRY_DEVICEMAP,
                                   L"Parallel Scanners",
                                   Extension->SymbolicLinkName.Buffer);
                                   
            ExFreePool(Extension->SymbolicLinkName.Buffer);
        }

        IoDeleteDevice(CurrentDevice);
    }

}

