/*++

Copyright (c) 1993-1997 Microsoft Corporation

Module Name:

    parport.c

Abstract:

    This module contains the code for the parallel port driver.

    This driver creates a device for each parallel port on the
    system.  It increments 'IoGetConfigurationInformation()->ParallelCount'
    once for each parallel port.  Each device created (\Device\ParallelPort0,
    \DeviceParallelPort1,...) supports a number of internal ioctls that
    allow parallel class drivers to get information about the parallel port and
    to share the parallel port.

    IOCTL_INTERNAL_GET_PARALLEL_PORT_INFO returns the location
    and span of the register set for the parallel port.  This ioctl
    also returns callback functions for 'FreePort', 'TryAllocatePort',
    and 'QueryNumWaiters' (more on these below).  This ioctl
    should be called by a class driver during 'DriverEntry' and the
    information added to the class driver's device extension.

    A parallel class driver should never touch the parallel port
    registers returned by IOCTL_INTERNAL_GET_PARALLEL_PORT_INFO
    without first allocating the port from the parallel port driver.

    The port can be allocated from IRQL <= DISPATCH_LEVEL by calling
    IOCTL_INTERNAL_PARALLEL_PORT_ALLOCATE, or 'TryAllocatePort'.
    The first call is the simplest:  the IRP will be queued in the
    parallel port driver until the port is free and then will return
    with a successful status.  The class driver may cancel this IRP
    at any time which serves as a mechanism to timeout an allocate
    request.  It is often convenient to use an incomming read or
    write IRP and pass it down to the port driver as an ALLOCATE.  That
    way the class driver may avoid having to do its own queueing.
    The 'TryAllocatePort' call returns immediately from the port
    driver with a TRUE status if the port was allocated or an
    FALSE status if the port was busy.

    Once the port is allocated, it is owned by the allocating class
    driver until a 'FreePort' call is made.  This releases the port
    and wakes up the next caller.

    The 'QueryNumWaiters' call which can be made at IRQL <= DISPATCH_LEVEL
    is useful to check and see if there are other class drivers waiting for the
    port.  In this way, a class driver that needs to hold on to the
    port for an extended period of time (e.g. tape backup) can let
    go of the port if it detects another class driver needing service.

    If a class driver wishes to use the parallel port's interrupt then
    it should connect to this interrupt by calling
    IOCTL_INTERNAL_PARALLEL_CONNECT_INTERRUPT during its DriverEntry
    routine.  Besides giving the port driver an interrupt service routine
    the class driver may optionally give the port driver a deferred
    port check routine.  Such a routine would be called whenever the
    port is left free.  This would allow the class driver to make sure
    that interrupts were turned on whenever the port was left idle.
    If the driver using the interrupt has an Unload routine
    then it should call IOCTL_INTERNAL_PARALLEL_DISCONNECT_INTERRUPT
    in its Unload routine.

    If a class driver's interrupt service routine is called when this
    class driver does not own the port, the class driver may attempt
    to grap the port quickly if it is free by calling the
    'TryAllocatePortAtInterruptLevel' function.  This function is returned
    when the class driver connects to the interrupt.  The class driver may
    also use the 'FreePortFromInterruptLevel' function to free the port.

    Please refer to the PARSIMP driver code for a template of a simple
    parallel class driver.  This template implements simple allocation and
    freeing of the parallel port on an IRP by IRP basis.  It also
    has optional code for timing out allocation requests and for
    using the parallel port interrupt.

Author:

    Anthony V. Ercolano 1-Aug-1992
    Norbert P. Kusters 18-Oct-1993

Environment:

    Kernel mode

Revision History :

    Timothy T. Wells (v-timtw) 23-Dec-1996
    
    Major revisions to this file.  Moved the older initialization code to parinit.c.  
    To build with the static initialization code, build without WANT_PNP defined.
    
    In the WDM world we will only add a controller to our list of managed controllers
    when told to do so by the configuration manager (as part of the plug and play
    process).  WDM initialization code is now in this file.
        
--*/

#include "ntddk.h"
#include "parallel.h"
#include "parport.h"
#include "parlog.h"

//
// This is the actual definition of ParDebugLevel.
// Note that it is only defined if this is a "debug"
// build.
//

#if DBG
extern ULONG PptDebugLevel = 0;
#endif

extern const PHYSICAL_ADDRESS PhysicalZero = {0};

typedef struct _SYNCHRONIZED_COUNT_CONTEXT {
    PLONG   Count;
    LONG    NewCount;
} SYNCHRONIZED_COUNT_CONTEXT, *PSYNCHRONIZED_COUNT_CONTEXT;

typedef struct _SYNCHRONIZED_LIST_CONTEXT {
    PLIST_ENTRY List;
    PLIST_ENTRY NewEntry;
} SYNCHRONIZED_LIST_CONTEXT, *PSYNCHRONIZED_LIST_CONTEXT;

typedef struct _SYNCHRONIZED_DISCONNECT_CONTEXT {
    PDEVICE_EXTENSION                   Extension;
    PPARALLEL_INTERRUPT_SERVICE_ROUTINE IsrInfo;
} SYNCHRONIZED_DISCONNECT_CONTEXT, *PSYNCHRONIZED_DISCONNECT_CONTEXT;

#ifndef STATIC_LOAD

NTSTATUS
DriverEntry(
    IN  PDRIVER_OBJECT  DriverObject,
    IN  PUNICODE_STRING RegistryPath
    );

NTSTATUS
PptPnpAddDevice(
    IN PDRIVER_OBJECT pDriverObject,
    IN PDEVICE_OBJECT pPhysicalDeviceObject
    );
    
NTSTATUS 
PptPortPnp (
    IN PDEVICE_OBJECT pDeviceObject,
    IN PIRP           pIrp
   );

NTSTATUS
PptSynchCompletionRoutine(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp,
    IN PKEVENT Event
    );
    
#endif

VOID
PptLogError(
    IN  PDRIVER_OBJECT      DriverObject,
    IN  PDEVICE_OBJECT      DeviceObject OPTIONAL,
    IN  PHYSICAL_ADDRESS    P1,
    IN  PHYSICAL_ADDRESS    P2,
    IN  ULONG               SequenceNumber,
    IN  UCHAR               MajorFunctionCode,
    IN  UCHAR               RetryCount,
    IN  ULONG               UniqueErrorValue,
    IN  NTSTATUS            FinalStatus,
    IN  NTSTATUS            SpecificIOStatus
    );

PVOID
PptGetMappedAddress(
    IN  INTERFACE_TYPE      BusType,
    IN  ULONG               BusNumber,
    IN  PHYSICAL_ADDRESS    IoAddress,
    IN  ULONG               NumberOfBytes,
    IN  ULONG               AddressSpace,
    OUT PBOOLEAN            MappedAddress
    );
    
VOID
PptReportResourcesDevice(
    IN  PDEVICE_EXTENSION   Extension,
    IN  BOOLEAN             ClaimInterrupt,
    OUT PBOOLEAN            ConflictDetected
    );

VOID
PptUnReportResourcesDevice(
    IN OUT  PDEVICE_EXTENSION   Extension
    );
    
NTSTATUS
PptConnectInterrupt(
    IN  PDEVICE_EXTENSION   Extension
    );
    
VOID
PptDisconnectInterrupt(
    IN  PDEVICE_EXTENSION   Extension
    );
    
NTSTATUS
PptDispatchCreateClose(
    IN  PDEVICE_OBJECT  DeviceObject,
    IN  PIRP            Irp
    );

BOOLEAN
PptSynchronizedIncrement(
    IN OUT  PVOID   SyncContext
    );
    
BOOLEAN
PptSynchronizedDecrement(
    IN OUT  PVOID   SyncContext
    );
    
BOOLEAN
PptSynchronizedRead(
    IN OUT  PVOID   SyncContext
    );
    
BOOLEAN
PptSynchronizedQueue(
    IN  PVOID   Context
    );
    
BOOLEAN
PptSynchronizedDisconnect(
    IN  PVOID   Context
    );
    
VOID
PptCancelRoutine(
    IN OUT  PDEVICE_OBJECT  DeviceObject,
    IN OUT  PIRP            Irp
    );
    
VOID
PptFreePortDpc(
    IN      PKDPC   Dpc,
    IN OUT  PVOID   Extension,
    IN      PVOID   SystemArgument1,
    IN      PVOID   SystemArgument2
    );
    
BOOLEAN
PptTryAllocatePortAtInterruptLevel(
    IN  PVOID   Context
    );
    
VOID
PptFreePortFromInterruptLevel(
    IN  PVOID   Context
    );
    
BOOLEAN
PptInterruptService(
    IN  PKINTERRUPT Interrupt,
    IN  PVOID       Extension
    );

BOOLEAN
PptTryAllocatePort(
    IN  PVOID   Extension
    );

BOOLEAN
PptTraversePortCheckList(
    IN  PVOID   Extension
    );
    
VOID
PptFreePort(
    IN  PVOID   Extension
    );

ULONG
PptQueryNumWaiters(
    IN  PVOID   Extension
    );
    
NTSTATUS
PptDispatchDeviceControl(
    IN  PDEVICE_OBJECT  DeviceObject,
    IN  PIRP            Irp
    );

VOID
PptCleanupDevice(
    IN  PDEVICE_EXTENSION   Extension
    );
    
NTSTATUS
PptDispatchCleanup(
    IN  PDEVICE_OBJECT  DeviceObject,
    IN  PIRP            Irp
    );

VOID
PptUnload(
    IN  PDRIVER_OBJECT  DriverObject
    );

BOOLEAN
PptIsNecR98Machine(
    void
    );


//
// PptGetMappedAddress can be in the INIT segment if PNP isn't going to load us.
//
#ifdef STATIC_LOAD
#ifdef ALLOC_PRAGMA
#pragma alloc_text(INIT,PptGetMappedAddress)
#endif
#endif

//
// Keep track of GET and RELEASE port info.
//
ULONG PortInfoReferenceCount = 0;
PFAST_MUTEX PortInfoMutex = NULL;

//
// The initialization code will vary depending upon whether the driver has been
// built for PNP or statically loaded.  Compile Flag STATIC_LOAD determines...
//

#ifndef STATIC_LOAD


NTSTATUS
DriverEntry(
    IN PDRIVER_OBJECT DriverObject,
    IN PUNICODE_STRING RegistryPath
    )

/*++

Routine Description:

    This routine is called at system initialization time to initialize
    this driver.

Arguments:

    DriverObject    - Supplies the driver object.

    RegistryPath    - Supplies the registry path for this driver.

Return Value:

    STATUS_SUCCESS

--*/

{
    //
    // Initialize the Driver Object with driver's entry points.
    // This is the first step in the PNP process.  We'll be told
    // by the configuration manager to add a device next.
    //

    DriverObject->MajorFunction[IRP_MJ_CREATE]                  = PptDispatchCreateClose;
    DriverObject->MajorFunction[IRP_MJ_CLOSE]                   = PptDispatchCreateClose;
    DriverObject->MajorFunction[IRP_MJ_INTERNAL_DEVICE_CONTROL] = PptDispatchDeviceControl;
    DriverObject->MajorFunction[IRP_MJ_CLEANUP]                 = PptDispatchCleanup;
    DriverObject->MajorFunction[IRP_MJ_PNP]                     = PptPortPnp;    
    DriverObject->DriverExtension->AddDevice                    = PptPnpAddDevice;
    DriverObject->DriverUnload                                  = PptUnload;

    return STATUS_SUCCESS;
}

NTSTATUS
PptPnpAddDevice(
    IN PDRIVER_OBJECT pDriverObject,
    IN PDEVICE_OBJECT pPhysicalDeviceObject
    )
/*++

Routine Description:

    This routine is called to create a new instance of the device.

Arguments:

    pDriverObject           - pointer to the driver object for this instance of parport.

    pPhysicalDeviceObject   - pointer to the device object that represents the port.

Return Value:

    STATUS_SUCCESS          - if successful.
    STATUS_UNSUCCESSFUL     - otherwise.

--*/
{
    UNICODE_STRING      uniNameString;
    UNICODE_STRING      uniDeviceString;
    UNICODE_STRING      uniBaseNameString;
    UNICODE_STRING      uniPortNumberString;
    WCHAR               wcPortNum[10];
    ULONG               ParallelCount;
    NTSTATUS            status;
    PDEVICE_EXTENSION   extension;
    PDEVICE_OBJECT      pDeviceObject;

    //
    // We'll do a number of things in this routine.  We'll create a device object,
    // setup some symbolic links, and initialize the device extension. We won't 
    // claim any resources until we get the start notification from Plug-N-Play.
    
    //
    // Form a name like \Device\ParallelPort0.
    // First we allocate space for the name.
    //

    RtlInitUnicodeString(&uniDeviceString, L"\\Device\\");
    RtlInitUnicodeString(&uniBaseNameString, DD_PARALLEL_PORT_BASE_NAME_U);
    RtlInitUnicodeString(&uniNameString, NULL);
    
    uniPortNumberString.Length = 0;
    uniPortNumberString.MaximumLength = 10;
    uniPortNumberString.Buffer = wcPortNum;
    
    ParallelCount = IoGetConfigurationInformation()->ParallelCount;
    
    status = RtlIntegerToUnicodeString (ParallelCount, 10, &uniPortNumberString);
    if (!NT_SUCCESS(status)) {

        PptLogError(pDriverObject, NULL, PhysicalZero, PhysicalZero,
                    0, 0, 0, 4, STATUS_SUCCESS, PAR_INSUFFICIENT_RESOURCES);

        ParDump(PARERRORS,
                ("PARPORT:  Couldn't convert controller number to UNICODE for Device Name\n"));

        return STATUS_INSUFFICIENT_RESOURCES;
    }
    
    uniNameString.MaximumLength = uniDeviceString.Length +
                                  uniBaseNameString.Length +
                                  uniPortNumberString.Length +
                                  sizeof(WCHAR);
                                  
    uniNameString.Buffer = ExAllocatePool(PagedPool, uniNameString.MaximumLength);
    
    if (!uniNameString.Buffer) {

        PptLogError(pDriverObject, NULL, PhysicalZero, PhysicalZero,
                    0, 0, 0, 8, STATUS_SUCCESS, PAR_INSUFFICIENT_RESOURCES);

        ParDump(PARERRORS,
                ("PARPORT:  Could not form Unicode name string for port %d\n",
                ParallelCount));

        return STATUS_INSUFFICIENT_RESOURCES;
    }

    RtlZeroMemory(uniNameString.Buffer, uniNameString.MaximumLength);
    RtlAppendUnicodeStringToString(&uniNameString, &uniDeviceString);
    RtlAppendUnicodeStringToString(&uniNameString, &uniBaseNameString);
    RtlAppendUnicodeStringToString(&uniNameString, &uniPortNumberString);

    //
    // Create the device object for this device.
    //

    status = IoCreateDevice(pDriverObject, 
                            sizeof(DEVICE_EXTENSION),
                            &uniNameString, 
                            FILE_DEVICE_PARALLEL_PORT,
                            0, 
                            FALSE, 
                            &pDeviceObject);

    ExFreePool(uniNameString.Buffer);
    
    if (!NT_SUCCESS(status)) {

        PptLogError(pDriverObject, NULL, PhysicalZero, PhysicalZero,
                    0, 0, 0, 9, STATUS_SUCCESS, PAR_INSUFFICIENT_RESOURCES);

        ParDump(PARERRORS,
                ("PARPORT:  Could not create a device for %d\n",
                ParallelCount));

        return status;
    }

    //
    // The device object has a pointer to an area of non-paged
    // pool allocated for this device.  This will be the device
    // extension.
    //

    extension = pDeviceObject->DeviceExtension;

    //
    // Zero all of the memory associated with the device
    // extension.
    //

    RtlZeroMemory(extension, sizeof(DEVICE_EXTENSION));

    //
    // Get a "back pointer" to the device object.
    //

    extension->DeviceObject = pDeviceObject;

    //
    // Attach our new Device to our parents stack.
    //
    extension->ParentDeviceObject = IoAttachDeviceToDeviceStack(
                                    pDeviceObject, 
                                    pPhysicalDeviceObject);
                                    
    if (NULL == extension->ParentDeviceObject) {
    
        IoDeleteDevice( pDeviceObject );
        
        return STATUS_NOT_SUPPORTED;
    }

    //
    // We have created the device, so increment the counter in the
    // IO system that keeps track.  Anyplace that we do an IoDeleteDevice
    // we need to decrement.
    //

    IoGetConfigurationInformation()->ParallelCount++;

    return STATUS_SUCCESS;

}

NTSTATUS
PptPortPnp (
    IN PDEVICE_OBJECT pDeviceObject,
    IN PIRP           pIrp
   )
/*++

Routine Description:

    This routine handles all PNP IRPs.

Arguments:

    pDeviceObject           - represents the port device
    
    pIrp                    - PNP irp 
    
Return Value:

    STATUS_SUCCESS          - if successful.
    STATUS_UNSUCCESSFUL     - otherwise.

--*/
{
    NTSTATUS                        Status = STATUS_NOT_SUPPORTED;
    PDEVICE_EXTENSION               Extension;
    PIO_STACK_LOCATION              pIrpStack;
    PIO_STACK_LOCATION              pNextIrpStack;
    BOOLEAN                         FoundPort = FALSE;
    BOOLEAN                         FoundInterrupt = FALSE;
    BOOLEAN                         FoundDma = FALSE;
    BOOLEAN                         Conflict = FALSE;
    PCM_RESOURCE_LIST               ResourceList;
    PCM_FULL_RESOURCE_DESCRIPTOR    FullResourceDescriptor;
    PCM_PARTIAL_RESOURCE_LIST       PartialResourceList;
    PCM_PARTIAL_RESOURCE_DESCRIPTOR PartialResourceDescriptor;
    KEVENT                          Event;
    ULONG                           i;
    
    pIrpStack = IoGetCurrentIrpStackLocation( pIrp );
    
    Extension = pDeviceObject->DeviceExtension;
    
    switch (pIrpStack->MinorFunction) {
        
        case IRP_MN_START_DEVICE:
        
            //
            // Prepare to call down to the parent with the PNP Request...
            //
        
            pNextIrpStack = IoGetNextIrpStackLocation(pIrp);
            pNextIrpStack->MajorFunction = pIrpStack->MajorFunction;
            pNextIrpStack->MinorFunction = pIrpStack->MinorFunction;
            RtlCopyMemory(&pNextIrpStack->Parameters, &pIrpStack->Parameters, sizeof(pIrpStack->Parameters.Others));

            KeInitializeEvent(&Event, NotificationEvent, FALSE);

            IoSetCompletionRoutine(
                pIrp,
                PptSynchCompletionRoutine,
                &Event,
                TRUE,
                TRUE,
                TRUE
                );

            //
            // Call down to our parent
            //

            Status = IoCallDriver(Extension->ParentDeviceObject, pIrp);

            if (Status == STATUS_PENDING) {

                //
                // Still pending, wait for the IRP to complete
                //

                KeWaitForSingleObject(&Event, Suspended, KernelMode, FALSE, NULL);

            }

            if (!NT_SUCCESS(Status)) {

                pIrp->IoStatus.Status = Status;
                pIrp->IoStatus.Information = 0;

                IoCompleteRequest(pIrp, IO_NO_INCREMENT);
                return (Status);

            }

            //
            // Get the configuration information from Plug-N-Play.
            //

            ResourceList = pIrpStack->Parameters.StartDevice.AllocatedResources;

            FullResourceDescriptor = &ResourceList->List[0];
            
            Extension->InterfaceType = FullResourceDescriptor->InterfaceType;
            Extension->BusNumber     = FullResourceDescriptor->BusNumber;
            
            PartialResourceList = &FullResourceDescriptor->PartialResourceList;
            
            for (i=0; i < PartialResourceList->Count; i++) {

                PartialResourceDescriptor = &PartialResourceList->PartialDescriptors[i];
                
                switch (PartialResourceDescriptor->Type) {

					case CmResourceTypePort:
                    
                        FoundPort = TRUE;
                        
                        Extension->PortInfo.OriginalController = PartialResourceDescriptor->u.Port.Start;
                        Extension->PortInfo.SpanOfController   = PartialResourceDescriptor->u.Port.Length;
                        Extension->AddressSpace                = PartialResourceDescriptor->Flags;    
                        
                        if ((Extension->PortInfo.SpanOfController == 0x1000) &&
                            (PptIsNecR98Machine())) {
                            
                            //
                            // Firmware bug with R98 machine.
                            // 
                            Extension->PortInfo.SpanOfController = 8;
                        }        
                        
                        break;
                        
                    case CmResourceTypeInterrupt:
                    
                        FoundInterrupt = TRUE;
                        
                        Extension->InterruptLevel       = PartialResourceDescriptor->u.Interrupt.Level;
                        Extension->InterruptVector      = PartialResourceDescriptor->u.Interrupt.Vector;
                        Extension->InterruptAffinity    = PartialResourceDescriptor->u.Interrupt.Affinity;
                        
                        if (PartialResourceDescriptor->Flags & CM_RESOURCE_INTERRUPT_LATCHED) {
                        
                            Extension->InterruptMode = Latched;
                        } else {
                        
                            Extension->InterruptMode = LevelSensitive;
                        }
                        
                        break;
                        
                    case CmResourceTypeDma:
                    
                        FoundDma = TRUE;
                        
                        Extension->DmaChannel   = PartialResourceDescriptor->u.Dma.Channel;
                        Extension->DmaPort      = PartialResourceDescriptor->u.Dma.Port;
                        Extension->DmaWidth     = PartialResourceDescriptor->Flags;
                        
                        break;
                        
                    default:
                    
                        break;
                        
                } 
            } 
               
            if (FoundPort) {
            
                //
                // Initialize 'WorkQueue' in extension.
                //
    
                InitializeListHead(&Extension->WorkQueue);
                Extension->WorkQueueCount = -1;


                //
                // Map the memory for the control registers for the parallel device
                // into virtual memory.
                //
                
                Extension->PortInfo.Controller = PptGetMappedAddress (
                        Extension->InterfaceType, Extension->BusNumber,
                        Extension->PortInfo.OriginalController, Extension->PortInfo.SpanOfController,
                        (BOOLEAN)Extension->AddressSpace, &Extension->UnMapRegisters);
                        
            
                if (!Extension->PortInfo.Controller) {
                    
                    PptLogError(pDeviceObject->DriverObject, pDeviceObject,
                            Extension->PortInfo.OriginalController, PhysicalZero, 0, 0, 0, 10,
                            STATUS_SUCCESS, PAR_REGISTERS_NOT_MAPPED);
                    
                    Extension->UnMapRegisters = FALSE;
                    Status = STATUS_NONE_MAPPED;
                    break;
                }            
                
                Extension->PortInfo.FreePort        = PptFreePort;
                Extension->PortInfo.TryAllocatePort = PptTryAllocatePort;
                Extension->PortInfo.QueryNumWaiters = PptQueryNumWaiters;
                Extension->PortInfo.Context         = Extension;
                
                //
                // Start off with an empty list of interrupt service routines.
                // Also, start off without the interrupt connected.
                //
                InitializeListHead(&Extension->IsrList);
                Extension->InterruptObject = NULL;
                Extension->InterruptRefCount = 0;

                //
                // Initialize the free port DPC.
                //
                KeInitializeDpc(&Extension->FreePortDpc, PptFreePortDpc, Extension);

                //
                // BUGBUG:  This is doing alot more work than it needs to, we should just pass
                //          along the resource list we already have...
                //
                
                PptReportResourcesDevice (Extension, FALSE, &Conflict);
                
                if (Conflict) {
                    
                    PptLogError(pDeviceObject->DriverObject, pDeviceObject,
                            Extension->PortInfo.OriginalController, PhysicalZero, 0, 0, 0, 11,
                            STATUS_SUCCESS, PAR_ADDRESS_CONFLICT);
                    
                    PptCleanupDevice (Extension);
                    
                    Status = STATUS_NO_SUCH_DEVICE;
                    break;
                }            
                
            } else {
            
                Status = STATUS_NO_SUCH_DEVICE;
                break;
                
            }      
            
#ifndef MEMPHIS
            //
            // Let this driver be paged until someone requests PORT_INFO.
            //                    
            PortInfoMutex = ExAllocatePool (NonPagedPool, sizeof(FAST_MUTEX));
            
            if (!PortInfoMutex) {
            
                PptUnReportResourcesDevice (Extension);
                
                PptCleanupDevice(Extension);
                
                Status = STATUS_INSUFFICIENT_RESOURCES;
                break;
            }
            
            ExInitializeFastMutex (PortInfoMutex);
            
            MmPageEntireDriver (DriverEntry);    
#endif
            
            Status = STATUS_SUCCESS;
            break;

        case IRP_MN_REMOVE_DEVICE:
            Status = STATUS_SUCCESS;
            break;            

        case IRP_MN_STOP_DEVICE:
            Status = STATUS_SUCCESS;
            break;
            
        default:
            break;            
    }
    
    //
    // Complete the IRP...
    //
    
    pIrp->IoStatus.Status = Status;
    pIrp->IoStatus.Information = 0;

    IoCompleteRequest(pIrp, IO_NO_INCREMENT);
                
    return( Status );
    
}

NTSTATUS
PptSynchCompletionRoutine(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp,
    IN PKEVENT Event
    )

/*++

Routine Description:

    This routine is for use with synchronous IRP processing.  
    All it does is signal an event, so the driver knows it
    can continue.

Arguments:

    DriverObject - Pointer to driver object created by system.

    Irp          - Irp that just completed

    Event        - Event we'll signal to say Irp is done

Return Value:

    None.

--*/

{

    KeSetEvent((PKEVENT) Event, 0, FALSE);
    
    return (STATUS_MORE_PROCESSING_REQUIRED);

}

#endif // STATIC_LOAD


VOID
PptLogError(
    IN  PDRIVER_OBJECT      DriverObject,
    IN  PDEVICE_OBJECT      DeviceObject OPTIONAL,
    IN  PHYSICAL_ADDRESS    P1,
    IN  PHYSICAL_ADDRESS    P2,
    IN  ULONG               SequenceNumber,
    IN  UCHAR               MajorFunctionCode,
    IN  UCHAR               RetryCount,
    IN  ULONG               UniqueErrorValue,
    IN  NTSTATUS            FinalStatus,
    IN  NTSTATUS            SpecificIOStatus
    )

/*++

Routine Description:

    This routine allocates an error log entry, copies the supplied data
    to it, and requests that it be written to the error log file.

Arguments:

    DriverObject        - Supplies a pointer to the driver object for the
                            device.

    DeviceObject        - Supplies a pointer to the device object associated
                            with the device that had the error, early in
                            initialization, one may not yet exist.

    P1,P2               - Supplies the physical addresses for the controller
                            ports involved with the error if they are available
                            and puts them through as dump data.

    SequenceNumber      - Supplies a ulong value that is unique to an IRP over
                            the life of the irp in this driver - 0 generally
                            means an error not associated with an irp.

    MajorFunctionCode   - Supplies the major function code of the irp if there
                            is an error associated with it.

    RetryCount          - Supplies the number of times a particular operation
                            has been retried.

    UniqueErrorValue    - Supplies a unique long word that identifies the
                            particular call to this function.

    FinalStatus         - Supplies the final status given to the irp that was
                            associated with this error.  If this log entry is
                            being made during one of the retries this value
                            will be STATUS_SUCCESS.

    SpecificIOStatus    - Supplies the IO status for this particular error.

Return Value:

    None.

--*/

{
    PIO_ERROR_LOG_PACKET    errorLogEntry;
    PVOID                   objectToUse;
    SHORT                   dumpToAllocate;

    if (ARGUMENT_PRESENT(DeviceObject)) {
        objectToUse = DeviceObject;
    } else {
        objectToUse = DriverObject;
    }

    dumpToAllocate = 0;

    // if (PptMemCompare(P1, 1, PhysicalZero, 1) != AddressesAreEqual) {
    //     dumpToAllocate = (SHORT) sizeof(PHYSICAL_ADDRESS);
    // }

    // if (PptMemCompare(P2, 1, PhysicalZero, 1) != AddressesAreEqual) {
    //     dumpToAllocate += (SHORT) sizeof(PHYSICAL_ADDRESS);
    // }

    if (P1.LowPart != 0 || P1.HighPart != 0) {
        dumpToAllocate = (SHORT) sizeof(PHYSICAL_ADDRESS);
    }

    if (P2.LowPart != 0 || P2.HighPart != 0) {
        dumpToAllocate += (SHORT) sizeof(PHYSICAL_ADDRESS);
    }
    
    errorLogEntry = IoAllocateErrorLogEntry(objectToUse,
            (UCHAR) (sizeof(IO_ERROR_LOG_PACKET) + dumpToAllocate));

    if (!errorLogEntry) {
        return;
    }

    errorLogEntry->ErrorCode = SpecificIOStatus;
    errorLogEntry->SequenceNumber = SequenceNumber;
    errorLogEntry->MajorFunctionCode = MajorFunctionCode;
    errorLogEntry->RetryCount = RetryCount;
    errorLogEntry->UniqueErrorValue = UniqueErrorValue;
    errorLogEntry->FinalStatus = FinalStatus;
    errorLogEntry->DumpDataSize = dumpToAllocate;

    if (dumpToAllocate) {

        RtlCopyMemory(errorLogEntry->DumpData, &P1, sizeof(PHYSICAL_ADDRESS));

        if (dumpToAllocate > sizeof(PHYSICAL_ADDRESS)) {

            RtlCopyMemory(((PUCHAR) errorLogEntry->DumpData) +
                          sizeof(PHYSICAL_ADDRESS), &P2,
                          sizeof(PHYSICAL_ADDRESS));
        }
    }

    IoWriteErrorLogEntry(errorLogEntry);
}


PVOID
PptGetMappedAddress(
    IN  INTERFACE_TYPE      BusType,
    IN  ULONG               BusNumber,
    IN  PHYSICAL_ADDRESS    IoAddress,
    IN  ULONG               NumberOfBytes,
    IN  ULONG               AddressSpace,
    OUT PBOOLEAN            MappedAddress
    )

/*++

Routine Description:

    This routine maps an IO address to system address space.

Arguments:

    BusType         - Supplies the type of bus - eisa, mca, isa.

    IoBusNumber     - Supplies the bus number.

    IoAddress       - Supplies the base device address to be mapped.

    NumberOfBytes   - Supplies the number of bytes for which the address is
                        valid.

    AddressSpace    - Supplies whether the address is in io space or memory.

    MappedAddress   - Supplies whether the address was mapped. This only has
                        meaning if the address returned is non-null.

Return Value:

    The mapped address.

--*/

{
    PHYSICAL_ADDRESS    cardAddress;
    PVOID               address;

    if (!HalTranslateBusAddress(BusType, BusNumber, IoAddress, &AddressSpace,
                                &cardAddress)) {

        AddressSpace = 1;
        cardAddress.QuadPart = 0;
    }

    //
    // Map the device base address into the virtual address space
    // if the address is in memory space.
    //

    if (!AddressSpace) {

        address = MmMapIoSpace(cardAddress, NumberOfBytes, FALSE);
        *MappedAddress = (address ? TRUE : FALSE);

    } else {

        address = (PVOID) cardAddress.LowPart;
        *MappedAddress = FALSE;
    }

    return address;
}

VOID
PptReportResourcesDevice(
    IN  PDEVICE_EXTENSION   Extension,
    IN  BOOLEAN             ClaimInterrupt,
    OUT PBOOLEAN            ConflictDetected
    )

/*++

Routine Description:

    This routine reports the resources used for a device that
    is "ready" to run.  If some conflict was detected, it doesn't
    matter, the reources are reported.

Arguments:

    Extension           - Supplies the device extension of the device we are
                            reporting resources for.

    ClaimInterrupt      - Supplies whether or not to claim the interrupt.

    ConflictDetected    - Returns whether or not a conflict was detected.

Return Value:

    None.

--*/

{
    PCM_RESOURCE_LIST               resourceList;
    PCM_PARTIAL_RESOURCE_DESCRIPTOR partial;
    UNICODE_STRING                  className;
    ULONG                           sizeofResourceList;

    ParDump(
        PARCONFIG,
        ("PARPORT:  In PptReportResourcesDevice\n"
         "--------  for extension %x\n",
         Extension)
        );

    //
    // The resource list for a device will consist of
    //
    // The resource list record itself with a count
    // of one for the single "built in" full resource
    // descriptor.
    //
    // The built-in full resource descriptor will contain
    // the bus type and busnumber and the built in partial
    // resource list.
    //
    // The built in partial resource list will have a count of 2:
    //
    //     1) The base register physical address and it's span.
    //
    //     2) The interrupt.
    //

    sizeofResourceList = sizeof(CM_RESOURCE_LIST);
    if (ClaimInterrupt) {
        sizeofResourceList += sizeof(CM_PARTIAL_RESOURCE_DESCRIPTOR);
    }
    resourceList = ExAllocatePool(PagedPool, sizeofResourceList);

    if (!resourceList) {

        //
        // Oh well, can't allocate the memory.  Act as though
        // we succeeded.
        //

        PptLogError(Extension->DeviceObject->DriverObject,
                    Extension->DeviceObject,
                    Extension->PortInfo.OriginalController,
                    PhysicalZero, 0, 0, 0, 13, STATUS_SUCCESS,
                    PAR_INSUFFICIENT_RESOURCES);

        return;
    }

    RtlZeroMemory(resourceList, sizeofResourceList);

    resourceList->Count = 1;

    resourceList->List[0].InterfaceType = Extension->InterfaceType;
    resourceList->List[0].BusNumber = Extension->BusNumber;
    resourceList->List[0].PartialResourceList.Count = 1;
    partial = &resourceList->List[0].PartialResourceList.PartialDescriptors[0];

    //
    // Account for the space used by the controller.
    //

    partial->Type = CmResourceTypePort;
    partial->ShareDisposition = CmResourceShareDeviceExclusive;
    partial->Flags = (USHORT) Extension->AddressSpace;
    partial->u.Port.Start = Extension->PortInfo.OriginalController;
    partial->u.Port.Length = Extension->PortInfo.SpanOfController;

    if (ClaimInterrupt) {
        partial++;
        resourceList->List[0].PartialResourceList.Count += 1;
        partial->Type = CmResourceTypeInterrupt;
        partial->ShareDisposition = CmResourceShareShared;
        if (Extension->InterruptMode == Latched) {
            partial->Flags = CM_RESOURCE_INTERRUPT_LATCHED;
        } else {
            partial->Flags = CM_RESOURCE_INTERRUPT_LEVEL_SENSITIVE;
        }
        partial->u.Interrupt.Vector = Extension->InterruptVector;
        partial->u.Interrupt.Level = Extension->InterruptLevel;
        partial->u.Interrupt.Affinity = Extension->InterruptAffinity;
    }

    RtlInitUnicodeString(&className, L"LOADED PARALLEL DRIVER RESOURCES");

    IoReportResourceUsage(&className, Extension->DeviceObject->DriverObject,
                          NULL, 0, Extension->DeviceObject, resourceList,
                          sizeofResourceList, FALSE, ConflictDetected);

    ExFreePool(resourceList);
}

VOID
PptUnReportResourcesDevice(
    IN OUT  PDEVICE_EXTENSION   Extension
    )

/*++

Routine Description:

    Purge the resources for this particular device.

Arguments:

    Extension - The device extension of the device we are *un*reporting
                resources for.

Return Value:

    None.

--*/

{
    CM_RESOURCE_LIST    resourceList;
    UNICODE_STRING      className;
    BOOLEAN             junkBoolean;

    ParDump(
        PARUNLOAD,
        ("PARPORT:  In PptUnreportResourcesDevice\n"
         "--------  for extension %x of port %x\n",
         Extension, Extension->PortInfo.OriginalController.LowPart)
        );

    RtlZeroMemory(&resourceList, sizeof(CM_RESOURCE_LIST));

    resourceList.Count = 0;

    RtlInitUnicodeString(&className, L"LOADED PARALLEL DRIVER RESOURCES");

    IoReportResourceUsage(&className, Extension->DeviceObject->DriverObject,
                          NULL, 0, Extension->DeviceObject, &resourceList,
                          sizeof(CM_RESOURCE_LIST), FALSE, &junkBoolean);
}

NTSTATUS
PptConnectInterrupt(
    IN  PDEVICE_EXTENSION   Extension
    )

/*++

Routine Description:

    This routine connects the port interrupt service routine
    to the interrupt.

Arguments:

    Extension   - Supplies the device extension.

Return Value:

    NTSTATUS code.

--*/

{
    BOOLEAN     conflict;
    ULONG       interruptVector;
    KIRQL       irql;
    KAFFINITY   affinity;
    NTSTATUS    status = STATUS_SUCCESS;

    PptReportResourcesDevice(Extension, TRUE, &conflict);

    if (conflict) {
        status = STATUS_NO_SUCH_DEVICE;
    }

    if (NT_SUCCESS(status)) {

        //
        // Connect the interrupt.
        //

        interruptVector = HalGetInterruptVector(Extension->InterfaceType,
                                                Extension->BusNumber,
                                                Extension->InterruptLevel,
                                                Extension->InterruptVector,
                                                &irql, &affinity);

        status = IoConnectInterrupt(&Extension->InterruptObject,
                                    PptInterruptService, Extension, NULL,
                                    interruptVector, irql, irql,
                                    Extension->InterruptMode, TRUE, affinity,
                                    FALSE);
    }

    if (!NT_SUCCESS(status)) {

        PptReportResourcesDevice(Extension, FALSE, &conflict);

        PptLogError(Extension->DeviceObject->DriverObject,
                    Extension->DeviceObject,
                    Extension->PortInfo.OriginalController,
                    PhysicalZero, 0, 0, 0, 14,
                    status, PAR_INTERRUPT_CONFLICT);

        ParDump(
            PARERRORS,
            ("PARPORT:  Could not connect to interrupt for %x\n",
              Extension->PortInfo.OriginalController.LowPart)
            );

    }

    return status;
}

VOID
PptDisconnectInterrupt(
    IN  PDEVICE_EXTENSION   Extension
    )

/*++

Routine Description:

    This routine disconnects the port interrupt service routine
    from the interrupt.

Arguments:

    Extension   - Supplies the device extension.

Return Value:

    None.

--*/

{
    BOOLEAN conflict;

    IoDisconnectInterrupt(Extension->InterruptObject);
    PptReportResourcesDevice(Extension, FALSE, &conflict);
}

NTSTATUS
PptDispatchCreateClose(
    IN  PDEVICE_OBJECT  DeviceObject,
    IN  PIRP            Irp
    )

/*++

Routine Description:

    This routine is the dispatch for create and close requests.  This
    request completes successfully.

Arguments:

    DeviceObject    - Supplies the device object.
    Irp             - Supplies the I/O request packet.

Return Value:

    STATUS_SUCCESS  - Success.

--*/

{
    ParDump(
        PARIRPPATH,
        ("PARPORT:  In create or close with IRP: %x\n",
         Irp)
        );

    Irp->IoStatus.Status = STATUS_SUCCESS;
    Irp->IoStatus.Information = 0;

    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    return STATUS_SUCCESS;
}

BOOLEAN
PptSynchronizedIncrement(
    IN OUT  PVOID   SyncContext
    )

/*++

Routine Description:

    This routine increments the 'Count' variable in the context and returns
    its new value in the 'NewCount' variable.

Arguments:

    SyncContext - Supplies the synchronize count context.

Return Value:

    TRUE

--*/

{
    ((PSYNCHRONIZED_COUNT_CONTEXT) SyncContext)->NewCount =
            ++(*(((PSYNCHRONIZED_COUNT_CONTEXT) SyncContext)->Count));
    return(TRUE);
}

BOOLEAN
PptSynchronizedDecrement(
    IN OUT  PVOID   SyncContext
    )

/*++

Routine Description:

    This routine decrements the 'Count' variable in the context and returns
    its new value in the 'NewCount' variable.

Arguments:

    SyncContext - Supplies the synchronize count context.

Return Value:

    TRUE

--*/

{
    ((PSYNCHRONIZED_COUNT_CONTEXT) SyncContext)->NewCount =
            --(*(((PSYNCHRONIZED_COUNT_CONTEXT) SyncContext)->Count));
    return(TRUE);
}

BOOLEAN
PptSynchronizedRead(
    IN OUT  PVOID   SyncContext
    )

/*++

Routine Description:

    This routine reads the 'Count' variable in the context and returns
    its value in the 'NewCount' variable.

Arguments:

    SyncContext - Supplies the synchronize count context.

Return Value:

    None.

--*/

{
    ((PSYNCHRONIZED_COUNT_CONTEXT) SyncContext)->NewCount =
            *(((PSYNCHRONIZED_COUNT_CONTEXT) SyncContext)->Count);
    return(TRUE);
}

BOOLEAN
PptSynchronizedQueue(
    IN  PVOID   Context
    )

/*++

Routine Description:

    This routine adds the given list entry to the given list.

Arguments:

    Context - Supplies the synchronized list context.

Return Value:

    TRUE

--*/

{
    PSYNCHRONIZED_LIST_CONTEXT  listContext;

    listContext = Context;
    InsertTailList(listContext->List, listContext->NewEntry);
    return(TRUE);
}

BOOLEAN
PptSynchronizedDisconnect(
    IN  PVOID   Context
    )

/*++

Routine Description:

    This routine removes the given list entry from the ISR
    list.

Arguments:

    Context - Supplies the synchronized disconnect context.

Return Value:

    FALSE   - The given list entry was not removed from the list.
    TRUE    - The given list entry was removed from the list.

--*/

{
    PSYNCHRONIZED_DISCONNECT_CONTEXT    disconnectContext;
    PKSERVICE_ROUTINE                   ServiceRoutine;
    PVOID                               ServiceContext;
    PLIST_ENTRY                         current;
    PISR_LIST_ENTRY                     listEntry;

    disconnectContext = Context;
    ServiceRoutine = disconnectContext->IsrInfo->InterruptServiceRoutine;
    ServiceContext = disconnectContext->IsrInfo->InterruptServiceContext;

    for (current = disconnectContext->Extension->IsrList.Flink;
         current != &(disconnectContext->Extension->IsrList);
         current = current->Flink) {

        listEntry = CONTAINING_RECORD(current, ISR_LIST_ENTRY, ListEntry);
        if (listEntry->ServiceRoutine == ServiceRoutine &&
            listEntry->ServiceContext == ServiceContext) {

            RemoveEntryList(current);
            return TRUE;
        }
    }

    return FALSE;
}

VOID
PptCancelRoutine(
    IN OUT  PDEVICE_OBJECT  DeviceObject,
    IN OUT  PIRP            Irp
    )

/*++

Routine Description:

    This routine is called on when the given IRP is cancelled.  It
    will dequeue this IRP off the work queue and complete the
    request as CANCELLED.  If it can't get if off the queue then
    this routine will ignore the CANCEL request since the IRP
    is about to complete anyway.

Arguments:

    DeviceObject    - Supplies the device object.

    Irp             - Supplies the IRP.

Return Value:

    None.

--*/

{
    PDEVICE_EXTENSION           extension;
    SYNCHRONIZED_COUNT_CONTEXT  syncContext;

    extension = DeviceObject->DeviceExtension;
    syncContext.Count = &extension->WorkQueueCount;
    if (extension->InterruptRefCount) {
        KeSynchronizeExecution(extension->InterruptObject,
                               PptSynchronizedDecrement,
                               &syncContext);
    } else {
        PptSynchronizedDecrement(&syncContext);
    }
    RemoveEntryList(&Irp->Tail.Overlay.ListEntry);
    IoReleaseCancelSpinLock(Irp->CancelIrql);

    Irp->IoStatus.Information = 0;
    Irp->IoStatus.Status = STATUS_CANCELLED;

    ParDump(
        PARIRPPATH,
        ("PARPORT:  About to complete IRP in cancel routine\n"
         "Irp: %x status: %x Information: %x\n",
         Irp,
         Irp->IoStatus.Status,
         Irp->IoStatus.Information)
        );

    IoCompleteRequest(Irp, IO_NO_INCREMENT);
}

VOID
PptFreePortDpc(
    IN      PKDPC   Dpc,
    IN OUT  PVOID   Extension,
    IN      PVOID   SystemArgument1,
    IN      PVOID   SystemArgument2
    )

/*++

Routine Description:

    This routine is a DPC that will free the port and if necessary
    complete an alloc request that is waiting.

Arguments:

    Dpc             - Not used.

    Extension       - Supplies the device extension.

    SystemArgument1 - Not used.

    SystemArgument2 - Not used.

Return Value:

    None.

--*/

{
    PptFreePort(Extension);
}

BOOLEAN
PptTryAllocatePortAtInterruptLevel(
    IN  PVOID   Context
    )

/*++

Routine Description:

    This routine is called at interrupt level to quickly allocate
    the parallel port if it is available.  This call will fail
    if the port is not available.

Arguments:

    Context - Supplies the device extension.

Return Value:

    FALSE   - The port was not allocated.
    TRUE    - The port was successfully allocated.

--*/

{
    if (((PDEVICE_EXTENSION) Context)->WorkQueueCount == -1) {
        ((PDEVICE_EXTENSION) Context)->WorkQueueCount = 0;
        return(TRUE);
    } else {
        return(FALSE);
    }
}

VOID
PptFreePortFromInterruptLevel(
    IN  PVOID   Context
    )

/*++

Routine Description:

    This routine frees the port that was allocated at interrupt level.

Arguments:

    Context - Supplies the device extension.

Return Value:

    None.

--*/

{
    // If no one is waiting for the port then this is simple operation,
    // otherwise queue a DPC to free the port later on.

    if (((PDEVICE_EXTENSION) Context)->WorkQueueCount == 0) {
        ((PDEVICE_EXTENSION) Context)->WorkQueueCount = -1;
    } else {
        KeInsertQueueDpc(&((PDEVICE_EXTENSION) Context)->FreePortDpc, NULL, NULL);
    }
}

BOOLEAN
PptInterruptService(
    IN  PKINTERRUPT Interrupt,
    IN  PVOID       Extension
    )

/*++

Routine Description:

    This routine services the interrupt for the parallel port.
    This routine will call out to all of the interrupt routines
    that connected with this device via
    IOCTL_INTERNAL_PARALLEL_CONNECT_INTERRUPT in order until
    one of them returns TRUE.

Arguments:

    Interrupt   - Supplies the interrupt object.

    Extension   - Supplies the device extension.

Return Value:

    FALSE   - The interrupt was not handled.
    TRUE    - The interrupt was handled.

--*/

{
    PDEVICE_EXTENSION   extension;
    PLIST_ENTRY         current;
    PISR_LIST_ENTRY     isrListEntry;

    extension = Extension;
    for (current = extension->IsrList.Flink;
         current != &extension->IsrList;
         current = current->Flink) {

        isrListEntry = CONTAINING_RECORD(current, ISR_LIST_ENTRY, ListEntry);
        if (isrListEntry->ServiceRoutine(Interrupt, isrListEntry->ServiceContext)) {
            return(TRUE);
        }
    }

    return(FALSE);
}

BOOLEAN
PptTryAllocatePort(
    IN  PVOID   Extension
    )

/*++

Routine Description:

    This routine attempts to allocate the port.  If the port is
    available then the call will succeed with the port allocated.
    If the port is not available the then call will fail
    immediately.

Arguments:

    Extension   - Supplies the device extension.

Return Value:

    FALSE   - The port was not allocated.
    TRUE    - The port was allocated.

--*/

{
    PDEVICE_EXTENSION   extension = Extension;
    KIRQL               cancelIrql;
    BOOLEAN             b;

    if (extension->InterruptRefCount) {
        b = KeSynchronizeExecution(extension->InterruptObject,
                                   PptTryAllocatePortAtInterruptLevel,
                                   extension);
    } else {
        IoAcquireCancelSpinLock(&cancelIrql);
        b = PptTryAllocatePortAtInterruptLevel(extension);
        IoReleaseCancelSpinLock(cancelIrql);
    }

    return(b);
}

BOOLEAN
PptTraversePortCheckList(
    IN  PVOID   Extension
    )

/*++

Routine Description:

    This routine traverses the deferred port check routines.  This
    call must be synchronized at interrupt level so that real
    interrupts are blocked until these routines are completed.

Arguments:

    Extension   - Supplies the device extension.

Return Value:

    FALSE   - The port is in use so no action taken by this routine.
    TRUE    - All of the deferred interrupt routines were called.

--*/

{
    PDEVICE_EXTENSION   extension = Extension;
    PLIST_ENTRY         current;
    PISR_LIST_ENTRY     checkEntry;

    // First check to make sure that the port is still free.

    if (extension->WorkQueueCount >= 0) {
        return FALSE;
    }

    for (current = extension->IsrList.Flink;
         current != &extension->IsrList;
         current = current->Flink) {
        
        checkEntry = CONTAINING_RECORD(current,
                                       ISR_LIST_ENTRY,
                                       ListEntry);

        if (checkEntry->DeferredPortCheckRoutine) {
            checkEntry->DeferredPortCheckRoutine(checkEntry->CheckContext);
        }
    }

    return TRUE;
}

VOID
PptFreePort(
    IN  PVOID   Extension
    )

/*++

Routine Description:

    This routine frees the port.

Arguments:

    Extension   - Supplies the device extension.

Return Value:

    None.

--*/

{
    PDEVICE_EXTENSION               extension = Extension;
    SYNCHRONIZED_COUNT_CONTEXT      syncContext;
    KIRQL                           cancelIrql;
    PLIST_ENTRY                     head;
    PIRP                            irp;
    ULONG                           interruptRefCount;

    syncContext.Count = &extension->WorkQueueCount;

    IoAcquireCancelSpinLock(&cancelIrql);

    if (extension->InterruptRefCount) {
        KeSynchronizeExecution(extension->InterruptObject,
                               PptSynchronizedDecrement,
                               &syncContext);
    } else {
        PptSynchronizedDecrement(&syncContext);
    }

    if (syncContext.NewCount >= 0) {
        head = RemoveHeadList(&extension->WorkQueue);
        irp = CONTAINING_RECORD(head, IRP, Tail.Overlay.ListEntry);
        IoSetCancelRoutine(irp, NULL);
        IoReleaseCancelSpinLock(cancelIrql);

        irp->IoStatus.Status = STATUS_SUCCESS;
        IoCompleteRequest(irp, IO_PARALLEL_INCREMENT);

    } else {
        interruptRefCount = extension->InterruptRefCount;
        IoReleaseCancelSpinLock(cancelIrql);
        if (interruptRefCount) {
            KeSynchronizeExecution(extension->InterruptObject,
                                   PptTraversePortCheckList,
                                   extension);
        }
    }
}

ULONG
PptQueryNumWaiters(
    IN  PVOID   Extension
    )

/*++

Routine Description:

    This routine returns the number of irps queued waiting for
    the parallel port.

Arguments:

    Extension   - Supplies the device extension.

Return Value:

    The number of irps queued waiting for the port.

--*/

{
    PDEVICE_EXTENSION           extension = Extension;
    KIRQL                       cancelIrql;
    SYNCHRONIZED_COUNT_CONTEXT  syncContext;

    syncContext.Count = &extension->WorkQueueCount;
    if (extension->InterruptRefCount) {
        KeSynchronizeExecution(extension->InterruptObject,
                               PptSynchronizedRead,
                               &syncContext);
    } else {
        IoAcquireCancelSpinLock(&cancelIrql);
        PptSynchronizedRead(&syncContext);
        IoReleaseCancelSpinLock(cancelIrql);
    }

    return((syncContext.NewCount >= 0) ? ((ULONG) syncContext.NewCount) : 0);
}

NTSTATUS
PptDispatchDeviceControl(
    IN  PDEVICE_OBJECT  DeviceObject,
    IN  PIRP            Irp
    )

/*++

Routine Description:

    This routine is the dispatch for INTERNAL_DEVICE_CONTROLs.

Arguments:

    DeviceObject    - Supplies the device object.

    Irp             - Supplies the I/O request packet.

Return Value:

    STATUS_SUCCESS              - Success.
    STATUS_PENDING              - The request is pending.
    STATUS_INVALID_PARAMETER    - Invalid parameter.
    STATUS_CANCELLED            - The request was cancelled.
    STATUS_BUFFER_TOO_SMALL     - The supplied buffer is too small.

--*/

{
    PIO_STACK_LOCATION                  irpSp;
    PDEVICE_EXTENSION                   extension;
    NTSTATUS                            status;
    PPARALLEL_PORT_INFORMATION          portInfo;
    PMORE_PARALLEL_PORT_INFORMATION     morePortInfo;
    KIRQL                               cancelIrql;
    SYNCHRONIZED_COUNT_CONTEXT          syncContext;
    PPARALLEL_INTERRUPT_SERVICE_ROUTINE isrInfo;
    PPARALLEL_INTERRUPT_INFORMATION     interruptInfo;
    PISR_LIST_ENTRY                     isrListEntry;
    SYNCHRONIZED_LIST_CONTEXT           listContext;
    SYNCHRONIZED_DISCONNECT_CONTEXT     disconnectContext;
    BOOLEAN                             disconnectInterrupt;

    irpSp = IoGetCurrentIrpStackLocation(Irp);

    ParDump(
        PARIRPPATH,
        ("PARPORT:  In internal device control dispatch with IRP: %x\n"
         "Io control code: %d\n",
         Irp,
         irpSp->Parameters.DeviceIoControl.IoControlCode)
        );

    extension = DeviceObject->DeviceExtension;
    Irp->IoStatus.Information = 0;

    switch (irpSp->Parameters.DeviceIoControl.IoControlCode) {

        case IOCTL_INTERNAL_PARALLEL_PORT_ALLOCATE:

            IoAcquireCancelSpinLock(&cancelIrql);

            if (Irp->Cancel) {
                status = STATUS_CANCELLED;
            } else {
                syncContext.Count = &extension->WorkQueueCount;
                if (extension->InterruptRefCount) {
                    KeSynchronizeExecution(extension->InterruptObject,
                                           PptSynchronizedIncrement,
                                           &syncContext);
                } else {
                    PptSynchronizedIncrement(&syncContext);
                }
                if (syncContext.NewCount) {

                    IoSetCancelRoutine(Irp, PptCancelRoutine);
                    IoMarkIrpPending(Irp);
                    InsertTailList(&extension->WorkQueue,
                                   &Irp->Tail.Overlay.ListEntry);
                    status = STATUS_PENDING;

                } else {

                    ParDump(
                        PARIRPPATH,
                        ("PARPORT:  Completing ALLOCATE request in dispatch\n")
                        );

                    status = STATUS_SUCCESS;
                }
            }

            IoReleaseCancelSpinLock(cancelIrql);
            break;

        case IOCTL_INTERNAL_GET_PARALLEL_PORT_INFO:

            if (irpSp->Parameters.DeviceIoControl.OutputBufferLength <
                sizeof(PARALLEL_PORT_INFORMATION)) {

                status = STATUS_BUFFER_TOO_SMALL;
            } else {

                Irp->IoStatus.Information = sizeof(PARALLEL_PORT_INFORMATION);
                portInfo = Irp->AssociatedIrp.SystemBuffer;
                *portInfo = extension->PortInfo;
                status = STATUS_SUCCESS;

#ifndef MEMPHIS
                ExAcquireFastMutex(PortInfoMutex);
                if (++PortInfoReferenceCount == 1) {
                    MmResetDriverPaging(DriverEntry);
                }
                ExReleaseFastMutex(PortInfoMutex);
#endif
            }
            break;

        case IOCTL_INTERNAL_RELEASE_PARALLEL_PORT_INFO:

#ifndef MEMPHIS
            ExAcquireFastMutex(PortInfoMutex);
            if (--PortInfoReferenceCount == 0) {
                MmPageEntireDriver(DriverEntry);
            }
            ExReleaseFastMutex(PortInfoMutex);
#endif
            status = STATUS_SUCCESS;
            break;

        case IOCTL_INTERNAL_GET_MORE_PARALLEL_PORT_INFO:

            if (irpSp->Parameters.DeviceIoControl.OutputBufferLength <
                sizeof(MORE_PARALLEL_PORT_INFORMATION)) {

                status = STATUS_BUFFER_TOO_SMALL;
            } else {

                Irp->IoStatus.Information = sizeof(MORE_PARALLEL_PORT_INFORMATION);
                morePortInfo = Irp->AssociatedIrp.SystemBuffer;
                morePortInfo->InterfaceType = extension->InterfaceType;
                morePortInfo->BusNumber = extension->BusNumber;
                morePortInfo->InterruptLevel = extension->InterruptLevel;
                morePortInfo->InterruptVector = extension->InterruptVector;
                morePortInfo->InterruptAffinity = extension->InterruptAffinity;
                morePortInfo->InterruptMode = extension->InterruptMode;
                status = STATUS_SUCCESS;
            }
            break;

        case IOCTL_INTERNAL_PARALLEL_CONNECT_INTERRUPT:

            if (irpSp->Parameters.DeviceIoControl.InputBufferLength <
                sizeof(PARALLEL_INTERRUPT_SERVICE_ROUTINE) ||
                irpSp->Parameters.DeviceIoControl.OutputBufferLength <
                sizeof(PARALLEL_INTERRUPT_INFORMATION)) {

                status = STATUS_BUFFER_TOO_SMALL;
            } else {
                isrInfo = Irp->AssociatedIrp.SystemBuffer;
                interruptInfo = Irp->AssociatedIrp.SystemBuffer;
                IoAcquireCancelSpinLock(&cancelIrql);
                if (extension->InterruptRefCount) {
                    ++extension->InterruptRefCount;
                    IoReleaseCancelSpinLock(cancelIrql);
                    status = STATUS_SUCCESS;
                } else {
                    IoReleaseCancelSpinLock(cancelIrql);
                    status = PptConnectInterrupt(extension);
                    if (NT_SUCCESS(status)) {
                        IoAcquireCancelSpinLock(&cancelIrql);
                        ++extension->InterruptRefCount;
                        IoReleaseCancelSpinLock(cancelIrql);
                    }
                }

                if (NT_SUCCESS(status)) {
                    isrListEntry = ExAllocatePool(NonPagedPool,
                                                  sizeof(ISR_LIST_ENTRY));

                    if (isrListEntry) {
                        
                        isrListEntry->ServiceRoutine =
                                isrInfo->InterruptServiceRoutine;
                        isrListEntry->ServiceContext =
                                isrInfo->InterruptServiceContext;
                        isrListEntry->DeferredPortCheckRoutine =
                                isrInfo->DeferredPortCheckRoutine;
                        isrListEntry->CheckContext =
                                isrInfo->DeferredPortCheckContext;

                        // Put the ISR_LIST_ENTRY onto the ISR list.

                        listContext.List = &extension->IsrList;
                        listContext.NewEntry = &isrListEntry->ListEntry;
                        KeSynchronizeExecution(extension->InterruptObject,
                                               PptSynchronizedQueue,
                                               &listContext);

                        interruptInfo->InterruptObject =
                                extension->InterruptObject;
                        interruptInfo->TryAllocatePortAtInterruptLevel =
                                PptTryAllocatePortAtInterruptLevel;
                        interruptInfo->FreePortFromInterruptLevel =
                                PptFreePortFromInterruptLevel;
                        interruptInfo->Context =
                                extension;

                        Irp->IoStatus.Information =
                                sizeof(PARALLEL_INTERRUPT_INFORMATION);
                        status = STATUS_SUCCESS;

                    } else {
                        status = STATUS_INSUFFICIENT_RESOURCES;
                    }
                }
            }
            break;

        case IOCTL_INTERNAL_PARALLEL_DISCONNECT_INTERRUPT:
            if (irpSp->Parameters.DeviceIoControl.InputBufferLength <
                sizeof(PARALLEL_INTERRUPT_SERVICE_ROUTINE)) {

                status = STATUS_BUFFER_TOO_SMALL;
            } else {
                isrInfo = Irp->AssociatedIrp.SystemBuffer;

                // Take the ISR out of the ISR list.

                IoAcquireCancelSpinLock(&cancelIrql);
                if (extension->InterruptRefCount) {
                    IoReleaseCancelSpinLock(cancelIrql);

                    disconnectContext.Extension = extension;
                    disconnectContext.IsrInfo = isrInfo;
                    if (KeSynchronizeExecution(extension->InterruptObject,
                                               PptSynchronizedDisconnect,
                                               &disconnectContext)) {

                        status = STATUS_SUCCESS;
                        IoAcquireCancelSpinLock(&cancelIrql);
                        if (--extension->InterruptRefCount == 0) {
                            disconnectInterrupt = TRUE;
                        } else {
                            disconnectInterrupt = FALSE;
                        }
                        IoReleaseCancelSpinLock(cancelIrql);

                    } else {
                        status = STATUS_INVALID_PARAMETER;
                        disconnectInterrupt = FALSE;
                    }
                } else {
                    IoReleaseCancelSpinLock(cancelIrql);
                    disconnectInterrupt = FALSE;
                    status = STATUS_INVALID_PARAMETER;
                }


                // Disconnect the interrupt if appropriate.

                if (disconnectInterrupt) {
                    PptDisconnectInterrupt(extension);
                }
            }
            break;

        default:
            status = STATUS_INVALID_PARAMETER;
            break;

    }

    if (status != STATUS_PENDING) {
        Irp->IoStatus.Status = status;
        IoCompleteRequest(Irp, IO_NO_INCREMENT);
    }

    return status;
}

VOID
PptCleanupDevice(
    IN  PDEVICE_EXTENSION   Extension
    )

/*++

Routine Description:

    This routine will deallocate all of the memory used for
    a particular device.  It will also disconnect any resources
    if need be.

Arguments:

    Extension - Pointer to the device extension which is getting
                rid of all it's resources.

Return Value:

    None.

--*/

{
    ParDump(
        PARUNLOAD,
        ("PARPORT:  in PptCleanupDevice for extension: %x\n",Extension)
        );

    if (Extension && Extension->UnMapRegisters) {
        MmUnmapIoSpace(Extension->PortInfo.Controller,
                       Extension->PortInfo.SpanOfController);
    }
}

NTSTATUS
PptDispatchCleanup(
    IN  PDEVICE_OBJECT  DeviceObject,
    IN  PIRP            Irp
    )

/*++

Routine Description:

    This routine cancels all of the IRPs currently queued on
    the given device.

Arguments:

    DeviceObject    - Supplies the device object.

    Irp             - Supplies the cleanup IRP.

Return Value:

    STATUS_SUCCESS  - Success.

--*/

{
    PDEVICE_EXTENSION   extension;
    PIRP                irp;
    KIRQL               cancelIrql;

    ParDump(
        PARIRPPATH,
        ("PARPORT:  In cleanup with IRP: %x\n",
         Irp)
        );

    extension = DeviceObject->DeviceExtension;

    IoAcquireCancelSpinLock(&cancelIrql);

    while (!IsListEmpty(&extension->WorkQueue)) {

        irp = CONTAINING_RECORD(extension->WorkQueue.Blink,
                                IRP, Tail.Overlay.ListEntry);

        irp->Cancel = TRUE;
        irp->CancelIrql = cancelIrql;
        irp->CancelRoutine = NULL;
        PptCancelRoutine(DeviceObject, irp);

        IoAcquireCancelSpinLock(&cancelIrql);
    }

    IoReleaseCancelSpinLock(cancelIrql);

    Irp->IoStatus.Status = STATUS_SUCCESS;
    Irp->IoStatus.Information = 0;

    ParDump(
        PARIRPPATH,
        ("PARPORT:  About to complete IRP in cleanup\n"
         "Irp: %x status: %x Information: %x\n",
         Irp,
         Irp->IoStatus.Status,
         Irp->IoStatus.Information)
        );

    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    return STATUS_SUCCESS;
}

VOID
PptUnload(
    IN  PDRIVER_OBJECT  DriverObject
    )

/*++

Routine Description:

    This routine cleans up all of the memory associated with
    any of the devices belonging to the driver.  It  will
    loop through the device list.

Arguments:

    DriverObject    - Supplies the driver object controling all of the
                        devices.

Return Value:

    None.

--*/

{
    PDEVICE_OBJECT                  currentDevice;
    PDEVICE_EXTENSION               extension;
    PLIST_ENTRY                     head;
    PISR_LIST_ENTRY                 entry;

    ParDump(
        PARUNLOAD,
        ("PARPORT:  In ParUnload\n")
        );

    while (currentDevice = DriverObject->DeviceObject) {

        extension = currentDevice->DeviceExtension;

        if (extension->InterruptRefCount) {
            PptDisconnectInterrupt(extension);
        }
        PptCleanupDevice(extension);

        while (!IsListEmpty(&extension->IsrList)) {
            head = RemoveHeadList(&extension->IsrList);
            entry = CONTAINING_RECORD(head, ISR_LIST_ENTRY, ListEntry);
            ExFreePool(entry);
        }

        PptUnReportResourcesDevice(extension);

        IoDeleteDevice(currentDevice);
        IoGetConfigurationInformation()->ParallelCount--;
    }

    ExFreePool(PortInfoMutex);
}

BOOLEAN
PptIsNecR98Machine(
    void
    )

/*++

Routine Description:

    This routine checks the machine type in the registry to determine
    if this is an Nec R98 machine.

Arguments:

    None.

Return Value:

    TRUE - this machine is an R98
    FALSE - this machine is not
    

--*/

{

    UNICODE_STRING path;
    RTL_QUERY_REGISTRY_TABLE paramTable[2];
    NTSTATUS status;

    UNICODE_STRING identifierString;
    UNICODE_STRING necR98Identifier;
    UNICODE_STRING necR98JIdentifier;

    RtlInitUnicodeString(&path, L"\\Registry\\Machine\\HARDWARE\\DESCRIPTION\\System");
    RtlInitUnicodeString(&necR98Identifier, L"NEC-R98");
    RtlInitUnicodeString(&necR98JIdentifier, L"NEC-J98");


    identifierString.Length = 0;
    identifierString.MaximumLength = 32;
    identifierString.Buffer = ExAllocatePool(PagedPool, identifierString.MaximumLength);

    if(!identifierString.Buffer)    return FALSE;

    RtlZeroMemory(paramTable, sizeof(paramTable));
    paramTable[0].Flags = RTL_QUERY_REGISTRY_DIRECT | 
                          RTL_QUERY_REGISTRY_REQUIRED;
    paramTable[0].Name = L"Identifier";
    paramTable[0].EntryContext = &identifierString;
    paramTable[0].DefaultType = REG_SZ;
    paramTable[0].DefaultData = &path;
    paramTable[0].DefaultLength = 0;

    status = RtlQueryRegistryValues(RTL_REGISTRY_ABSOLUTE,
                                    path.Buffer,
                                    paramTable,
                                    NULL,
                                    NULL);


    if(NT_SUCCESS(status))  {

        if((RtlCompareUnicodeString(&identifierString, 
                                    &necR98Identifier, FALSE) == 0) ||
           (RtlCompareUnicodeString(&identifierString, 
                                    &necR98JIdentifier, FALSE) == 0)) {

            ParDump(0, ("parport!PptIsNecR98Machine - this an R98 machine\n"));
            ExFreePool(identifierString.Buffer);
            return TRUE;
        }
    } else {

        ParDump(0, ("parport!PptIsNecR98Machine - "
                    "RtlQueryRegistryValues failed [status 0x%x]\n", status));
        ExFreePool(identifierString.Buffer);
        return FALSE;
    }

    ParDump(0,  ("parport!PptIsNecR98Machine - "
                 "this is not an R98 machine\n"));
    ExFreePool(identifierString.Buffer);
    return FALSE;
}
        

