/*++

Copyright (c) 1993-1997 Microsoft Corporation

Module Name:

    parinit.c

Abstract:

    This module contains the code for the parallel port driver.

    The code in this module contains all of the static load code
    (from Microsoft Windows NT 4.0).  WDM Plug-N-Play code now
    exists in the main module (parport.c) in its place.  This
    code will be used if STATIC_LOAD is defined at compile time.

Author:

    Timothy T. Wells (WestTek, LLC) 16-Jan-1997

Environment:

    Kernel mode

Revision History :

--*/

#ifdef STATIC_LOAD

#include "ntddk.h"
#include "parallel.h"
#include "parport.h"
#include "parlog.h"

typedef struct _PARALLEL_FIRMWARE_DATA {
    PDRIVER_OBJECT  DriverObject;
    ULONG           ControllersFound;
    LIST_ENTRY      ConfigList;
} PARALLEL_FIRMWARE_DATA, *PPARALLEL_FIRMWARE_DATA;

typedef enum _PPT_MEM_COMPARES {
    AddressesAreEqual,
    AddressesOverlap,
    AddressesAreDisjoint
} PPT_MEM_COMPARES, *PPPT_MEM_COMPARES;

NTSTATUS
DriverEntry(
    IN  PDRIVER_OBJECT  DriverObject,
    IN  PUNICODE_STRING RegistryPath
    );

PPT_MEM_COMPARES
PptMemCompare(
    IN  PHYSICAL_ADDRESS    A,
    IN  ULONG               SpanOfA,
    IN  PHYSICAL_ADDRESS    B,
    IN  ULONG               SpanOfB
    );

VOID
PptGetConfigInfo(
    IN  PDRIVER_OBJECT  DriverObject,
    IN  PUNICODE_STRING RegistryPath,
    OUT PLIST_ENTRY     ConfigList
    );

NTSTATUS
PptConfigCallBack(
    IN  PVOID                           Context,
    IN  PUNICODE_STRING                 PathName,
    IN  INTERFACE_TYPE                  BusType,
    IN  ULONG                           BusNumber,
    IN  PKEY_VALUE_FULL_INFORMATION*    BusInformation,
    IN  CONFIGURATION_TYPE              ControllerType,
    IN  ULONG                           ControllerNumber,
    IN  PKEY_VALUE_FULL_INFORMATION*    ControllerInformation,
    IN  CONFIGURATION_TYPE              PeripheralType,
    IN  ULONG                           PeripheralNumber,
    IN  PKEY_VALUE_FULL_INFORMATION*    PeripheralInformation
    );

NTSTATUS
PptItemCallBack(
    IN  PVOID                           Context,
    IN  PUNICODE_STRING                 PathName,
    IN  INTERFACE_TYPE                  BusType,
    IN  ULONG                           BusNumber,
    IN  PKEY_VALUE_FULL_INFORMATION*    BusInformation,
    IN  CONFIGURATION_TYPE              ControllerType,
    IN  ULONG                           ControllerNumber,
    IN  PKEY_VALUE_FULL_INFORMATION*    ControllerInformation,
    IN  CONFIGURATION_TYPE              PeripheralType,
    IN  ULONG                           PeripheralNumber,
    IN  PKEY_VALUE_FULL_INFORMATION*    PeripheralInformation
    );

BOOLEAN
PptPutInConfigList(
    IN      PDRIVER_OBJECT  DriverObject,
    IN OUT  PLIST_ENTRY     ConfigList,
    IN      PCONFIG_DATA    New
    );

NTSTATUS
PptInitializeController(
    IN  PDRIVER_OBJECT  DriverObject,
    IN  PUNICODE_STRING RegistryPath,
    IN  PCONFIG_DATA    ConfigData
    );


#ifdef ALLOC_PRAGMA
#pragma alloc_text(INIT,DriverEntry)
#pragma alloc_text(INIT,PptGetConfigInfo)
#pragma alloc_text(INIT,PptConfigCallBack)
#pragma alloc_text(INIT,PptItemCallBack)
#pragma alloc_text(INIT,PptInitializeController)
#pragma alloc_text(INIT,PptPutInConfigList)
#endif


NTSTATUS
DriverEntry(
    IN PDRIVER_OBJECT DriverObject,
    IN PUNICODE_STRING RegistryPath
    )

/*++

Routine Description:

    This routine is called at system initialization time to initialize
    this driver.

Arguments:

    DriverObject    - Supplies the driver object.

    RegistryPath    - Supplies the registry path for this driver.

Return Value:

    STATUS_SUCCESS          - We could initialize at least one device.
    STATUS_NO_SUCH_DEVICE   - We could not initialize even one device.

--*/

{
    LIST_ENTRY                  configList;
    PCONFIG_DATA                currentConfig;
    PLIST_ENTRY                 head;

    PptGetConfigInfo(DriverObject, RegistryPath, &configList);

    //
    // Initialize each item in the list of configuration records.
    //

    while (!IsListEmpty(&configList)) {

        head = RemoveHeadList(&configList);
        currentConfig = CONTAINING_RECORD(head, CONFIG_DATA, ConfigList);

        PptInitializeController(DriverObject, RegistryPath, currentConfig);
    }

    if (!DriverObject->DeviceObject) {
        return STATUS_NO_SUCH_DEVICE;
    }


    //
    // Initialize the Driver Object with driver's entry points
    //

    DriverObject->MajorFunction[IRP_MJ_CREATE] = PptDispatchCreateClose;
    DriverObject->MajorFunction[IRP_MJ_CLOSE] = PptDispatchCreateClose;
    DriverObject->MajorFunction[IRP_MJ_INTERNAL_DEVICE_CONTROL] =
            PptDispatchDeviceControl;
    DriverObject->MajorFunction[IRP_MJ_CLEANUP] = PptDispatchCleanup;
    DriverObject->DriverUnload = PptUnload;

    //
    // Let this driver be paged until someone requests PORT_INFO.
    //
    PortInfoMutex = ExAllocatePool(NonPagedPool, sizeof(FAST_MUTEX));
    if (!PortInfoMutex) {
        return STATUS_INSUFFICIENT_RESOURCES;
    }
    ExInitializeFastMutex(PortInfoMutex);

    MmPageEntireDriver(DriverEntry);

    return STATUS_SUCCESS;
}

PPT_MEM_COMPARES
PptMemCompare(
    IN  PHYSICAL_ADDRESS    A,
    IN  ULONG               SpanOfA,
    IN  PHYSICAL_ADDRESS    B,
    IN  ULONG               SpanOfB
    )

/*++

Routine Description:

    This routine compares two phsical address.

Arguments:

    A       - Supplies one half of the comparison.

    SpanOfA - Supplies the span of A in units of bytes.

    B       - Supplies the other half of the comparison.

    SpanOfB - Supplies the span of B in units of bytes.

Return Value:

    The result of the comparison.

--*/

{
    LARGE_INTEGER   a, b;
    LARGE_INTEGER   lower, higher;
    ULONG           lowerSpan;

    a.LowPart = A.LowPart;
    a.HighPart = A.HighPart;
    b.LowPart = B.LowPart;
    b.HighPart = B.HighPart;

    if (a.QuadPart == b.QuadPart) {
        return AddressesAreEqual;
    }

    if (a.QuadPart > b.QuadPart) {
        higher = a;
        lower = b;
        lowerSpan = SpanOfB;
    } else {
        higher = b;
        lower = a;
        lowerSpan = SpanOfA;
    }

    if (higher.QuadPart - lower.QuadPart >= lowerSpan) {
        return AddressesAreDisjoint;
    }

    return AddressesOverlap;
}

VOID
PptGetConfigInfo(
    IN  PDRIVER_OBJECT  DriverObject,
    IN  PUNICODE_STRING RegistryPath,
    OUT PLIST_ENTRY     ConfigList
    )

/*++

Routine Description:

    This routine will "return" a list of configuration records
    for the parallel ports to initialize.

    It will first query the firmware data.  It will then look
    for "user" specified parallel ports in the registry.  It
    will place the user specified parallel ports in the passed
    in list.

    After it finds all of the user specified ports, it will
    attempt to add the firmware parallel ports into the passed
    in lists.  The insert in the list code detects conflicts
    and rejects a new port.  In this way we can prevent
    firmware found ports from overiding information specified
    by the "user".

Arguments:

    DriverObject - Supplies the driver object.

    RegistryPath - Supplies the registry path for this driver.

    ConfigList   - Returns a list of configuration records for
                    the parallel ports to initialize.

Return Value:

    None.

--*/

{
    PARALLEL_FIRMWARE_DATA  firmware;
    CONFIGURATION_TYPE      sc;
    INTERFACE_TYPE          iType;
    PLIST_ENTRY             head;
    PCONFIG_DATA            firmwareData;

    InitializeListHead(ConfigList);

    RtlZeroMemory(&firmware, sizeof(PARALLEL_FIRMWARE_DATA));
    firmware.DriverObject = DriverObject;
    firmware.ControllersFound = IoGetConfigurationInformation()->ParallelCount;
    InitializeListHead(&firmware.ConfigList);

    //
    // First we query the hardware registry for all of
    // the firmware defined ports.  We loop over all of
    // the busses.
    //

    sc = ParallelController;
    for (iType = 0; iType < MaximumInterfaceType; iType++) {
        IoQueryDeviceDescription(&iType, NULL, &sc, NULL, NULL, NULL,
                                 PptConfigCallBack, &firmware);
    }

    while (!IsListEmpty(&firmware.ConfigList)) {

        head = RemoveHeadList(&firmware.ConfigList);
        firmwareData = CONTAINING_RECORD(head, CONFIG_DATA, ConfigList);

        if (!PptPutInConfigList(DriverObject, ConfigList, firmwareData)) {

            PptLogError(DriverObject, NULL, PhysicalZero, PhysicalZero,
                        0, 0, 0, 1, STATUS_SUCCESS, PAR_USER_OVERRIDE);

            ParDump(PARERRORS,
                    ("PARPORT:  Conflict detected with user data for firmware port %wZ\n"
                     "--------  User data will overides firmware data\n",
                     &firmwareData->NtNameForPort));

            ExFreePool(firmwareData->NtNameForPort.Buffer);
            ExFreePool(firmwareData);
        }
    }
}

NTSTATUS
PptConfigCallBack(
    IN  PVOID                           Context,
    IN  PUNICODE_STRING                 PathName,
    IN  INTERFACE_TYPE                  BusType,
    IN  ULONG                           BusNumber,
    IN  PKEY_VALUE_FULL_INFORMATION*    BusInformation,
    IN  CONFIGURATION_TYPE              ControllerType,
    IN  ULONG                           ControllerNumber,
    IN  PKEY_VALUE_FULL_INFORMATION*    ControllerInformation,
    IN  CONFIGURATION_TYPE              PeripheralType,
    IN  ULONG                           PeripheralNumber,
    IN  PKEY_VALUE_FULL_INFORMATION*    PeripheralInformation
    )

/*++

Routine Description:

    This routine is used to acquire all of the configuration
    information for each parallel controller found by the firmware

Arguments:

    Context                 - Supplies the pointer to the list head of the list
                                of configuration records that we are building up.

    PathName                - Not Used.

    BusType                 - Supplies the bus type.  Internal, Isa, ...

    BusNumber               - Supplies which bus number if we are on a multibus
                                system.

    BusInformation          - Not Used.

    ControllerType          - Supplies the controller type.  Should always be
                                ParallelController.

    ControllerNumber        - Supplies which controller number if there is more
                               than one controller in the system.

    ControllerInformation   - Supplies an array of pointers to the three pieces
                               of registry information.

    PeripheralType          - Undefined for this call.

    PeripheralNumber        - Undefined for this call.

    PeripheralInformation   - Undefined for this call.

Return Value:

    STATUS_SUCCESS                  - Success.
    STATUS_INSUFFICIENT_RESOURCES   - Not all of the resource information
                                        could be acquired.
--*/

{
    PPARALLEL_FIRMWARE_DATA         config = Context;
    PCM_FULL_RESOURCE_DESCRIPTOR    controllerData;
    PCONFIG_DATA                    controller;
    ULONG                           i;
    PCM_PARTIAL_RESOURCE_DESCRIPTOR partial;
    BOOLEAN                         foundPort, foundInterrupt;
    WCHAR                           ntNumberBuffer[100];
    UNICODE_STRING                  ntNumberString;
    UNICODE_STRING                  temp;

    ASSERT(ControllerType == ParallelController);

    //
    // Bail if some fool wrote a loader.
    //

    if (!ControllerInformation[IoQueryDeviceConfigurationData]->DataLength) {
        return STATUS_SUCCESS;
    }

    controllerData =
        (PCM_FULL_RESOURCE_DESCRIPTOR)
        (((PUCHAR)ControllerInformation[IoQueryDeviceConfigurationData]) +
         ControllerInformation[IoQueryDeviceConfigurationData]->DataOffset);

    controller = ExAllocatePool(PagedPool, sizeof(CONFIG_DATA));
    if (!controller) {
        PptLogError(config->DriverObject, NULL, PhysicalZero, PhysicalZero, 0,
                    0, 0, 2, STATUS_SUCCESS, PAR_INSUFFICIENT_RESOURCES);

        ParDump(
            PARERRORS,
            ("PARPORT:  Couldn't allocate memory for the configuration data\n"
             "--------  for firmware data\n")
            );

        return STATUS_INSUFFICIENT_RESOURCES;
    }
    RtlZeroMemory(controller, sizeof(CONFIG_DATA));
    InitializeListHead(&controller->ConfigList);
    controller->InterfaceType = BusType;
    controller->BusNumber = BusNumber;

    //
    // We need to get the following information out of the partial
    // resource descriptors.
    //
    // The base address and span covered by the parallel controllers
    // registers.
    //
    // It is not defined how these appear in the partial resource
    // lists, so we will just loop over all of them.  If we find
    // something we don't recognize, we drop that information on
    // the floor.  When we have finished going through all the
    // partial information, we validate that we got the above
    // information.
    //

    foundInterrupt = foundPort = FALSE;
    for (i = 0; i < controllerData->PartialResourceList.Count; i++) {

        partial = &controllerData->PartialResourceList.PartialDescriptors[i];

        if (partial->Type == CmResourceTypePort) {

            foundPort = TRUE;

            controller->SpanOfController = partial->u.Port.Length;
            controller->Controller = partial->u.Port.Start;
            controller->AddressSpace = partial->Flags;

            if((controller->SpanOfController == 0x1000) &&
               (PptIsNecR98Machine()))  {

                ParDump(0, ("parport!PptConfigCallBack - "
                            "found R98 machine with firmware bug. " 
                            "Truncating SpanOfController to 8\n"));
                controller->SpanOfController = 8;
            }

        } else if (partial->Type == CmResourceTypeInterrupt) {

            foundInterrupt = TRUE;

            controller->InterruptLevel = partial->u.Interrupt.Level;
            controller->InterruptVector = partial->u.Interrupt.Vector;
            controller->InterruptAffinity = partial->u.Interrupt.Affinity;
            if (partial->Flags & CM_RESOURCE_INTERRUPT_LATCHED) {
                controller->InterruptMode = Latched;
            } else {
                controller->InterruptMode = LevelSensitive;
            }
        }
    }

    if (!foundPort || !foundInterrupt) {

        PptLogError(config->DriverObject, NULL, PhysicalZero, PhysicalZero, 0,
                    0, 0, 3, STATUS_SUCCESS, PAR_NOT_ENOUGH_CONFIG_INFO);

        ExFreePool(controller);

        return STATUS_SUCCESS;
    }

    //
    // The following are so the we can form the
    // name following the \Device
    // and the default name that will be symbolic
    // linked to the device and the object directory
    // that link will go in.
    //

    ntNumberString.Length = 0;
    ntNumberString.MaximumLength = 100;
    ntNumberString.Buffer = ntNumberBuffer;

    if (!NT_SUCCESS(RtlIntegerToUnicodeString(config->ControllersFound,
                                              10, &ntNumberString))) {

        PptLogError(config->DriverObject, NULL, controller->Controller,
                    PhysicalZero, 0, 0, 0, 4, STATUS_SUCCESS,
                    PAR_INSUFFICIENT_RESOURCES);

        ParDump(
            PARERRORS,
            ("PARPORT:  Couldn't convert NT controller number\n"
             "--------  to unicode for firmware data: %d\n",
             config->ControllersFound)
            );

        ExFreePool(controller);

        return STATUS_SUCCESS;
    }

    RtlInitUnicodeString(&controller->NtNameForPort, NULL);
    RtlInitUnicodeString(&temp, DD_PARALLEL_PORT_BASE_NAME_U);

    controller->NtNameForPort.MaximumLength = temp.Length +
            ntNumberString.Length + sizeof(WCHAR);

    controller->NtNameForPort.Buffer = ExAllocatePool(PagedPool,
            controller->NtNameForPort.MaximumLength);

    if (!controller->NtNameForPort.Buffer) {

        PptLogError(config->DriverObject, NULL, controller->Controller,
                    PhysicalZero, 0, 0, 0, 5, STATUS_SUCCESS,
                    PAR_INSUFFICIENT_RESOURCES);

        ParDump(
            PARERRORS,
            ("PARPORT:  Couldn't allocate memory for NT\n"
             "--------  name for NT firmware data: %d\n",
             config->ControllersFound)
            );

        ExFreePool(controller);

        return STATUS_SUCCESS;
    }

    RtlZeroMemory(controller->NtNameForPort.Buffer,
                  controller->NtNameForPort.MaximumLength);

    RtlAppendUnicodeStringToString(&controller->NtNameForPort, &temp);

    RtlAppendUnicodeStringToString(&controller->NtNameForPort, &ntNumberString);

    InsertTailList(&config->ConfigList, &controller->ConfigList);

    config->ControllersFound++;

    return STATUS_SUCCESS;
}

NTSTATUS
PptItemCallBack(
    IN  PVOID                           Context,
    IN  PUNICODE_STRING                 PathName,
    IN  INTERFACE_TYPE                  BusType,
    IN  ULONG                           BusNumber,
    IN  PKEY_VALUE_FULL_INFORMATION*    BusInformation,
    IN  CONFIGURATION_TYPE              ControllerType,
    IN  ULONG                           ControllerNumber,
    IN  PKEY_VALUE_FULL_INFORMATION*    ControllerInformation,
    IN  CONFIGURATION_TYPE              PeripheralType,
    IN  ULONG                           PeripheralNumber,
    IN  PKEY_VALUE_FULL_INFORMATION*    PeripheralInformation
    )

/*++

Routine Description:

    This IoQueryDeviceDescription callback merely sets the context
    argument to TRUE.

Arguments:

    Context - Supplies a boolean to set to TRUE.

Return Value:

    STATUS_SUCCESS  - Success.

--*/

{
    *((BOOLEAN *)Context) = TRUE;
    return STATUS_SUCCESS;
}

BOOLEAN
PptPutInConfigList(
    IN      PDRIVER_OBJECT  DriverObject,
    IN OUT  PLIST_ENTRY     ConfigList,
    IN      PCONFIG_DATA    New
    )

/*++

Routine Description:

    Given a new config record and a config list, this routine
    will perform a check to make sure that the new record doesn't
    conflict with old records.

    If everything checks out it will insert the new config record
    into the config list.

Arguments:

    DriverObject    - Supplies the driver we are attempting to get
                       configuration information for.

    ConfigList      - Supplies the listhead for a list of configuration
                       records for ports to control.

    New             - Supplies a pointer to new configuration record to add.

Return Value:

    FALSE   - The new record was not added to the config list.
    TRUE    - The new record was added to the config list.

--*/

{
    PHYSICAL_ADDRESS    PhysicalMax;
    PLIST_ENTRY         current;
    PCONFIG_DATA        oldConfig;

    ParDump(
        PARCONFIG,
        ("PARPORT:  Attempting to add %wZ\n"
         "--------  to the config list\n"
         "--------  PortAddress is %x\n"
         "--------  BusNumber is %d\n"
         "--------  BusType is %d\n"
         "--------  AddressSpace is %d\n",
         &New->NtNameForPort,
         New->Controller.LowPart,
         New->BusNumber,
         New->InterfaceType,
         New->AddressSpace
         )
        );

    //
    // We don't support any boards whose memory wraps around
    // the physical address space.
    //

    PhysicalMax.LowPart = (ULONG)~0;
    PhysicalMax.HighPart = ~0;
    if (PptMemCompare(New->Controller, New->SpanOfController,
                      PhysicalMax, 0) != AddressesAreDisjoint) {

        PptLogError(DriverObject, NULL, New->Controller, PhysicalZero,
                    0, 0, 0, 6, STATUS_SUCCESS, PAR_DEVICE_TOO_HIGH);

        ParDump(
            PARERRORS,
            ("PARPORT:  Error in config record for %wZ\n"
             "--------  registers rap around physical memory\n",
             &New->NtNameForPort)
            );

        return FALSE;
    }


    //
    // Go through the list looking for previous devices
    // with the same address.
    //

    for (current = ConfigList->Flink; current != ConfigList;
         current = current->Flink) {


        oldConfig = CONTAINING_RECORD(current, CONFIG_DATA, ConfigList);

        //
        // We only care about ports that are on the same bus.
        //

        if (oldConfig->InterfaceType == New->InterfaceType &&
            oldConfig->BusNumber == New->BusNumber) {

            ParDump(
                    PARCONFIG,
                    ("PARPORT:  Comparing it to %wZ\n"
                     "--------  already in the config list\n"
                     "--------  PortAddress is %x\n"
                     "--------  BusNumber is %d\n"
                     "--------  BusType is %d\n"
                     "--------  AddressSpace is %d\n",
                     &oldConfig->NtNameForPort,
                     oldConfig->Controller.LowPart,
                     oldConfig->BusNumber,
                     oldConfig->InterfaceType,
                     oldConfig->AddressSpace
                     )
                    );

            if (PptMemCompare(New->Controller, New->SpanOfController,
                              oldConfig->Controller,
                              oldConfig->SpanOfController) !=
                              AddressesAreDisjoint) {

                PptLogError(DriverObject, NULL, New->Controller,
                            oldConfig->Controller, 0, 0, 0, 7,
                            STATUS_SUCCESS, PAR_CONTROL_OVERLAP);

                return FALSE;
            }
        }
    }

    InsertTailList(ConfigList, &New->ConfigList);

    return TRUE;
}

NTSTATUS
PptInitializeController(
    IN  PDRIVER_OBJECT  DriverObject,
    IN  PUNICODE_STRING RegistryPath,
    IN  PCONFIG_DATA    ConfigData
    )

/*++

Routine Description:

    Really too many things to mention here.  In general, it forms
    and sets up names, creates the device, translates bus relative
    items...

Arguments:

    DriverObject    - Supplies the driver object to be used to create the
                        device object.

    RegistryPath    - Supplies the registry path for this port driver.

    ConfigData      - Supplies the configuration record for this port.

Return Value:

    STATUS_SUCCCESS if everything went ok.  A !NT_SUCCESS status
    otherwise.

Notes:

    This routine will deallocate the config data.

--*/

{
    UNICODE_STRING      uniNameString, uniDeviceString;
    NTSTATUS            status;
    PDEVICE_EXTENSION   extension;
    BOOLEAN             conflict;
    PDEVICE_OBJECT      deviceObject;

    ParDump(
        PARCONFIG,
        ("PARPORT:  Initializing for configuration record of %wZ\n",
         &ConfigData->NtNameForPort)
        );

    //
    // Form a name like \Device\ParallelPort0.
    //
    // First we allocate space for the name.
    //

    RtlInitUnicodeString(&uniNameString, NULL);
    RtlInitUnicodeString(&uniDeviceString, L"\\Device\\");
    uniNameString.MaximumLength = uniDeviceString.Length +
                                  ConfigData->NtNameForPort.Length +
                                  sizeof(WCHAR);
    uniNameString.Buffer = ExAllocatePool(PagedPool, uniNameString.MaximumLength);
    if (!uniNameString.Buffer) {

        PptLogError(DriverObject, NULL, ConfigData->Controller, PhysicalZero,
                    0, 0, 0, 8, STATUS_SUCCESS, PAR_INSUFFICIENT_RESOURCES);

        ParDump(
            PARERRORS,
            ("PARPORT:  Could not form Unicode name string for %wZ\n",
              &ConfigData->NtNameForPort)
            );

        ExFreePool(ConfigData->NtNameForPort.Buffer);
        ExFreePool(ConfigData);

        return STATUS_INSUFFICIENT_RESOURCES;
    }

    RtlZeroMemory(uniNameString.Buffer, uniNameString.MaximumLength);
    RtlAppendUnicodeStringToString(&uniNameString, &uniDeviceString);
    RtlAppendUnicodeStringToString(&uniNameString, &ConfigData->NtNameForPort);

    //
    // Create the device object for this device.
    //

    status = IoCreateDevice(DriverObject, sizeof(DEVICE_EXTENSION),
                            &uniNameString, FILE_DEVICE_PARALLEL_PORT,
                            0, FALSE, &deviceObject);

    if (!NT_SUCCESS(status)) {

        PptLogError(DriverObject, NULL, ConfigData->Controller, PhysicalZero,
                    0, 0, 0, 9, STATUS_SUCCESS, PAR_INSUFFICIENT_RESOURCES);

        ParDump(
            PARERRORS,
            ("PARPORT:  Could not create a device for %wZ\n",
             &ConfigData->NtNameForPort)
            );

        ExFreePool(uniNameString.Buffer);
        ExFreePool(ConfigData->NtNameForPort.Buffer);
        ExFreePool(ConfigData);

        return status;
    }

    //
    // We have created the device, increment the counter in the
    // IO system that keep track.  Anyplace that we do an IoDeleteDevice
    // we need to decrement.
    //

    IoGetConfigurationInformation()->ParallelCount++;

    //
    // The device object has a pointer to an area of non-paged
    // pool allocated for this device.  This will be the device
    // extension.
    //

    extension = deviceObject->DeviceExtension;

    //
    // Zero all of the memory associated with the device
    // extension.
    //

    RtlZeroMemory(extension, sizeof(DEVICE_EXTENSION));

    //
    // Get a "back pointer" to the device object.
    //

    extension->DeviceObject = deviceObject;

    //
    // Initialize 'WorkQueue' in extension.
    //

    InitializeListHead(&extension->WorkQueue);
    extension->WorkQueueCount = -1;

    //
    // Map the memory for the control registers for the parallel device
    // into virtual memory.
    //

    extension->PortInfo.Controller = PptGetMappedAddress(
            ConfigData->InterfaceType, ConfigData->BusNumber,
            ConfigData->Controller, ConfigData->SpanOfController,
            (BOOLEAN) ConfigData->AddressSpace, &extension->UnMapRegisters);

    if (!extension->PortInfo.Controller) {

        PptLogError(deviceObject->DriverObject, deviceObject,
                    ConfigData->Controller, PhysicalZero, 0, 0, 0, 10,
                    STATUS_SUCCESS, PAR_REGISTERS_NOT_MAPPED);

        ParDump(
            PARERRORS,
            ("PARPORT:  Could not map memory for device registers for %wZ\n",
              &ConfigData->NtNameForPort)
            );

        extension->UnMapRegisters = FALSE;
        status = STATUS_NONE_MAPPED;

        goto ExtensionCleanup;
    }

    extension->PortInfo.OriginalController = ConfigData->Controller;
    extension->PortInfo.SpanOfController = ConfigData->SpanOfController;
    extension->PortInfo.FreePort = PptFreePort;
    extension->PortInfo.TryAllocatePort = PptTryAllocatePort;
    extension->PortInfo.QueryNumWaiters = PptQueryNumWaiters;
    extension->PortInfo.Context = extension;

    //
    // Save the configuration information about the interrupt.
    //
    extension->AddressSpace = ConfigData->AddressSpace;
    extension->InterfaceType = ConfigData->InterfaceType;
    extension->BusNumber = ConfigData->BusNumber;
    extension->InterruptLevel = ConfigData->InterruptLevel;
    extension->InterruptVector = ConfigData->InterruptVector;
    extension->InterruptAffinity = ConfigData->InterruptAffinity;
    extension->InterruptMode = ConfigData->InterruptMode;

    //
    // Start off with an empty list of interrupt service routines.
    // Also, start off without the interrupt connected.
    //
    InitializeListHead(&extension->IsrList);
    extension->InterruptObject = NULL;
    extension->InterruptRefCount = 0;

    //
    // Initialize the free port DPC.
    //
    KeInitializeDpc(&extension->FreePortDpc, PptFreePortDpc, extension);

    //
    // Now the extension is all set up.  Just report the resources.
    //

    PptReportResourcesDevice(extension, FALSE, &conflict);

    if (conflict) {

        status = STATUS_NO_SUCH_DEVICE;

        PptLogError(deviceObject->DriverObject, deviceObject,
                    ConfigData->Controller, PhysicalZero, 0, 0, 0, 11,
                    STATUS_SUCCESS, PAR_ADDRESS_CONFLICT);

        ParDump(
            PARERRORS,
            ("PARPORT:  Could not claim the device registers for %wZ\n",
              &ConfigData->NtNameForPort)
            );

        goto ExtensionCleanup;
    }

    //
    // If it was requested that the port be disabled, now is the
    // time to do it.
    //

    if (ConfigData->DisablePort) {

        status = STATUS_NO_SUCH_DEVICE;

        PptLogError(deviceObject->DriverObject, deviceObject,
                    ConfigData->Controller, PhysicalZero, 0, 0, 0, 12,
                    STATUS_SUCCESS, PAR_DISABLED_PORT);

        ParDump(
            PARERRORS,
            ("PARPORT:  Port %wZ disabled as requested\n",
              &ConfigData->NtNameForPort)
            );

        goto ExtensionCleanup;
    }

    //
    // Common error path cleanup.  If the status is
    // bad, get rid of the device extension, device object
    // and any memory associated with it.
    //

ExtensionCleanup:

    ExFreePool(uniNameString.Buffer);
    ExFreePool(ConfigData->NtNameForPort.Buffer);
    ExFreePool(ConfigData);

    if (NT_ERROR(status)) {

        PptCleanupDevice(extension);
        IoDeleteDevice(deviceObject);
        IoGetConfigurationInformation()->ParallelCount--;
    }

    return status;
}

#endif // STATIC_LOAD