/*++

Copyright (c) 1993-1997 Microsoft Corporation

Module Name:

    parinit.c

Abstract:

    This module contains the code for the parallel class driver.
    This is the static initialization code taken from the oringinal
    parclass (and replaced with WDM code).

Author:

    Timothy T. Wells (WestTek, L.L.C.)  20 January 1997
    
Environment:

    Kernel mode

Revision History :

--*/

#ifdef STATIC_LOAD

#include "ntddk.h"
#include "ntddser.h"
#include "parallel.h"
#include "parclass.h"
#include "parlog.h"

NTSTATUS
DriverEntry(
    IN  PDRIVER_OBJECT  DriverObject,
    IN  PUNICODE_STRING RegistryPath
    );

BOOLEAN
ParMakeNames(
    IN  ULONG           ParallelPortNumber,
    OUT PUNICODE_STRING PortName,
    OUT PUNICODE_STRING ClassName,
    OUT PUNICODE_STRING LinkName
    );

VOID
ParInitializeClassDevice(
    IN  PDRIVER_OBJECT  DriverObject,
    IN  PUNICODE_STRING RegistryPath,
    IN  ULONG           ParallelPortNumber
    );

VOID
ParCheckParameters(
    IN      PUNICODE_STRING     RegistryPath,
    IN OUT  PDEVICE_EXTENSION   Extension
    );


#ifdef ALLOC_PRAGMA
#pragma alloc_text(INIT,DriverEntry)
#pragma alloc_text(INIT,ParMakeNames)
#pragma alloc_text(INIT,ParInitializeClassDevice)
#pragma alloc_text(INIT,ParCheckParameters)
#endif


NTSTATUS
DriverEntry(
    IN PDRIVER_OBJECT DriverObject,
    IN PUNICODE_STRING RegistryPath
    )

/*++

Routine Description:

    This routine is called at system initialization time to initialize
    this driver.

Arguments:

    DriverObject    - Supplies the driver object.

    RegistryPath    - Supplies the registry path for this driver.

Return Value:

    STATUS_SUCCESS          - We could initialize at least one device.
    STATUS_NO_SUCH_DEVICE   - We could not initialize even one device.

--*/

{
    ULONG       i;

    //
    // allocate the mutex to protect driver reference count
    //

    OpenCloseMutex = ExAllocatePool(NonPagedPool, sizeof(FAST_MUTEX));
    if (!OpenCloseMutex) {

        //
        // NOTE - we could probably do without bailing here and just 
        // leave a note for ourselves to never page out, but since we
        // don't have enough memory to allocate a mutex we should probably
        // avoid leaving the driver paged in at all times
        //

        return STATUS_INSUFFICIENT_RESOURCES;
    }

    ExInitializeFastMutex(OpenCloseMutex);

    for (i = 0; i < IoGetConfigurationInformation()->ParallelCount; i++) {
        ParInitializeClassDevice(DriverObject, RegistryPath, i);
    }

    if (!DriverObject->DeviceObject) {
        return STATUS_NO_SUCH_DEVICE;
    }

    //
    // Initialize the Driver Object with driver's entry points
    //

    DriverObject->MajorFunction[IRP_MJ_CREATE] = ParCreateOpen;
    DriverObject->MajorFunction[IRP_MJ_CLOSE] = ParClose;
    DriverObject->MajorFunction[IRP_MJ_CLEANUP] = ParCleanup;
    DriverObject->MajorFunction[IRP_MJ_READ] = ParReadWrite;
    DriverObject->MajorFunction[IRP_MJ_WRITE] = ParReadWrite;
    DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL] = ParDeviceControl;
    DriverObject->MajorFunction[IRP_MJ_QUERY_INFORMATION] =
            ParQueryInformationFile;
    DriverObject->MajorFunction[IRP_MJ_SET_INFORMATION] =
            ParSetInformationFile;
    DriverObject->DriverUnload = ParUnload;

    //
    // page out the driver if we can
    //

    ParReleaseDriver();

    return STATUS_SUCCESS;
}

BOOLEAN
ParMakeNames(
    IN  ULONG           ParallelPortNumber,
    OUT PUNICODE_STRING PortName,
    OUT PUNICODE_STRING ClassName,
    OUT PUNICODE_STRING LinkName
    )

/*++

Routine Description:

    This routine generates the names \Device\ParallelPortN,
    \Device\ParallelN, and \DosDevices\LPT(N+1) where N is
    'ParallelPortNumber'.  This routine will allocate pool
    so that the buffers of these unicode strings need to
    be eventually freed.

Arguments:

    ParallelPortNumber  - Supplies the port number.

    PortName            - Returns the port name.

    ClassName           - Returns the class name.

    LinkName            - Returns the link name.

Return Value:

    FALSE   - Failure.
    TRUE    - Success.

--*/
{
    UNICODE_STRING  prefix, digits, linkPrefix, linkDigits;
    WCHAR           digitsBuffer[10], linkDigitsBuffer[10];
    UNICODE_STRING  portSuffix, classSuffix, linkSuffix;
    NTSTATUS        status;

    // Put together local variables for constructing names.

    RtlInitUnicodeString(&prefix, L"\\Device\\");
    RtlInitUnicodeString(&linkPrefix, L"\\DosDevices\\");
    RtlInitUnicodeString(&portSuffix, DD_PARALLEL_PORT_BASE_NAME_U);
    RtlInitUnicodeString(&classSuffix, DEFAULT_NT_SUFFIX);
    RtlInitUnicodeString(&linkSuffix, DEFAULT_PARALLEL_NAME);
    digits.Length = 0;
    digits.MaximumLength = 20;
    digits.Buffer = digitsBuffer;
    linkDigits.Length = 0;
    linkDigits.MaximumLength = 20;
    linkDigits.Buffer = linkDigitsBuffer;
    status = RtlIntegerToUnicodeString(ParallelPortNumber, 10, &digits);
    if (!NT_SUCCESS(status)) {
        return FALSE;
    }
    status = RtlIntegerToUnicodeString(ParallelPortNumber + 1, 10, &linkDigits);
    if (!NT_SUCCESS(status)) {
        return FALSE;
    }


    // Make the port name.

    PortName->Length = 0;
    PortName->MaximumLength = prefix.Length + portSuffix.Length +
                              digits.Length + sizeof(WCHAR);
    PortName->Buffer = ExAllocatePool(PagedPool, PortName->MaximumLength);
    if (!PortName->Buffer) {
        return FALSE;
    }
    RtlZeroMemory(PortName->Buffer, PortName->MaximumLength);
    RtlAppendUnicodeStringToString(PortName, &prefix);
    RtlAppendUnicodeStringToString(PortName, &portSuffix);
    RtlAppendUnicodeStringToString(PortName, &digits);


    // Make the class name.

    ClassName->Length = 0;
    ClassName->MaximumLength = prefix.Length + classSuffix.Length +
                               digits.Length + sizeof(WCHAR);
    ClassName->Buffer = ExAllocatePool(PagedPool, ClassName->MaximumLength);
    if (!ClassName->Buffer) {
        ExFreePool(PortName->Buffer);
        return FALSE;
    }
    RtlZeroMemory(ClassName->Buffer, ClassName->MaximumLength);
    RtlAppendUnicodeStringToString(ClassName, &prefix);
    RtlAppendUnicodeStringToString(ClassName, &classSuffix);
    RtlAppendUnicodeStringToString(ClassName, &digits);


    // Make the link name.

    LinkName->Length = 0;
    LinkName->MaximumLength = linkPrefix.Length + linkSuffix.Length +
                             linkDigits.Length + sizeof(WCHAR);
    LinkName->Buffer = ExAllocatePool(PagedPool, LinkName->MaximumLength);
    if (!LinkName->Buffer) {
        ExFreePool(PortName->Buffer);
        ExFreePool(ClassName->Buffer);
        return FALSE;
    }
    RtlZeroMemory(LinkName->Buffer, LinkName->MaximumLength);
    RtlAppendUnicodeStringToString(LinkName, &linkPrefix);
    RtlAppendUnicodeStringToString(LinkName, &linkSuffix);
    RtlAppendUnicodeStringToString(LinkName, &linkDigits);

    return TRUE;
}

VOID
ParInitializeClassDevice(
    IN  PDRIVER_OBJECT  DriverObject,
    IN  PUNICODE_STRING RegistryPath,
    IN  ULONG           ParallelPortNumber
    )

/*++

Routine Description:

    This routine is called for every parallel port in the system.  It
    will create a class device upon connecting to the port device
    corresponding to it.

Arguments:

    DriverObject        - Supplies the driver object.

    RegistryPath        - Supplies the registry path.

    ParallelPortNumber  - Supplies the number for this port.

Return Value:

    None.

--*/

{
    UNICODE_STRING      portName, className, linkName;
    NTSTATUS            status;
    PDEVICE_OBJECT      deviceObject;
    PDEVICE_EXTENSION   extension;

    //
    // Cobble together the port, class, and windows symbolic names.
    //

    if (!ParMakeNames(ParallelPortNumber, &portName, &className, &linkName)) {

        ParLogError(DriverObject, NULL, PhysicalZero, PhysicalZero, 0, 0, 0, 1,
                    STATUS_SUCCESS, PAR_INSUFFICIENT_RESOURCES);

        ParDump(
            PARERRORS,
            ("PARALLEL: Could not form Unicode name string.\n")
            );

        return;
    }


    // Create the device object.

    status = IoCreateDevice(DriverObject, sizeof(DEVICE_EXTENSION),
                            &className, FILE_DEVICE_PARALLEL_PORT, 0, TRUE,
                            &deviceObject);

    if (!NT_SUCCESS(status)) {

        ParLogError(DriverObject, NULL, PhysicalZero, PhysicalZero, 0, 0, 0, 2,
                    STATUS_SUCCESS, PAR_INSUFFICIENT_RESOURCES);

        ParDump(
            PARERRORS,
            ("PARALLEL: Could not create a device for %wZ\n",
             &className)
            );

        ExFreePool(linkName.Buffer);
        goto Cleanup;
    }


    // Now that the device has been created,
    // set up the device extension.

    extension = deviceObject->DeviceExtension;

    RtlZeroMemory(extension, sizeof(DEVICE_EXTENSION));

    extension->DeviceObject = deviceObject;
    deviceObject->Flags |= DO_BUFFERED_IO;

    status = IoGetDeviceObjectPointer(&portName, FILE_READ_ATTRIBUTES,
                                      &extension->PortDeviceFileObject,
                                      &extension->PortDeviceObject);

    if (!NT_SUCCESS(status)) {

        ParLogError(DriverObject, NULL, PhysicalZero, PhysicalZero, 0, 0, 0, 3,
                    STATUS_SUCCESS, PAR_CANT_FIND_PORT_DRIVER);

        ParDump(
            PARERRORS,
            ("PARALLEL: Unable to get device object pointer for port object.\n")
            );

        IoDeleteDevice(deviceObject);

        ExFreePool(linkName.Buffer);
        goto Cleanup;
    }

    extension->DeviceObject->StackSize =
            extension->PortDeviceObject->StackSize + 1;

    InitializeListHead(&extension->WorkQueue);

    KeInitializeSemaphore(&extension->RequestSemaphore, 0, MAXLONG);

    extension->TimerStart = PAR_WRITE_TIMEOUT_VALUE;

    extension->Initialized = FALSE;
    extension->Initializing = FALSE;
    extension->DeferredWorkItem = NULL;

    status = ParGetPortInfoFromPortDevice(extension);

    if (!NT_SUCCESS(status)) {

        ParLogError(DriverObject, NULL, PhysicalZero, PhysicalZero, 0, 0, 0, 4,
                    STATUS_SUCCESS, PAR_CANT_FIND_PORT_DRIVER);

        ParDump(
            PARERRORS,
            ("PARALLEL: Can't get port info from port device.\n")
            );

        ObDereferenceObject(extension->PortDeviceFileObject);

        IoDeleteDevice(deviceObject);

        ExFreePool(linkName.Buffer);
        goto Cleanup;
    }

    extension->BusyDelay = 0;
    extension->BusyDelayDetermined = FALSE;

    if (extension->OriginalController.HighPart == 0 &&
        extension->OriginalController.LowPart == (ULONG) extension->Controller) {

        extension->UsePIWriteLoop = FALSE;
    } else {
        extension->UsePIWriteLoop = TRUE;
    }


    // Set up some constants.

    extension->AbsoluteOneSecond.QuadPart = 10*1000*1000;
    extension->OneSecond.QuadPart = -(extension->AbsoluteOneSecond.QuadPart);

    // Try to initialize the printer here.  This is done here so that printers that
    // change modes upon reset can be set to the desired mode before the first print.

    //
    // Queue up a workitem to initialize the device (so we don't take forever loading).
    // 

    ParDeferDeviceInitialization(extension);

    // Now setup the symbolic link for windows.

    status = IoCreateUnprotectedSymbolicLink(&linkName, &className);

    if (NT_SUCCESS(status)) {

        // We were able to create the symbolic link, so record this
        // value in the extension for cleanup at unload time.

        extension->CreatedSymbolicLink = TRUE;
        extension->SymbolicLinkName = linkName;

        // Write out the result of the symbolic link to the registry.

        status = RtlWriteRegistryValue(RTL_REGISTRY_DEVICEMAP,
                                       L"PARALLEL PORTS",
                                       className.Buffer,
                                       REG_SZ,
                                       linkName.Buffer,
                                       linkName.Length + sizeof(WCHAR));
    } else {

        //
        // Oh well, couldn't create the symbolic link.
        //

        extension->CreatedSymbolicLink = FALSE;

        ParDump(
            PARERRORS,
            ("PARALLEL: Couldn't create the symbolic link\n"
             "--------  for port %wZ\n",
             &className)
            );

        ParLogError(DriverObject, deviceObject, extension->OriginalController,
                    PhysicalZero, 0, 0, 0, 5, status, PAR_NO_SYMLINK_CREATED);

    }

    ExFreePool(linkName.Buffer);

    if (!NT_SUCCESS(status)) {

        //
        // Oh well, it didn't work.  Just go to cleanup.
        //

        ParDump(
            PARERRORS,
            ("PARALLEL: Couldn't create the device map entry\n"
             "--------  for port %wZ\n",
             &className)
            );

        ParLogError(DriverObject, deviceObject, extension->OriginalController,
                    PhysicalZero, 0, 0, 0, 6, status,
                    PAR_NO_DEVICE_MAP_CREATED);
    }


    // Check the registry for parameters.

    ParCheckParameters(RegistryPath, extension);


Cleanup:
    ExFreePool(portName.Buffer);
    ExFreePool(className.Buffer);
}

VOID
ParCheckParameters(
    IN      PUNICODE_STRING     RegistryPath,
    IN OUT  PDEVICE_EXTENSION   Extension
    )

/*++

Routine Description:

    This routine reads the parameters section of the registry and modifies
    the device extension as specified by the parameters.

Arguments:

    RegistryPath    - Supplies the registry path.

    Extension       - Supplies the device extension.

Return Value:

    None.

--*/

{
    UNICODE_STRING parameters;
    UNICODE_STRING path;
    RTL_QUERY_REGISTRY_TABLE paramTable[4];
    ULONG usePIWriteLoop;
    ULONG zero = 0;
    ULONG useNT35Priority;
    NTSTATUS status;

    RtlInitUnicodeString(&parameters, L"\\Parameters");
    path.Length = 0;
    path.MaximumLength = RegistryPath->Length +
                         parameters.Length +
                         sizeof(WCHAR);
    path.Buffer = ExAllocatePool(PagedPool, path.MaximumLength);

    if (!path.Buffer) {
        return;
    }
    RtlZeroMemory(path.Buffer, path.MaximumLength);
    RtlAppendUnicodeStringToString(&path, RegistryPath);
    RtlAppendUnicodeStringToString(&path, &parameters);

    RtlZeroMemory(paramTable, sizeof(paramTable));

    paramTable[0].Flags = RTL_QUERY_REGISTRY_DIRECT;
    paramTable[0].Name = L"UsePIWriteLoop";
    paramTable[0].EntryContext = &usePIWriteLoop;
    paramTable[0].DefaultType = REG_DWORD;
    paramTable[0].DefaultData = &zero;
    paramTable[0].DefaultLength = sizeof(ULONG);

    paramTable[1].Flags = RTL_QUERY_REGISTRY_DIRECT;
    paramTable[1].Name = L"UseNT35Priority";
    paramTable[1].EntryContext = &useNT35Priority;
    paramTable[1].DefaultType = REG_DWORD;
    paramTable[1].DefaultData = &zero;
    paramTable[1].DefaultLength = sizeof(ULONG);

    paramTable[2].Flags = RTL_QUERY_REGISTRY_DIRECT;
    paramTable[2].Name = L"InitializationTimeout";
    paramTable[2].EntryContext = &(Extension->InitializationTimeout);
    paramTable[2].DefaultType = REG_DWORD;
    paramTable[2].DefaultData = &zero;
    paramTable[2].DefaultLength = sizeof(ULONG);

    status = RtlQueryRegistryValues(RTL_REGISTRY_ABSOLUTE | RTL_REGISTRY_OPTIONAL,
                                    path.Buffer, paramTable, NULL, NULL);

    if (NT_SUCCESS(status)) {

        if(usePIWriteLoop) {
            Extension->UsePIWriteLoop = TRUE;
        }

        if(useNT35Priority) {
            Extension->UseNT35Priority = TRUE;
        }

        if(Extension->InitializationTimeout == 0) {
            Extension->InitializationTimeout = 15;
        }
    } else {

        Extension->InitializationTimeout = 15;

    }

    ExFreePool(path.Buffer);
}

#endif