/******************************************************************************
******************************************************************************/

/******************************************************************************

  wsti.cpp
  WDM Still Imaging interface

  Copyright (c) 1997 Microsoft Corporation
  All rights reserved

Notes:
  THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
  KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
  PURPOSE.

******************************************************************************/

/*****************************************************************************

    globals

*****************************************************************************/

#define OFF		0
#define ON		1

//
// scanner command strings
//
CHAR SCLReset[]			= "E";
CHAR SetXRes[]			= "*a%dR";
CHAR SetYRes[]			= "*a%dS";
CHAR SetXExtPix[]		= "*f%dP";
CHAR SetYExtPix[]		= "*f%dQ";
CHAR InqXRes[]			= "*s10323R";
CHAR SetBitsPerPixel[]  = "*a%dG";
CHAR SetIntensity[]     = "*a%dL";
CHAR SetContrast[]     	= "*a%dK";
CHAR SetNegative[]     	= "*a%dI";
CHAR SetMirror[]     	= "*a%dM";
CHAR SetDataType[]     	= "*a%dT";
CHAR ScanCmd[]			= "*f0S";
CHAR LampOn[]			= "*f1L";
CHAR LampOff[]			= "*f0L";
CHAR PollButton[]		= "*s1044E";

//
// STI.H - STI_DEVICE_MJ_TYPE
//
STRINGTABLE StStiDeviceType[] =
{
	0, "StiDeviceTypeDefault",0,
	1, "StiDeviceTypeScanner",0,
	2, "StiDeviceTypeDigitalCamera",0,
    0, "",-1
};

//
// STIERR.H - errors
//
STRINGTABLE StStiError[] =
{
	STI_OK,						"STI_OK",0,
	STI_NOTCONNECTED,			"STI_NOTCONNECTED",0,
	STI_CHANGENOEFFECT,			"STI_CHANGENOEFFECT",0,
	STIERR_OLD_VERSION,			"STIERR_OLD_VERSION",0,
	STIERR_BETA_VERSION,		"STIERR_BETA_VERSION",0,
	STIERR_BADDRIVER,			"STIERR_BADDRIVER",0,
	STIERR_DEVICENOTREG,		"STIERR_DEVICENOTREG",0,
	STIERR_OBJECTNOTFOUND,		"STIERR_OBJECTNOTFOUND",0,
	STIERR_INVALID_PARAM,		"STIERR_INVALID_PARAM",0,
	STIERR_NOINTERFACE,			"STIERR_NOINTERFACE",0,
    STIERR_GENERIC,				"STIERR_GENERIC", 0,
    STIERR_OUTOFMEMORY,			"STIERR_OUTOFMEMORY", 0,
    STIERR_UNSUPPORTED,			"STIERR_UNSUPPORTED", 0,
    STIERR_NOT_INITIALIZED,		"STIERR_NOT_INITIALIZED", 0,
    STIERR_ALREADY_INITIALIZED, "STIERR_ALREADY_INITIALIZED", 0,
    STIERR_DEVICE_LOCKED,		"STIERR_DEVICE_LOCKED", 0,
    STIERR_READONLY,			"STIERR_READONLY", 0,
    STIERR_NOTINITIALIZED,		"STIERR_NOTINITIALIZED", 0,
    STIERR_NEEDS_LOCK,			"STIERR_NEEDS_LOCK", 0,
	STIERR_SHARING_VIOLATION,	"STIERR_SHARING_VIOLATION", 0,
	STIERR_HANDLEEXISTS,		"STIERR_HANDLEEXISTS", 0,
    0, "",-1
};

//
// global still image
//
PSTI					pSti = NULL;		// handle to STI subsystem
PVOID					pStiInfo = NULL;	// STI device info buffer
PSTI_DEVICE_INFORMATION pStiInfoPtr = NULL;	// pointer to device in pStiBuffer
int						nStiNumber = 0;		// 0 based index into pStiInfo
DWORD					dwStiTotal = 0;		// total number of STI devices found
PSTIDEVICE				pStiDevice = NULL;	// STI device being used
STI_DEVICE_STATUS       StiStatus;			// status of STI device being used

LPBITMAPINFO			pDIB;				// pointer to DIB bitmap header
HBITMAP					hDIBSection;		// handle to DIB
LPBYTE					pDIBBits;			// pointer to DIB bit data
int						m_XSize = 800,		// horizontal size in pixels
   						m_YSize = 800;		// vertical size in pixels


/*****************************************************************************

    prototypes

*****************************************************************************/

INT		CreateScanDIB(HWND);
INT		DisplayScanDIB(HWND);

INT		StiEnum();
void	StiLamp(int);
INT		StiSelect(HWND);
INT		StiScan(HWND);
void	StiClose();

HRESULT WINAPI		SendDeviceCommandString(PSTIDEVICE,LPSTR,...);
HRESULT WINAPI		TransactDevice(PSTIDEVICE,LPSTR,UINT,LPSTR,...);

BOOL FAR PASCAL		SelectDevice(HWND,UINT,WPARAM,LPARAM);


/*****************************************************************************
	INT StiEnum( )
		Opens STI subsystem and enumerates any still image devices found

	Parameters:
		none

	Return:
		0 on success, -1 on error

*****************************************************************************/
INT StiEnum( )
{
    HRESULT hReturn;
    UINT    iDev;


	//
	// any errors?
	//
	if (LastError(TRUE))
		DisplayOutput("Found an error before opening STI subsystem");

	//
	// close any open devices before enumeration
	//
	StiClose();

	//
    // Open the Sti subsystem
	//
    hReturn = StiCreateInstance(
		GetModuleHandle(NULL),	// current instance
		STI_VERSION,			// STI version
		&pSti,					// handle to STI subsystem
		NULL					// Controlling unk
		);

    if (!SUCCEEDED(hReturn)) {
		DisplayOutput("StiCreateInstance returned %xh %u",hReturn,hReturn);
        DisplayOutput("* call failed");
        return -1;
    }

	//
	// any errors?
	//
	if (LastError(TRUE))
		DisplayOutput("Found an error after StiCreateInstance");

	//
    // Enumerate devices
	//
    hReturn = pSti->GetDeviceList(
		0,				// Type
		0,				// Flags
		&dwStiTotal,	// number of devices found
		&pStiInfo		// STI device info buffer
		);

	//
	// any errors?
	//
	if (LastError(TRUE))
		DisplayOutput("Found an error after ->GetDeviceList");

    if (!SUCCEEDED(hReturn) || !pStiInfo) {
		DisplayOutput("->GetDeviceList returned %xh %u",hReturn,hReturn);
        DisplayOutput("* call failed");
		StiClose();
        return -1;
    }

	if (dwStiTotal = 1)
		strcpy(pszString,"");
	else
		strcpy(pszString,"s");
    DisplayOutput("->GetDeviceList found %d device%s",dwStiTotal,pszString);

	//
    // Display Sti info on each device found
	//
    pStiInfoPtr = (PSTI_DEVICE_INFORMATION) pStiInfo;

	//
	// make sure the buffer is valid
	//
	if (pStiInfoPtr->dwSize != sizeof(STI_DEVICE_INFORMATION)) {
		DisplayOutput("* invalid buffer returned from ->GetDeviceList");
		StiClose();
		return -1;
	}

    for (iDev=0;iDev < dwStiTotal;iDev++, pStiInfoPtr++) {
        DisplayOutput("  Device number %2d :",iDev);
		pszString = StrFromTable(GET_STIDEVICE_TYPE(pStiInfoPtr->DeviceType),
			StStiDeviceType);
		DisplayOutput("  Device type %xh %s",
			GET_STIDEVICE_TYPE(pStiInfoPtr->DeviceType),
			pszString);
		DisplayOutput("  Device subtype %xh",
			GET_STIDEVICE_SUBTYPE(pStiInfoPtr->DeviceType));
        DisplayOutput("  Internal name %S",
			pStiInfoPtr->szDeviceInternalName);
    }

	//
	// point to first device again
	//
	nStiNumber = 0;
    pStiInfoPtr = (PSTI_DEVICE_INFORMATION) pStiInfo + nStiNumber;

    return (0);
}


/*****************************************************************************
	INT StiSelect(HWND hWnd)
		Select and open a specific Still Image device

	Parameters:
		handle to current window

	Return:
		0 on success, -1 on error

*****************************************************************************/
INT StiSelect(HWND hWnd)
{
    HRESULT hReturn;
	BOOL	bReturn;


	//
	// check that STI subsystem is loaded
	//
	if (! pSti)
		return -1;

	//
	// any errors?
	//
	if (LastError(TRUE))
		DisplayOutput("Found an error before selecting STI device");

	bReturn = fDialog(IDD_SELECT, hWnd, (FARPROC) SelectDevice);

	//
	// just return if user pressed CANCEL in dialog
	//
	if (bReturn == FALSE)
		return 0;

	//
	// close any currently active imaging device
	//
    if (pStiDevice) {
        pStiDevice->Release();
		pStiDevice = 0;
    }

	//
	// get pointer to device selected in dialog
	//
    pStiInfoPtr = (PSTI_DEVICE_INFORMATION) pStiInfo + nStiNumber;

	if (pStiInfoPtr->szDeviceInternalName == NULL) {
		DisplayOutput("* invalid device name");
		return -1;
	}

    //
    // Create object for selected device
    //
    hReturn = pSti->CreateDevice(
		pStiInfoPtr->szDeviceInternalName,	// device name
        STI_DEVICE_CREATE_DATA,				// Mode
        &pStiDevice,						// STI device being used
        NULL );								// Controlling unk

	//
	// any errors?
	//
	if (LastError(TRUE))
		DisplayOutput("Found an error after ->CreateDevice");

    if (!SUCCEEDED(hReturn) || !pStiDevice) {
	    DisplayOutput("->CreateDevice on %S returned %xh %u",
           pStiInfoPtr->szDeviceInternalName,hReturn,hReturn);
		pszString = StrFromTable(hReturn,StStiError);
        DisplayOutput("* call failed %s",pszString);
        return -1;
    }

    //
    // Try to communicate with device in raw mode
    //
    pStiDevice->LockDevice(2000);

	//
	// any errors?
	//
	if (LastError(TRUE))
		DisplayOutput("Found an error after ->LockDevice");

    // Resetting device
    pStiDevice->DeviceReset();

	//
	// any errors?
	//
	if (LastError(TRUE))
		DisplayOutput("Found an error after ->DeviceReset");

    //
    // Get and display status
    //
    ZeroMemory(&StiStatus,sizeof(StiStatus));
    hReturn = pStiDevice->GetStatus(&StiStatus);

	//
	// any errors?
	//
	if (LastError(TRUE))
		DisplayOutput("Found an error after ->GetStatus");

    if (!SUCCEEDED(hReturn) ) {
		DisplayOutput("->GetStatus returned %xh %u",hReturn,hReturn);
		pszString = StrFromTable(hReturn,StStiError);
        DisplayOutput("* call failed %s",pszString);
        return -1;
    }

	// Let user see menu items
	EnableMenuItem(hMenu, IDM_LAMPON,  MF_ENABLED);
	EnableMenuItem(hMenu, IDM_LAMPOFF, MF_ENABLED);
	EnableMenuItem(hMenu, IDM_SCAN,    MF_ENABLED);

	//
	// tell user we are ready
	//
	wsprintf(pszString,"%ls",pStiInfoPtr->szDeviceInternalName);
	DisplayOutput("  \"%s\" is ready for Action",pszString);

	return(0);
}


/*****************************************************************************
	INT StiClose( )
		Close any open devices and Sti subsystem

	Parameters:
		none

	Return:
		none

*****************************************************************************/
void StiClose( )
{
	// if a device is selected..
    if (pStiDevice) {
		// unselect the device
		pStiDevice->UnLockDevice();
        pStiDevice->Release();

		// clear the device buffers
		pStiDevice = NULL;
		pStiInfo = NULL;

		// disable Actions menu
		EnableMenuItem(hMenu, IDM_LAMPON, MF_GRAYED);
		EnableMenuItem(hMenu, IDM_LAMPOFF, MF_GRAYED);
		EnableMenuItem(hMenu, IDM_SCAN, MF_GRAYED);
    }
	// if Sti subsystem is open..
    if ( pSti) {
		// close subsystem
        pSti->Release();
		// clear buffer
		pSti = NULL;
    }
}


/*****************************************************************************
	void StiLamp(int nOnOff)
		Turn the scanner lamp on and off

	Parameters:
		Send "ON" to turn lamp on, "OFF" to turn it off.

	Return:
		none

*****************************************************************************/
void StiLamp(int nOnOff)
{
    HRESULT hReturn;

	
    //
	// check that an Sti device is selected
    //
	if (pStiDevice == NULL)
		return;

    //
    // Test lamp on/off capability
    //
	if (nOnOff == ON) {
		strcpy(pszString,LampOn);
		strcpy(pszMsg,"On");
	}
	else {
		strcpy(pszString,LampOff);
		strcpy(pszMsg,"Off");
	}

	if (hReturn = SendDeviceCommandString(pStiDevice,pszString))
		DisplayOutput("Failed to turn Lamp %s",pszMsg);
	else
		DisplayOutput("Turned Lamp  %s",pszMsg);
}


/******************************************************************************
	HRESULT
	WINAPI
	SendDeviceCommandString(
		PSTIDEVICE  pStiDevice,
		LPSTR       pszFormat,
		...
		)
	Send formatted SCL string to the device

	Parameters:
		StiDevice buffer and the command string

	Return:
		Result of the call.

******************************************************************************/
HRESULT
WINAPI
SendDeviceCommandString(
    PSTIDEVICE  pStiDevice,
    LPSTR       pszFormat,
    ...
    )
{

    CHAR    ScanCommand[255];
    UINT    cbChar = 1;
    HRESULT hReturn;


	//
    // Format command string
	// 
    ZeroMemory(ScanCommand,sizeof(ScanCommand));
    ScanCommand[0]='\033';							

    va_list ap;
    va_start(ap, pszFormat);
    cbChar += wvsprintfA(ScanCommand+1, pszFormat, ap);
    va_end(ap);

    DisplayOutput("->RawWriteData sending \"%2x %s\"",
		ScanCommand[0],ScanCommand+1);

	//
    // Send command string to the device
	//
    hReturn = pStiDevice->RawWriteData(
		ScanCommand,	//
		cbChar,			//
		NULL			//
		);

	//
	// any errors?
	//
	if (LastError(TRUE))
		DisplayOutput("Found an error after ->RawWriteData");

    if (!SUCCEEDED(hReturn)) {
		DisplayOutput("->RawWriteData returned %xh %u",hReturn,hReturn);
		pszString = StrFromTable(hReturn,StStiError);
        DisplayOutput("* call failed %s",pszString);
    }

    return hReturn;
}


/******************************************************************************
	HRESULT
	WINAPI
	TransactDevice(
		PSTIDEVICE  pStiDevice,
		LPSTR	    lpResultBuffer,
	    UINT        cbResultBufferSize,
		LPSTR       pszFormat,
		...
		)
	Send formatted SCL string to the device and return data in a buffer.

	Parameters:
		StiDevice buffer, data buffer, sizeof databuffer and the command string.
		
	Return:
		Result of the call.

******************************************************************************/
HRESULT
WINAPI
TransactDevice(
    PSTIDEVICE  pStiDevice,
    LPSTR	    lpResultBuffer,
    UINT        cbResultBufferSize,
    LPSTR       pszFormat,
    ...
    )
{

    CHAR    ScanCommand[255];
    UINT    cbChar = 1;
    ULONG   cbActual = 0;
    HRESULT hReturn;


   	//
	// Format command string
	//
    ZeroMemory(ScanCommand,sizeof(ScanCommand));
    ScanCommand[0]='\033';

    va_list ap;
    va_start(ap, pszFormat);
    cbChar += wvsprintfA(ScanCommand+1, pszFormat, ap);
    va_end(ap);

    DisplayOutput("->Escape sending \"%2x %s\"",
		ScanCommand[0],ScanCommand+1);

	//
    // Send command string to the device
	//
    hReturn = pStiDevice->Escape(
		StiTransact,		//
		ScanCommand,		//
		cbChar,				//
		lpResultBuffer,		//
		cbResultBufferSize,	//
		&cbActual);			//

	//
	// any errors?
	//
	if (LastError(TRUE))
		DisplayOutput("Found an error after ->Escape");

    if (!SUCCEEDED(hReturn)) {
		DisplayOutput("->RawWriteData returned %xh %u",hReturn,hReturn);
		pszString = StrFromTable(hReturn,StStiError);
        DisplayOutput("* call failed %s",pszString);
    }
	if (cbActual != 0)
		DisplayOutput("  cbActual %xh",cbActual);

    return hReturn;
}


/******************************************************************************
	BOOL FAR PASCAL SelectDevice(HWND,UINT,WPARAM,LPARAM)
		Put up a dialog for user to select a Still Image device

	Parameters:
		The usual dialog box parameters.

	Return:
		Result of the call.

******************************************************************************/
BOOL FAR PASCAL SelectDevice(HWND hDlg,UINT msg,WPARAM wParam,LPARAM lParam)
{
    PSTI_DEVICE_INFORMATION pTmpInfoPtr;
    UINT					iDev;
    int						iIndex;


    switch (msg) {

        case WM_INITDIALOG:

			//
			// point to first Sti device
			//
			pTmpInfoPtr = (PSTI_DEVICE_INFORMATION) pStiInfo;

			//
			// loop through and fill dialog with Sti Device Internal Name
			//
			for (iDev=0;iDev < dwStiTotal;iDev++, pTmpInfoPtr++) {
				//
				// convert UNICODE string to ANSI
				//
				wsprintf(pszString,"%ls",pTmpInfoPtr->szDeviceInternalName);
				
				iIndex = SendDlgItemMessage(hDlg,IDC_SELECT_DEVICE,
					CB_ADDSTRING,0,(LPARAM) (LPCTSTR) pszString);
				}
			SendDlgItemMessage(hDlg,IDC_SELECT_DEVICE,CB_SETCURSEL,0,0);

            return TRUE;

        case WM_COMMAND:
            switch (wParam) {
                case IDOK:
                    nStiNumber = SendDlgItemMessage(hDlg,IDC_SELECT_DEVICE,
						CB_GETCURSEL,0,0);

					//
					// ensure device number not greater than total
					//
					if (nStiNumber >= (int) dwStiTotal)
						nStiNumber = (int) dwStiTotal;

                    EndDialog(hDlg, TRUE);
                    return TRUE;

                case IDCANCEL:
                    EndDialog(hDlg, FALSE);
                    return TRUE;
            }

    }
    return FALSE;
}


/*****************************************************************************
	INT		CreateScanDIB(HWND);
		Create a DIB to display scanned image..

	Parameters:
		Handle to the window to display image in.

	Return:
		0 on success, -1 on error

*****************************************************************************/
INT CreateScanDIB(HWND hWnd)
{
	HDC					hScreenDC;
	RGBTRIPLE			*pTriplet;
	LPBITMAPINFOHEADER	pHdr;
	int					x,
						y;


	//
	// initialize the DIB
	//
	pDIB = (LPBITMAPINFO) GlobalAlloc(GPTR,sizeof(BITMAPINFO));

	pHdr = &pDIB->bmiHeader;

	pHdr->biSize			= sizeof(BITMAPINFOHEADER);
	pHdr->biWidth			= m_XSize;
	pHdr->biHeight			= -m_YSize;	// indicate top-down dib
	pHdr->biPlanes			= 1;
	pHdr->biBitCount		= 24;
	pHdr->biCompression		= BI_RGB;
	pHdr->biSizeImage		= 0;
	pHdr->biXPelsPerMeter	= 0;
	pHdr->biYPelsPerMeter	= 0;
	pHdr->biClrUsed			= 0;
	pHdr->biClrImportant	= 0;

	//
	// create the DIB
	//
	hScreenDC = GetDC(hWnd);
	if (NULL == (hDIBSection = CreateDIBSection(hScreenDC,	
		(PBITMAPINFO) pDIB,
		DIB_RGB_COLORS,
		(void **) &pDIBBits,
		NULL,
		0)))
	{
		LastError(TRUE);
		DisplayOutput("*failed to create DIB");
		ReleaseDC(hWnd,hScreenDC);
		return -1;
	}
	ReleaseDC(hWnd,hScreenDC);

	//
	// Fill the DIB with colors
	//
	pTriplet = (RGBTRIPLE *) pDIBBits;

	for (x = 0;x < m_XSize;x++) {
		for (y = 0;y < m_YSize;y++,pTriplet++) {
			pTriplet->rgbtBlue  = 200;
			pTriplet->rgbtRed   = 50;
			pTriplet->rgbtGreen = 0;
		}
	}

	return(0);
}


/*****************************************************************************
	INT		DisplayScanDIB(HWND);
		Show the DIB.

	Parameters:
		Handle to the window to display image in.

	Return:
		0 on success, -1 on error

*****************************************************************************/
INT DisplayScanDIB(HWND hWnd)
{
	HDC					hScreenDC;


	//
	// display the DIB
	//
	hScreenDC = GetDC(hWnd);
	SetDIBitsToDevice(hScreenDC,
		0,0,
		m_XSize,m_YSize,
		0,0,
		0,m_YSize,
		pDIBBits,
		(LPBITMAPINFO) pDIB,
		DIB_RGB_COLORS);
	ReleaseDC(hWnd,hScreenDC);

	return(0);
}


/*****************************************************************************
	INT StiScan(HWND hWnd)
		Scan and display an image from device.

	Parameters:
		Handle to the window to display image in.

	Return:
		0 on success, -1 on error

*****************************************************************************/
INT StiScan(HWND hWnd)
{
	HRESULT				hReturn;
	ULONG				cbDataSize,
						ulDIBSize,
						ulScanSize;
	RGBTRIPLE			*pTriplet;
	LPBYTE				pDIBPtr;
	UINT				i,
						iPixel,
						xRes = 0;
	int					m_XResolution = 100,
	   				    m_YResolution = 100;
	CHAR				ScanData[1024*16];


	//
	// ensure there is an active still imaging device open
	//
	if (pStiDevice == NULL)
		return -1;

	//
	// Set basic parameters
	//
	hReturn = SendDeviceCommandString(pStiDevice,SetBitsPerPixel,24);
	hReturn = SendDeviceCommandString(pStiDevice,SetIntensity,0);
	hReturn = SendDeviceCommandString(pStiDevice,SetContrast,0);
	hReturn = SendDeviceCommandString(pStiDevice,SetNegative,1);
	hReturn = SendDeviceCommandString(pStiDevice,SetMirror,0);
	hReturn = SendDeviceCommandString(pStiDevice,SetDataType,5);	// Color

	hReturn = SendDeviceCommandString(pStiDevice,SetXRes,m_XResolution);
	hReturn = SendDeviceCommandString(pStiDevice,SetYRes,m_YResolution);

	hReturn = SendDeviceCommandString(pStiDevice,SetXExtPix,(m_XSize*300/m_XResolution));
	hReturn = SendDeviceCommandString(pStiDevice,SetYExtPix,(m_YSize*300/m_YResolution));

	//
    // Inquire commands ( X and Y resolution)
	//
    cbDataSize = sizeof(ScanData);
    ZeroMemory(ScanData,sizeof(ScanData));
	hReturn = TransactDevice(pStiDevice,ScanData,cbDataSize,InqXRes);

	//
	// calculate the size of the DIB
	//
	ulDIBSize = pDIB->bmiHeader.biWidth * (-pDIB->bmiHeader.biHeight);

	//
    // start the scan
	//
    hReturn = SendDeviceCommandString(pStiDevice,ScanCmd);

    for (i = 0,pDIBPtr = pDIBBits,cbDataSize = sizeof(ScanData);
		cbDataSize == sizeof(ScanData);i++) {

        hReturn = pStiDevice->RawReadData(ScanData,&cbDataSize,NULL);

		if ((cbDataSize * i) < ulDIBSize) {
			//
			// copy this scanline into the DIB until it is full
			//
			memcpy(pDIBPtr,ScanData,cbDataSize);
			pDIBPtr += cbDataSize;
		}
	}

	//
	// how large was the scan?
	//
	ulScanSize = (sizeof(ScanData))*i+cbDataSize;

    DisplayOutput("Scan done. Total passes %d, bytes %lu.",
		i,ulScanSize);

	//
	// Triplets coming in from scanner are inverted from DIB format
	//
	for (iPixel = 0,pTriplet = (RGBTRIPLE *) pDIBBits;
		iPixel < ulDIBSize/3;iPixel++,pTriplet++) {
		BYTE	bTemp;

		bTemp = pTriplet->rgbtBlue;
		pTriplet->rgbtBlue = pTriplet->rgbtRed;
		pTriplet->rgbtRed = bTemp;
		}

	//
	// display the DIB
	//
	DisplayScanDIB(hWnd);

	return(0);
}


