/*++

Copyright (c) 1986-1997  Microsoft Corporation

Module Name:

    sti.h

Abstract:

    This module contains the user mode still image APIs in COM format

Author:


Revision History:


--*/

#ifndef _STICOM_
#define _STICOM_

// Set packing
#include <pshpack8.h>

// Include COM definitions
#define COM_NO_WINDOWS_H
#include <objbase.h>

#include <stireg.h>
#include <stierr.h>

//
// Compiler pragmas
//
#pragma warning(disable:4200)       // warning about zero-sized arrays being non-stadard C extension

#define DLLEXP __declspec( dllexport )

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Class IID's
 */

// B323F8E0-2E68-11D0-90EA-00AA0060F86C
DEFINE_GUID(CLSID_Sti, 0xB323F8E0L, 0x2E68, 0x11D0, 0x90, 0xEA, 0x00, 0xAA, 0x00, 0x60, 0xF8, 0x6C);

// {7C9086E0-2DC8-11D0-90EA-00AA0060F86C}
DEFINE_GUID(CLSID_StiDevice, 0x7C9086E0L, 0x2DC8, 0x11D0, 0x90, 0xEA, 0x00, 0xAA, 0x00, 0x60, 0xF8, 0x6C);

/*
 * Interface IID's
 */
#if defined( _WIN32 ) && !defined( _NO_COM)

// {641BD880-2DC8-11D0-90EA-00AA0060F86C}
DEFINE_GUID(IID_IStillImage, 0x641BD880L, 0x2DC8, 0x11D0, 0x90, 0xEA, 0x00, 0xAA, 0x00, 0x60, 0xF8, 0x6C);

// {6CFA5A80-2DC8-11D0-90EA-00AA0060F86C}
DEFINE_GUID(IID_IStiDevice, 0x6CFA5A80L, 0x2DC8, 0x11D0, 0x90, 0xEA, 0x00, 0xAA, 0x00, 0x60, 0xF8, 0x6C);

#endif


//
// Generic constants and definitions
//
#define STI_VERSION                 0x00000001

// begin sti_device_information

//
//  Device information definitions and prototypes
// ----------------------------------------------
//

//
//  Following information is used for enumerating still image devices , currently configured
//  in the system. Presence of the device in the enumerated list does not mean availability
// of the device, it only means that device was installed at least once and had not been removed since.
//

//
// Type of device ( scanner, camera) is represented by DWORD value with
// hi word containing generic device type , and lo word containing sub-type
//
typedef enum _STI_DEVICE_MJ_TYPE {
    StiDeviceTypeDefault          = 0,
    StiDeviceTypeScanner          = 1,
    StiDeviceTypeDigitalCamera    = 2
} STI_DEVICE_MJ_TYPE;

typedef DWORD STI_DEVICE_TYPE;

// Macros to extract device type/subtype from single field
#define GET_STIDEVICE_TYPE(dwDevType)   HIWORD(dwDevType)
#define GET_STIDEVICE_SUBTYPE(dwDevType)   LOWORD(dwDevType)

//
// Device capabilities bits.
// Various capabilities are grouped into separate bitmasks
//

typedef struct _STI_DEV_CAPS {
    DWORD   dwGeneric;
} STI_DEV_CAPS, *PSTI_DEV_CAPS;


//
// Notifications are supported.
// If this capability set , device can be subscribed to .
//
#define STI_GENCAP_NOTIFICATIONS    0x00000001

//
// Polling required .
// This capability is used when previous is set to TRUE. Presence of it means
// that device is not capable of issuing "truly" asyncronous notifications, but can
// be polled to determine the moment when event happened
#define STI_GENCAP_POLLING_NEEDED   0x00000002


//
// Type of bus connection for those in need to know
//
#define STI_HW_CONFIG_SCSI      0x0001
#define STI_HW_CONFIG_USB       0x0002
#define STI_HW_CONFIG_SERIAL    0x0004
#define STI_HW_CONFIG_PARALLEL  0x0008

//
// Device information structure, this is not configurable. This data is returned from
// device enumeration API and is used for populating UI or selecting which device
// should be used in current session
//
typedef struct _STI_DEVICE_INFORMATION {
    DWORD   dwSize;

    // Type of the hardware imaging device
    STI_DEVICE_TYPE   DeviceType;

    // Device identifier for reference when creating device object
    WCHAR   szDeviceInternalName[32];

    // Set of capabilities flags
    STI_DEV_CAPS   DeviceCapabilities;

    // This includes bus type
    DWORD   dwHardwareConfiguration;

    // Vendor description string
    LPWSTR    pszVendorDescription;

    // BUGBUG
    // Additional strings, which may be available through getDeviceData APIs
    // for now put them here to avoid braking control panel
    LPWSTR    pszDeviceDescription;
    LPWSTR    pszPortName;
    LPWSTR    pszTwainDataSource;
    LPWSTR    pszEventList;
    LPWSTR    pszPropProvider;

    // Name of the ICM profile, associated with the device
    LPWSTR    pszICMProfileName;

    // END BUGBUG

    // Local specific ("friendly") name of the device, mainly used for showing in the UI
    LPWSTR    pszLocalName;

} STI_DEVICE_INFORMATION, *PSTI_DEVICE_INFORMATION;

// end sti_device_information

//
// Device state information.
// ------------------------
//
// Following types  are used to inquire state characteristics of the device after
// it had been opened.
//
// Device configuration structure contains configurable parameters reflecting
// current state of the device
//
//
// Device hardware status.
//

//
// Individual bits for state acquiring  through StatusMask
//

// State of hardware as known to USD
#define STI_DEVSTATUS_ONLINE_STATE      0x0001

// State of pending events ( as known to USD)
#define STI_DEVSTATUS_EVENTS_STATE      0x0002

//
// Online state values
//
#define STI_ONLINESTATE_OPERATIONAL         0x00000001
#define STI_ONLINESTATE_PENDING             0x00000002
#define STI_ONLINESTATE_ERROR               0x00000004
#define STI_ONLINESTATE_PAUSED              0x00000008
#define STI_ONLINESTATE_PAPER_JAM           0x00000010
#define STI_ONLINESTATE_PAPER_PROBLEM       0x00000020
#define STI_ONLINESTATE_OFFLINE             0x00000040
#define STI_ONLINESTATE_IO_ACTIVE           0x00000080
#define STI_ONLINESTATE_BUSY                0x00000100
#define STI_ONLINESTATE_TRANSFERRING        0x00000200
#define STI_ONLINESTATE_INITIALIZING        0x00000400
#define STI_ONLINESTATE_WARMING_UP          0x00000800
#define STI_ONLINESTATE_USER_INTERVENTION   0x00001000
#define STI_ONLINESTATE_POWER_SAVE          0x00002000

//
// Event processing parameters
//
#define STI_EVENTHANDLING_ENABLED           0x00000001
#define STI_EVENTHANDLING_POLLING           0x00000002
#define STI_EVENTHANDLING_PENDING           0x00000004

typedef struct _STI_DEVICE_STATUS {

    DWORD   StatusMask;

    DWORD   dwOnlineState;
    DWORD   dwHardwareStatusCode;

    DWORD   dwEventHandlingState;
    DWORD   dwPollingInterval;


} STI_DEVICE_STATUS,*PSTI_DEVICE_STATUS;

//
// Structure to describe diagnostic ( test ) request to be processed by USD
//

// Basic test for presence of associated hardware
#define STI_DIAGCODE_HWPRESENCE         0x00000001

//
// Status bits for diagnostic
//

//
// generic diagnostic errors
//

typedef struct _DIAG {

    DWORD   dwSize;

    // Diagnostic request fields
    DWORD   dwBasicDiagCode;
    DWORD   dwVendorDiagCode;

    // Response fields
    DWORD   dwStatusMask;

    DWORD   dwGenericError;
    DWORD   dwVendorError;

    WCHAR szExtendedErrorText[255];

} DIAG,*LPDIAG;

// end device state information.


//
// Event notification mechansims.
// ------------------------------
//
// Those are used to inform last subscribed caller of the changes in device state, initiated by
// device.
//
// The only supported discipline of notification is stack. The last caller to subscribe will be notified
// and will receive notification data. After caller unsubscribes , the previously subscribed caller will
// become active.
//
typedef struct _STISUBSCRIBE {

    HWND    hWndNotify;
    HANDLE  hEvent;

} STISUBSCRIBE,*LPSTISUBSCRIBE;


typedef struct _STINOTIFY {

    DWORD   dwSize;                 // Total size of the notification

    DWORD   dwNotificationCode;     // Standard code
    DWORD   dwSubNotificationCode;  // Vendor specific extension

    BYTE    bNotificationData[];

} STINOTIFY,*LPSTINOTIFY;


// end event_mechanisms

//
// Device create modes
//
#define STI_DEVICE_CREATE_STATUS         0x00000001
#define STI_DEVICE_CREATE_DATA           0x00000002
#define STI_DEVICE_CREATE_BOTH           0x00000003

//
// Flags controlling device enumeration
//
#define STIEDFL_ALLDEVICES             0x00000000
#define STIEDFL_ATTACHEDONLY           0x00000001

//
// Control code , sent to the device through raw control interface
//
typedef  DWORD STI_RAW_CONTROL_CODE;

//
// All raw codes below this one are reserved for future use.
//
#define STI_RAW_RESERVED    0x1000

//
// Device instance value names
//
#define STI_DEVICE_VALUE_TWAIN_NAME    L"TwainDS"
#define STI_DEVICE_VALUE_ISIS_NAME     L"ISISDriverName"
#define STI_DEVICE_VALUE_ICM_PROFILE   L"ICMProfile"
#define STI_DEVICE_VALUE_DEFAULT_LAUNCHAPP  L"DefaultLaunchApp"


 /*
  * COM Interfaces to STI
  */

#ifdef __cplusplus

/* 'struct' not 'class' per the way DECLARE_INTERFACE_ is defined */
interface IStillImage;
interface IStiDevice;

#endif

typedef interface IStillImage             *PSTI;
typedef interface IStiDevice              *PSTIDEVICE;


DLLEXP STDMETHODIMP StiCreateInstance(HINSTANCE hinst, DWORD dwVer, PSTI *ppSti, LPUNKNOWN punkOuter);

/*
 * IStillImage interface
 *
 * Top level STI access interface.
 *
 */


#undef INTERFACE
#define INTERFACE IStillImage
DECLARE_INTERFACE_(IStillImage, IUnknown)
{
    /*** IUnknown methods ***/
    STDMETHOD(QueryInterface) (THIS_ REFIID riid, LPVOID* ppvObj) PURE;
    STDMETHOD_(ULONG, AddRef) (THIS) PURE;
    STDMETHOD_(ULONG, Release) (THIS) PURE;

    /*** IStillImage methods ***/
    STDMETHOD(Initialize) (THIS_ HINSTANCE hinst,DWORD dwVersion) PURE;

    STDMETHOD(GetDeviceList)(THIS_ DWORD dwType,DWORD dwFlags,DWORD *pdwItemsReturned,LPVOID *ppBuffer) PURE;
    STDMETHOD(GetDeviceInfo)(THIS_ LPWSTR  pwszDeviceName, LPVOID *ppBuffer) PURE;

    STDMETHOD(CreateDevice) (THIS_ LPWSTR  pwszDeviceName, DWORD   dwMode, PSTIDEVICE *pDevice,LPUNKNOWN punkOuter) PURE;

    //
    // Device instance values. Used to associate various data with device.
    //
    STDMETHOD(GetDeviceValue)(THIS_ LPWSTR  pwszDeviceName,LPWSTR    pValueName,LPDWORD  pType,LPBYTE   pData,LPDWORD    cbData);
    STDMETHOD(SetDeviceValue)(THIS_ LPWSTR  pwszDeviceName,LPWSTR   pValueName,DWORD   Type,LPBYTE  pData,DWORD   cbData);

    //
    // For appllication started through push model launch, returns associated information
    //
    STDMETHOD(GetSTILaunchInformation)(THIS_ LPWSTR  pwszDeviceName, DWORD *pdwEventCode,LPWSTR  pwszEventName) PURE;


    #ifdef NOT_IMPLEMENTED

        //
        // TO register application for receiving various STI notifications
        //
        STIMETHOD(RegisterDeviceNotification(THIS_ LPWSTR  *ppwszAppName,LPSUBSCRIBE lpSubscribe) PURE;
        STIMETHOD(UnregisterDeviceNotification(THIS_ ) PURE;

        //
        // To control state of notification handling. For polled devices this means state of monitor
        // polling, for true notification devices means enabling/disabling notification flow
        // from monitor to registered applications
        //
        STDMETHOD(EnableHwNotifications)(THIS_ LPWSTR  *ppwszDeviceName,BOOL bNewState) PURE;
        STDMETHOD(GetHwNotificationState)(THIS_ LPWSTR  *ppwszDeviceName,BOOL *pbCurrentState) PURE;

        //
        // When device is installed but not accessible, application may request bus refresh
        // which in some cases will make device known. This is mainly used for nonPnP buses
        // like SCSI, when device was powered on after PnP enumeration
        //
        //
        STIMETHOD(RefreshDeviceBus(THIS_ LPWSTR  *ppwszDeviceName) PURE;

    #endif //NOT_IMPLEMENTED

};

#if !defined(__cplusplus) || defined(CINTERFACE)
#define IStillImage_QueryInterface(p,a,b)       (p)->lpVtbl->QueryInterface(p,a,b)
#define IStillImage_AddRef(p)                   (p)->lpVtbl->AddRef(p)
#define IStillImage_Release(p)                  (p)->lpVtbl->Release(p)
#define IStillImage_Initialize(p,a,b)           (p)->lpVtbl->Initialize(p,a,b)

#define IStillImage_GetDeviceList(p,a,b,c,d)    (p)->lpVtbl->GetDeviceList(p,a,b,c,d)
#define IStillImage_GetDeviceInfo(p,a,b)        (p)->lpVtbl->GetDeviceInfo(p,a,b)
#define IStillImage_CreateDevice(p,a,b,c,d)     (p)->lpVtbl->CreateDevice(p,a,b,c,d)
#endif

/*
 * IStillImage_Device interface
 *
 * This is generic per device interface. Specialized interfaces are also
 * available
 */
#undef INTERFACE
#define INTERFACE IStiDevice
DECLARE_INTERFACE_(IStiDevice, IUnknown)
{
    /*** IUnknown methods ***/
    STDMETHOD(QueryInterface) (THIS_ REFIID riid, LPVOID* ppvObj) PURE;
    STDMETHOD_(ULONG, AddRef) (THIS) PURE;
    STDMETHOD_(ULONG, Release) (THIS) PURE;

    /*** IStiDevice methods ***/
    STDMETHOD(Initialize) (THIS_ HINSTANCE hinst,LPCWSTR pwszDeviceName,DWORD dwVersion,DWORD  dwMode) PURE;

    STDMETHOD(GetCapabilities) (THIS_ PSTI_DEV_CAPS pDevCaps) PURE;

    STDMETHOD(GetStatus) (THIS_ PSTI_DEVICE_STATUS pDevStatus) PURE;

    STDMETHOD(DeviceReset)(THIS ) PURE;
    STDMETHOD(Diagnostic)(THIS_ LPDIAG pBuffer) PURE;

    STDMETHOD(Escape)(THIS_ STI_RAW_CONTROL_CODE    EscapeFunction,LPVOID  lpInData,DWORD   cbInDataSize,LPVOID pOutData,DWORD dwOutDataSize,LPDWORD pdwActualData) PURE ;

    STDMETHOD(GetLastError) (THIS_ LPDWORD pdwLastDeviceError) PURE;

    STDMETHOD(LockDevice) (THIS_ DWORD dwTimeOut) PURE;
    STDMETHOD(UnLockDevice) (THIS ) PURE;

    STDMETHOD(RawReadData)(THIS_ LPVOID lpBuffer,LPDWORD lpdwNumberOfBytes,LPOVERLAPPED lpOverlapped) PURE;
    STDMETHOD(RawWriteData)(THIS_ LPVOID lpBuffer,DWORD nNumberOfBytes,LPOVERLAPPED lpOverlapped) PURE;

    STDMETHOD(RawReadCommand)(THIS_ LPVOID lpBuffer,LPDWORD lpdwNumberOfBytes,LPOVERLAPPED lpOverlapped) PURE;
    STDMETHOD(RawWriteCommand)(THIS_ LPVOID lpBuffer,DWORD nNumberOfBytes,LPOVERLAPPED lpOverlapped) PURE;

    //
    // Subscription is used to enable "control center" style applications , where flow of
    // notifications should be redirected from monitor itself to another "launcher"
    //
    STDMETHOD(Subscribe)(THIS_ LPSTISUBSCRIBE lpSubsribe) PURE;
    STDMETHOD(GetNotificationData)(THIS_ LPSTINOTIFY   lpNotify) PURE;
    STDMETHOD(UnSubscribe)(THIS ) PURE;
};

#if !defined(__cplusplus) || defined(CINTERFACE)
#define IStiDevice_QueryInterface(p,a,b)        (p)->lpVtbl->QueryInterface(p,a,b)
#define IStiDevice_AddRef(p)                    (p)->lpVtbl->AddRef(p)
#define IStiDevice_Release(p)                   (p)->lpVtbl->Release(p)
#define IStiDevice_Initialize(p,a,b,c,d)        (p)->lpVtbl->Initialize(p,a,b,c,d)

#define IStiDevice_GetCapabilities(p,a)         (p)->lpVtbl->GetCapabilities(p,a)
#define IStiDevice_GetStatus(p,a)               (p)->lpVtbl->GetStatus(p,a)
#define IStiDevice_DeviceReset(p)               (p)->lpVtbl->DeviceReset(p)
#define IStiDevice_LockDevice(p,a)              (p)->lpVtbl->LockDevice(p,a)
#define IStiDevice_UnLockDevice(p)              (p)->lpVtbl->LockDevice(p)

#define IStiDevice_Diagnostic(p,a)              (p)->lpVtbl->Diagnostic(p,a)
#define IStiDevice_Escape(p,a,b,c,d,e,f)        (p)->lpVtbl->Escape(p,a,b,c,d,e,f)
#define IStiDevice_GetLastError(p,a)            (p)->lpVtbl->GetLastError(p,a)
#define IStiDevice_RawReadData(p,a,b,c)         (p)->lpVtbl->RawReadData(p,a,b,c)
#define IStiDevice_RawWriteData(p,a,b,c)        (p)->lpVtbl->RawWriteData(p,a,b,c)
#define IStiDevice_RawReadCommand(p,a,b,c)      (p)->lpVtbl->RawReadCommand(p,a,b,c)
#define IStiDevice_RawWriteCommand(p,a,b,c)     (p)->lpVtbl->RawWriteCommand(p,a,b,c)

#define IStiDevice_Subscribe(p,a)               (p)->lpVtbl->Subscribe(p,a)
#define IStiDevice_GetNotificationData(p,a)     (p)->lpVtbl->GetNotificationData(p,a)
#define IStiDevice_UnSubscribe(p)               (p)->lpVtbl->UnSubscribe(p)

#endif




#ifdef __cplusplus
};
#endif

// Reset packing
#include <poppack.h>

#endif // _STICOM_



