PRO Bar3D, Z, Ax=Ax, Az=Az, Title=Title, Xtitle=Xtitle, $
    Ytitle=Ytitle, Ztitle=Ztitle, $
    RowColors=RowColors, ColumnColors=ColumnColors, Background=Background
;+
; NAME: BAR3D
; Author: Ed Kase, PVI Sales
; Modified:  Peggy E. Reed, PVI Sales
; Modified:  Steve Lang, 12/96 - Added ColumnColors & Background keywords.
;                                Also rewrote to display shaded bars & work w/ Z buffer

; This procedure will create a 3D bar plot.  The Z parameter is a 2D array of
; elevation values.
;
;
; Parameters:
; Z is the array of Z values.  By default Z is displayed up the display.
;
; Keywords:
; Ax is the rotation about the X axis, 30 degrees by default.
; Az is the rotation about the Z axis, 30 degrees by default.
; Title output using XYOUTS, if defined.
; XTitle, YTitle, and ZTitle are passed to surface, if defined.
;
; Set values if the keyword is not defined
;
; Example:
;       xx = dist(5)
;       tek_color
;       bar_3d, xx, title= 'Buffalo Securities Demo', xtitle = 'X Axis', $
;           ytitle = 'Y Axis', ztitle = 'Z Axis'
;
;-

Size_Z = Size(Z)
nrows = Size_Z(1)
ncols = Size_Z(2)

If (N_ELEMENTS(Ax) EQ 0) then Ax = 30
If (N_ELEMENTS(Az) EQ 0) then Az = 30
If (not keyword_set(Title)) then Title = ' '
If (not keyword_set(Xtitle)) then Xtitle = ' '
If (not keyword_set(Ytitle)) then Ytitle = ' '
If (not keyword_set(Ztitle)) then Ztitle = ' '
If ((not keyword_set(RowColors)) and (not keyword_set(ColumnColors))) then $
   RowColors = INDGEN(nrows)+2
If (not keyword_set(Background)) then Background = !P.Background

If N_ELEMENTS(RowColors) ne 0 then rowflag=1 else rowflag=0

;
; Create a temporary array to define the grid space.
;
Del = 0.35
Max_Z = Max(Z, Min=Min_Z)
Temp_Array = Intarr(Size_Z(1)+1, Size_Z(2)+1) + Max_Z
Temp_Array(1, 1) = Min_Z
;
; Set the major tick lines to grid mode.
;
!X.Ticklen = 1
!X.Ticks = Size_Z(1)
!X.Minor = -1
!X.Style = 1
!Y.Ticklen = 1
!Y.Ticks = Size_Z(2)
!Y.Minor = -1
!Y.Style = 1
!Z.Range = [Min_Z, Max_Z]
!Z.Style = 1
;
; Project the grid into 3-space.
;
Surface, Temp_Array, /Nodata, /Save, Ax=Ax, Az=Az, $
   Xtitle=Xtitle, Ytitle=Ytitle, Ztitle=Ztitle, Background=Background 
;
;  Put up title 
;
Xyouts, .5, .95, Title, align = .5, /normal

; Workaround because Z buffer doesn't handle hidden lines correctly in Wave 6.04
axis_img = TVRD(0,0,!D.X_VSize,!D.Y_VSize)
erase, 0
tv, axis_img

; Set up color palette based on first 32 colors to create 
; darker shades for top and sides
SET_PLOT,'PS'

TVLCT, r, g, b, /Get

rf = r(0:31)
gf = g(0:31)
bf = b(0:31)

darker = 80
rl = (rf - darker) > 0
gl = (gf - darker) > 0
bl = (bf - darker) > 0
TVLCT, rl, gl, bl, 32

darker = 40
rp = (rf - darker) > 0
gp = (gf - darker) > 0
bp = (bf - darker) > 0
TVLCT, rp, gp, bp, 64

SET_PLOT,'Z'

;
; Draw the 3D bars from back to front.
;
Icount = Size_Z(1) - 1
Jcount = Size_Z(2) - 1
For I = Icount, 0, -1 Do Begin
   For J = Jcount, 0, -1 Do Begin

      If (rowflag) then col = RowColors(I) else col = ColumnColors(J)

      ; Front
      Xdraw = [I+Del,   I+2*Del,  I+2*Del,   I+Del,   I+Del]
      Ydraw = [J+Del,   J+Del,    J+Del,     J+Del,   J+Del]
      Zdraw = [Min_Z,   Min_Z,    Z(I,J),    Z(I,J),  Min_Z]
      Polyfill, Xdraw, Ydraw, Zdraw, /T3D, Color=col

      ; Left
      Xdraw = [I+Del,   I+Del,    I+Del,     I+Del,   I+Del]
      Ydraw = [J+2*Del, J+Del,    J+Del,     J+2*Del, J+2*Del]
      Zdraw = [Min_Z,   Min_Z,    Z(I,J),    Z(I,J),  Min_Z]
      Polyfill, Xdraw, Ydraw, Zdraw, /T3D, Color=col+32

      ; Top
      Xdraw = [I+Del,   I+2*Del,  I+2*Del,   I+Del,   I+Del]
      Ydraw = [J+Del,   J+Del,    J+2*Del,   J+2*Del, J+Del]
      Zdraw = [Z(I,J),  Z(I,J),   Z(I,J),    Z(I,J),  Z(I,J)]
      Polyfill, Xdraw, Ydraw, Zdraw, /T3D, Color=col+64

   Endfor
Endfor

;
Return
End

